/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#ifdef __KERNEL__
#include <linux/kernel.h>  // for uint32_t
#else
#include <stdlib.h>  // for malloc
#include <stdint.h>
#include <stdbool.h>
#include <stdio.h>
#include <fcntl.h>   // for open
#endif

#include "PIC_regheaders.h"  // for the regmasks
#include "pic_full_subblock_list.h"  // for detailed pic_handle
#include "pic_constants.h"
#include "pic_handle.h"      // for pic_fixup_shadow_pointers
#include "pic_if.h"          // for pic irq structures
#include "pic_driverlib_if.h"
//*****************************************************
// interrupt support functions
//*****************************************************

//----------------------------------------------
// set all interrupts in the passed in irqstruct
//----------------------------------------------
void pic_common_set_all_irqstruct(struct pic_common_ints *irqstruct, bool setval)
{
    irqstruct->instance = 0; // the caller can overwrite this - we just want to init
    irqstruct->skew = setval;
    irqstruct->skew_odma = setval;
    irqstruct->af_wdma_common = setval;
    irqstruct->af_dma2 = setval;
    irqstruct->af_dma1 = setval;
    irqstruct->af_dma0 = setval;
    irqstruct->wdma_common = setval;
    irqstruct->wdma2 = setval;
    irqstruct->wdma1 = setval;
    irqstruct->wdma0 = setval;
    irqstruct->ps_esd = setval;
    irqstruct->idma_2d = setval;
}

void pic_output_set_all_irqstruct(struct pic_output_dma_interrupt_info *irqstruct,
                                  bool setval)
{
    irqstruct->bad_rresp = setval;
    irqstruct->bad_bresp = setval;
    irqstruct->soft_rst_cmpl = setval;
    irqstruct->eoi = setval;
    irqstruct->dir_err = setval;
    irqstruct->chg_line_align_err = setval;
    irqstruct->eol_align_err = setval;
    irqstruct->eoi_align_err = setval;
    irqstruct->eoi_err = setval;
    irqstruct->xfer_end = setval;
    irqstruct->own = setval;
}

void pic_idma2d_set_all_irqstruct(struct pic_idma2d_interrupt_info *irqstruct, bool setval)
{
    irqstruct->eol = setval;
    irqstruct->rst = setval;
    irqstruct->eoi = setval;
    irqstruct->fin = setval;
    irqstruct->who = setval;
}

//------------------------------------------------------------------------------
// Convert irqstructs to register bits, and update from the passed in reg value.
//
// pass in the current value for the interrupt register, and only the set fields
// in the irqstruct will update it.
// These functions are structured for the case where the interrupt registers all have
// the same bit mappings between enable, pending, and ack, so passing in the type
// of masks to use for conversion is unnecessary. (e.g. parameter operation is not used
// as it is in icetest_common.c)
//------------------------------------------------------------------------------

uint32_t pic_common_convert_irqstruct_to_update_reg(struct pic_common_ints *irqstruct,
                                                    uint32_t ints,
                                                    bool set_to_one)
{
    uint8_t set_val;

    set_val = set_to_one?1:0;
    
    if (irqstruct->skew)
        ints =  PIC_IPR_SKEW_REPLACE_VAL(ints, set_val);
    if (irqstruct->skew_odma)
        ints =  PIC_IPR_SKEW_ODMA_REPLACE_VAL(ints, set_val);
    if (irqstruct->af_wdma_common)
        ints =  PIC_IPR_AF_WDMA_COMMON_REPLACE_VAL(ints, set_val);
    if (irqstruct->af_dma2)
        ints =  PIC_IPR_AF_DMA2_REPLACE_VAL(ints, set_val);
    if (irqstruct->af_dma1)
        ints =  PIC_IPR_AF_DMA1_REPLACE_VAL(ints, set_val);
    if (irqstruct->af_dma0)
        ints =  PIC_IPR_AF_DMA0_REPLACE_VAL(ints, set_val);
    if (irqstruct->wdma_common)
        ints =  PIC_IPR_WDMA_COMMON_REPLACE_VAL(ints, set_val);
    if (irqstruct->wdma2)
        ints =  PIC_IPR_WDMA2_REPLACE_VAL(ints, set_val);
    if (irqstruct->wdma1)
        ints =  PIC_IPR_WDMA1_REPLACE_VAL(ints, set_val);
    if (irqstruct->wdma0)
        ints =  PIC_IPR_WDMA0_REPLACE_VAL(ints, set_val);
    if (irqstruct->ps_esd)
        ints =  PIC_IPR_PS_ESD_REPLACE_VAL(ints, set_val);
    if (irqstruct->idma_2d)
        ints =  PIC_IPR_IDMA_2D_REPLACE_VAL(ints, set_val);

    return ints;
}


uint32_t pic_output_convert_irqstruct_to_update_reg(struct pic_output_dma_interrupt_info *irqstruct,
                                                    uint32_t ints,
                                                    bool set_to_one)
{
    uint8_t set_val;

    set_val = set_to_one?1:0;
    if (irqstruct->bad_rresp)
        ints = PIC_WDMA_DMA0_INT_PEND_RRESP_INT_PEND_REPLACE_VAL(ints, set_val);
    if (irqstruct->bad_bresp)
        ints = PIC_WDMA_DMA0_INT_PEND_BRESP_INT_PEND_REPLACE_VAL(ints, set_val);
    if (irqstruct->soft_rst_cmpl)
        ints = PIC_WDMA_DMA0_INT_PEND_RST_INT_PEND_REPLACE_VAL(ints, set_val);
    if (irqstruct->eoi)
        ints = PIC_WDMA_DMA0_INT_PEND_EOI_INT_PEND_REPLACE_VAL(ints, set_val);
    if (irqstruct->dir_err)
        ints = PIC_WDMA_DMA0_INT_PEND_DIR_INT_PEND_REPLACE_VAL(ints, set_val);
    if (irqstruct->chg_line_align_err)
        ints = PIC_WDMA_DMA0_INT_PEND_CL_ALI_INT_PEND_REPLACE_VAL(ints, set_val);
    if (irqstruct->eol_align_err)
        ints = PIC_WDMA_DMA0_INT_PEND_EOL_ALI_INT_PEND_REPLACE_VAL(ints, set_val);
    if (irqstruct->eoi_align_err)
        ints = PIC_WDMA_DMA0_INT_PEND_EOI_ALI_INT_PEND_REPLACE_VAL(ints, set_val);
    if (irqstruct->eoi_err)
        ints = PIC_WDMA_DMA0_INT_PEND_EOI_ERR_INT_PEND_REPLACE_VAL(ints, set_val);
    if (irqstruct->xfer_end)
        ints = PIC_WDMA_DMA0_INT_PEND_FIN_INT_PEND_REPLACE_VAL(ints, set_val);
    if (irqstruct->own)
        ints = PIC_WDMA_DMA0_INT_PEND_WHO_INT_PEND_REPLACE_VAL(ints, set_val);
    return ints;
}

uint32_t pic_idma2d_convert_irqstruct_to_update_reg(struct pic_idma2d_interrupt_info *irqstruct,
                                                    uint32_t curr_irqs,
                                                    bool enable)
{
    uint8_t set_val;

    // NOTE: lots of functions use this routine (enable,disable,force,clear), which
    //   is OK as long as all of the IRQ registers use the same IRQ bit positions
    //   as the status register.  Otherwise will need to create operation parameter
    //   as is used in icetest_common.c

    // Are we enabling or disabling IRQs?
    set_val = enable ? 1 : 0; 

    // Any IRQ's in irqstruct that are set to true will get set to set_val, other
    // bits will be left as-is.

    if (irqstruct->eol)
    {
        curr_irqs = IDMA_2D_INT_ST_EOL_INT_ST_REPLACE_VAL(curr_irqs, set_val);
    }

    if (irqstruct->rst)
    {
        curr_irqs = IDMA_2D_INT_ST_RST_INT_ST_REPLACE_VAL(curr_irqs, set_val);
    }

    if (irqstruct->eoi)
    {
        curr_irqs = IDMA_2D_INT_ST_EOI_INT_ST_REPLACE_VAL(curr_irqs, set_val);
    }

    if (irqstruct->fin)
    {
        curr_irqs = IDMA_2D_INT_ST_FIN_INT_ST_REPLACE_VAL(curr_irqs, set_val);
    }

    if (irqstruct->who)
    {
        curr_irqs = IDMA_2D_INT_ST_WHO_INT_ST_REPLACE_VAL(curr_irqs, set_val);
    }

    return curr_irqs;
}


void pic_common_convert_uintarray_to_irqstruct(struct pic_common_ints *irqstruct,
                                               uint32_t ints)
                                               
{
    // if the given bit is set, the interrupt field gets a true, false otherwise
    irqstruct->skew = (PIC_IPR_SKEW_MASK_SHIFT(ints) != 0);
    irqstruct->skew_odma = (PIC_IPR_SKEW_ODMA_MASK_SHIFT(ints) != 0);
    irqstruct->af_wdma_common = (PIC_IPR_AF_WDMA_COMMON_MASK_SHIFT(ints) != 0);
    irqstruct->af_dma2 = (PIC_IPR_AF_DMA2_MASK_SHIFT(ints) != 0);
    irqstruct->af_dma1 = (PIC_IPR_AF_DMA1_MASK_SHIFT(ints) != 0);
    irqstruct->af_dma0 = (PIC_IPR_AF_DMA0_MASK_SHIFT(ints) != 0);
    irqstruct->wdma_common = (PIC_IPR_WDMA_COMMON_MASK_SHIFT(ints) != 0);
    irqstruct->wdma2 = (PIC_IPR_WDMA2_MASK_SHIFT(ints) != 0);
    irqstruct->wdma1 = (PIC_IPR_WDMA1_MASK_SHIFT(ints) != 0);
    irqstruct->wdma0 = (PIC_IPR_WDMA0_MASK_SHIFT(ints) != 0);
    irqstruct->ps_esd = (PIC_IPR_PS_ESD_MASK_SHIFT(ints) != 0);
    irqstruct->idma_2d = (PIC_IPR_IDMA_2D_MASK_SHIFT(ints) != 0);
    irqstruct->int_array = ints;  // for debug only
}

void pic_output_convert_uintarray_to_irqstruct(struct pic_output_dma_interrupt_info *irqstruct,
                                               uint32_t ints)
                                               
{
    // if the given bit is set, the interrupt field gets a true, false otherwise
    irqstruct->bad_rresp = (PIC_WDMA_DMA0_INT_PEND_RRESP_INT_PEND_MASK_SHIFT(ints) != 0);
    irqstruct->bad_bresp = (PIC_WDMA_DMA0_INT_PEND_BRESP_INT_PEND_MASK_SHIFT(ints) != 0);
    irqstruct->soft_rst_cmpl = (PIC_WDMA_DMA0_INT_PEND_RST_INT_PEND_MASK_SHIFT(ints) != 0);
    irqstruct->eoi = (PIC_WDMA_DMA0_INT_PEND_EOI_INT_PEND_MASK_SHIFT(ints) != 0);
    irqstruct->dir_err = (PIC_WDMA_DMA0_INT_PEND_DIR_INT_PEND_MASK_SHIFT(ints) != 0);
    irqstruct->chg_line_align_err = (PIC_WDMA_DMA0_INT_PEND_CL_ALI_INT_PEND_MASK_SHIFT(ints) != 0);
    irqstruct->eol_align_err = (PIC_WDMA_DMA0_INT_PEND_EOL_ALI_INT_PEND_MASK_SHIFT(ints) != 0);
    irqstruct->eoi_align_err = (PIC_WDMA_DMA0_INT_PEND_EOI_ALI_INT_PEND_MASK_SHIFT(ints) != 0);
    irqstruct->eoi_err = (PIC_WDMA_DMA0_INT_PEND_EOI_ERR_INT_PEND_MASK_SHIFT(ints) != 0);
    irqstruct->xfer_end = (PIC_WDMA_DMA0_INT_PEND_FIN_INT_PEND_MASK_SHIFT(ints) != 0);
    irqstruct->own = (PIC_WDMA_DMA0_INT_PEND_WHO_INT_PEND_MASK_SHIFT(ints) != 0);
}

void pic_idma2d_convert_uintarray_to_irqstruct(struct pic_idma2d_interrupt_info *irqstruct,
                                               uint32_t curr_irqs)

{
    // If the given bit is set the interrupt field gets a true, false otherwise

    irqstruct->eol = (IDMA_2D_INT_ST_EOL_INT_ST_MASK_SHIFT(curr_irqs) != 0);
    irqstruct->rst = (IDMA_2D_INT_ST_RST_INT_ST_MASK_SHIFT(curr_irqs) != 0);
    irqstruct->eoi = (IDMA_2D_INT_ST_EOI_INT_ST_MASK_SHIFT(curr_irqs) != 0);
    irqstruct->fin = (IDMA_2D_INT_ST_FIN_INT_ST_MASK_SHIFT(curr_irqs) != 0);
    irqstruct->who = (IDMA_2D_INT_ST_WHO_INT_ST_MASK_SHIFT(curr_irqs) != 0);
}

#define COMPUTE_SHADOW_POINTER(pointer_name, subblock_index)            \
    shadow_size = pic_handle->subblock_size_table[subblock_index];      \
    if (shadow_size == 0)                                               \
    {                                                                   \
        /* this block was never registered - mark it NULL */            \
        pic_handle->pointer_name = NULL;                                \
    }                                                                   \
    else                                                                \
    {                                                                   \
        pic_handle->pointer_name = (void *) shadow_reg_addr;            \
        /* compute the next subblock of shadow registers */             \
        shadow_reg_addr += shadow_size;                                 \
    }


// create the pointers into the shadow register space from the subblock sizes table
void pic_fixup_shadow_pointers(struct pic_handle_t *pic_handle)
{
    int shadow_size;  // size in bytes of the subblock shadow registers
    // 8 bit pointer for pointer math since shadow_size is in bytes    
    uint8_t *shadow_reg_addr; // current address of shadow registers

    // Set the shadow_regs pointer to point to the next word address
    // Since we allocated the size of the struct + size of the shadow registers,
    // the structure's shadow_regs pointer is the last word in the structure
    // before the actual registers - so point this at them (pointer math ++ uint32)
    pic_handle->shadow_regs = (uint32_t *)  &(pic_handle->shadow_regs);
    pic_handle->shadow_regs++;  // point at the word after my field    
    // the start of the shadow registers - used by COMPUTE_SHADOW_POINTER
    shadow_reg_addr = (uint8_t *) pic_handle->shadow_regs;

    COMPUTE_SHADOW_POINTER(pic_common, common_index);
    COMPUTE_SHADOW_POINTER(pic_adcnorm, adcnorm_index);
    COMPUTE_SHADOW_POINTER(pic_bulbmon, bulbmon_index);
    COMPUTE_SHADOW_POINTER(pic_lrmargin[0], lrmargin0_index);
    COMPUTE_SHADOW_POINTER(pic_idma_2d, idma2d_index);
    COMPUTE_SHADOW_POINTER(pic_pd, pd_index);
    COMPUTE_SHADOW_POINTER(pic_chipgap, chipgap_index);
    COMPUTE_SHADOW_POINTER(pic_ps_esd, ps_esd_index);
    COMPUTE_SHADOW_POINTER(pic_hfir, hfir_index);
    COMPUTE_SHADOW_POINTER(pic_hscale, hscale_index);
    COMPUTE_SHADOW_POINTER(pic_lrmargin[1], lrmargin1_index);    
    COMPUTE_SHADOW_POINTER(pic_wdma, wdma_top_index);    
    COMPUTE_SHADOW_POINTER(pic_wdma_channel[0], wdma_dma0_index);
    COMPUTE_SHADOW_POINTER(pic_wdma_channel[1], wdma_dma1_index);
    COMPUTE_SHADOW_POINTER(pic_wdma_channel[2], wdma_dma2_index);
    COMPUTE_SHADOW_POINTER(pic_skewdet, skew_index);
    COMPUTE_SHADOW_POINTER(pic_skew_odma, skew_odma_index);    
    COMPUTE_SHADOW_POINTER(pic_antiflin, flare_index);
    COMPUTE_SHADOW_POINTER(pic_antiflin_wdma0, flare_dma0_index);
    COMPUTE_SHADOW_POINTER(pic_antiflin_wdma1, flare_dma1_index);
    COMPUTE_SHADOW_POINTER(pic_antiflin_wdma2, flare_dma2_index);    
    COMPUTE_SHADOW_POINTER(pic_bitreduct, bdr_index);
}

// function to get the subblock offsets array from the driver.  If this is called from
// kernel space, it just calls the kernel function directly to get the data.  If this is
// called from user space, it will use the sysfs to get the data
uint32_t *pic_get_subblock_sizes(void)
{
    uint32_t *subblock_array;
    
   // Memory will be allocated for the subblock array.  It needs to be
    // deallocated when pic_handle is created (see pic_create_new_default_handle)

#ifdef __KERNEL__
    // since we are in the kernel, we can just call the kernel function
    pic_get_subblock_sizes_array(&subblock_array);
#else

    // note that the subblock sizes array is the same for pic0 and pic1
    // since they should be identical, this should not be a problem
#define PATH_TO_PICHANDLE "/sys/devices/f95f0000.pic0/subblock_sizes_array"
#define PATH_TO_PICHANDLE64 "/sys/devices/f95f000000010000.pic0/subblock_sizes_array"
#define BUFSIZE_MAX 255    
    
    // read the data from the sysfs file
    char path[(sizeof PATH_TO_PICHANDLE) + 1];
    int fd,  i;
    int readret;

    subblock_array = (uint32_t *) allocate_memory(BUFSIZE_MAX, NULL);

    fd = open(PATH_TO_PICHANDLE, O_RDONLY);
    if (-1 == fd)
    {
        // try to open the 64-bit version
        fd = open(PATH_TO_PICHANDLE64, O_RDONLY);
        if (-1 == fd)
        {
            error_print("Failed to open pic_handle (to get subblock) for reading!\n");
            return(NULL);
        }
    }
    
    readret = read(fd, subblock_array, BUFSIZE_MAX);
    
    if (-1 == readret)    
    {
	error_print("Failed to read value!\n");
	return(NULL);
    }

    // now read to verify we get a 0 back.  If we don't we have a problem, since
    // we used up our buffer already....
    if (0 != read(fd, subblock_array, BUFSIZE_MAX))
    {
        error_print("%s: warning: sysfs return bigger than expected: first read=%d, 2nd is non_zero\n",
                    __func__, readret);
    }
    close(fd);

#endif
    return subblock_array;
}


