/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

/**
 * \file pic_convenience_if.h
 *
 * \brief Convenience functions to manipulate PIC handle values
 *
 * After creating a pic_handle through \link picPublicApi PIC Public API \endlink, you may
 * use the convenience functions defined in this file to change the values in pic_handle.
 *
 */

#ifndef _PIC_CONVENIENCE_IF_H_
#define _PIC_CONVENIENCE_IF_H_

#ifndef __KERNEL__
#include <stdint.h>
#include <stdbool.h>
#endif



/**
 * \defgroup picCallbacks PIC Callback Registration
 * @{
 */

/**
 * \brief To register a callback on PIC common interrupts
 * \param pic_handle the pic handle to set the callback
 * \param callbackfcn the callback function to trigger
 * \param callback_data the callback data to pass in the function 
 */
void pic_common_register_callback(struct pic_handle_t *pic_handle, void *callbackfcn,
                                  void *callback_data);

/**
 * \brief To register a callback on PIC idma2d interrupts
 * \param pic_handle the pic handle to set the callback
 * \param callbackfcn the callback function to trigger
 * \param callback_data the callback data to pass in the function
 */
void pic_idma2d_register_callback(struct pic_handle_t *pic_handle, void *callbackfcn,
                                  void *callback_data);

/**
 * \brief To register a callback on a output dma channel interrupt
 * \param pic_handle the pic handle to set the callback
 * \param callbackfcn the callback function to trigger when the output dma channel receives an interrupt signal
 * \param callback_data the callback data to pass in the function 
 * \param channel the dma channel
 */
void pic_odma_register_callback_dma_channel(struct pic_handle_t *pic_handle,
                                            void *callbackfcn,
                                            void *callback_data,
                                            uint8_t channel);
/** @} */

/**
 * \defgroup picConvenience PIC Convenience Functions
 * @{
 */
void pic_enable_common_irqs(struct pic_handle_t *pic_handle,
                            struct pic_common_ints *irqstruct,
                            bool enable);

void pic_force_common_irqs(struct pic_handle_t *pic_handle, struct pic_common_ints *irqstruct, bool force);

void pic_adc_set_norm(struct pic_handle_t *pic_handle, bool bypass, bool invert,
                      bool left_justify, uint8_t pad_fill, uint8_t numadcbits);

void pic_adc_get_norm(struct pic_handle_t *pic_handle, bool *bypass, bool *invert,
                      bool *left_justify, uint8_t *pad_fill, uint8_t *numadcbits);

void pic_adc_set_invert(struct pic_handle_t *pic_handle, bool invert);

void pic_hs_set_bypass(struct pic_handle_t *pic_handle, bool bypass);

void pic_hs_set_hscale_factor(struct pic_handle_t *pic_handle, uint32_t sfactor);

void pic_hs_get_hscale_factor(struct pic_handle_t *pic_handle, uint32_t *sfactor);

void pic_pd_copy_settings(struct pic_handle_t *pic_handle_dst, struct pic_handle_t *pic_handle_src);

void pic_pd_set_gain(struct pic_handle_t *pic_handle, uint8_t *leftgainintc, uint8_t *leftgaindecc,
		uint8_t *rightgainintc, uint8_t *rightgaindecc);

void pic_pd_get_gain(struct pic_handle_t *pic_handle, uint8_t *leftgainintc, uint8_t *leftgaindecc,
		uint8_t *rightgainintc, uint8_t *rightgaindecc);

void pic_pd_set_left_slope(struct pic_handle_t *pic_handle, uint8_t *slope);

void pic_pd_get_left_slope(struct pic_handle_t *pic_handle, uint8_t *slope);

void pic_pd_set_left_max_c0(struct pic_handle_t *pic_handle, uint16_t leftcorr);

void pic_pd_get_left_max_c0(struct pic_handle_t *pic_handle, uint16_t *leftcorr);

void pic_pd_set_left_max_c1(struct pic_handle_t *pic_handle, uint16_t leftcorr);

void pic_pd_set_left_max_c2(struct pic_handle_t *pic_handle, uint16_t leftcorr);

void pic_pd_set_right_slope(struct pic_handle_t *pic_handle, uint8_t *slope);

void pic_pd_get_right_slope(struct pic_handle_t *pic_handle, uint8_t *slope);

void pic_pd_set_right_start_c0(struct pic_handle_t *pic_handle, uint16_t rightstart);

void pic_pd_get_right_start_c0(struct pic_handle_t *pic_handle, uint16_t *rightstart);

void pic_pd_set_right_start_c1(struct pic_handle_t *pic_handle, uint16_t rightstart);

void pic_pd_set_right_start_c2(struct pic_handle_t *pic_handle, uint16_t rightstart);

void pic_pd_get_cfg1(struct pic_handle_t *pic_handle, uint8_t *corrbits, uint8_t *allocc);

void pic_pd_get_dsnu_offset_c0(struct pic_handle_t *pic_handle, uint16_t *offset);

void pic_pd_get_dsnu_mult_scale_c0(struct pic_handle_t *pic_handle, uint8_t  *scale, uint16_t *mult);

void pic_pd_get_prnu_offset_c0(struct pic_handle_t *pic_handle, uint16_t *offset);

void pic_pd_get_prnu_mult_scale_c0(struct pic_handle_t *pic_handle, uint8_t *scale, uint16_t *mult);

void pic_pd_set_config1(struct pic_handle_t *pic_handle, uint8_t corrbits, int8_t allocc2,
                        int8_t allocc1, int8_t allocc0);

void pic_pd_set_coeffwidth(struct pic_handle_t *pic_handle, uint8_t coeffwidth);

int pic_pd_get_coeffwidth(struct pic_handle_t *pic_handle);

void pic_pd_set_bypass_all(struct pic_handle_t *pic_handle, bool bypass_val);

bool pic_pd_get_bypass_all(struct pic_handle_t *pic_handle);

void pic_pd_set_bypass_prnu(struct pic_handle_t *pic_handle, bool bypass_val);

bool pic_pd_get_bypass_prnu(struct pic_handle_t *pic_handle);

void pic_pd_set_bypass_dsnu(struct pic_handle_t *pic_handle, bool bypass_val);

bool pic_pd_get_bypass_dsnu(struct pic_handle_t *pic_handle);

void pic_pd_set_bypass_bad_pix_replace(struct pic_handle_t *pic_handle, bool bypass_val);

bool pic_pd_get_bypass_bad_pix_replace(struct pic_handle_t *pic_handle);

void pic_pd_set_bypass_exp_comp(struct pic_handle_t *pic_handle, bool bypass_val);

bool pic_pd_get_bypass_exp_comp(struct pic_handle_t *pic_handle);

void pic_pd_set_bypass_quadratic(struct pic_handle_t *pic_handle, bool bypass_val);

bool pic_pd_get_bypass_quadratic(struct pic_handle_t *pic_handle);

void pic_pd_set_dsnu_offset(struct pic_handle_t *pic_handle, uint16_t offset0, uint16_t offset1, uint16_t offset2);

void pic_pd_set_dsnu_mult_scale(struct pic_handle_t *pic_handle, uint16_t mult0, uint8_t scale0, uint16_t mult1, uint8_t scale1, uint16_t mult2, uint8_t scale2);

void pic_pd_set_prnu_offset(struct pic_handle_t *pic_handle, uint16_t offset0, uint16_t offset1, uint16_t offset2);

void pic_pd_set_prnu_mult_scale(struct pic_handle_t *pic_handle, uint16_t mult0, uint8_t scale0, uint16_t mult1, uint8_t scale1, uint16_t mult2, uint8_t scale2);

void pic_pd_set_quad_cfg(struct pic_handle_t *pic_handle, uint8_t sf1, uint8_t sf2, uint8_t q_shift);

void pic_pd_idma_enable_lut_dma(struct pic_handle_t *pic_handle, bool enable);

void pic_pd_idma2d_set_line_width(struct pic_handle_t *pic_handle, uint8_t repeat_forever,
		uint8_t repeat_line_cnt, uint32_t line_width);

void pic_pd_idma2d_get_line_width(struct pic_handle_t *pic_handle, uint8_t *repeat_forever,
		uint8_t *repeat_line_cnt, uint32_t *line_width);

void pic_enable_idma2d_irqs(struct pic_handle_t *pic_handle,
                            struct pic_idma2d_interrupt_info *irqstruct,
                            bool enable);
void pic_clear_idma2d_irqs(struct pic_handle_t *pic_handle,
                           struct pic_idma2d_interrupt_info *irqstruct);

void pic_lrmargin_set_cfg(struct pic_handle_t *pic_handle, uint8_t lrmargin_instance, bool bypass);

void pic_lrmargin_get_cfg(struct pic_handle_t *pic_handle, uint8_t lrmargin_instance, bool *bypass);

void pic_lrmargin_set_margin(struct pic_handle_t *pic_handle, uint8_t lrmargin_instance,
		uint16_t rightmargin, uint16_t leftmargin, uint8_t color);

void pic_lrmargin_get_margin(struct pic_handle_t *pic_handle, uint8_t lrmargin_instance,
		uint16_t *rightmargin, uint16_t *leftmargin, uint8_t color);

/////////////////// public API functions for pic BDR ///////////////////

void pic_bdr_set_bypass(struct pic_handle_t *pic_handle, bool bypass);

// SANDRA FIXME, where should define for PIC_BDR_LUT_SIZE go??  not pichw.h
void pic_bdr_set_lut(struct pic_handle_t *pic_handle, uint32_t *lut);

void pic_bdr_set_num_luts(struct pic_handle_t *pic_handle, int num_luts);

void pic_bdr_select_lut(struct pic_handle_t *pic_handle, int lutsel);

void pic_chipgap_set_CFG(struct pic_handle_t *pic_handle, uint8_t bypass);

void pic_chipgap_set_LOCAT(struct pic_handle_t *pic_handle, uint16_t width, uint16_t start);

// gap must be an array of 4
void pic_chipgap_set_INSERT0(struct pic_handle_t *pic_handle, uint8_t *gap);

void pic_chipgap_set_INSERT1(struct pic_handle_t *pic_handle, uint8_t *gap);

void pic_chipgap_set_INSERT2(struct pic_handle_t *pic_handle, uint8_t *gap);

void pic_chipgap_set_INSERT3(struct pic_handle_t *pic_handle, uint8_t *gap);

void pic_output_dma_set_disable_mode(struct pic_handle_t *pic_handle, uint8_t mode);

void pic_output_dma_get_disable_mode(struct pic_handle_t *pic_handle, uint8_t *mode);

void pic_output_dma_set_common_int(struct pic_handle_t *pic_handle, uint8_t channels, bool add);

void pic_enable_output_dma_channel_irqs(struct pic_handle_t *pic_handle,
                                        struct pic_output_dma_interrupt_info *irqstruct,
                                        bool enable,
                                        uint8_t channel);

void pic_output_dma_channel_get_status(struct pic_handle_t *pic_handle,
				struct pic_wdma_status_info *info, uint8_t channel);

void pic_output_dma_channel_get_trans_addr(struct pic_handle_t *pic_handle, uint32_t *trans_addr,
                                           uint8_t channel);
void pic_output_dma_channel_get_trans_len(struct pic_handle_t *pic_handle, uint32_t *trans_len,
                                          uint8_t channel);

void pic_output_dma_set_routing(struct pic_handle_t *pic_handle, int monoDMA, int color2DMA,
                                int color1DMA, int color0DMA);

void pic_output_dma_channel_set_linesize(struct pic_handle_t *pic_handle, int line_size,
                                         uint8_t channel);

void pic_output_dma_channel_get_linesize(struct pic_handle_t *pic_handle, uint32_t *line_size,
                                         uint8_t channel);

void pic_output_dma_channel_set_burst_len(struct pic_handle_t *pic_handle, uint8_t burst_len,
                                          uint8_t channel);

void pic_output_dma_channel_set_upper_half(struct pic_handle_t *pic_handle, uint8_t val,
                                           uint8_t channel);

void pic_output_dma_channel_set_enable_dma(struct pic_handle_t *pic_handle, uint8_t enable,
                                           uint8_t channel);

void pic_output_dma_channel_set_input_data_width(struct pic_handle_t *pic_handle,
                                                 uint8_t in_width, uint8_t channel);

uint8_t pic_output_dma_channel_get_input_data_width(struct pic_handle_t *pic_handle,
                                                    uint8_t channel);

void pic_output_dma_channel_enable_line_reverse(struct pic_handle_t *pic_handle,
                                                uint8_t enable, uint8_t channel);

void pic_output_dma_channel_set_input_data_width(struct pic_handle_t *pic_handle,
                                                 uint8_t in_width, uint8_t channel);
void pic_output_dma_channel_set_handshake_enable(struct pic_handle_t *pic_handle, bool enable,
                                                 uint8_t channel);



void pic_force_output_dma_irqs(struct pic_handle_t *pic_handle,
                               struct pic_output_dma_interrupt_info *irqstruct,
                               bool force,
                               uint8_t channel);

struct pic_handle_t *pic_do_get_default(struct pic_handle_t *pic_handle);
struct pic_handle_t *pic_create_new_default_handle(void);


/** @} */


// revision values - this driver is designed to work with the version specified (and earlier)
#define PIC_COMMON_MAJ_REV0       (3 << PIC_REV0_MAJ_SHIFT)
#define PIC_ADCNORM_MAJ_REV0      (2 << ADCNORM_REV0_MAJ_SHIFT)
#define PIC_BDR_MAJ_REV0          (2 << BITREDUCT_REV0_MAJ_SHIFT)
#define PIC_BULBMON_MAJ_REV0      (0 << BULBMON_REV0_MAJ_SHIFT)
#define PIC_CHIPGAP_MAJ_REV0      (0 << NEWMAN_REV0_MAJ_SHIFT)
#define PIC_HSCALE_MAJ_REV0       (2 << HSCALE_REV0_MAJ_SHIFT)
#define PIC_IDMA2D_MAJ_REV0       (1 << IDMA_2D_REV0_MAJ_SHIFT)
#define PIC_LRMARGIN_MAJ_REV0     (2 << LRMARGIN0_REV0_MAJ_SHIFT)
#define PIC_PD_MAJ_REV0           (2 << PRNUDSNU_REV0_MAJ_SHIFT)
#define PIC_WDMA_MAJ_REV0         (2 << PIC_WDMA_REV0_MAJ_SHIFT)
#define PIC_WDMA_CHANNEL_MAJ_REV0 (4 << PIC_WDMA_DMA0_REV0_MAJ_SHIFT)

#endif /*_PIC_CONVENIENCE_IF_H_ */
