/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

// this file needs to compile in both user and kernel space so that convenience
// functions are available from either a kernel loadable module or an app

#ifdef __KERNEL__
#include <linux/kernel.h>
#include <linux/export.h>     // for EXPORT_SYMBOL
#include <linux/module.h>
#else
#include <stdint.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#endif

#include "PIC_regheaders.h"

#include "pic_constants.h"
#include "pic_full_subblock_list.h"  // needed for detailed pic_handle
#include "pic_handle.h" 
#include "pic_if.h"
#include "pic_driverlib_if.h"
#include "pic_driver.h"
#include "pic_convenience_if.h"

#include "pic_driverlib_if.h"  // driver utilities and print macros

#define PIC_PTRCHECK(subblockname, retval)        \
    BUG_ON(pic_handle == NULL);                   \
    if (pic_handle->subblockname == NULL)         \
    {                                             \
        print ("NULL subblockname\n");            \
        return retval;                            \
    }

#define PIC_PTRCHECK_ARRY_NR(subblockname, index)  \
    BUG_ON(pic_handle == NULL);                    \
    if (pic_handle->subblockname[index] == NULL)   \
    {                                              \
        print ("NULL subblockname %d\n", index);   \
        return;                                    \
    }

#define PIC_PTRCHECK_ARRY(subblockname, index, retval)  \
    BUG_ON(pic_handle == NULL);                         \
    if (pic_handle->subblockname[index] == NULL)        \
    {                                                   \
        print ("NULL subblockname %d\n", index);        \
        return retval;                                  \
    }

#define PIC_PTRCHECK_NR(subblockname)             \
    BUG_ON(pic_handle == NULL);                   \
    if (pic_handle->subblockname == NULL)         \
    {                                             \
        print ("NULL subblockname\n");            \
        return;                                   \
    }

#define PIC_PTRCHECK_NAME_NR(handle_name, subblockname) \
    BUG_ON(handle_name == NULL);                        \
    if (handle_name->subblockname == NULL)              \
    {                                                   \
        print("NULL subblockname\n");                   \
        return;                                         \
    }

///////////////////////////////////////////
// to get the size of the whole pic_handle
///////////////////////////////////////////

int pic_get_handle_size(struct pic_handle_t *pic_handle)
{
    return pic_handle->pic_handle_size;
}
EXPORT_SYMBOL(pic_get_handle_size);

////////////////////////////////////
// pic common
////////////////////////////////////

void pic_common_dump_handle_regs(struct pic_handle_t *pic_handle)
{
    print("PIC common regs (pic_handle)\n");
    PIC_PTRCHECK_NR(pic_common);
    
    print("PCR=0x%X\n", pic_handle->pic_common->PCR);
    print("IENR=0x%X\n", pic_handle->pic_common->IENR);
    print("IPR=0x%X\n", pic_handle->pic_common->IPR);
    print("IFR=0x%X\n", pic_handle->pic_common->IFR);
    print("REV0=0x%X\n", pic_handle->pic_common->REV0);
    print("REV1=0x%X\n", pic_handle->pic_common->REV1);
}
EXPORT_SYMBOL(pic_common_dump_handle_regs);

void pic_set_common_PCR(struct pic_handle_t *pic_handle, uint8_t debug_enable,
                        uint8_t softreset)
{
    PIC_PTRCHECK_NR(pic_common);
    
    pic_handle->pic_common->PCR =
        PIC_PCR_DEBUGENB_REPLACE_VAL(pic_handle->pic_common->PCR, debug_enable);
    // the PIC html file says about the soft_reset_valid bit:
    // "Setting this bit will soft reset (synchronous) all blocks in this PIC PIPE"
    pic_handle->pic_common->PCR =
        PIC_PCR_SOFTRESET_REPLACE_VAL(pic_handle->pic_common->PCR, softreset);
}
EXPORT_SYMBOL(pic_set_common_PCR);

// if force == false, unforce requested irqs
// if force == true, force requested irqs
// if irqstruct == NULL, then force/unforce all irqs
void pic_force_common_irqs(struct pic_handle_t *pic_handle,
                           struct pic_common_ints *irqstruct,
                           bool force)
{
    struct pic_common_ints irq;
    PIC_PTRCHECK_NR(pic_common);

    if (irqstruct == NULL)
    {
        pic_common_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
    // set/clear force irq bits
    pic_handle->pic_common->IFR =
        pic_common_convert_irqstruct_to_update_reg(irqstruct,
                                                   pic_handle->pic_common->IFR,
                                                   force);
}
EXPORT_SYMBOL(pic_force_common_irqs);


// if enable == false, disable requested irqs
// if enable == true, disable requested irqs
// if irqstruct == NULL, then enable/disable all irqs
void pic_enable_common_irqs(struct pic_handle_t *pic_handle,
                            struct pic_common_ints *irqstruct,
                            bool enable)
{
    struct pic_common_ints irq;
    PIC_PTRCHECK_NR(pic_common);

    if (irqstruct == NULL)
    {
        pic_common_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
    // set/clear enable/irq bits
    pic_handle->pic_common->IENR =
        pic_common_convert_irqstruct_to_update_reg(irqstruct,
                                                   pic_handle->pic_common->IENR,
                                                   enable);
}
EXPORT_SYMBOL(pic_enable_common_irqs);

void pic_common_clone_regs(struct pic_handle_t *src_pic_handle,
                               struct pic_handle_t *dest_pic_handle)
{
    PIC_PTRCHECK_NAME_NR(src_pic_handle, pic_common);
    PIC_PTRCHECK_NAME_NR(dest_pic_handle, pic_common);

    dest_pic_handle->pic_common->PCR = src_pic_handle->pic_common->PCR;
    dest_pic_handle->pic_common->IENR = src_pic_handle->pic_common->IENR;
    dest_pic_handle->pic_common->IPR = src_pic_handle->pic_common->IPR;
    dest_pic_handle->pic_common->IFR = src_pic_handle->pic_common->IFR;
    dest_pic_handle->pic_common->REV0 = src_pic_handle->pic_common->REV0;
    dest_pic_handle->pic_common->REV1 = src_pic_handle->pic_common->REV1;
}
EXPORT_SYMBOL(pic_common_clone_regs);


////////////////////////////////////
// pic adcnorm
////////////////////////////////////
void pic_adc_dump_handle_regs(struct pic_handle_t *pic_handle)
{
    print("PIC ADCNorm regs (pic_handle)\n");
    PIC_PTRCHECK_NR(pic_adcnorm);

    print("ANR=0x%X\n", pic_handle->pic_adcnorm->ANR);
    print("REV0=0x%X\n", pic_handle->pic_adcnorm->REV0);
    print("REV1=0x%X\n", pic_handle->pic_adcnorm->REV1);    
}
EXPORT_SYMBOL(pic_adc_dump_handle_regs);

void pic_adc_set_norm(struct pic_handle_t *pic_handle, bool bypass, bool invert,
                      bool left_justify, uint8_t pad_fill, uint8_t numadcbits)
{
    PIC_PTRCHECK_NR(pic_adcnorm);

    pic_handle->pic_adcnorm->ANR =
        ADCNORM_ANR_BYPASS_REPLACE_VAL(pic_handle->pic_adcnorm->ANR, bypass ? 1 : 0);
    pic_handle->pic_adcnorm->ANR =
        ADCNORM_ANR_INVERT_REPLACE_VAL(pic_handle->pic_adcnorm->ANR, invert ? 1 : 0);
    pic_handle->pic_adcnorm->ANR =
        ADCNORM_ANR_JUSTIFY_REPLACE_VAL(pic_handle->pic_adcnorm->ANR, left_justify ? 1 : 0);
    pic_handle->pic_adcnorm->ANR =
        ADCNORM_ANR_PADFILL_REPLACE_VAL(pic_handle->pic_adcnorm->ANR, pad_fill);
    pic_handle->pic_adcnorm->ANR =
        ADCNORM_ANR_ADCBITS_REPLACE_VAL(pic_handle->pic_adcnorm->ANR, numadcbits);
}
EXPORT_SYMBOL(pic_adc_set_norm);

void pic_adc_get_norm(struct pic_handle_t *pic_handle, bool *bypass, bool *invert,
                      bool *left_justify, uint8_t *pad_fill, uint8_t *numadcbits)
{
    PIC_PTRCHECK_NR(pic_adcnorm);

    *bypass = ADCNORM_ANR_BYPASS_MASK_SHIFT(pic_handle->pic_adcnorm->ANR);
    *invert = ADCNORM_ANR_INVERT_MASK_SHIFT(pic_handle->pic_adcnorm->ANR);
    *left_justify = ADCNORM_ANR_JUSTIFY_MASK_SHIFT(pic_handle->pic_adcnorm->ANR);
    *pad_fill = ADCNORM_ANR_PADFILL_MASK_SHIFT(pic_handle->pic_adcnorm->ANR);
    *numadcbits = ADCNORM_ANR_ADCBITS_MASK_SHIFT(pic_handle->pic_adcnorm->ANR);
}
EXPORT_SYMBOL(pic_adc_get_norm);

void pic_adc_set_invert(struct pic_handle_t *pic_handle, bool invert)
{
    PIC_PTRCHECK_NR(pic_adcnorm);

    pic_handle->pic_adcnorm->ANR =
        ADCNORM_ANR_INVERT_REPLACE_VAL(pic_handle->pic_adcnorm->ANR, invert ? 1 : 0);
}
EXPORT_SYMBOL(pic_adc_set_invert);

////////////////////////////////////
// pic bulbmon
////////////////////////////////////

void pic_bulbmon_dump_handle_regs(struct pic_handle_t *pic_handle)
{
    print("PIC Bulbmon regs (pic_handle)\n");
    PIC_PTRCHECK_NR(pic_bulbmon);

    print("BMCR=0x%X\n", pic_handle->pic_bulbmon->BMCR);
    print("BMR0=0x%X\n", pic_handle->pic_bulbmon->BMR0);
    print("BMR1=0x%X\n", pic_handle->pic_bulbmon->BMR1);
    print("BMR2=0x%X\n", pic_handle->pic_bulbmon->BMR2);
    print("REV0=0x%X\n", pic_handle->pic_bulbmon->REV0);
    print("REV1=0x%X\n", pic_handle->pic_bulbmon->REV1);
}
EXPORT_SYMBOL(pic_bulbmon_dump_handle_regs);

////////////////////////////////////
// pic lrmargin
////////////////////////////////////
void pic_lrmargin_dump_handle_regs(struct pic_handle_t *pic_handle, int marginnum)
{
    print("PIC lrmargin regs (pic_handle)\n");
    PIC_PTRCHECK_ARRY_NR(pic_lrmargin, marginnum);

    print("LRCR=0x%X\n", pic_handle->pic_lrmargin[marginnum]->LRCR);
    print("LRMR0=0x%X\n", pic_handle->pic_lrmargin[marginnum]->LRMR0);
    print("LRMR1=0x%X\n", pic_handle->pic_lrmargin[marginnum]->LRMR1);
    print("LRMR2=0x%X\n", pic_handle->pic_lrmargin[marginnum]->LRMR2);
    print("REV0=0x%X\n", pic_handle->pic_lrmargin[marginnum]->REV0);
    print("REV1=0x%X\n", pic_handle->pic_lrmargin[marginnum]->REV1);
}
EXPORT_SYMBOL(pic_lrmargin_dump_handle_regs);

void pic_lrmargin_set_cfg(struct pic_handle_t *pic_handle, uint8_t lrmargin_instance,
                          bool bypass)
{
    PIC_PTRCHECK_NR(pic_lrmargin);

    pic_handle->pic_lrmargin[lrmargin_instance]->LRCR =
        LRMARGIN0_LRCR_BYPASS_REPLACE_VAL(pic_handle->pic_lrmargin[lrmargin_instance]->LRCR,
                                          bypass ? 1 : 0);
}
EXPORT_SYMBOL(pic_lrmargin_set_cfg);

void pic_lrmargin_get_cfg(struct pic_handle_t *pic_handle, uint8_t lrmargin_instance,
                          bool *bypass)
{
    PIC_PTRCHECK_NR(pic_lrmargin);

    *bypass =
        (LRMARGIN0_LRCR_BYPASS_MASK_SHIFT(pic_handle->pic_lrmargin[lrmargin_instance]->LRCR) == 1);
}
EXPORT_SYMBOL(pic_lrmargin_get_cfg);

void pic_lrmargin_set_margin(struct pic_handle_t *pic_handle, uint8_t lrmargin_instance,
		uint16_t rightmargin, uint16_t leftmargin, uint8_t color)
{
    PIC_PTRCHECK_NR(pic_lrmargin);

    switch(color) {
    case 0:
	pic_handle->pic_lrmargin[lrmargin_instance]->LRMR0 =
            LRMARGIN0_LRMR0_RIGHTMARGINC0_REPLACE_VAL(pic_handle->pic_lrmargin[lrmargin_instance]->LRMR0,
                                                      rightmargin);
	pic_handle->pic_lrmargin[lrmargin_instance]->LRMR0 =
            LRMARGIN0_LRMR0_LEFTMARGINC0_REPLACE_VAL(pic_handle->pic_lrmargin[lrmargin_instance]->LRMR0,
                                                     leftmargin);
        break;
    case 1:
	pic_handle->pic_lrmargin[lrmargin_instance]->LRMR1 =
            LRMARGIN0_LRMR1_RIGHTMARGINC1_REPLACE_VAL(pic_handle->pic_lrmargin[lrmargin_instance]->LRMR1,
                                                      rightmargin);
	pic_handle->pic_lrmargin[lrmargin_instance]->LRMR1 =
            LRMARGIN0_LRMR1_LEFTMARGINC1_REPLACE_VAL(pic_handle->pic_lrmargin[lrmargin_instance]->LRMR1,
                                                     leftmargin);
        break;
    case 2:
	pic_handle->pic_lrmargin[lrmargin_instance]->LRMR2 =
            LRMARGIN0_LRMR2_RIGHTMARGINC2_REPLACE_VAL(pic_handle->pic_lrmargin[lrmargin_instance]->LRMR2,
                                                      rightmargin);
	pic_handle->pic_lrmargin[lrmargin_instance]->LRMR2 =
            LRMARGIN0_LRMR2_LEFTMARGINC2_REPLACE_VAL(pic_handle->pic_lrmargin[lrmargin_instance]->LRMR2,
                                                     leftmargin);
        break;
    default:
        error_print("%s: bad color request %d\n", __func__, color);
        BUG_ON(1);
        break;
    }
}
EXPORT_SYMBOL(pic_lrmargin_set_margin);

void pic_lrmargin_get_margin(struct pic_handle_t *pic_handle, uint8_t lrmargin_instance,
		uint16_t *rightmargin, uint16_t *leftmargin, uint8_t color)
{
    uint32_t reg;

    PIC_PTRCHECK_NR(pic_lrmargin);

    switch(color) {
    case 0:
        reg = pic_handle->pic_lrmargin[lrmargin_instance]->LRMR0;
        break;
    case 1:
        reg = pic_handle->pic_lrmargin[lrmargin_instance]->LRMR1;
        break;
    case 2:
        reg = pic_handle->pic_lrmargin[lrmargin_instance]->LRMR2;
        break;
    default:
        error_print("%s: bad color request %d\n", __func__, color);
        BUG_ON(1);
        return;
    }

    *rightmargin = LRMARGIN0_LRMR0_RIGHTMARGINC0_MASK_SHIFT(reg);
    *leftmargin = LRMARGIN0_LRMR0_LEFTMARGINC0_MASK_SHIFT(reg);
}
EXPORT_SYMBOL(pic_lrmargin_get_margin);

////////////////////////////////////
// pic idma2d
////////////////////////////////////
void pic_idma2d_dump_handle_regs(struct pic_handle_t *pic_handle)
{
    print("PIC idma2d regs (pic_handle)\n");
    PIC_PTRCHECK_NR(pic_idma_2d);

    print("cfg=0x%X\n", pic_handle->pic_idma_2d->cfg);
    print("status=0x%X\n", pic_handle->pic_idma_2d->status);
    print("line_width=0x%X\n", pic_handle->pic_idma_2d->line_width);
    print("int_en=0x%X\n", pic_handle->pic_idma_2d->int_en);
    print("int_st=0x%X\n", pic_handle->pic_idma_2d->int_st);
    // int_cl, int_fo, and desc_write are write-only
    print("desc_read=0x%X\n", pic_handle->pic_idma_2d->desc_read);
    print("xfer_length=0x%X\n", pic_handle->pic_idma_2d->xfer_length);
    print("xfer_addr=0x%X\n", pic_handle->pic_idma_2d->xfer_addr);
    print("ctrl_word=0x%X\n", pic_handle->pic_idma_2d->ctrl_word);
    // reset is write-only
    print("REV0=0x%X\n", pic_handle->pic_idma_2d->REV0);
    print("REV1=0x%X\n", pic_handle->pic_idma_2d->REV1);    
}
EXPORT_SYMBOL(pic_idma2d_dump_handle_regs);

void pic_pd_idma2d_set_line_width(struct pic_handle_t *pic_handle, uint8_t repeat_forever,
		uint8_t repeat_line_cnt, uint32_t line_width)
{
    PIC_PTRCHECK_NR(pic_idma_2d);

    pic_handle->pic_idma_2d->line_width =
        IDMA_2D_LINE_WIDTH_REPEAT_FOREVER_REPLACE_VAL(pic_handle->pic_idma_2d->line_width, repeat_forever);
    pic_handle->pic_idma_2d->line_width =
        IDMA_2D_LINE_WIDTH_REPEAT_LINE_CNT_REPLACE_VAL(pic_handle->pic_idma_2d->line_width, repeat_line_cnt);
    pic_handle->pic_idma_2d->line_width =
        IDMA_2D_LINE_WIDTH_LINE_WIDTH_REPLACE_VAL(pic_handle->pic_idma_2d->line_width, line_width);
}
EXPORT_SYMBOL(pic_pd_idma2d_set_line_width);

void pic_pd_idma_enable_lut_dma(struct pic_handle_t *pic_handle, bool enable)
{
    PIC_PTRCHECK_NR(pic_idma_2d);

    pic_handle->pic_idma_2d->cfg =
        IDMA_2D_CFG_ENABLE_REPLACE_VAL(pic_handle->pic_idma_2d->cfg, enable ? 1 : 0);
}
EXPORT_SYMBOL(pic_pd_idma_enable_lut_dma);

void pic_pd_idma2d_get_line_width(struct pic_handle_t *pic_handle, uint8_t *repeat_forever,
		uint8_t *repeat_line_cnt, uint32_t *line_width)
{
    PIC_PTRCHECK_NR(pic_idma_2d);

    *repeat_forever = IDMA_2D_LINE_WIDTH_REPEAT_FOREVER_MASK_SHIFT(pic_handle->pic_idma_2d->line_width);
    *repeat_line_cnt = IDMA_2D_LINE_WIDTH_REPEAT_LINE_CNT_MASK_SHIFT(pic_handle->pic_idma_2d->line_width);
    *line_width = IDMA_2D_LINE_WIDTH_LINE_WIDTH_MASK_SHIFT(pic_handle->pic_idma_2d->line_width);
}
EXPORT_SYMBOL(pic_pd_idma2d_get_line_width);

// if force == false, unforce requested irqs
// if force == true, force requested irqs
// if irqstruct == NULL, then force/unforce all irqs
void pic_force_idma2d_irqs(struct pic_handle_t *pic_handle,
                           struct pic_idma2d_interrupt_info *irqstruct,
                           bool force)
{
    struct pic_idma2d_interrupt_info irq;
    PIC_PTRCHECK_NR(pic_idma_2d);

    if (irqstruct == NULL)
    {
        pic_idma2d_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
    // set/clear force irq bits
    pic_handle->pic_idma_2d->int_fo =
        pic_idma2d_convert_irqstruct_to_update_reg(irqstruct,
                                                   pic_handle->pic_idma_2d->int_fo,
                                                   force);
}
EXPORT_SYMBOL(pic_force_idma2d_irqs);

void pic_enable_idma2d_irqs(struct pic_handle_t *pic_handle,
                            struct pic_idma2d_interrupt_info *irqstruct,
                            bool enable)
{
    struct pic_idma2d_interrupt_info irq;
    PIC_PTRCHECK_NR(pic_idma_2d);

    if (irqstruct == NULL)
    {
        pic_idma2d_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
    // enable/disable requested irqs
    pic_handle->pic_idma_2d->int_en =
        pic_idma2d_convert_irqstruct_to_update_reg(irqstruct,
                                                   pic_handle->pic_idma_2d->int_en,
                                                   enable);
}
EXPORT_SYMBOL(pic_enable_idma2d_irqs);

void pic_clear_idma2d_irqs(struct pic_handle_t *pic_handle,
                           struct pic_idma2d_interrupt_info *irqstruct)
{
    struct pic_idma2d_interrupt_info irq;
    PIC_PTRCHECK_NR(pic_idma_2d);

    if (irqstruct == NULL)
    {
        pic_idma2d_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
    // clear requested irqs
    pic_handle->pic_idma_2d->int_cl =
        pic_idma2d_convert_irqstruct_to_update_reg(irqstruct,
                                                   pic_handle->pic_idma_2d->int_en,
                                                   true);  // set the int_cl bits to clear irq
}

EXPORT_SYMBOL(pic_clear_idma2d_irqs);


////////////////////////////////////
// pic PRNU/DSNU (pd)
////////////////////////////////////

void pic_pd_dump_handle_regs(struct pic_handle_t *pic_handle)
{
    print("PIC PRNU/DSNU regs (pic_handle)\n");
    PIC_PTRCHECK_NR(pic_pd);

    print("RLSCR=0x%X\n", pic_handle->pic_pd->RLSCR);
    print("LSVCR=0x%X\n", pic_handle->pic_pd->LSVCR);
    print("LMPCR0=0x%X\n", pic_handle->pic_pd->LMPCR0);
    print("LMPCR1=0x%X\n", pic_handle->pic_pd->LMPCR1);
    print("LMPCR2=0x%X\n", pic_handle->pic_pd->LMPCR2);
    print("RSVCR=0x%X\n", pic_handle->pic_pd->RSVCR);
    print("RSPCR0=0x%X\n", pic_handle->pic_pd->RSPCR0);
    print("RSPCR1=0x%X\n", pic_handle->pic_pd->RSPCR1);
    print("RSPCR2=0x%X\n", pic_handle->pic_pd->RSPCR2);
    print("PDCR1=0x%X\n", pic_handle->pic_pd->PDCR1);
    print("PDCR2=0x%X\n", pic_handle->pic_pd->PDCR2);
    print("PONCR=0x%X\n", pic_handle->pic_pd->PONCR);
    print("QUAD=0x%X\n", pic_handle->pic_pd->QUAD);
    print("DOFF0=0x%X\n", pic_handle->pic_pd->DOFF0);
    print("DOFF1=0x%X\n", pic_handle->pic_pd->DOFF1);
    print("DOFF2=0x%X\n", pic_handle->pic_pd->DOFF2);
    print("DMULT0=0x%X\n", pic_handle->pic_pd->DMULT0);
    print("DMULT1=0x%X\n", pic_handle->pic_pd->DMULT1);
    print("DMULT2=0x%X\n", pic_handle->pic_pd->DMULT2);    
    print("POFF0=0x%X\n", pic_handle->pic_pd->POFF0);
    print("POFF1=0x%X\n", pic_handle->pic_pd->POFF1);
    print("POFF2=0x%X\n", pic_handle->pic_pd->POFF2);    
    print("PMULT0=0x%X\n", pic_handle->pic_pd->PMULT0);
    print("PMULT1=0x%X\n", pic_handle->pic_pd->PMULT1);
    print("PMULT2=0x%X\n", pic_handle->pic_pd->PMULT2);    
    print("REV0=0x%X\n", pic_handle->pic_pd->REV0);
    print("REV1=0x%X\n", pic_handle->pic_pd->REV1);
}
EXPORT_SYMBOL(pic_pd_dump_handle_regs);

void pic_pd_copy_settings(struct pic_handle_t *pic_handle_dst, struct pic_handle_t *pic_handle_src)
{
    memcpy(pic_handle_dst->pic_pd, pic_handle_src->pic_pd, sizeof(*(pic_handle_src->pic_pd)) );
}
EXPORT_SYMBOL(pic_pd_copy_settings);

void pic_pd_set_gain(struct pic_handle_t *pic_handle, uint8_t *leftgainintc, uint8_t *leftgaindecc,
		uint8_t *rightgainintc, uint8_t *rightgaindecc)
{
    PIC_PTRCHECK_NR(pic_pd);

    pic_handle->pic_pd->RLSCR = 
        PRNUDSNU_RLSCR_LEFTGAININTC2_REPLACE_VAL(pic_handle->pic_pd->RLSCR, leftgainintc[2]);
    pic_handle->pic_pd->RLSCR =
        PRNUDSNU_RLSCR_LEFTGAININTC1_REPLACE_VAL(pic_handle->pic_pd->RLSCR, leftgainintc[1]);
    pic_handle->pic_pd->RLSCR =
        PRNUDSNU_RLSCR_LEFTGAININTC0_REPLACE_VAL(pic_handle->pic_pd->RLSCR, leftgainintc[0]);
    pic_handle->pic_pd->RLSCR =
        PRNUDSNU_RLSCR_LEFTGAINDECC2_REPLACE_VAL(pic_handle->pic_pd->RLSCR, leftgaindecc[2]);
    pic_handle->pic_pd->RLSCR =
        PRNUDSNU_RLSCR_LEFTGAINDECC1_REPLACE_VAL(pic_handle->pic_pd->RLSCR, leftgaindecc[1]);
    pic_handle->pic_pd->RLSCR =
        PRNUDSNU_RLSCR_LEFTGAINDECC0_REPLACE_VAL(pic_handle->pic_pd->RLSCR, leftgaindecc[0]);
    pic_handle->pic_pd->RLSCR =
        PRNUDSNU_RLSCR_RIGHTGAININTC2_REPLACE_VAL(pic_handle->pic_pd->RLSCR, rightgainintc[2]);
    pic_handle->pic_pd->RLSCR =
        PRNUDSNU_RLSCR_RIGHTGAININTC1_REPLACE_VAL(pic_handle->pic_pd->RLSCR, rightgainintc[1]);
    pic_handle->pic_pd->RLSCR =
        PRNUDSNU_RLSCR_RIGHTGAININTC0_REPLACE_VAL(pic_handle->pic_pd->RLSCR, rightgainintc[0]);
    pic_handle->pic_pd->RLSCR =
        PRNUDSNU_RLSCR_RIGHTGAINDECC2_REPLACE_VAL(pic_handle->pic_pd->RLSCR, rightgaindecc[2]);
    pic_handle->pic_pd->RLSCR =
        PRNUDSNU_RLSCR_RIGHTGAINDECC1_REPLACE_VAL(pic_handle->pic_pd->RLSCR, rightgaindecc[1]);
    pic_handle->pic_pd->RLSCR = 
        PRNUDSNU_RLSCR_RIGHTGAINDECC0_REPLACE_VAL(pic_handle->pic_pd->RLSCR, rightgaindecc[0]);
}
EXPORT_SYMBOL(pic_pd_set_gain);

void pic_pd_get_gain(struct pic_handle_t *pic_handle, uint8_t *leftgainintc, uint8_t *leftgaindecc,
		uint8_t *rightgainintc, uint8_t *rightgaindecc)
{
    PIC_PTRCHECK_NR(pic_pd);

    leftgainintc[2] = PRNUDSNU_RLSCR_LEFTGAININTC2_MASK_SHIFT(pic_handle->pic_pd->RLSCR);
    leftgainintc[1] = PRNUDSNU_RLSCR_LEFTGAININTC1_MASK_SHIFT(pic_handle->pic_pd->RLSCR);
    leftgainintc[0] = PRNUDSNU_RLSCR_LEFTGAININTC0_MASK_SHIFT(pic_handle->pic_pd->RLSCR);
    leftgaindecc[2] = PRNUDSNU_RLSCR_LEFTGAINDECC2_MASK_SHIFT(pic_handle->pic_pd->RLSCR);
    leftgaindecc[1] = PRNUDSNU_RLSCR_LEFTGAINDECC1_MASK_SHIFT(pic_handle->pic_pd->RLSCR);
    leftgaindecc[0] = PRNUDSNU_RLSCR_LEFTGAINDECC0_MASK_SHIFT(pic_handle->pic_pd->RLSCR);
    rightgainintc[2] = PRNUDSNU_RLSCR_RIGHTGAININTC2_MASK_SHIFT(pic_handle->pic_pd->RLSCR);
    rightgainintc[1] = PRNUDSNU_RLSCR_RIGHTGAININTC1_MASK_SHIFT(pic_handle->pic_pd->RLSCR);
    rightgainintc[0] = PRNUDSNU_RLSCR_RIGHTGAININTC0_MASK_SHIFT(pic_handle->pic_pd->RLSCR);
    rightgaindecc[2] = PRNUDSNU_RLSCR_RIGHTGAINDECC2_MASK_SHIFT(pic_handle->pic_pd->RLSCR);
    rightgaindecc[1] = PRNUDSNU_RLSCR_RIGHTGAINDECC1_MASK_SHIFT(pic_handle->pic_pd->RLSCR);
    rightgaindecc[0] = PRNUDSNU_RLSCR_RIGHTGAINDECC0_MASK_SHIFT(pic_handle->pic_pd->RLSCR);
}
EXPORT_SYMBOL(pic_pd_get_gain);

void pic_pd_set_left_slope(struct pic_handle_t *pic_handle, uint8_t *slope)
{
    PIC_PTRCHECK_NR(pic_pd);

    pic_handle->pic_pd->LSVCR =
        PRNUDSNU_LSVCR_LEFTSLOPEC2_REPLACE_VAL(pic_handle->pic_pd->LSVCR, slope[2]);
    pic_handle->pic_pd->LSVCR =
        PRNUDSNU_LSVCR_LEFTSLOPEC1_REPLACE_VAL(pic_handle->pic_pd->LSVCR, slope[1]);
    pic_handle->pic_pd->LSVCR =
        PRNUDSNU_LSVCR_LEFTSLOPEC0_REPLACE_VAL(pic_handle->pic_pd->LSVCR, slope[0]);
}
EXPORT_SYMBOL(pic_pd_set_left_slope);

void pic_pd_get_left_slope(struct pic_handle_t *pic_handle, uint8_t *slope)
{
    PIC_PTRCHECK_NR(pic_pd);

    slope[2] = PRNUDSNU_LSVCR_LEFTSLOPEC2_MASK_SHIFT(pic_handle->pic_pd->LSVCR);
    slope[1] = PRNUDSNU_LSVCR_LEFTSLOPEC1_MASK_SHIFT(pic_handle->pic_pd->LSVCR);
    slope[0] = PRNUDSNU_LSVCR_LEFTSLOPEC0_MASK_SHIFT(pic_handle->pic_pd->LSVCR);
}
EXPORT_SYMBOL(pic_pd_get_left_slope);

void pic_pd_set_left_max_c0(struct pic_handle_t *pic_handle, uint16_t leftcorr)
{
    PIC_PTRCHECK_NR(pic_pd);
    
    pic_handle->pic_pd->LMPCR0 =
        PRNUDSNU_LMPCR0_LEFTCORR_REPLACE_VAL(pic_handle->pic_pd->LMPCR0, leftcorr);
}
EXPORT_SYMBOL(pic_pd_set_left_max_c0);

void pic_pd_get_left_max_c0(struct pic_handle_t *pic_handle, uint16_t *leftcorr)
{
    PIC_PTRCHECK_NR(pic_pd);
    
    pic_handle->pic_pd->LMPCR0 =
        PRNUDSNU_LMPCR0_LEFTCORR_MASK_SHIFT(pic_handle->pic_pd->LMPCR0);
}
EXPORT_SYMBOL(pic_pd_get_left_max_c0);

void pic_pd_set_left_max_c1(struct pic_handle_t *pic_handle, uint16_t leftcorr)
{
    PIC_PTRCHECK_NR(pic_pd);
    
    pic_handle->pic_pd->LMPCR1 =
        PRNUDSNU_LMPCR1_LEFTCORR_REPLACE_VAL(pic_handle->pic_pd->LMPCR1, leftcorr);
}
EXPORT_SYMBOL(pic_pd_set_left_max_c1);

void pic_pd_set_left_max_c2(struct pic_handle_t *pic_handle, uint16_t leftcorr)
{
    PIC_PTRCHECK_NR(pic_pd);
    
    pic_handle->pic_pd->LMPCR2 =
        PRNUDSNU_LMPCR2_LEFTCORR_REPLACE_VAL(pic_handle->pic_pd->LMPCR2, leftcorr);
}
EXPORT_SYMBOL(pic_pd_set_left_max_c2);

void pic_pd_set_right_slope(struct pic_handle_t *pic_handle, uint8_t *slope)
{
    PIC_PTRCHECK_NR(pic_pd);
    
    pic_handle->pic_pd->RSVCR =
        PRNUDSNU_RSVCR_RIGHTSLOPEC2_REPLACE_VAL(pic_handle->pic_pd->RSVCR, slope[2]);
    pic_handle->pic_pd->RSVCR =
        PRNUDSNU_RSVCR_RIGHTSLOPEC1_REPLACE_VAL(pic_handle->pic_pd->RSVCR, slope[1]);
    pic_handle->pic_pd->RSVCR =
        PRNUDSNU_RSVCR_RIGHTSLOPEC0_REPLACE_VAL(pic_handle->pic_pd->RSVCR, slope[0]);
}
EXPORT_SYMBOL(pic_pd_set_right_slope);

void pic_pd_get_right_slope(struct pic_handle_t *pic_handle, uint8_t *slope)
{
    PIC_PTRCHECK_NR(pic_pd);
    
    slope[2] = PRNUDSNU_RSVCR_RIGHTSLOPEC2_MASK_SHIFT(pic_handle->pic_pd->RSVCR);
    slope[1] = PRNUDSNU_RSVCR_RIGHTSLOPEC1_MASK_SHIFT(pic_handle->pic_pd->RSVCR);
    slope[0] = PRNUDSNU_RSVCR_RIGHTSLOPEC0_MASK_SHIFT(pic_handle->pic_pd->RSVCR);
}
EXPORT_SYMBOL(pic_pd_get_right_slope);

void pic_pd_set_right_start_c0(struct pic_handle_t *pic_handle, uint16_t rightstart)
{
    PIC_PTRCHECK_NR(pic_pd);
    
    pic_handle->pic_pd->RSPCR0 =
        PRNUDSNU_RSPCR0_RIGHTSTART_REPLACE_VAL(pic_handle->pic_pd->RSPCR0, rightstart);
}
EXPORT_SYMBOL(pic_pd_set_right_start_c0);

void pic_pd_get_right_start_c0(struct pic_handle_t *pic_handle, uint16_t *rightstart)
{
    PIC_PTRCHECK_NR(pic_pd);

    *rightstart = PRNUDSNU_RSPCR0_RIGHTSTART_MASK_SHIFT(pic_handle->pic_pd->RSPCR0);
}
EXPORT_SYMBOL(pic_pd_get_right_start_c0);

void pic_pd_set_right_start_c1(struct pic_handle_t *pic_handle, uint16_t rightstart)
{
    PIC_PTRCHECK_NR(pic_pd);

    pic_handle->pic_pd->RSPCR1 =
        PRNUDSNU_RSPCR1_RIGHTSTART_REPLACE_VAL(pic_handle->pic_pd->RSPCR1, rightstart);
}
EXPORT_SYMBOL(pic_pd_set_right_start_c1);

void pic_pd_set_right_start_c2(struct pic_handle_t *pic_handle, uint16_t rightstart)
{
    PIC_PTRCHECK_NR(pic_pd);

    pic_handle->pic_pd->RSPCR2 =
        PRNUDSNU_RSPCR2_RIGHTSTART_REPLACE_VAL(pic_handle->pic_pd->RSPCR2, rightstart);
}
EXPORT_SYMBOL(pic_pd_set_right_start_c2);

void pic_pd_get_cfg1(struct pic_handle_t *pic_handle, uint8_t *corrbits, uint8_t *allocc)
{
    PIC_PTRCHECK_NR(pic_pd);

    *corrbits = PRNUDSNU_PDCR1_CORRBITS_MASK_SHIFT(pic_handle->pic_pd->PDCR1);
    allocc[2] = PRNUDSNU_PDCR1_ALLOCC2_MASK_SHIFT(pic_handle->pic_pd->PDCR1);
    allocc[1] = PRNUDSNU_PDCR1_ALLOCC1_MASK_SHIFT(pic_handle->pic_pd->PDCR1);
    allocc[0] = PRNUDSNU_PDCR1_ALLOCC0_MASK_SHIFT(pic_handle->pic_pd->PDCR1);
}
EXPORT_SYMBOL(pic_pd_get_cfg1);

static void pic_pd_set_cfg1(struct pic_handle_t *pic_handle, uint8_t corrbits, uint8_t *allocc)
{
    PIC_PTRCHECK_NR(pic_pd);

    pic_handle->pic_pd->PDCR1 =
        PRNUDSNU_PDCR1_CORRBITS_REPLACE_VAL(pic_handle->pic_pd->PDCR1, corrbits);
    pic_handle->pic_pd->PDCR1 =
        PRNUDSNU_PDCR1_ALLOCC2_REPLACE_VAL(pic_handle->pic_pd->PDCR1, allocc[2]);
    pic_handle->pic_pd->PDCR1 =
        PRNUDSNU_PDCR1_ALLOCC1_REPLACE_VAL(pic_handle->pic_pd->PDCR1, allocc[1]);
    pic_handle->pic_pd->PDCR1 =
        PRNUDSNU_PDCR1_ALLOCC0_REPLACE_VAL(pic_handle->pic_pd->PDCR1, allocc[0]);
}

static void pic_pd_set_cfg2(struct pic_handle_t *pic_handle, uint8_t coeffwidth,
		uint8_t bypass, uint8_t bypassquad, uint8_t bypassexp,
		uint8_t bypassprnu, uint8_t bypassdsnu, uint8_t bypassbadpix)
{
    PIC_PTRCHECK_NR(pic_pd);

    pic_handle->pic_pd->PDCR2 =
        PRNUDSNU_PDCR2_COEFFWIDTH_REPLACE_VAL(pic_handle->pic_pd->PDCR2, coeffwidth);
    pic_handle->pic_pd->PDCR2 =
        PRNUDSNU_PDCR2_BYPASS_REPLACE_VAL(pic_handle->pic_pd->PDCR2, bypass);
    pic_handle->pic_pd->PDCR2 =
        PRNUDSNU_PDCR2_BYPASSQUAD_REPLACE_VAL(pic_handle->pic_pd->PDCR2, bypassquad);
    pic_handle->pic_pd->PDCR2 =
        PRNUDSNU_PDCR2_BYPASSEXP_REPLACE_VAL(pic_handle->pic_pd->PDCR2, bypassexp);
    pic_handle->pic_pd->PDCR2 =
        PRNUDSNU_PDCR2_BYPASSPRNU_REPLACE_VAL(pic_handle->pic_pd->PDCR2, bypassprnu);
    pic_handle->pic_pd->PDCR2 =
        PRNUDSNU_PDCR2_BYPASSDSNU_REPLACE_VAL(pic_handle->pic_pd->PDCR2, bypassdsnu);
    pic_handle->pic_pd->PDCR2 =
        PRNUDSNU_PDCR2_BYPASSBADPIX_REPLACE_VAL(pic_handle->pic_pd->PDCR2, bypassbadpix);
}

static void pic_pd_get_cfg2(struct pic_handle_t *pic_handle, uint8_t *coeffwidth,
		uint8_t *bypass, uint8_t *bypassquad, uint8_t *bypassexp,
		uint8_t *bypassprnu, uint8_t *bypassdsnu, uint8_t *bypassbadpix)
{
    PIC_PTRCHECK_NR(pic_pd);

    *coeffwidth = PRNUDSNU_PDCR2_COEFFWIDTH_MASK_SHIFT(pic_handle->pic_pd->PDCR2);
    *bypass = PRNUDSNU_PDCR2_BYPASS_MASK_SHIFT(pic_handle->pic_pd->PDCR2);
    *bypassquad = PRNUDSNU_PDCR2_BYPASSQUAD_MASK_SHIFT(pic_handle->pic_pd->PDCR2);
    *bypassexp = PRNUDSNU_PDCR2_BYPASSEXP_MASK_SHIFT(pic_handle->pic_pd->PDCR2);
    *bypassprnu = PRNUDSNU_PDCR2_BYPASSPRNU_MASK_SHIFT(pic_handle->pic_pd->PDCR2);
    *bypassdsnu = PRNUDSNU_PDCR2_BYPASSDSNU_MASK_SHIFT(pic_handle->pic_pd->PDCR2);
    *bypassbadpix = PRNUDSNU_PDCR2_BYPASSBADPIX_MASK_SHIFT(pic_handle->pic_pd->PDCR2);
}
EXPORT_SYMBOL(pic_pd_get_cfg2);

void pic_pd_get_dsnu_offset_c0(struct pic_handle_t *pic_handle, uint16_t *offset)
{
    PIC_PTRCHECK_NR(pic_pd);

    *offset = PRNUDSNU_DOFF0_OFFSET_MASK_SHIFT(pic_handle->pic_pd->DOFF0);
}
EXPORT_SYMBOL(pic_pd_get_dsnu_offset_c0);

void pic_pd_get_dsnu_mult_scale_c0(struct pic_handle_t *pic_handle, uint8_t  *scale, uint16_t *mult)
{
    PIC_PTRCHECK_NR(pic_pd);

    *scale = PRNUDSNU_DMULT0_SCALE_MASK_SHIFT(pic_handle->pic_pd->DMULT0);
    *mult = PRNUDSNU_DMULT0_MULT_MASK_SHIFT(pic_handle->pic_pd->DMULT0);
}
EXPORT_SYMBOL(pic_pd_get_dsnu_mult_scale_c0);

void pic_pd_get_prnu_offset_c0(struct pic_handle_t *pic_handle, uint16_t *offset)
{
    PIC_PTRCHECK_NR(pic_pd);

    *offset = PRNUDSNU_POFF0_OFFSET_MASK_SHIFT(pic_handle->pic_pd->POFF0);
}
EXPORT_SYMBOL(pic_pd_get_prnu_offset_c0);

void pic_pd_get_prnu_mult_scale_c0(struct pic_handle_t *pic_handle, uint8_t *scale, uint16_t *mult)
{
    PIC_PTRCHECK_NR(pic_pd);

    *scale = PRNUDSNU_PMULT0_SCALE_MASK_SHIFT(pic_handle->pic_pd->PMULT0);
    *mult = PRNUDSNU_PMULT0_MULT_MASK_SHIFT(pic_handle->pic_pd->PMULT0);
}
EXPORT_SYMBOL(pic_pd_get_prnu_mult_scale_c0);

static uint8_t pic_pd_allocbits_regval(int8_t value)
{
    uint8_t regval;

    switch(value)
    {
    	case -3:
    	    regval = PD_DSNU_PLUS_3;
    	     break;
    	case -2:
            regval = PD_DSNU_PLUS_2;
            break;
        case -1:
            regval = PD_DSNU_PLUS_1;
            break;
        case 0:
            regval = PD_50_50;
            break;
        case 1:
            regval = PD_PRNU_PLUS_1;
            break;
        case 2:
            regval = PD_PRNU_PLUS_2;
            break;
        case 3:
             regval = PD_PRNU_PLUS_3;
             break;
        default:
            error_print("ERR:%s: bad allocbits value %d\n",__func__,value);
            return -1;
    }
    return regval;
}

void pic_pd_set_config1(struct pic_handle_t *pic_handle, uint8_t corrbits, int8_t allocc2,
                        int8_t allocc1, int8_t allocc0)
{
    uint8_t corrbits_regval;
    uint8_t allocc_regval[3];

    // Translate passed values into register values
    switch(corrbits)
    {
        case 32:
            corrbits_regval = PD_LUT_32bits;
            break;
        case 28:
            corrbits_regval = PD_LUT_28bits;
            break;
        case 24:
            corrbits_regval = PD_LUT_24bits;
            break;
        case 20:
            corrbits_regval = PD_LUT_20bits;
            break;
        case 16:
            corrbits_regval = PD_LUT_16bits;
            break;
        case 12:
            corrbits_regval = PD_LUT_12bits;
            break;
        default:
            error_print("ERR:%s: bad corrbits value %d\n",__func__,corrbits);
            BUG();
            break;
    }

    allocc_regval[0] = pic_pd_allocbits_regval(allocc0);
    allocc_regval[1] = pic_pd_allocbits_regval(allocc1);
    allocc_regval[2] = pic_pd_allocbits_regval(allocc2);

    pic_pd_set_cfg1(pic_handle, corrbits_regval, allocc_regval);
}
EXPORT_SYMBOL(pic_pd_set_config1);

void pic_pd_set_coeffwidth(struct pic_handle_t *pic_handle, uint8_t coeffwidth)
{
    uint8_t coeffwidth_regval;
    uint8_t bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth_regval, &bypass, &bypassquad, &bypassexp,
		&bypassprnu, &bypassdsnu, &bypassbadpix);


    switch(coeffwidth)
    {
        case 32:
            coeffwidth_regval = PD_COEFF_32bits;
            break;
        case 28:
            coeffwidth_regval = PD_COEFF_28bits;
            break;
        case 24:
            coeffwidth_regval = PD_COEFF_24bits;
            break;
        case 20:
            coeffwidth_regval = PD_COEFF_20bits;
            break;
        case 16:
            coeffwidth_regval = PD_COEFF_16bits;
            break;
        case 12:
            coeffwidth_regval = PD_COEFF_12bits;
            break;
        default:
            error_print("ERR:%s: bad coefficient value %d\n",__func__,coeffwidth);
            return;
    }

    pic_pd_set_cfg2(pic_handle, coeffwidth_regval, bypass, bypassquad, bypassexp,
			bypassprnu, bypassdsnu, bypassbadpix);
}
EXPORT_SYMBOL(pic_pd_set_coeffwidth);

int pic_pd_get_coeffwidth(struct pic_handle_t *pic_handle)
{
    uint8_t coeffwidth = 0;
    uint8_t bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
			&bypassprnu, &bypassdsnu, &bypassbadpix);

    switch(coeffwidth) {
    	   case PD_COEFF_32bits:
    		   coeffwidth = 32;
    		   break;
           case PD_COEFF_28bits:
               coeffwidth= 28;
               break;
           case PD_COEFF_24bits:
        	   coeffwidth = 24;
               break;
           case PD_COEFF_20bits:
        	   coeffwidth = 20;
               break;
           case PD_COEFF_16bits:
        	   coeffwidth = 16;
               break;
           case PD_COEFF_12bits:
               coeffwidth = 12;
               break;
           default:
               error_print("ERR:%s: bad coefficient value %d\n",__func__,coeffwidth);

       }
    return coeffwidth;
}
EXPORT_SYMBOL(pic_pd_get_coeffwidth);

void pic_pd_set_bypass_all(struct pic_handle_t *pic_handle, bool bypass_val)
{
    uint8_t coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
		&bypassprnu, &bypassdsnu, &bypassbadpix);

    bypass =  (bypass_val ? 1 : 0);

    pic_pd_set_cfg2(pic_handle, coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix);
}
EXPORT_SYMBOL(pic_pd_set_bypass_all);

bool pic_pd_get_bypass_all(struct pic_handle_t *pic_handle)
{
    uint8_t coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
			&bypassprnu, &bypassdsnu, &bypassbadpix);

    return (bypass ? true : false);
}
EXPORT_SYMBOL(pic_pd_get_bypass_all);

void pic_pd_set_bypass_prnu(struct pic_handle_t *pic_handle, bool bypass_val)
{
    uint8_t coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
		&bypassprnu, &bypassdsnu, &bypassbadpix);

    bypassprnu =  (bypass_val ? 1 : 0);

    pic_pd_set_cfg2(pic_handle, coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix);
}
EXPORT_SYMBOL(pic_pd_set_bypass_prnu);

bool pic_pd_get_bypass_prnu(struct pic_handle_t *pic_handle)
{
    uint8_t coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
			&bypassprnu, &bypassdsnu, &bypassbadpix);

    return (bypassprnu ? true : false);
}
EXPORT_SYMBOL(pic_pd_get_bypass_prnu);

void pic_pd_set_bypass_dsnu(struct pic_handle_t *pic_handle, bool bypass_val)
{
    uint8_t coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
		&bypassprnu, &bypassdsnu, &bypassbadpix);

    bypassdsnu =  (bypass_val ? 1 : 0);

    pic_pd_set_cfg2(pic_handle, coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix);
}
EXPORT_SYMBOL(pic_pd_set_bypass_dsnu);

bool pic_pd_get_bypass_dsnu(struct pic_handle_t *pic_handle)
{
    uint8_t coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
			&bypassprnu, &bypassdsnu, &bypassbadpix);

    return (bypassdsnu ? true : false);
}
EXPORT_SYMBOL(pic_pd_get_bypass_dsnu);

void pic_pd_set_bypass_bad_pix_replace(struct pic_handle_t *pic_handle, bool bypass_val)
{
    uint8_t coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
			&bypassprnu, &bypassdsnu, &bypassbadpix);

    bypassbadpix =  (bypass_val ? 1 : 0);

    pic_pd_set_cfg2(pic_handle, coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix);
}
EXPORT_SYMBOL(pic_pd_set_bypass_bad_pix_replace);

bool pic_pd_get_bypass_bad_pix_replace(struct pic_handle_t *pic_handle)
{
    uint8_t coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
			&bypassprnu, &bypassdsnu, &bypassbadpix);

    return (bypassbadpix ? true : false);
}
EXPORT_SYMBOL(pic_pd_get_bypass_bad_pix_replace);

void pic_pd_set_bypass_exp_comp(struct pic_handle_t *pic_handle, bool bypass_val)
{
    uint8_t coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
				&bypassprnu, &bypassdsnu, &bypassbadpix);

    bypassexp =  (bypass_val ? 1 : 0);

    pic_pd_set_cfg2(pic_handle, coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix);
}
EXPORT_SYMBOL(pic_pd_set_bypass_exp_comp);


bool pic_pd_get_bypass_exp_comp(struct pic_handle_t *pic_handle)
{
    uint8_t coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
			&bypassprnu, &bypassdsnu, &bypassbadpix);

    return (bypassexp ? true : false);
}
EXPORT_SYMBOL(pic_pd_get_bypass_exp_comp);

void pic_pd_set_bypass_quadratic(struct pic_handle_t *pic_handle, bool bypass_val)
{
    uint8_t coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
				&bypassprnu, &bypassdsnu, &bypassbadpix);

    bypassquad =  (bypass_val ? 1 : 0);

    pic_pd_set_cfg2(pic_handle, coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix);
}
EXPORT_SYMBOL(pic_pd_set_bypass_quadratic);

bool pic_pd_get_bypass_quadratic(struct pic_handle_t *pic_handle)
{
    uint8_t coeffwidth, bypass, bypassquad, bypassexp, bypassprnu, bypassdsnu, bypassbadpix;

    pic_pd_get_cfg2(pic_handle, &coeffwidth, &bypass, &bypassquad, &bypassexp,
			&bypassprnu, &bypassdsnu, &bypassbadpix);

    return (bypassquad ? true : false);
}
EXPORT_SYMBOL(pic_pd_get_bypass_quadratic);

void pic_pd_set_dsnu_offset(struct pic_handle_t *pic_handle, uint16_t offset0, uint16_t offset1, uint16_t offset2)
{
    PIC_PTRCHECK_NR(pic_pd);

    // Fill out virtual register structure - color 0
    pic_handle->pic_pd->DOFF0 =
        PRNUDSNU_DOFF0_OFFSET_REPLACE_VAL(pic_handle->pic_pd->DOFF0, offset0);

    // Fill out virtual register structure - color 1
    pic_handle->pic_pd->DOFF1 =
        PRNUDSNU_DOFF1_OFFSET_REPLACE_VAL(pic_handle->pic_pd->DOFF1, offset1);

    // Fill out virtual register structure - color 2
    pic_handle->pic_pd->DOFF2 =
        PRNUDSNU_DOFF2_OFFSET_REPLACE_VAL(pic_handle->pic_pd->DOFF2, offset2);
}
EXPORT_SYMBOL(pic_pd_set_dsnu_offset);

void pic_pd_set_dsnu_mult_scale(struct pic_handle_t *pic_handle, uint16_t mult0, uint8_t scale0, uint16_t mult1, uint8_t scale1, uint16_t mult2, uint8_t scale2)
{
    PIC_PTRCHECK_NR(pic_pd);

    // Fill out virtual register structure - color 0
    pic_handle->pic_pd->DMULT0 =
        PRNUDSNU_DMULT0_SCALE_REPLACE_VAL(pic_handle->pic_pd->DMULT0, scale0);
    pic_handle->pic_pd->DMULT0 =
        PRNUDSNU_DMULT0_MULT_REPLACE_VAL(pic_handle->pic_pd->DMULT0, mult0);

    // Fill out virtual register structure - color 1
    pic_handle->pic_pd->DMULT1 =
        PRNUDSNU_DMULT1_SCALE_REPLACE_VAL(pic_handle->pic_pd->DMULT1, scale1);
    pic_handle->pic_pd->DMULT1 =
        PRNUDSNU_DMULT1_MULT_REPLACE_VAL(pic_handle->pic_pd->DMULT1, mult1);

    // Fill out virtual register structure - color 2
    pic_handle->pic_pd->DMULT2 =
        PRNUDSNU_DMULT2_SCALE_REPLACE_VAL(pic_handle->pic_pd->DMULT2, scale2);
    pic_handle->pic_pd->DMULT2 =
        PRNUDSNU_DMULT2_MULT_REPLACE_VAL(pic_handle->pic_pd->DMULT2, mult2);
}
EXPORT_SYMBOL(pic_pd_set_dsnu_mult_scale);

void pic_pd_set_prnu_offset(struct pic_handle_t *pic_handle, uint16_t offset0, uint16_t offset1, uint16_t offset2)
{
    PIC_PTRCHECK_NR(pic_pd);

    // Fill out virtual register structure - color 0
    pic_handle->pic_pd->POFF0 =
        PRNUDSNU_POFF0_OFFSET_REPLACE_VAL(pic_handle->pic_pd->POFF0, offset0);

    // Fill out virtual register structure - color 1
    pic_handle->pic_pd->POFF1 =
        PRNUDSNU_POFF1_OFFSET_REPLACE_VAL(pic_handle->pic_pd->POFF1, offset1);

    // Fill out virtual register structure - color 2
    pic_handle->pic_pd->POFF2 =
        PRNUDSNU_POFF2_OFFSET_REPLACE_VAL(pic_handle->pic_pd->POFF2, offset2);
}
EXPORT_SYMBOL(pic_pd_set_prnu_offset);

void pic_pd_set_prnu_mult_scale(struct pic_handle_t *pic_handle, uint16_t mult0, uint8_t scale0, uint16_t mult1, uint8_t scale1, uint16_t mult2, uint8_t scale2)
{
    PIC_PTRCHECK_NR(pic_pd);

    // Fill out virtual register structure - color 0
    pic_handle->pic_pd->PMULT0 =
        PRNUDSNU_PMULT0_SCALE_REPLACE_VAL(pic_handle->pic_pd->PMULT0, scale0);
    pic_handle->pic_pd->PMULT0 =
        PRNUDSNU_PMULT0_MULT_REPLACE_VAL(pic_handle->pic_pd->PMULT0, mult0);

    // Fill out virtual register structure - color 1
    pic_handle->pic_pd->PMULT1 =
        PRNUDSNU_PMULT1_SCALE_REPLACE_VAL(pic_handle->pic_pd->PMULT1, scale1);
    pic_handle->pic_pd->PMULT1 =
        PRNUDSNU_PMULT1_MULT_REPLACE_VAL(pic_handle->pic_pd->PMULT1, mult1);

    // Fill out virtual register structure - color 2
    pic_handle->pic_pd->PMULT2 =
        PRNUDSNU_PMULT2_SCALE_REPLACE_VAL(pic_handle->pic_pd->PMULT2, scale2);
    pic_handle->pic_pd->PMULT2 =
        PRNUDSNU_PMULT2_MULT_REPLACE_VAL(pic_handle->pic_pd->PMULT2, mult2);
}
EXPORT_SYMBOL(pic_pd_set_prnu_mult_scale);

void pic_pd_set_quad_cfg(struct pic_handle_t *pic_handle, uint8_t sf1, uint8_t sf2, uint8_t q_shift)
{
    PIC_PTRCHECK_NR(pic_pd);

    pic_handle->pic_pd->QUAD =
        PRNUDSNU_QUAD_SF2_REPLACE_VAL(pic_handle->pic_pd->QUAD, sf2);
    pic_handle->pic_pd->QUAD =
        PRNUDSNU_QUAD_SF1_REPLACE_VAL(pic_handle->pic_pd->QUAD, sf1);
    pic_handle->pic_pd->QUAD =
        PRNUDSNU_QUAD_Q_SHIFT_REPLACE_VAL(pic_handle->pic_pd->QUAD, q_shift);
}
EXPORT_SYMBOL(pic_pd_set_quad_cfg);

////////////////////////////////////
// pic Newman (chipgap)
////////////////////////////////////

void pic_chipgap_dump_handle_regs(struct pic_handle_t *pic_handle)
{
    print("PIC Chipgap (Newman) regs (pic_handle)\n");
    PIC_PTRCHECK_NR(pic_chipgap);

    print("CFG=0x%X\n", pic_handle->pic_chipgap->CFG);
    print("LOCAT=0x%X\n", pic_handle->pic_chipgap->LOCAT);
    print("INSERT0=0x%X\n", pic_handle->pic_chipgap->INSERT0);
    print("INSERT1=0x%X\n", pic_handle->pic_chipgap->INSERT1);
    print("INSERT2=0x%X\n", pic_handle->pic_chipgap->INSERT2);
    print("INSERT3=0x%X\n", pic_handle->pic_chipgap->INSERT3);    
    print("REV0=0x%X\n", pic_handle->pic_chipgap->REV0);
    print("REV1=0x%X\n", pic_handle->pic_chipgap->REV1);    
}
EXPORT_SYMBOL(pic_chipgap_dump_handle_regs);

void pic_chipgap_set_CFG(struct pic_handle_t *pic_handle, uint8_t bypass)
{
    PIC_PTRCHECK_NR(pic_chipgap);

    pic_handle->pic_chipgap->CFG =
        NEWMAN_CFG_BYPASS_REPLACE_VAL(pic_handle->pic_chipgap->CFG, bypass);
}
EXPORT_SYMBOL(pic_chipgap_set_CFG);

void pic_chipgap_set_LOCAT(struct pic_handle_t *pic_handle, uint16_t width, uint16_t start)
{
    PIC_PTRCHECK_NR(pic_chipgap);

    pic_handle->pic_chipgap->LOCAT =
        NEWMAN_LOCAT_WIDTH_REPLACE_VAL(pic_handle->pic_chipgap->LOCAT, width);
    pic_handle->pic_chipgap->LOCAT =
        NEWMAN_LOCAT_START_REPLACE_VAL(pic_handle->pic_chipgap->LOCAT, start);
}
EXPORT_SYMBOL(pic_chipgap_set_LOCAT);

// gap must be an array of 4
void pic_chipgap_set_INSERT0(struct pic_handle_t *pic_handle, uint8_t *gap)
{
    PIC_PTRCHECK_NR(pic_chipgap);
        
    pic_handle->pic_chipgap->INSERT0 =
        NEWMAN_INSERT0_GAP3_REPLACE_VAL(pic_handle->pic_chipgap->INSERT0, gap[3]);
    pic_handle->pic_chipgap->INSERT0 =
        NEWMAN_INSERT0_GAP2_REPLACE_VAL(pic_handle->pic_chipgap->INSERT0, gap[2]);
    pic_handle->pic_chipgap->INSERT0 =
        NEWMAN_INSERT0_GAP1_REPLACE_VAL(pic_handle->pic_chipgap->INSERT0, gap[1]);
    pic_handle->pic_chipgap->INSERT0 =
        NEWMAN_INSERT0_GAP0_REPLACE_VAL(pic_handle->pic_chipgap->INSERT0, gap[0]);
}
EXPORT_SYMBOL(pic_chipgap_set_INSERT0);

void pic_chipgap_set_INSERT1(struct pic_handle_t *pic_handle, uint8_t *gap)
{
    PIC_PTRCHECK_NR(pic_chipgap);

    pic_handle->pic_chipgap->INSERT1 =
        NEWMAN_INSERT1_GAP7_REPLACE_VAL(pic_handle->pic_chipgap->INSERT1, gap[3]);
    pic_handle->pic_chipgap->INSERT1 =
        NEWMAN_INSERT1_GAP6_REPLACE_VAL(pic_handle->pic_chipgap->INSERT1, gap[2]);
    pic_handle->pic_chipgap->INSERT1 =
        NEWMAN_INSERT1_GAP5_REPLACE_VAL(pic_handle->pic_chipgap->INSERT1, gap[1]);
    pic_handle->pic_chipgap->INSERT1 =
        NEWMAN_INSERT1_GAP4_REPLACE_VAL(pic_handle->pic_chipgap->INSERT1, gap[0]);
}
EXPORT_SYMBOL(pic_chipgap_set_INSERT1);

void pic_chipgap_set_INSERT2(struct pic_handle_t *pic_handle, uint8_t *gap)
{
    PIC_PTRCHECK_NR(pic_chipgap);

    pic_handle->pic_chipgap->INSERT2 =
        NEWMAN_INSERT2_GAP11_REPLACE_VAL(pic_handle->pic_chipgap->INSERT2, gap[3]);
    pic_handle->pic_chipgap->INSERT2 =
        NEWMAN_INSERT2_GAP10_REPLACE_VAL(pic_handle->pic_chipgap->INSERT2, gap[2]);
    pic_handle->pic_chipgap->INSERT2 =
        NEWMAN_INSERT2_GAP9_REPLACE_VAL(pic_handle->pic_chipgap->INSERT2, gap[1]);
    pic_handle->pic_chipgap->INSERT2 =
        NEWMAN_INSERT2_GAP8_REPLACE_VAL(pic_handle->pic_chipgap->INSERT2, gap[0]);
}
EXPORT_SYMBOL(pic_chipgap_set_INSERT2);

void pic_chipgap_set_INSERT3(struct pic_handle_t *pic_handle, uint8_t *gap)
{
    PIC_PTRCHECK_NR(pic_chipgap);

    pic_handle->pic_chipgap->INSERT3 =
        NEWMAN_INSERT3_GAP15_REPLACE_VAL(pic_handle->pic_chipgap->INSERT3, gap[3]);
    pic_handle->pic_chipgap->INSERT3 =
        NEWMAN_INSERT3_GAP14_REPLACE_VAL(pic_handle->pic_chipgap->INSERT3, gap[2]);
    pic_handle->pic_chipgap->INSERT3 =
        NEWMAN_INSERT3_GAP13_REPLACE_VAL(pic_handle->pic_chipgap->INSERT3, gap[1]);
    pic_handle->pic_chipgap->INSERT3 =
        NEWMAN_INSERT3_GAP12_REPLACE_VAL(pic_handle->pic_chipgap->INSERT3, gap[0]);
}
EXPORT_SYMBOL(pic_chipgap_set_INSERT3);

////////////////////////////////////
// pic hscale
////////////////////////////////////
void pic_hscale_dump_handle_regs(struct pic_handle_t *pic_handle)
{
    print("PIC Hscale regs (pic_handle)\n");
    PIC_PTRCHECK_NR(pic_hscale);

    print("HSCR=0x%X\n", pic_handle->pic_hscale->HSCR);
    print("HSSR=0x%X\n", pic_handle->pic_hscale->HSSR);    
    print("REV0=0x%X\n", pic_handle->pic_hscale->REV0);
    print("REV1=0x%X\n", pic_handle->pic_hscale->REV1);    
}
EXPORT_SYMBOL(pic_hscale_dump_handle_regs);

void pic_hs_set_bypass(struct pic_handle_t *pic_handle, bool bypass)
{
    PIC_PTRCHECK_NR(pic_hscale);

    pic_handle->pic_hscale->HSCR =
        HSCALE_HSCR_BYPASS_REPLACE_VAL(pic_handle->pic_hscale->HSCR, bypass ? 1 : 0);
}
EXPORT_SYMBOL(pic_hs_set_bypass);

void pic_hs_set_hscale_factor(struct pic_handle_t *pic_handle, uint32_t sfactor)
{
    PIC_PTRCHECK_NR(pic_hscale);

    pic_handle->pic_hscale->HSSR =
        HSCALE_HSSR_SCALE_REPLACE_VAL(pic_handle->pic_hscale->HSSR, sfactor);
}
EXPORT_SYMBOL(pic_hs_set_hscale_factor);

void pic_hs_get_hscale_factor(struct pic_handle_t *pic_handle, uint32_t *sfactor)
{
    PIC_PTRCHECK_NR(pic_hscale);

    *sfactor = HSCALE_HSSR_SCALE_MASK_SHIFT(pic_handle->pic_hscale->HSSR);
}
EXPORT_SYMBOL(pic_hs_get_hscale_factor);

////////////////////////////////////
// pic wdma (pic dma write)
////////////////////////////////////

void pic_wdma_dump_handle_regs(struct pic_handle_t *pic_handle)
{
    print("PIC WDMA regs (pic_handle)\n");
    PIC_PTRCHECK_NR(pic_wdma);

    print("DWOCR=0x%X\n", pic_handle->pic_wdma->DWOCR);
    print("DWRTBL=0x%X\n", pic_handle->pic_wdma->DWRTBL);
    print("DWCIS=0x%X\n", pic_handle->pic_wdma->DWCIS);    
    print("REV0=0x%X\n", pic_handle->pic_wdma->REV0);
    print("REV1=0x%X\n", pic_handle->pic_wdma->REV1);    
}
EXPORT_SYMBOL(pic_wdma_dump_handle_regs);

void pic_output_dma_set_disable_mode(struct pic_handle_t *pic_handle, uint8_t mode)
{
    char picmode;
    PIC_PTRCHECK_NR(pic_wdma);

    switch (mode)
    {
    case PIC_OUT_DMA_PROCESS_DATA:
        picmode = WDMA_PROCESS_DATA;
        break;
    case PIC_OUT_DMA_DISCARD_DATA:
        picmode = WDMA_DISCARD_DATA;
        break;
    case PIC_OUT_DMA_DISALLOW_DATA:
        picmode = WDMA_DISALLOW_DATA;
        break;
    default:
        error_print("ERROR, %s doesn't accept mode = %d\n",__func__, mode);
        return;
    }

    pic_handle->pic_wdma->DWOCR =
        PIC_WDMA_DWOCR_DISABLEMODE_REPLACE_VAL(pic_handle->pic_wdma->DWOCR, picmode);
}
EXPORT_SYMBOL(pic_output_dma_set_disable_mode);

void pic_output_dma_get_disable_mode(struct pic_handle_t *pic_handle, uint8_t *mode)
{
    uint32_t disableMode;
    PIC_PTRCHECK_NR(pic_wdma);

    disableMode = PIC_WDMA_DWOCR_DISABLEMODE_MASK_SHIFT(pic_handle->pic_wdma->DWOCR);

    switch (disableMode)
    {
    case PIC_OUT_DMA_PROCESS_DATA:
        *mode = WDMA_PROCESS_DATA;
        break;
    case PIC_OUT_DMA_DISCARD_DATA:
        *mode = WDMA_DISCARD_DATA;
        break;
    case PIC_OUT_DMA_DISALLOW_DATA:
        *mode = WDMA_DISALLOW_DATA;
        break;
    default:
        *mode = 0;
        error_print("ERROR, %s doesn't accept mode = %d\n",__func__, disableMode);
        break;
    }
}
EXPORT_SYMBOL(pic_output_dma_get_disable_mode);

void pic_output_dma_set_common_int(struct pic_handle_t *pic_handle, uint8_t channels, bool add)
{
    PIC_PTRCHECK_NR(pic_wdma);

    if (add)
        pic_handle->pic_wdma->DWCIS |=
            PIC_WDMA_DWCIS_SELECT_REPLACE_VAL(pic_handle->pic_wdma->DWCIS, channels);
    else
        pic_handle->pic_wdma->DWCIS =
            PIC_WDMA_DWCIS_SELECT_REPLACE_VAL(pic_handle->pic_wdma->DWCIS, channels);
}
EXPORT_SYMBOL(pic_output_dma_set_common_int);

void pic_output_dma_set_routing(struct pic_handle_t *pic_handle,
                                int monoDMA, int color2DMA, int color1DMA, int color0DMA)
{
    PIC_PTRCHECK_NR(pic_wdma);

    pic_handle->pic_wdma->DWRTBL =
        PIC_WDMA_DWRTBL_MONODMACRV_REPLACE_VAL(pic_handle->pic_wdma->DWRTBL, monoDMA);
    pic_handle->pic_wdma->DWRTBL =
        PIC_WDMA_DWRTBL_COLOR2DMACRV_REPLACE_VAL(pic_handle->pic_wdma->DWRTBL, color2DMA);
    pic_handle->pic_wdma->DWRTBL =
        PIC_WDMA_DWRTBL_COLOR1DMACRV_REPLACE_VAL(pic_handle->pic_wdma->DWRTBL, color1DMA);
    pic_handle->pic_wdma->DWRTBL =
        PIC_WDMA_DWRTBL_COLOR0DMACRV_REPLACE_VAL(pic_handle->pic_wdma->DWRTBL, color0DMA);
}
EXPORT_SYMBOL(pic_output_dma_set_routing);

////////////////////////////////////
// pic BDR (bit depth reduction)
////////////////////////////////////

void pic_bdr_dump_handle_regs(struct pic_handle_t *pic_handle)
{
    print("PIC BDR (bit depth reduction) regs (pic_handle)\n");
    PIC_PTRCHECK_NR(pic_bitreduct);

    print("Note: not dumping the LUT\n");
    print("BDRLUTR=0x%X\n", pic_handle->pic_bitreduct->BDRLUTR);
    print("BDRCR=0x%X\n", pic_handle->pic_bitreduct->BDRCR);
    print("REV0=0x%X\n", pic_handle->pic_bitreduct->REV0);
    print("REV1=0x%X\n", pic_handle->pic_bitreduct->REV1);    
}
EXPORT_SYMBOL(pic_bdr_dump_handle_regs);

void pic_bdr_set_bypass(struct pic_handle_t *pic_handle, bool bypass)
{
    int val;
    PIC_PTRCHECK_NR(pic_bitreduct);
    
    val = bypass ? 1 : 0;
    pic_handle->pic_bitreduct->BDRCR =
        BITREDUCT_BDRCR_BYPASS_REPLACE_VAL(pic_handle->pic_bitreduct->BDRCR, val);
}
EXPORT_SYMBOL(pic_bdr_set_bypass);

void pic_bdr_set_lut(struct pic_handle_t *pic_handle, uint32_t *lut)
{
    int i;
    PIC_PTRCHECK_NR(pic_bitreduct);
    
    // -1 because the first 1024 values (0-1023) go into the BDRLUT ...
    for (i=0; i<PIC_BDR_LUT_SIZE-1; i++)
    {
        pic_handle->pic_bitreduct->BDRLUT[i] = BITREDUCT_BDRLUT_VAL_MASK_SHIFT(lut[i]);
    }
    // and the 1025th value (#1024) of the LUT goes into BDRLUTR LUT(addr+1)
    pic_handle->pic_bitreduct->BDRLUTR = BITREDUCT_BDRLUTR_VAL_MASK_SHIFT(lut[PIC_BDR_LUT_SIZE-1]);
}
EXPORT_SYMBOL(pic_bdr_set_lut);

void pic_bdr_set_num_luts(struct pic_handle_t *pic_handle, int num_luts)
{
    PIC_PTRCHECK_NR(pic_bitreduct);
    
    if (num_luts == 1)
        pic_handle->pic_bitreduct->BDRCR =
            BITREDUCT_BDRCR_NUMLUTS_REPLACE_VAL(pic_handle->pic_bitreduct->BDRCR,
                                                PIC_BDR_BDRCR_NUMLUTS_ONE);
    else if (num_luts == 3)
        pic_handle->pic_bitreduct->BDRCR =
            BITREDUCT_BDRCR_NUMLUTS_REPLACE_VAL(pic_handle->pic_bitreduct->BDRCR,
                                                PIC_BDR_BDRCR_NUMLUTS_THREE);
    else
    {
        error_print("ERR:num_luts = %d\n",num_luts);
        BUG();
    }
}
EXPORT_SYMBOL(pic_bdr_set_num_luts);

void pic_bdr_select_lut(struct pic_handle_t *pic_handle, int lutsel)
{
    PIC_PTRCHECK_NR(pic_bitreduct);
    
    if (lutsel==PIC_BDR_BDRCR_LUTSEL_COLOR0MONO
        || lutsel==PIC_BDR_BDRCR_LUTSEL_COLOR1
        || lutsel==PIC_BDR_BDRCR_LUTSEL_COLOR2)
    {
        pic_handle->pic_bitreduct->BDRCR =
            BITREDUCT_BDRCR_LUTSEL_REPLACE_VAL(pic_handle->pic_bitreduct->BDRCR, lutsel);
    }
    else
    {
        error_print("ERR:lutsel=%d\n",lutsel);
        BUG();
    }
    // call lo level function to set pic_bdr_bdrcr for selected lut
}
EXPORT_SYMBOL(pic_bdr_select_lut);

////////////////////////////////////
// pic wdma channel
////////////////////////////////////

void pic_wdma_channel_dump_handle_regs(struct pic_handle_t *pic_handle, int channelnum)
{
    print("PIC WDMA Channel %d regs (pic_handle)\n", channelnum);
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channelnum);

    print("cfg=0x%X\n", pic_handle->pic_wdma_channel[channelnum]->cfg);
    print("status=0x%X\n", pic_handle->pic_wdma_channel[channelnum]->status);
    print("line_size=0x%X\n", pic_handle->pic_wdma_channel[channelnum]->line_size);
    print("int_en=0x%X\n", pic_handle->pic_wdma_channel[channelnum]->int_en);
    print("int_pend=0x%X\n", pic_handle->pic_wdma_channel[channelnum]->int_pend);
    // int_ack, int_fo, desc_write are write-only
    print("desc_read=0x%X\n", pic_handle->pic_wdma_channel[channelnum]->desc_read);
    print("desc_length=0x%X\n", pic_handle->pic_wdma_channel[channelnum]->xfer_length);
    print("xfer_addr=0x%X\n", pic_handle->pic_wdma_channel[channelnum]->xfer_addr);
    print("xfer_burst=0x%X\n", pic_handle->pic_wdma_channel[channelnum]->xfer_burst);
    // reset is write-only
    print("REV0=0x%X\n", pic_handle->pic_wdma_channel[channelnum]->REV0);
    print("REV1=0x%X\n", pic_handle->pic_wdma_channel[channelnum]->REV1);
}
EXPORT_SYMBOL(pic_wdma_channel_dump_handle_regs);

// if force == false, unforce requested irqs
// if force == true, force requested irqs
// if irqstruct == NULL, then force/unforce all irqs
void pic_force_output_dma_irqs(struct pic_handle_t *pic_handle,
                               struct pic_output_dma_interrupt_info *irqstruct,
                               bool force,
                               uint8_t channel)
{
    struct pic_output_dma_interrupt_info irq;
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);

    if (irqstruct == NULL)
    {
        pic_output_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
    // set/clear force irq bits
    pic_handle->pic_wdma_channel[channel]->int_fo =
        pic_output_convert_irqstruct_to_update_reg(irqstruct,
                                                   pic_handle->pic_wdma_channel[channel]->int_fo,
                                                   force);
}
EXPORT_SYMBOL(pic_force_output_dma_irqs);


void pic_enable_output_dma_channel_irqs(struct pic_handle_t *pic_handle,
                                        struct pic_output_dma_interrupt_info *irqstruct,
                                        bool enable,
                                        uint8_t channel)
{
    struct pic_output_dma_interrupt_info irq;
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);

    if (irqstruct == NULL)
    {
        pic_output_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
    // set/clear enable/irq bits
    pic_handle->pic_wdma_channel[channel]->int_en =
        pic_output_convert_irqstruct_to_update_reg(irqstruct,
                                                   pic_handle->pic_wdma_channel[channel]->int_en,
                                                   enable);
}
EXPORT_SYMBOL(pic_enable_output_dma_channel_irqs);


void pic_output_dma_channel_set_enable_dma(struct pic_handle_t *pic_handle,
                                           uint8_t enable, uint8_t channel)
{
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);

    pic_handle->pic_wdma_channel[channel]->cfg =    
        PIC_WDMA_DMA0_CFG_ENABLE_REPLACE_VAL(pic_handle->pic_wdma_channel[channel]->cfg, enable);

}
EXPORT_SYMBOL(pic_output_dma_channel_set_enable_dma);

bool pic_output_dma_channel_is_dma_enabled(struct pic_handle_t *pic_handle, uint8_t channel)
{
    uint8_t dma_enable;
    PIC_PTRCHECK_ARRY(pic_wdma_channel, channel, false);
    
    dma_enable = PIC_WDMA_DMA0_CFG_ENABLE_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->cfg);
    if (dma_enable == PIC_WDMA_CFG_DISABLE)
        return false;
    else
        return true;
}

void pic_output_dma_channel_enable_line_reverse(struct pic_handle_t *pic_handle,
                                                uint8_t enable, uint8_t channel)
{
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);

    pic_handle->pic_wdma_channel[channel]->cfg =    
        PIC_WDMA_DMA0_CFG_LINE_REV_REPLACE_VAL(pic_handle->pic_wdma_channel[channel]->cfg, enable);
}
EXPORT_SYMBOL(pic_output_dma_channel_enable_line_reverse);

void pic_output_dma_channel_set_input_data_width(struct pic_handle_t *pic_handle,
                                                 uint8_t in_width, uint8_t channel)
{
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);

    pic_handle->pic_wdma_channel[channel]->cfg =
        PIC_WDMA_DMA0_CFG_IN_WIDTH_REPLACE_VAL(pic_handle->pic_wdma_channel[channel]->cfg, in_width);
}
EXPORT_SYMBOL(pic_output_dma_channel_set_input_data_width);

uint8_t pic_output_dma_channel_get_input_data_width(struct pic_handle_t *pic_handle,
                                                    uint8_t channel)
{
    uint8_t in_width;
    PIC_PTRCHECK_ARRY(pic_wdma_channel, channel, 0);

    in_width = PIC_WDMA_DMA0_CFG_IN_WIDTH_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->cfg);
    return in_width;
}
EXPORT_SYMBOL(pic_output_dma_channel_get_input_data_width);

void pic_output_dma_channel_set_burst_len(struct pic_handle_t *pic_handle,
                                          uint8_t burst_len, uint8_t channel)
{
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);

    pic_handle->pic_wdma_channel[channel]->cfg =
        PIC_WDMA_DMA0_CFG_BURST_LEN_REPLACE_VAL(pic_handle->pic_wdma_channel[channel]->cfg, burst_len);
}
EXPORT_SYMBOL(pic_output_dma_channel_set_burst_len);

void pic_output_dma_channel_set_upper_half(struct pic_handle_t *pic_handle,
                                           uint8_t val, uint8_t channel)
{
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);

    pic_handle->pic_wdma_channel[channel]->cfg =
        PIC_WDMA_DMA0_CFG_UPPER_HALF_REPLACE_VAL(pic_handle->pic_wdma_channel[channel]->cfg, val);
}
EXPORT_SYMBOL(pic_output_dma_channel_set_upper_half);

void pic_output_dma_channel_get_status(struct pic_handle_t *pic_handle, struct pic_wdma_status_info *info, uint8_t channel)
{
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);

    info->bytes_waiting =
        PIC_WDMA_DMA0_STATUS_BYTESWAITING_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->force_burst_busy =
        PIC_WDMA_DMA0_STATUS_FORCE_BURST_BUSY_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->dma_paused =
        PIC_WDMA_DMA0_STATUS_DMAPAUSED_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->pause_after =
        PIC_WDMA_DMA0_STATUS_PAUSEAFTER_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->pause_before =
        PIC_WDMA_DMA0_STATUS_PAUSEBEFORE_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->desc_own =
        PIC_WDMA_DMA0_STATUS_DESC_OWN_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->desc_eoi =
        PIC_WDMA_DMA0_STATUS_DESC_EOI_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->desc_soi =
        PIC_WDMA_DMA0_STATUS_DESC_SOI_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->reverse =
        PIC_WDMA_DMA0_STATUS_REVERSE_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->softreset =
        PIC_WDMA_DMA0_STATUS_SOFTRESET_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->full_cbuf =
        PIC_WDMA_DMA0_STATUS_FULL_CBUF_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->empty_cbuf =
        PIC_WDMA_DMA0_STATUS_EMPTY_CBUF_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->full_dbuf =
        PIC_WDMA_DMA0_STATUS_FULL_DBUF_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->empty_dbuf =
        PIC_WDMA_DMA0_STATUS_EMPTY_DBUF_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->packer_empty =
        PIC_WDMA_DMA0_STATUS_PACKER_EMPTY_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->dma_busy =
        PIC_WDMA_DMA0_STATUS_DMA_BUSY_MASK_SHIFT(pic_handle->pic_wdma_channel[channel]->status);
    info->debug_array =
        pic_handle->pic_wdma_channel[channel]->status;
}
EXPORT_SYMBOL(pic_output_dma_channel_get_status);

void pic_output_dma_channel_get_trans_addr(struct pic_handle_t *pic_handle, uint32_t *trans_addr, uint8_t channel)
{
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);
    *trans_addr = pic_handle->pic_wdma_channel[channel]->xfer_addr;
}
EXPORT_SYMBOL(pic_output_dma_channel_get_trans_addr);


void pic_output_dma_channel_get_trans_len(struct pic_handle_t *pic_handle, uint32_t *trans_len, uint8_t channel)
{
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);
    *trans_len = pic_handle->pic_wdma_channel[channel]->xfer_length;
}
EXPORT_SYMBOL(pic_output_dma_channel_get_trans_len);


void pic_output_dma_channel_set_linesize(struct pic_handle_t *pic_handle, int line_size,
                                         uint8_t channel)
{
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);

    pic_handle->pic_wdma_channel[channel]->line_size = line_size;
}
EXPORT_SYMBOL(pic_output_dma_channel_set_linesize);


void pic_output_dma_channel_get_linesize(struct pic_handle_t *pic_handle, uint32_t *line_size,
                                         uint8_t channel)
{
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);

    *line_size = pic_handle->pic_wdma_channel[channel]->line_size;
}
EXPORT_SYMBOL(pic_output_dma_channel_get_linesize);

void pic_output_dma_channel_set_handshake_enable(struct pic_handle_t *pic_handle, bool enable,
                                                 uint8_t channel)
{
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);

    pic_handle->pic_wdma_channel[channel]->cfg =
        PIC_WDMA_DMA0_CFG_HANDSHAKEENABLE_REPLACE_VAL(pic_handle->pic_wdma_channel[channel]->cfg, enable);
}
EXPORT_SYMBOL(pic_output_dma_channel_set_handshake_enable);


////////////////////////////////////
// pic dump registers function
////////////////////////////////////

void pic_dump_handle_regs(struct pic_handle_t *pic_handle, uint8_t num_dma_channels)
{
    int i;
    
    print("Dumping the whole pic handle\n");
    pic_common_dump_handle_regs(pic_handle);
    pic_adc_dump_handle_regs(pic_handle);
    pic_bulbmon_dump_handle_regs(pic_handle);
    pic_lrmargin_dump_handle_regs(pic_handle, 0);
    pic_lrmargin_dump_handle_regs(pic_handle, 1);    
    pic_idma2d_dump_handle_regs(pic_handle);
    pic_pd_dump_handle_regs(pic_handle);
    pic_chipgap_dump_handle_regs(pic_handle);
    pic_hscale_dump_handle_regs(pic_handle);
    pic_wdma_dump_handle_regs(pic_handle);
    pic_bdr_dump_handle_regs(pic_handle);
    for (i=0;i<num_dma_channels;i++)
        pic_wdma_channel_dump_handle_regs(pic_handle, i);
}
EXPORT_SYMBOL(pic_dump_handle_regs);


//////////////////////////////////////////////////////////////
// pic callback functions
//////////////////////////////////////////////////////////////

void pic_common_register_callback(struct pic_handle_t *pic_handle, void *callbackfcn,
                                  void *callback_data)
{
    PIC_PTRCHECK_NR(pic_common);
    pic_handle->pic_common_callback = callbackfcn;
    pic_handle->pic_common_callback_data = callback_data;
}
EXPORT_SYMBOL(pic_common_register_callback);


void pic_idma2d_register_callback(struct pic_handle_t *pic_handle, void *callbackfcn,
                                  void *callback_data)
{
    PIC_PTRCHECK_NR(pic_idma_2d);
    pic_handle->pic_idma2d_callback = callbackfcn;
    pic_handle->pic_idma2d_callback_data = callback_data;
}
EXPORT_SYMBOL(pic_idma2d_register_callback);

void pic_odma_register_callback_dma_channel(struct pic_handle_t *pic_handle,
                                            void *callbackfcn,
                                            void *callback_data,
                                            uint8_t channel)
{
    PIC_PTRCHECK_ARRY_NR(pic_wdma_channel, channel);
    pic_handle->pic_odma_callback[channel] = callbackfcn;
    pic_handle->pic_odma_callback_data[channel] = callback_data;
}
EXPORT_SYMBOL(pic_odma_register_callback_dma_channel);

///////////////////////////////////////
// get defaults
///////////////////////////////////////
// all default values from the 88PA6270RA-Registers doc - Aug 28, 2014

static void get_pic_common_default(struct pic_handle_t *pic_handle)
{

    pic_handle->pic_common->PCR = 0;
    pic_handle->pic_common->IENR = 0;
    pic_handle->pic_common->IPR = 0;
    pic_handle->pic_common->IFR = 0;    
    pic_handle->pic_common->REV0 = PIC_COMMON_MAJ_REV0;
    pic_handle->pic_common->REV1 = 0;
}

static void get_pic_adcnorm_default(struct pic_handle_t *pic_handle)
{
    uint32_t regval;
    
    regval = 0;
    regval = ADCNORM_ANR_BYPASS_REPLACE_VAL(regval, 1);
    regval = ADCNORM_ANR_ADCBITS_REPLACE_VAL(regval, 0xD);
    pic_handle->pic_adcnorm->ANR = regval; // 0x10D;

    // Read the revision from the registers
    pic_handle->pic_adcnorm->REV0 = PIC_ADCNORM_MAJ_REV0;
    pic_handle->pic_adcnorm->REV1 = 0;
}

static void get_pic_bulbmon_default(struct pic_handle_t *pic_handle)
{
    pic_handle->pic_bulbmon->BMCR = BULBMON_BMCR_BYPASS_REPLACE_VAL(0, 1);
    pic_handle->pic_bulbmon->BMR0 = 0;
    pic_handle->pic_bulbmon->BMR1 = 0;    
    pic_handle->pic_bulbmon->BMR2 = 0;    
    pic_handle->pic_bulbmon->REV0 = PIC_BULBMON_MAJ_REV0;
    pic_handle->pic_bulbmon->REV1 = 0;
}

static void get_pic_lrmargin_default(struct pic_handle_t *pic_handle)
{
    uint32_t regval;
    int i;

    regval = 0;
    
    regval = LRMARGIN0_LRCR_BYPASS_REPLACE_VAL(regval, 1);

    for (i=0;i<MAX_NUM_LRMARGIN_BLOCKS;i++)
    {
        pic_handle->pic_lrmargin[i]->LRCR = regval; // 0x10000000;
        pic_handle->pic_lrmargin[i]->LRMR0 = 0;
        pic_handle->pic_lrmargin[i]->LRMR1 = 0;
        pic_handle->pic_lrmargin[i]->LRMR2 = 0;    
        pic_handle->pic_lrmargin[i]->REV0 = PIC_LRMARGIN_MAJ_REV0;
        pic_handle->pic_lrmargin[i]->REV1 = 0;
    }
}

static void get_pic_pd_default(struct pic_handle_t *pic_handle)
{
    pic_handle->pic_pd->RLSCR = 0;
    pic_handle->pic_pd->LSVCR = 0;    
    pic_handle->pic_pd->LMPCR0 = PRNUDSNU_LMPCR0_LEFTCORR_REPLACE_VAL(0, 200);
    pic_handle->pic_pd->LMPCR1 = PRNUDSNU_LMPCR1_LEFTCORR_REPLACE_VAL(0, 200);
    pic_handle->pic_pd->LMPCR2 = PRNUDSNU_LMPCR2_LEFTCORR_REPLACE_VAL(0, 200);
    pic_handle->pic_pd->RSVCR = 0;
    pic_handle->pic_pd->RSPCR0 = PRNUDSNU_RSPCR0_RIGHTSTART_REPLACE_VAL(0, 200);
    pic_handle->pic_pd->RSPCR1 = PRNUDSNU_RSPCR1_RIGHTSTART_REPLACE_VAL(0, 200);
    pic_handle->pic_pd->RSPCR2 = PRNUDSNU_RSPCR2_RIGHTSTART_REPLACE_VAL(0, 200);
    pic_handle->pic_pd->PDCR1 = 0;
    pic_handle->pic_pd->PDCR2 = PRNUDSNU_PDCR2_BYPASS_REPLACE_VAL(0, 1);    
    pic_handle->pic_pd->PONCR = PRNUDSNU_PONCR_PONNORM_REPLACE_VAL(0, 38);
    pic_handle->pic_pd->QUAD = 0;
    pic_handle->pic_pd->DOFF0 = 0;
    pic_handle->pic_pd->DOFF1 = 0;
    pic_handle->pic_pd->DOFF2 = 0;    
    pic_handle->pic_pd->POFF0 = PRNUDSNU_POFF0_OFFSET_REPLACE_VAL(0, 0x8000);
    pic_handle->pic_pd->POFF1 = PRNUDSNU_POFF1_OFFSET_REPLACE_VAL(0, 0x8000);
    pic_handle->pic_pd->POFF2 = PRNUDSNU_POFF2_OFFSET_REPLACE_VAL(0, 0x8000);
    pic_handle->pic_pd->PMULT0 = 0;
    pic_handle->pic_pd->PMULT1 = 0;
    pic_handle->pic_pd->PMULT2 = 0;    
    pic_handle->pic_pd->REV0 = PIC_PD_MAJ_REV0;
    pic_handle->pic_pd->REV1 = 0;
}

static void get_pic_hscale_default(struct pic_handle_t *pic_handle)
{
    pic_handle->pic_hscale->HSCR = HSCALE_HSCR_BYPASS_REPLACE_VAL(0, 1);
    pic_handle->pic_hscale->HSSR = HSCALE_HSSR_SCALE_REPLACE_VAL(0, 0x10000);
    pic_handle->pic_hscale->REV0 = PIC_HSCALE_MAJ_REV0;
    pic_handle->pic_hscale->REV1 = 0;
}

static void get_pic_chipgap_default(struct pic_handle_t *pic_handle)
{
    pic_handle->pic_chipgap->CFG = NEWMAN_CFG_BYPASS_REPLACE_VAL(0, 1);
    pic_handle->pic_chipgap->LOCAT = 0;
    pic_handle->pic_chipgap->INSERT0 = 0;
    pic_handle->pic_chipgap->INSERT1 = 0;
    pic_handle->pic_chipgap->INSERT2 = 0;
    pic_handle->pic_chipgap->INSERT3 = 0;
    pic_handle->pic_chipgap->REV0 = PIC_CHIPGAP_MAJ_REV0;
    pic_handle->pic_chipgap->REV1 = 0;
}

static void get_pic_idma2d_default(struct pic_handle_t *pic_handle)
{
    uint32_t regval;

    pic_handle->pic_idma_2d->cfg = 0;
    regval = 0;
    regval = IDMA_2D_STATUS_SOFTRESET_REPLACE_VAL(regval, 1);
    regval = IDMA_2D_STATUS_EMPTY_CBUF_REPLACE_VAL(regval, 1);
    regval = IDMA_2D_STATUS_EMPTY_DBUF_REPLACE_VAL(regval, 1);
    regval = IDMA_2D_STATUS_PACKER_EMPTY_REPLACE_VAL(regval, 1);

    pic_handle->pic_idma_2d->status = regval; // 0x56;
    pic_handle->pic_idma_2d->line_width = 0;
    pic_handle->pic_idma_2d->int_en = 0;
    pic_handle->pic_idma_2d->int_st = 0;
    pic_handle->pic_idma_2d->int_cl = 0;
    pic_handle->pic_idma_2d->int_fo = 0;
    pic_handle->pic_idma_2d->desc_write = 0;
    pic_handle->pic_idma_2d->desc_read = 0;
    pic_handle->pic_idma_2d->xfer_length = 0;
    pic_handle->pic_idma_2d->xfer_addr = 0;
    pic_handle->pic_idma_2d->ctrl_word = 0;
    pic_handle->pic_idma_2d->reset = 0;
    pic_handle->pic_idma_2d->REV0 = PIC_IDMA2D_MAJ_REV0;
    pic_handle->pic_idma_2d->REV1 = 0;
}

static void get_pic_wdma_default(struct pic_handle_t *pic_handle)
{
    uint32_t regval;
    
    pic_handle->pic_wdma->DWOCR = PIC_WDMA_DWOCR_DISABLEMODE_REPLACE_VAL(0, 2);
    regval = PIC_WDMA_DWRTBL_COLOR2DMACRV_REPLACE_VAL(0, 4);
    regval = PIC_WDMA_DWRTBL_COLOR1DMACRV_REPLACE_VAL(regval, 2);
    pic_handle->pic_wdma->DWRTBL = regval; // 0x000420;
    pic_handle->pic_wdma->DWCIS = 0;
    pic_handle->pic_wdma->REV0 = PIC_WDMA_MAJ_REV0;
    pic_handle->pic_wdma->REV1 = 0;
}

static void get_pic_wdma_channel_default(struct pic_handle_t *pic_handle)
{
    uint32_t regval;
    uint8_t chan;

    regval = PIC_WDMA_DMA0_CFG_UPPER_HALF_REPLACE_VAL(0, 1);
    regval = PIC_WDMA_DMA0_CFG_MSB_IN_REPLACE_VAL(regval, 1);
    
    for (chan=0;chan<MAX_OUTPUT_DMAS;chan++)
    {
        pic_handle->pic_wdma_channel[chan]->cfg = regval; // 0x00060000;
        pic_handle->pic_wdma_channel[chan]->status = 0;
        pic_handle->pic_wdma_channel[chan]->line_size = 0;
        pic_handle->pic_wdma_channel[chan]->int_en = 0;
        pic_handle->pic_wdma_channel[chan]->int_pend = 0;
        pic_handle->pic_wdma_channel[chan]->int_ack = 0;
        pic_handle->pic_wdma_channel[chan]->int_fo = 0;
        pic_handle->pic_wdma_channel[chan]->desc_write = 0;
        pic_handle->pic_wdma_channel[chan]->desc_read = 0;
        pic_handle->pic_wdma_channel[chan]->xfer_length = 0;
        pic_handle->pic_wdma_channel[chan]->xfer_addr = 0;
        pic_handle->pic_wdma_channel[chan]->xfer_burst = 0;
        pic_handle->pic_wdma_channel[chan]->reset = 0;    
        // verify that we have the matching hardware version
        pic_handle->pic_wdma_channel[chan]->REV0 = PIC_WDMA_CHANNEL_MAJ_REV0;
        pic_handle->pic_wdma_channel[chan]->REV1 = 0;
    }
}

static void get_pic_bdr_default(struct pic_handle_t *pic_handle)
{
    int i;
    
    // write the lut (0-addr) - see registers guide BDR Look Up Table
    for (i=0; i<(PIC_BDR_LUT_SIZE-1); i++)
        pic_handle->pic_bitreduct->BDRLUT[i] = 0;
    // write the final entry (addr+1)
    pic_handle->pic_bitreduct->BDRLUTR = 0;

    pic_handle->pic_bitreduct->BDRCR = BITREDUCT_BDRCR_BYPASS_REPLACE_VAL(0, 1);
    pic_handle->pic_bitreduct->REV0 = PIC_BDR_MAJ_REV0;
    pic_handle->pic_bitreduct->REV1 = 0;
}

struct pic_handle_t *pic_do_get_default(struct pic_handle_t *pic_handle)
{
    int i;
    
    get_pic_common_default(pic_handle);
    get_pic_adcnorm_default(pic_handle);
    get_pic_bulbmon_default(pic_handle);
    get_pic_lrmargin_default(pic_handle);
    get_pic_pd_default(pic_handle);
    get_pic_hscale_default(pic_handle);
    get_pic_chipgap_default(pic_handle);
    get_pic_idma2d_default(pic_handle);
    get_pic_wdma_default(pic_handle);
    get_pic_wdma_channel_default(pic_handle);
    get_pic_bdr_default(pic_handle);    
    // as new subblocks are added, create their get_default function
    // and add the call here

    // init all of the interrupt callbacks and callback data
    // if a new callback is added, be sure to add space in the
    // pic_handle for it (pic_create_new_default_handle), as well
    // as adding it to the clone function.
    pic_handle->pic_common_callback = NULL;
    pic_handle->pic_common_callback_data = NULL;
    for (i=0;i<MAX_OUTPUT_DMAS;i++)
    {
        pic_handle->pic_odma_callback[i] = NULL;
        pic_handle->pic_odma_callback_data[i] = NULL;
    }
    pic_handle->pic_idma2d_callback = NULL;
    pic_handle->pic_idma2d_callback_data = NULL;    
    return pic_handle;
}
EXPORT_SYMBOL(pic_do_get_default);

// This function allocates the pic handle and returns a pointer to it
// When the caller is done with the pic handle in kernel space,
// calling pic_do_free_handle() must be done to avoid a memory leak.
// 
// For user space, just freeing the memory is all that needs to be done

struct pic_handle_t *pic_create_new_default_handle(void)
{
    struct pic_handle_t *pic_handle;
    int pic_handle_size;
    uint32_t *subblock_size_table;
    enum pic_full_subblock_list i;

    // create a table of the size of every subblock
    subblock_size_table = pic_get_subblock_sizes();

    // We need to compute the size of the pic_handle to allocate it.
    // Start out with the size of the pic_handle structure
    pic_handle_size = sizeof(struct pic_handle_t);

    // Now add the size of the shadow registers
    // The shadow registers are in the pic_handle
    // (any non-existent ones will have a 0 entry)
    for (i=common_index; i<max_subblock_pic; i++)
        pic_handle_size += subblock_size_table[i];
    
    pic_handle = allocate_memory(pic_handle_size, GFP_KERNEL);

    // write the number of bytes in the whole pic_handle table to the pic_handle
    pic_handle->pic_handle_size = pic_handle_size;
    
    // copy the subblock_size_table into the pic_handle - it's needed
    // to create the pointer list (required when moving to kernel space)
    for (i=common_index; i<max_subblock_pic; i++)
        pic_handle->subblock_size_table[i] = subblock_size_table[i];

    pic_fixup_shadow_pointers(pic_handle);

    free_memory(subblock_size_table);

    pic_do_get_default(pic_handle); // now put the default values in
    
    return pic_handle;
}
EXPORT_SYMBOL(pic_create_new_default_handle);

MODULE_LICENSE("GPL");
