/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/irqreturn.h> // needed for irqreturn_t and IRQ_HANDLED
#include <linux/spinlock.h>  // for spinlock_t
#include <linux/io.h>        // for ioread/write32 (pic Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/delay.h>     // for msleep

#include "PIC_regheaders.h"
#include "pic_full_subblock_list.h"  // needed for detailed pic_handle
#include "pic_constants.h"
#include "pic_handle.h"
#include "pic_if.h"
#include "pic_data.h"
#include "pic_driver.h"
#include "pic_common.h"
#include "pic_common_if.h"
#include "pic_driverlib_if.h"


//  NOTE THERE IS NO FUNCTION clear_pic_common_int - since the interrupt
//  register is READ only.  Each bit must be cleared at the source (WDMA, etc)

// enable each field that is set
void enable_pic_common_irqs(picCommonData *device_data,
                            struct pic_common_ints *irqstruct)
{
    unsigned long flags;
    uint32_t reg;
    struct pic_common_ints irq;

    if (irqstruct == NULL)
    {
        pic_common_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
    // enable interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = picCommonRead(IENR);
    reg = pic_common_convert_irqstruct_to_update_reg(irqstruct, reg, true);
    picCommonWrite(IENR, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

// disable each field that is set to true
void disable_pic_common_irqs(picCommonData *device_data,
                             struct pic_common_ints *irqstruct)
{
    unsigned long flags;
    uint32_t reg;
    struct pic_common_ints irq;

    if (irqstruct == NULL)
    {
        pic_common_set_all_irqstruct(&irq, true);
        irqstruct = &irq;
    }
    // enable interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = picCommonRead(IENR);
    reg = pic_common_convert_irqstruct_to_update_reg(irqstruct, reg, false);
    picCommonWrite(IENR, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

/*************************************
 * pic_platform_irq: interrupt handler
 * @irq:  irq number
 * @pdev: interrupt source
 *
 * This function returns IRQ_HANDLED if the IRQ has been handled
 * This is an ISR don't trace, use attribute interface instead
 **************************************/
///  NOTE - This function runs in interrupt context - no long operations allowed
irqreturn_t pic_platform_irq(int irq, void *pdev)
{
    uint32_t reg, forcereg, intval;
    picCommonData *device_data;
    struct pic_common_ints irqstruct;
    unsigned long flags;  // for IRQ protection macro

    device_data = pdev;
    // check for the test register - f/w can use to cause an interrupt - clear it
    PROTECT_INTREG_ACCESS_IRQ;
    forcereg = picCommonRead(IFR);
    if (forcereg != 0)
    {
        debug_print("Clearing the IFR=0x%X\n",forcereg);
        picCommonWrite(IFR,0);
    }
    
    reg = picCommonRead(IPR);
    UNPROTECT_INTREG_ACCESS_IRQ;
    // print("IRQ:IPR=0x%X\n", reg);

    reg |= forcereg; // add in any f/w force interrupts
    
    // Check to see if Write DMA is causing the interrupt
    intval = PIC_WDMA_INT_MASK(reg);
    if (intval != 0)
    {
        // clear any pic WDMA interrupts
        pic_output_dma_channel_handle_irqs(device_data->instance);
    }

    // Check to see if PD IDMA2D is causing the interrupt
    intval = PIC_IDMA2D_INT_MASK(reg);
    if (intval != 0)
    {
       pic_pd_idma2d_handle_irqs(device_data->instance);
    }

    if (reg != 0)
    {
        pic_common_convert_uintarray_to_irqstruct(&irqstruct, reg);
        // now that we have a list of interrupts that were set,
        // execute the registered callback function
        if (device_data->interrupt_callback != NULL)
        {
            debug_print("calling callback with struct (reg=0x%X)\n",reg);
            device_data->interrupt_callback(&irqstruct, device_data->interrupt_callback_data);
        }
    }
    return IRQ_HANDLED;
}

// RevA had no Debug field, so shift down references to the REV* fields
uint32_t rev0_value(picCommonData *device_data)
{
    if (device_data->asicrev != ASICREVA)
    {
        return(picCommonRead(REV0));
    }
    else
    {
        return(picCommonRead(Debug));
    }
}

uint32_t rev1_value(picCommonData *device_data)
{
    if (device_data->asicrev != ASICREVA)
    {
        return(picCommonRead(REV1));
    }
    else
    {
        return(picCommonRead(REV0));
    }
}   

void dump_pic_common_regs(picCommonData *device_data)
{
    print("print the PIC common regs for pic %d\n", device_data->instance);
    print("PCR=0x%X\n", picCommonRead(PCR));
    print("IENR=0x%X\n", picCommonRead(IENR));
    print("IPR=0x%X\n", picCommonRead(IPR));
    print("IFR=0x%X\n", picCommonRead(IFR));
    print("Debug=0x%X\n", picCommonRead(Debug));
    print("REV0=0x%X\n", rev0_value(device_data));
    print("REV1=0x%X\n", rev1_value(device_data));
}
EXPORT_SYMBOL(dump_pic_common_regs);

static void pic_configure(picCommonData *device_data, struct pic_handle_t *pic_handle)
{
    device_data->interrupt_callback = pic_handle->pic_common_callback;
    device_data->interrupt_callback_data = pic_handle->pic_common_callback_data;
    
    PROTECT_REG_ACCESS;
    picCommonWrite(PCR, pic_handle->pic_common->PCR);
    picCommonWrite(IENR, pic_handle->pic_common->IENR);
    // IPR is read only
    picCommonWrite(IFR, pic_handle->pic_common->IFR);
    // Debug, REV0 and REV1 are read only
    UNPROTECT_REG_ACCESS;
}

static void pic_get_current(picCommonData *device_data, struct pic_handle_t *pic_handle)
{
    PROTECT_REG_ACCESS;
    pic_handle->pic_common->PCR = picCommonRead(PCR);
    pic_handle->pic_common->IENR = picCommonRead(IENR);
    pic_handle->pic_common->IPR = picCommonRead(IPR);
    pic_handle->pic_common->IFR = picCommonRead(IFR);
    pic_handle->pic_common->Debug = picCommonRead(Debug);
    pic_handle->pic_common->REV0 = rev0_value(device_data);
    pic_handle->pic_common->REV1 = rev1_value(device_data);
    
    UNPROTECT_REG_ACCESS;
}

// set, then clear the reset bit
static void pic_reset(picCommonData *device_data)
{
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    reg = picCommonRead(PCR);
    reg = PIC_PCR_SOFTRESET_REPLACE_VAL(reg, 1);
    picCommonWrite(PCR, reg);
    UNPROTECT_REG_ACCESS;
    
    msleep(5); // wait for the reset to occur
    
    PROTECT_REG_ACCESS;
    reg = picCommonRead(PCR);
    reg = PIC_PCR_SOFTRESET_REPLACE_VAL(reg, 0);
    picCommonWrite(PCR, reg);
    UNPROTECT_REG_ACCESS;
}

static void pic_do_not_reset(picCommonData *device_data)
{
    // pic soft reset of pic 0 affects pic 1
    // allow user to turn off the soft reset of pic common
}

// disable each field that is set to true
void disable_pic_common_soft_reset(picCommonData *device_data)
{
    picCommonDeviceHandle *pic_device_handle;

    // unregister with the parent
    pic_device_handle = unregister_pic_subblock(common, device_data->instance, 0);
    pic_device_handle->fcn_tbl->pic_reset = pic_do_not_reset;
    register_pic_subblock(common, pic_device_handle, device_data->instance, 0);
}

static int pic_revcheck(picCommonData *device_data, struct pic_handle_t *pic_handle)
{
    uint32_t rev0;
    uint32_t pich_rev;

    rev0 = PIC_REV0_MAJ_MASK_SHIFT(rev0_value(device_data));
        
    pich_rev = PIC_REV0_MAJ_MASK_SHIFT(pic_handle->pic_common->REV0);

    // pic_common driver code should work with older ASIC revs
    if (rev0 > pich_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pich_rev);
        return -1;
    }
    else
        return 0;
}

struct pic_common_function_struct pic_common_functions =
{
    .pic_reset = pic_reset,
    .pic_configure = pic_configure,
    .pic_get_current = pic_get_current,
    .pic_revcheck = pic_revcheck,
    
    .pic_dump_regs = dump_pic_common_regs,

    .enable_pic_common_irqs = enable_pic_common_irqs,
    .disable_pic_common_irqs = disable_pic_common_irqs,
    .disable_pic_common_soft_reset = disable_pic_common_soft_reset,
};

void pic_common_init(picCommonData *device_data)
{
    picCommonDeviceHandle *pic_device_handle;

    pic_device_handle = allocate_memory(sizeof(picCommonDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pic_device_handle->fcn_tbl = &pic_common_functions;
    pic_device_handle->device_data = device_data;
    register_pic_subblock(common, pic_device_handle, device_data->instance, 0);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses reg_spinlock
    spin_lock_init(&(pic_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pic_device_handle->device_data->reg_spinlock));
    pic_device_handle->device_data->interrupt_callback = NULL;
    pic_device_handle->device_data->interrupt_callback_data = NULL;
}
EXPORT_SYMBOL(pic_common_init);

void pic_common_exit(picCommonData *device_data)
{
    picCommonDeviceHandle *pic_device_handle;
    
    // unregister with the parent
    pic_device_handle = unregister_pic_subblock(common, device_data->instance, 0);
    free_memory(pic_device_handle);
}
EXPORT_SYMBOL(pic_common_exit);
