/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/spinlock.h>  // for spinlock_t, print, BUG_ON, and memset
#include <linux/io.h>        // for ioread/write32 (pic Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL

// Note: When looking at the ASIC headers, the chip gap hardware is also known
//       as the newman block!
#include "PIC_regheaders.h"
#include "pic_full_subblock_list.h"
#include "pic_constants.h"
#include "pic_handle.h"
#include "pic_if.h"
#include "pic_driverlib_if.h"
#include "pic_data.h"
#include "pic_driver.h"
#include "pic_chipgap.h"
#include "pic_chipgap_if.h"
#include "pic_driverlib_if.h"

void dump_chipgap_regs(picChipgapData *device_data)
{
    print("Chipgap dump for pic=%d\n", device_data->instance);

    print( "    CFG=0x%08x   LOCAT=0x%08x    REV0=0x%08x    REV1=0x%08x\n", 
            chipgapRead(CFG),  chipgapRead(LOCAT),
            chipgapRead(REV0), chipgapRead(REV1) );
    print( "INSERT0=0x%08x INSERT1=0x%08x INSERT2=0x%08x INSERT3=0x%08x\n", 
            chipgapRead(INSERT0), chipgapRead(INSERT1),
            chipgapRead(INSERT2), chipgapRead(INSERT3) );
}

static void pic_configure(picChipgapData *device_data, struct pic_handle_t *pic_handle)
{
    PROTECT_REG_ACCESS;
    chipgapWrite(CFG, pic_handle->pic_chipgap->CFG);
    chipgapWrite(LOCAT, pic_handle->pic_chipgap->LOCAT);
    chipgapWrite(INSERT0, pic_handle->pic_chipgap->INSERT0);
    chipgapWrite(INSERT1, pic_handle->pic_chipgap->INSERT1);
    chipgapWrite(INSERT2, pic_handle->pic_chipgap->INSERT2);
    chipgapWrite(INSERT3, pic_handle->pic_chipgap->INSERT3);
    // REV0, REV1 are read-only
    UNPROTECT_REG_ACCESS;
}

static void pic_get_current(picChipgapData *device_data, struct pic_handle_t *pic_handle)
{
    PROTECT_REG_ACCESS;
    pic_handle->pic_chipgap->CFG = chipgapRead(CFG);
    pic_handle->pic_chipgap->LOCAT = chipgapRead(LOCAT);
    pic_handle->pic_chipgap->INSERT0 = chipgapRead(INSERT0);
    pic_handle->pic_chipgap->INSERT1 = chipgapRead(INSERT1);
    pic_handle->pic_chipgap->INSERT2 = chipgapRead(INSERT2);
    pic_handle->pic_chipgap->INSERT3 = chipgapRead(INSERT3);
    pic_handle->pic_chipgap->REV0 = chipgapRead(REV0);
    pic_handle->pic_chipgap->REV1 = chipgapRead(REV1);
    UNPROTECT_REG_ACCESS;
}

static void pic_reset(picChipgapData *device_data)
{
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    reg = chipgapRead(CFG);
    reg = NEWMAN_CFG_BYPASS_REPLACE_VAL(reg, 1); // bypass to do a reset
    chipgapWrite(CFG, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static int pic_revcheck(picChipgapData *device_data, struct pic_handle_t *pic_handle)
{
    uint32_t rev0;
    uint32_t pich_rev;

    rev0 = NEWMAN_REV0_MAJ_MASK_SHIFT(chipgapRead(REV0));
    pich_rev = NEWMAN_REV0_MAJ_MASK_SHIFT(pic_handle->pic_chipgap->REV0);

    if (rev0 != pich_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pich_rev);
        return -1;
    }
    else
        return 0;
}

struct pic_chipgap_function_struct pic_chipgap_functions =
{
    .pic_reset       = pic_reset,
    .pic_configure   = pic_configure,
    .pic_get_current = pic_get_current,
    .pic_revcheck    = pic_revcheck,
    
    .pic_dump_regs   = dump_chipgap_regs,
};

void pic_chipgap_init(picChipgapData *device_data)
{
    chipgapDeviceHandle *pic_device_handle;

    pic_device_handle = allocate_memory(sizeof(chipgapDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pic_device_handle->fcn_tbl = &pic_chipgap_functions;
    pic_device_handle->device_data = device_data;
    register_pic_subblock(chipgap, pic_device_handle, device_data->instance, device_data->submodinstance);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pic_device_handle->device_data->reg_spinlock));
    // chipgap has no interrupts associated with it, so no callback
    pic_device_handle->device_data->interrupt_callback = NULL;
    pic_device_handle->device_data->interrupt_callback_data = NULL;
    debug_print("PIC Chipgap Driver Loaded, pic=%d\n", device_data->instance);
}
EXPORT_SYMBOL(pic_chipgap_init);


void pic_chipgap_exit(picChipgapData *device_data)
{
    chipgapDeviceHandle *pic_device_handle;

    // unregister with the parent
    pic_device_handle = unregister_pic_subblock(chipgap, device_data->instance,
                                                device_data->submodinstance);
    free_memory(pic_device_handle);
}
EXPORT_SYMBOL(pic_chipgap_exit);

