/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/spinlock.h>  // for spinlock_t, print, BUG_ON, and memset
#include <linux/io.h>        // for ioread/write32 (pie Read/Write macros)
#include <linux/export.h>    // for EXPORT_SYMBOL

#include "PIC_regheaders.h"
#include "pic_full_subblock_list.h"
#include "pic_constants.h"
#include "pic_handle.h"
#include "pic_if.h"
#include "pic_data.h"
#include "pic_driver.h"
#include "pic_bdr.h"
#include "pic_bdr_if.h"
#include "pic_driverlib_if.h"

void dump_bdr_regs(picBdrData *device_data)
{
    int i;
    uint32_t reg_LUT;
    
    print("BDR dump for pic=%d\n", device_data->instance);
    
    print("BDRCR=0x%08x\n", bdrRead(BDRCR));
    print("BDRLUT\n");
    for (i=0; i<PIC_BDR_LUT_SIZE-1; i++)
    {
        if (i%16 == 0) print("\n%d-%d: ", i, i+15);
        reg_LUT = bdrReadArray(BDRLUT, i);
        print("0x%X ", BITREDUCT_BDRLUT_VAL_MASK_SHIFT(reg_LUT));
    }
    print("BDRLUTR=0x%08x\n", bdrRead(BDRLUTR));
    print("\n");
}


static void pic_configure(picBdrData *device_data, struct pic_handle_t *pic_handle)
{
    uint32_t reg;
    uint8_t use_luts;
    int i, j, numluts;

    // Not bypassing means using the LUTs - we don't spend time writing the LUTS
    // if the pic_handle says bypass.
    if (BITREDUCT_BDRCR_BYPASS_MASK_SHIFT(pic_handle->pic_bitreduct->BDRCR) == 0)
        use_luts = 1;
    else
        use_luts = 0;
    
    PROTECT_REG_ACCESS;
    
    if (use_luts)
    {
        // mono (1 LUT) or color (3 LUTs)?  If the h/w doesn't support 3 independent ones, it will
        // map to the same address
        numluts = BITREDUCT_BDRCR_NUMLUTS_MASK_SHIFT(pic_handle->pic_bitreduct->BDRCR) ? 3 : 1;
        for (i=0; i<numluts; i++)
        {
            // enable cpu write
            reg = BITREDUCT_BDRCR_CPU_REPLACE_VAL(pic_handle->pic_bitreduct->BDRCR, 1);
            reg = BITREDUCT_BDRCR_LUTSEL_REPLACE_VAL(reg, i);
            bdrWrite(BDRCR, reg);
            // write the lut (0-addr) - see registers guide BDR Look Up Table
            for (j=0; j<PIC_BDR_LUT_SIZE-1; j++)
            {
                bdrWriteArray(BDRLUT, pic_handle->pic_bitreduct->BDRLUT[j], j);
            }
            // now write the final entry (addr+1)
            bdrWrite(BDRLUTR, pic_handle->pic_bitreduct->BDRLUTR);
            // Outside the loop we will write the BDRCR as requested in the pic handle
        }
    }
    // write the BDRCR, as requested by caller (they should have set the bypass and CPU properly)
    bdrWrite(BDRCR, pic_handle->pic_bitreduct->BDRCR);
    
    UNPROTECT_REG_ACCESS;
}

static void pic_get_current(picBdrData *device_data, struct pic_handle_t *pic_handle)
{
    int i;
    PROTECT_REG_ACCESS;
    
    pic_handle->pic_bitreduct->BDRCR = bdrRead(BDRCR);

    for (i=0; i<PIC_BDR_LUT_SIZE-1; i++)
    {
        pic_handle->pic_bitreduct->BDRLUT[i] = bdrReadArray(BDRLUT, i);
    }

    pic_handle->pic_bitreduct->BDRLUTR = bdrRead(BDRLUTR);
    
    pic_handle->pic_bitreduct->REV0 = bdrRead(REV0);
    pic_handle->pic_bitreduct->REV1 = bdrRead(REV1);
    
    UNPROTECT_REG_ACCESS;
}

static void pic_reset(picBdrData *device_data)
{
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    reg = bdrRead(BDRCR);
    reg = BITREDUCT_BDRCR_BYPASS_REPLACE_VAL(reg, 1); // bypass to do a reset
    bdrWrite(BDRCR, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static int pic_revcheck(picBdrData *device_data, struct pic_handle_t *pic_handle)
{
    uint32_t rev0;
    uint32_t pich_rev;

    rev0 = BITREDUCT_REV0_MAJ_MASK_SHIFT(bdrRead(REV0));
    pich_rev = BITREDUCT_REV0_MAJ_MASK_SHIFT(pic_handle->pic_bitreduct->REV0);

    if (rev0 != pich_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pich_rev);
        return -1;
    }
    else
        return 0;
}

struct pic_bdr_function_struct pic_bdr_functions =
{
    .pic_reset       = pic_reset,
    .pic_configure   = pic_configure,
    .pic_get_current = pic_get_current,
    .pic_revcheck    = pic_revcheck,

    .pic_dump_regs   = dump_bdr_regs,
};

void pic_bdr_init(picBdrData *device_data)
{
    bdrDeviceHandle *pic_device_handle;

    pic_device_handle = allocate_memory(sizeof(bdrDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pic_device_handle->fcn_tbl = &pic_bdr_functions;
    pic_device_handle->device_data = device_data;
    register_pic_subblock(bdr, pic_device_handle, device_data->instance, device_data->submodinstance);
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pic_device_handle->device_data->reg_spinlock));
    // bdr has no interrupts associated with it, so no callback
    pic_device_handle->device_data->interrupt_callback = NULL;
    pic_device_handle->device_data->interrupt_callback_data = NULL;
    debug_print("PIC bdr Driver Loaded, pic=%d\n", device_data->instance);
}
EXPORT_SYMBOL(pic_bdr_init);

void pic_bdr_exit(picBdrData *device_data)
{
    bdrDeviceHandle *pic_device_handle;

    // unregister with the parent
    pic_device_handle = unregister_pic_subblock(bdr, device_data->instance, device_data->submodinstance);
    free_memory(pic_device_handle);
}
EXPORT_SYMBOL(pic_bdr_exit);

