/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/spinlock.h>   // for spinlock_t
#include <linux/io.h> // for ioread/write32 (pic Read/Write macros)
#include <linux/export.h> // for EXPORT_SYMBOL

#include "PIC_regheaders.h"
#include "pic_full_subblock_list.h"  // for detailed pic_handle
#include "pic_constants.h"
#include "pic_handle.h"
#include "pic_if.h"
#include "pic_data.h"
#include "pic_driver.h"
#include "pic_adcnorm.h"
#include "pic_adcnorm_if.h"
#include "pic_driverlib_if.h"   // for print macros

void dump_pic_adcnorm_regs(picAdcnormData *device_data)
{
    print("pic_adcnorm.c dump_regs\n");
    print("   ANR=0x%X\n", picAdcnormRead(ANR));
    print("   REV0=0x%X\n", picAdcnormRead(REV0));
    print("   REV1=0x%X\n", picAdcnormRead(REV1));
}
EXPORT_SYMBOL(dump_pic_adcnorm_regs);

static void pic_configure(picAdcnormData *device_data, struct pic_handle_t *pic_handle)
{
    PROTECT_REG_ACCESS;
    picAdcnormWrite(ANR, pic_handle->pic_adcnorm->ANR);
    // REV0 and REV1 are read-only
    UNPROTECT_REG_ACCESS;
}

static void pic_get_current(picAdcnormData *device_data, struct pic_handle_t *pic_handle)
{
    PROTECT_REG_ACCESS;
    pic_handle->pic_adcnorm->ANR = picAdcnormRead(ANR);
    pic_handle->pic_adcnorm->REV0 = picAdcnormRead(REV0);
    pic_handle->pic_adcnorm->REV1 = picAdcnormRead(REV1);
    UNPROTECT_REG_ACCESS;
}

static void pic_reset(picAdcnormData *device_data)
{
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    reg = picAdcnormRead(ANR);
    reg = ADCNORM_ANR_BYPASS_REPLACE_VAL(reg, 1);  // bypass to do a reset
    picAdcnormWrite(ANR, reg);
    // leaving in bypass - if someone wants this subblock, they need to
    // enable it (take out of bypass)
    UNPROTECT_REG_ACCESS;
}

static int pic_revcheck(picAdcnormData *device_data, struct pic_handle_t *pic_handle)
{
    uint32_t rev0;
    uint32_t pich_rev;

    rev0 = ADCNORM_REV0_MAJ_MASK_SHIFT(picAdcnormRead(REV0));
    pich_rev = ADCNORM_REV0_MAJ_MASK_SHIFT(pic_handle->pic_adcnorm->REV0);

    if (rev0 != pich_rev)
    {
        error_print("%s: %s failed, rev0=%d, handle rev=%d\n",
                    __FILE__, __func__, rev0, pich_rev);
        return -1;
    }
    else
        return 0;
}

struct pic_adcnorm_function_struct pic_adcnorm_functions =
{
    .pic_reset = pic_reset,
    .pic_configure = pic_configure,
    .pic_get_current = pic_get_current,
    .pic_revcheck = pic_revcheck,
    
    .pic_dump_regs = dump_pic_adcnorm_regs,
};

void pic_adcnorm_init(picAdcnormData *device_data)
{
    adcNormDeviceHandle *pic_device_handle;

    pic_device_handle = allocate_memory(sizeof(adcNormDeviceHandle), GFP_KERNEL);
    
    // register with the parent
    pic_device_handle->fcn_tbl = &pic_adcnorm_functions;
    pic_device_handle->device_data = device_data;
    register_pic_subblock(adcnorm, pic_device_handle, device_data->instance, 0);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses reg_spinlock
    spin_lock_init(&(pic_device_handle->device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(pic_device_handle->device_data->reg_spinlock));
    pic_device_handle->device_data->interrupt_callback = NULL;
    pic_device_handle->device_data->interrupt_callback_data = NULL;
}
EXPORT_SYMBOL(pic_adcnorm_init);

void pic_adcnorm_exit(picAdcnormData *device_data)
{
    adcNormDeviceHandle *pic_device_handle;

    // unregister with the parent
    pic_device_handle = unregister_pic_subblock(adcnorm, device_data->instance, 0);
    free_memory(pic_device_handle);
}
EXPORT_SYMBOL(pic_adcnorm_exit);
