/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/kernel.h>    // for printk and uint32_t
#include <linux/bug.h>       // for BUG_ON/BUG
#include <linux/export.h>    // for EXPORT_SYMBOL
#include <linux/vmalloc.h>

#include "pic_constants.h"
#include "pic_handle_if.h"
#include "pic_if.h"
#include "pic.h"
#include "pic_driver.h"

// for subblock register sizes, include the headers
#include "PIC_regheaders.h"
#include "pic_full_subblock_list.h"  // needed for detailed pic_handle
// these include files are for the subblock function_structs
#include "pic_common_if.h"
#include "pic_adcnorm_if.h"
#include "pic_bulbmon_if.h"
#include "pic_lrmargin_if.h"
#include "pic_idma2d_if.h"
#include "pic_pd_if.h"
#include "pic_chipgap_if.h"
#include "pic_ps_esd_if.h"
#include "pic_hfir_if.h"
#include "pic_hscale_if.h"
#include "pic_output_dma_if.h"
#include "pic_skew_if.h"
#include "pic_skew_odma_if.h"
#include "pic_flare_if.h"
#include "pic_flare_dma_if.h"
#include "pic_bdr_if.h"
// end of include files for the subblock function_structs

#include "pic_driverlib_if.h"

struct pic_object_struct
{
    picCommonDeviceHandle *pic_common_handle;
    adcNormDeviceHandle   *pic_adcnorm_handle;
    bulbmonDeviceHandle   *pic_bulbmon_handle;
    lrMarginDeviceHandle  *lrmargin_handle[MAX_NUM_LRMARGIN_BLOCKS];
    picIdma2dDeviceHandle *pic_idma2d_handle;
    prnuDsnuDeviceHandle  *pic_pd_handle;
    chipgapDeviceHandle   *chipgap_handle;
    psesdDeviceHandle     *ps_esd_handle;
    hfirDeviceHandle      *hfir_handle;
    hscaleDeviceHandle    *hscale_handle;
    outputDmaDeviceHandle *output_dma_handle;
    skewDeviceHandle      *skew_handle;
    skewOdmaDeviceHandle  *skew_odma_handle;
    flareDeviceHandle     *flare_handle;
    flareWdmaDeviceHandle *flare_wdma_handle[MAX_FLARE_DMAS];
    bdrDeviceHandle       *bdr_handle;
};

struct pic_object_struct pic_object[MAX_NUM_PICBLOCKS];

// called by lo level driver initialization - before any chance
// of the subblock drivers registering
void init_pic(uint32_t pic_instance)
{
    int j;
    debug_print("%s: initializing pic %d\n", __func__, pic_instance);
    pic_object[pic_instance].pic_common_handle = NULL;
    pic_object[pic_instance].pic_adcnorm_handle = NULL;
    pic_object[pic_instance].pic_bulbmon_handle = NULL;
    for (j=0;j<MAX_NUM_LRMARGIN_BLOCKS;j++)
    {
        pic_object[pic_instance].lrmargin_handle[j] = NULL;
    }
    pic_object[pic_instance].pic_idma2d_handle = NULL;
    pic_object[pic_instance].pic_pd_handle     = NULL;
    pic_object[pic_instance].chipgap_handle    = NULL;
    pic_object[pic_instance].ps_esd_handle = NULL;
    pic_object[pic_instance].hfir_handle = NULL;
    pic_object[pic_instance].hscale_handle = NULL;
    pic_object[pic_instance].output_dma_handle = NULL;
    pic_object[pic_instance].skew_handle = NULL;
    pic_object[pic_instance].skew_odma_handle = NULL;
    pic_object[pic_instance].flare_handle = NULL;
    for (j=0;j<MAX_FLARE_DMAS;j++)
    {
        pic_object[pic_instance].flare_wdma_handle[j] = NULL;
    }
    pic_object[pic_instance].bdr_handle = NULL;
    
    // initialize any subsystems that need it
    pic_output_dma_init(pic_instance);
}

void exit_pic(int pic_instance)
{
    // we don't need to NULL out the pic_object, since the unregister takes care of that.
    
    // call cleanup of any subsystems that need it
    pic_output_dma_exit(pic_instance);
}

void register_pic_subblock(enum pic_internal_subblock picsubblock, void *pic_subblock_data, int instance,
                           int subblock_instance)
{
    if (picsubblock == common)
    {
        debug_print("pic.c:register common, instance=%d\n", instance);
        pic_object[instance].pic_common_handle = pic_subblock_data;
    }
    else if (picsubblock == adcnorm)
    {
        debug_print("pic.c:register adcnorm, instance=%d\n", instance);
	pic_object[instance].pic_adcnorm_handle = pic_subblock_data;
    }
    else if (picsubblock == bulbmon)
    {
        debug_print("pic.c:register bulbmon, instance=%d\n", instance);
        pic_object[instance].pic_bulbmon_handle = pic_subblock_data;
    }
    else if (picsubblock == lrmargin)
    {
        debug_print("pic.c:register lrmargin %s, instance=%d\n",
                    (subblock_instance==0)?"upper":"lower", instance);
        pic_object[instance].lrmargin_handle[subblock_instance] = pic_subblock_data;
    }
    else if (picsubblock == idma2d)
    {
        debug_print("pic.c:register ldma2d, instance=%d\n", instance);
        pic_object[instance].pic_idma2d_handle = pic_subblock_data;
    }
    else if (picsubblock == pd)
    {
        debug_print("pic.c:register prnu/dsnu, instance=%d\n", instance);
        pic_object[instance].pic_pd_handle = pic_subblock_data;
    }
    else if (picsubblock == chipgap)
    {
        debug_print("pic.c:register chipgap, instance=%d\n", instance);
        pic_object[instance].chipgap_handle = pic_subblock_data;
    }
    else if (picsubblock == ps_esd)
    {
        debug_print("pic.c:register ps_esd, instance=%d\n", instance);
        pic_object[instance].ps_esd_handle = pic_subblock_data;
    }
    else if (picsubblock == hfir)
    {
        debug_print("pic.c:register hfir, instance=%d\n", instance);
        pic_object[instance].hfir_handle = pic_subblock_data;
    }
    else if (picsubblock == hscale)
    {
        debug_print("pic.c:register hscale, instance=%d\n", instance);
        pic_object[instance].hscale_handle = pic_subblock_data;
    }
    else if (picsubblock == output_dma)
    {
        if (pic_subblock_data == NULL)
            debug_print("pic.c: initializing output dma to NULL, instance=%d\n", instance);
        else
            debug_print("pic.c: registered output dma, instance=%d\n", instance);
        pic_object[instance].output_dma_handle = pic_subblock_data;
    }
    else if (picsubblock == skew)
    {
        debug_print("pic.c:register skew instance=%d\n", instance);
        pic_object[instance].skew_handle = pic_subblock_data;
    }
    else if (picsubblock == skew_odma)
    {
        debug_print("pic.c:register skew_odma instance=%d\n", instance);
        pic_object[instance].skew_odma_handle = pic_subblock_data;
    }
    else if (picsubblock == flare)
    {
        debug_print("pic.c:register flare instance=%d\n", instance);
        pic_object[instance].flare_handle = pic_subblock_data;
    }
    else if (picsubblock == flare_wdma)
    {
        debug_print("pic.c:register flare_wdma instance=%d\n", instance);
        pic_object[instance].flare_wdma_handle[subblock_instance] = pic_subblock_data;
    }
    else if (picsubblock == bdr)
    {
        debug_print("pic.c:register bdr, instance=%d\n", instance);
        pic_object[instance].bdr_handle = pic_subblock_data;
    }
    else
    {
        error_print("Error!!! no support for block %d\n", picsubblock);
    }
}

void *unregister_pic_subblock(enum pic_internal_subblock picsubblock, int instance, int subblock_instance)
{
    void *subblock_data;
    
    if (picsubblock == common)
    {
        subblock_data = pic_object[instance].pic_common_handle;
        pic_object[instance].pic_common_handle = NULL;
        debug_print("pic.c: unregistered common, instance=%d\n", instance);
    }
    else if (picsubblock == adcnorm)
    {
        subblock_data = pic_object[instance].pic_adcnorm_handle;
        pic_object[instance].pic_adcnorm_handle = NULL;
        debug_print("pic.c: unregistered adcnorm, instance=%d\n", instance);
    }
    else if (picsubblock == bulbmon)
    {
        subblock_data = pic_object[instance].pic_bulbmon_handle;
        pic_object[instance].pic_bulbmon_handle = NULL;
        debug_print("pic.c: unregistered bulbmon, instance=%d\n", instance);
    }
    else if (picsubblock == lrmargin)
    {
        subblock_data = pic_object[instance].lrmargin_handle[subblock_instance];
        pic_object[instance].lrmargin_handle[subblock_instance] = NULL;
        debug_print("pic.c: unregistered lrmargin %s, instance %d\n",
                    (subblock_instance==0)?"upper":"lower", instance);
    }
    else if (picsubblock == idma2d)
    {
        subblock_data = pic_object[instance].pic_idma2d_handle;
        pic_object[instance].pic_idma2d_handle = NULL;
        debug_print("pic:c: unregistered idma2d, instance=%d\n", instance);
    }
    else if (picsubblock == pd)
    {
        subblock_data = pic_object[instance].pic_pd_handle;
        pic_object[instance].pic_pd_handle = NULL;
        debug_print("pic_c: unregistered prnudsnu, instance=%d\n", instance);
    }
    else if (picsubblock == chipgap)
    {
        subblock_data = pic_object[instance].chipgap_handle;
        pic_object[instance].chipgap_handle = NULL;
        debug_print("pic.c: unregistered chipgap, instance=%d\n", instance);
    }
    else if (picsubblock == ps_esd)
    {
        subblock_data = pic_object[instance].ps_esd_handle;
        pic_object[instance].ps_esd_handle = NULL;
        debug_print("pic_c: unregistered ps_esd, instance=%d\n", instance);
    }
    else if (picsubblock == hfir)
    {
        subblock_data = pic_object[instance].hfir_handle;
        pic_object[instance].hfir_handle = NULL;
        debug_print("pic_c: unregistered hfir, instance=%d\n", instance);
    }
    else if (picsubblock == hscale)
    {
        subblock_data = pic_object[instance].hscale_handle;
        pic_object[instance].hscale_handle = NULL;
        debug_print("pic_c: unregistered hscale, instance=%d\n", instance);
    }
    else if (picsubblock == skew)
    {
        subblock_data = pic_object[instance].skew_handle;
        pic_object[instance].skew_handle = NULL;
        debug_print("pic_c: unregistered skew, instance=%d\n", instance);
    }
    else if (picsubblock == skew_odma)
    {
        subblock_data = pic_object[instance].skew_odma_handle;
        pic_object[instance].skew_odma_handle = NULL;
        debug_print("pic_c: unregistered skew_odma, instance=%d\n", instance);
    }
    else if (picsubblock == flare)
    {
        subblock_data = pic_object[instance].flare_handle;
        pic_object[instance].flare_handle = NULL;
        debug_print("pic_c: unregistered flare, instance=%d\n", instance);
    }
    else if (picsubblock == flare_wdma)
    {
        subblock_data = pic_object[instance].flare_wdma_handle[subblock_instance];
        pic_object[instance].flare_wdma_handle[subblock_instance] = NULL;
        debug_print("pic_c: unregistered flare_wdma, instance=%d\n", instance);
    }
    else if (picsubblock == bdr)
    {
        subblock_data = pic_object[instance].bdr_handle;
        pic_object[instance].bdr_handle = NULL;
        debug_print("pic.c: unregistered bdr, instance=%d\n", instance);
    }
    else if (picsubblock == output_dma)
    {
        subblock_data = pic_object[instance].output_dma_handle;
        pic_object[instance].output_dma_handle = NULL;
        debug_print("pic.c: unregistered output_dma, instance=%d\n", instance);
    }
    else
    {
        error_print("pic.c: Fail to unregister %d, instance=%d\n", picsubblock, instance);
        return NULL;
    }
    return subblock_data;
}

// NOTE! This function is only for use by pic_output_dma.c - when registering or
// unregistering its own subblocks
outputDmaDeviceHandle *get_output_dma_device(int pic_instance)
{
    return pic_object[pic_instance].output_dma_handle;
}

#define CHECK4NULL(pointer)                                             \
    if (pointer == NULL)                                                \
    {                                                                   \
        print("ERROR %s, pointer %s is NULL\n", __func__,               \
              #pointer);                                                \
        BUG();                                                          \
    }

// macro to find the function table and subblock data
#define PIC_RETRIEVE_function_table(data_type, function_struct, pic_object_field, pic_instance) \
    struct function_struct *ft;                                         \
    data_type *pic_subblock_data;                                       \
                                                                        \
    if (pic_object[pic_instance].pic_object_field == NULL)              \
        return;  /* subblock doesn't exist */                           \
    pic_subblock_data = pic_object[pic_instance].pic_object_field->device_data; \
    CHECK4NULL(pic_subblock_data);                                      \
    ft = pic_object[pic_instance].pic_object_field->fcn_tbl;            \
    CHECK4NULL(ft);

#define PIC_RETRIEVE_function_table_retval(data_type, function_struct, pic_object_field, pic_instance, ret_on_null) \
    struct function_struct *ft;                                         \
    data_type *pic_subblock_data;                                       \
                                                                        \
    if (pic_object[pic_instance].pic_object_field == NULL)              \
        return ret_on_null;  /* subblock doesn't exist */               \
    pic_subblock_data = pic_object[pic_instance].pic_object_field->device_data; \
    CHECK4NULL(pic_subblock_data);                                      \
    ft = pic_object[pic_instance].pic_object_field->fcn_tbl;            \
    CHECK4NULL(ft);

// ok, now that we have our function table and subblock data in hand
// (the ft and pic_subblock_data from the macro), figure out
// which standard function to call
// Note that retval is only set if the requested function returns a value
#define EXECUTE_FUNCTION(the_function, retval)                          \
    if (the_function == do_reset)                                       \
        ft->pic_reset(pic_subblock_data);                               \
    else if (the_function == do_configure)                              \
        ft->pic_configure(pic_subblock_data, pic_handle);               \
    else if (the_function == do_get_current)                            \
        ft->pic_get_current(pic_subblock_data, pic_handle);             \
    else if (the_function == do_revcheck)                               \
        *retval = ft->pic_revcheck(pic_subblock_data, pic_handle);      \
    else                                                                \
        error_print("%s:Error!! no support for function %d\n", __func__, the_function);


static int do_function_subblock(enum pic_internal_subblock picsubblock,
                                enum function_type do_function,
                                struct pic_handle_t *pic_handle,
                                int pic_instance, int subblock_instance)
{
    int retval = -1;
    
    if (picsubblock == common)
    {
        PIC_RETRIEVE_function_table_retval(picCommonData, pic_common_function_struct,
                                           pic_common_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == adcnorm)
    {
        PIC_RETRIEVE_function_table_retval(picAdcnormData, pic_adcnorm_function_struct,
                                           pic_adcnorm_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == bulbmon)
    {
        PIC_RETRIEVE_function_table_retval(picBulbmonData, pic_bulbmon_function_struct,
                                           pic_bulbmon_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == lrmargin)
    {
        PIC_RETRIEVE_function_table_retval(picLrmarginData, pic_lrmargin_function_struct,
                                           lrmargin_handle[subblock_instance], pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == idma2d)
    {
        PIC_RETRIEVE_function_table_retval(picIdma2dData, pic_idma2d_function_struct,
                                           pic_idma2d_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == pd)
    {
        PIC_RETRIEVE_function_table_retval(picPrnuDsnuData, pic_pd_function_struct,
                                           pic_pd_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == chipgap)
    {
        PIC_RETRIEVE_function_table_retval(picChipgapData, pic_chipgap_function_struct,
                                           chipgap_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == ps_esd)
    {
        PIC_RETRIEVE_function_table_retval(picPsEsdData, pic_ps_esd_function_struct,
                                           ps_esd_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == hfir)
    {
        PIC_RETRIEVE_function_table_retval(picHfirData, pic_hfir_function_struct,
                                           hfir_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == hscale)
    {
        PIC_RETRIEVE_function_table_retval(picHscaleData, pic_hscale_function_struct,
                                           hscale_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == output_dma)
    {
        PIC_RETRIEVE_function_table_retval(outputDmaData, output_dma_function_struct,
                                           output_dma_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == skew)
    {
        PIC_RETRIEVE_function_table_retval(picSkewData, pic_skew_function_struct,
                                           skew_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == skew_odma)
    {
        PIC_RETRIEVE_function_table_retval(picSkewOdmaData, pic_skew_odma_function_struct,
                                           skew_odma_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == flare)
    {
        PIC_RETRIEVE_function_table_retval(picFlareData, pic_flare_function_struct,
                                           flare_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == flare_wdma)
    {
        PIC_RETRIEVE_function_table_retval(picFlareWdmaData, pic_flare_wdma_function_struct,
                                           flare_wdma_handle[subblock_instance], pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else if (picsubblock == bdr)
    {
        PIC_RETRIEVE_function_table_retval(picBdrData, pic_bdr_function_struct,
                                           bdr_handle, pic_instance, -ENODEV);
        EXECUTE_FUNCTION(do_function, &retval);
    }
    else
    {
        error_print("%s:Error!!! no support for block %d\n", __func__, picsubblock);
    }
    return retval;
}

//////////////////////////////////////////////////////////////////////////////
//                       start of public API functions                      //
//////////////////////////////////////////////////////////////////////////////

// For those subblocks that have multiple instances, the first instance will be
// picked up in the main loop, and the rest in a subsequent loop
#define FIRST_SUBBLOCK_INSTANCE 0

// for functions that don't care which pic instance (e.g. revcheck doesn't care which one)
#define ANY_PIC_INSTANCE 0
#define CHECK_RET                                                       \
    if (retval == -ENODEV)                                              \
    {                                                                   \
        debug_print("%s: non-existent subblock=%d\n", __func__, pic_subblock); \
    }                                                                   \
    else if (retval < 0)                                                \
    {                                                                   \
        print("%s: error, return value %d\n", __func__, retval);        \
        final_ret = retval;                                             \
    }

int pic_do_revcheck(struct pic_handle_t *pic_handle)
{
    enum pic_internal_subblock pic_subblock;
    int i, retval, final_ret;

    final_ret = 0;
    
    for (pic_subblock=common; pic_subblock<last_pic_subblock; pic_subblock++)
    {
        retval = do_function_subblock(pic_subblock, do_revcheck, pic_handle, ANY_PIC_INSTANCE, FIRST_SUBBLOCK_INSTANCE);
        CHECK_RET;
    }

    // Now revcheck the rest of the lrmargin and flare_wdma, since we already did 0
    for (i=1; i<MAX_NUM_LRMARGIN_BLOCKS; i++)
    {
        retval = do_function_subblock(lrmargin, do_revcheck, pic_handle, ANY_PIC_INSTANCE, i);
        CHECK_RET;
    }
    for (i=1; i<MAX_FLARE_DMAS; i++)
    {
        retval = do_function_subblock(flare_wdma, do_revcheck, pic_handle, ANY_PIC_INSTANCE, i);
        CHECK_RET;
    }
            
    return final_ret;
}
EXPORT_SYMBOL(pic_do_revcheck);

// soft reset - This is sets whatever bits necessary to
// reset pic's subblocks
void pic_do_reset(int instance)
{
    enum pic_internal_subblock pic_subblock;
    int i;
    
    for (pic_subblock=common; pic_subblock<last_pic_subblock; pic_subblock++)
    {
        do_function_subblock(pic_subblock, do_reset, NULL, instance, FIRST_SUBBLOCK_INSTANCE);
    }
    // Now reset the rest of the lrmargin and flare_wdma, since we already did 0
    for (i=1; i<MAX_NUM_LRMARGIN_BLOCKS; i++)
        do_function_subblock(lrmargin, do_reset, NULL, instance, i);
    for (i=1; i<MAX_FLARE_DMAS; i++)
        do_function_subblock(flare_wdma, do_reset, NULL, instance, i);
}
EXPORT_SYMBOL(pic_do_reset);

// NOTE: pic_do_configure writes all of the configuration registers,
// but not the side effects registers.  e.g. we don't write any registers
// that start the DMA.  
void pic_do_configure(struct pic_handle_t *pic_handle, int instance)
{
    enum pic_internal_subblock pic_subblock;
    int i;
    
    for (pic_subblock=common; pic_subblock<last_pic_subblock; pic_subblock++)
    {
        do_function_subblock(pic_subblock, do_configure, pic_handle, instance, FIRST_SUBBLOCK_INSTANCE);
    }
    for (i=1; i<MAX_NUM_LRMARGIN_BLOCKS; i++)
        do_function_subblock(lrmargin, do_configure, pic_handle, instance, i);
    for (i=1; i<MAX_FLARE_DMAS; i++)
        do_function_subblock(flare_wdma, do_configure, pic_handle, instance, i);

}
EXPORT_SYMBOL(pic_do_configure);

void pic_do_free_handle(struct pic_handle_t *pic_handle)
{
    vfree(pic_handle);
}
EXPORT_SYMBOL(pic_do_free_handle);

// Send the pic_handle down to each subblock to fill in the
// current values of all the registers
void pic_do_get_current(struct pic_handle_t *pic_handle, int instance)
{
    enum pic_internal_subblock pic_subblock;
    int i;    
    
    for (pic_subblock=common; pic_subblock<last_pic_subblock; pic_subblock++)
    {
        debug_print("%s:subblock=%d\n", __func__, pic_subblock);
        do_function_subblock(pic_subblock, do_get_current, pic_handle, instance, FIRST_SUBBLOCK_INSTANCE);
    }
    for (i=1; i<MAX_NUM_LRMARGIN_BLOCKS; i++)
    {
        debug_print("%s:subblock=lrmargin, instance=%d\n", __func__, i);
        do_function_subblock(lrmargin, do_get_current, pic_handle, instance, i);
    }
    for (i=1; i<MAX_FLARE_DMAS; i++)
    {
        debug_print("%s:subblock=flare_wdma, instance=%d\n", __func__, i);
        do_function_subblock(flare_wdma, do_get_current, pic_handle, instance, i);
    }
}
EXPORT_SYMBOL(pic_do_get_current);

//////////////////////////////////////////////////////////
// dumping registers - this is not for everyday use
// but for debug and bringup
//////////////////////////////////////////////////////////
void pic_dump_common_regs(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picCommonData, pic_common_function_struct,
                                pic_common_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_dump_common_regs);

void pic_dump_adcnorm(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picAdcnormData, pic_adcnorm_function_struct,
                                pic_adcnorm_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_dump_adcnorm);

void pic_dump_bulbmon(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picBulbmonData, pic_bulbmon_function_struct,
                                pic_bulbmon_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_dump_bulbmon);

void pic_lrmargin_dump(uint8_t pic_instance, uint8_t lrmargin_instance)
{
    PIC_RETRIEVE_function_table(picLrmarginData, pic_lrmargin_function_struct,
                                lrmargin_handle[lrmargin_instance], pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_lrmargin_dump);

void pic_pd_idma_dump(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picIdma2dData, pic_idma2d_function_struct,
                                pic_idma2d_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_pd_idma_dump);

void pic_pd_dump(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picPrnuDsnuData, pic_pd_function_struct,
                                pic_pd_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_pd_dump);

void pic_chipgap_drv_dump(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picChipgapData, pic_chipgap_function_struct,
                                chipgap_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_chipgap_drv_dump);

void pic_dump_ps_esd(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picPsEsdData, pic_ps_esd_function_struct,
                                ps_esd_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_dump_ps_esd);

void pic_dump_hfir(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picHfirData, pic_hfir_function_struct,
                                hfir_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_dump_hfir);

void pic_dump_hscale(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picHscaleData, pic_hscale_function_struct,
                                hscale_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_dump_hscale);

void pic_dump_output_dma_regs(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(outputDmaData, output_dma_function_struct,
                                output_dma_handle, pic_instance);
    ft->dump_all_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_dump_output_dma_regs);

void pic_dump_skew(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picSkewData, pic_skew_function_struct,
                                skew_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_dump_skew);

void pic_dump_skew_odma(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picSkewOdmaData, pic_skew_odma_function_struct,
                                skew_odma_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_dump_skew_odma);

void pic_dump_flare(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picFlareData, pic_flare_function_struct,
                                flare_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_dump_flare);

void pic_dump_flare_wdma(uint8_t pic_instance, int flare_instance)
{
    PIC_RETRIEVE_function_table(picFlareWdmaData, pic_flare_wdma_function_struct,
                                flare_wdma_handle[flare_instance], pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
}
EXPORT_SYMBOL(pic_dump_flare_wdma);

void pic_bdr_dump(uint8_t pic_instance, bool dump_lut)
{
    PIC_RETRIEVE_function_table(picBdrData, pic_bdr_function_struct,
                                bdr_handle, pic_instance);
    ft->pic_dump_regs(pic_subblock_data);
    if (dump_lut)
    {
    // FIXME        
        /* The 'lut_size' is number of entries in the table, not bytes. */
        // call lo level bdr lut dump function
        // pic_bdr_dump_lut
        // pic_bdr_dump_lut(PIC_BDR_LUT_SIZE*sizeof(uint32_t)); FIXME!!! implement
    }
}
EXPORT_SYMBOL(pic_bdr_dump);

void pic_dump(uint8_t pic_instance)
{
    int i;
    
    pic_dump_common_regs(pic_instance);
    pic_dump_adcnorm(pic_instance);
    pic_dump_bulbmon(pic_instance);
    for (i=1; i<MAX_NUM_LRMARGIN_BLOCKS; i++)
    {
        pic_lrmargin_dump(pic_instance, i);
    }
    pic_pd_idma_dump(pic_instance);
    pic_pd_dump(pic_instance);
    pic_chipgap_drv_dump(pic_instance);
    pic_dump_ps_esd(pic_instance);
    pic_dump_hfir(pic_instance);
    pic_dump_hscale(pic_instance);
    pic_dump_output_dma_regs(pic_instance);
    pic_dump_skew(pic_instance);
    pic_dump_skew_odma(pic_instance);
    pic_dump_flare(pic_instance);
    for (i=1; i<MAX_FLARE_DMAS; i++)
    {    
        pic_dump_flare_wdma(pic_instance, i);
    }
    pic_bdr_dump(pic_instance, false);
}
EXPORT_SYMBOL(pic_dump);

void pic_do_clear_all_irqs(int pic_instance)
{
    int i;
    // FIXME - Need to clear all of the other interrupt sources as well, once
    // their subblock drivers are implemented:  Skew, Skew odma, Antiflin,
    // Antiflin DMA, ps_esd
    PIC_RETRIEVE_function_table(outputDmaData, output_dma_function_struct,
                                output_dma_handle, pic_instance);

    for (i=0; i<MAX_OUTPUT_DMAS; i++)
        ft->clear_output_dma_irqs(pic_subblock_data, NULL, i);
}
EXPORT_SYMBOL(pic_do_clear_all_irqs);

// disable the pic common soft reset
void pic_common_disable_soft_reset(int pic_instance)
{
    PIC_RETRIEVE_function_table(picCommonData, pic_common_function_struct,
                                pic_common_handle, pic_instance);
    
    ft->disable_pic_common_soft_reset(pic_subblock_data);
}
EXPORT_SYMBOL(pic_common_disable_soft_reset);


/////////////////////////////////////////////////////////////////
// ONLY FOR USE BY PIPE CUTTER!!!! - NOT A GENERAL PUBLIC API
/////////////////////////////////////////////////////////////////
void pic_get_trans_len_output_dma_channel(uint8_t pic_instance, uint32_t *trans_len,
                                          uint8_t channel)
{
    PIC_RETRIEVE_function_table(outputDmaData, output_dma_function_struct,
                                output_dma_handle, pic_instance);
    
    ft->output_dma_channel_get_trans_len(pic_subblock_data, trans_len, channel);
}
EXPORT_SYMBOL(pic_get_trans_len_output_dma_channel);


/////////////////////////////////////////////////////////////////
// ONLY FOR USE BY THE DMA CODE!!!! - NOT A GENERAL PUBLIC API
// DMA functions used by dma-handling code after do_configure
/////////////////////////////////////////////////////////////////

// starts the prnu/dsnu dma
void pic_start_idma2d_dma(uint8_t pic_instance, dma_addr_t descriptor_phys)
{
    PIC_RETRIEVE_function_table(picIdma2dData, pic_idma2d_function_struct,
                                pic_idma2d_handle, pic_instance);
    
    ft->pic_start_idma2d_dma(pic_subblock_data, descriptor_phys);
}
EXPORT_SYMBOL(pic_start_idma2d_dma);

// starts the pic output dma
void pic_start_output_dma(uint8_t pic_instance, uint8_t channel,
                          dma_addr_t descriptor_phys)
{
    PIC_RETRIEVE_function_table(outputDmaData, output_dma_function_struct,
                                output_dma_handle, pic_instance);
    
    ft->start_output_dma(pic_subblock_data, channel, descriptor_phys);
}
EXPORT_SYMBOL(pic_start_output_dma);

bool pic_output_dma_channel_busy(uint8_t pic_instance, uint8_t channel)
{
    PIC_RETRIEVE_function_table_retval(outputDmaData, output_dma_function_struct,
                                    output_dma_handle, pic_instance, false);
    
    return(ft->output_dma_channel_status_busy(pic_subblock_data, channel));
}
EXPORT_SYMBOL(pic_output_dma_channel_busy);


bool pic_output_dma_channel_is_enabled(uint8_t pic_instance, uint8_t channel)
{
    PIC_RETRIEVE_function_table_retval(outputDmaData, output_dma_function_struct,
                                    output_dma_handle, pic_instance, false);
    return ft->output_dma_channel_is_enabled(pic_subblock_data, channel);
}
EXPORT_SYMBOL(pic_output_dma_channel_is_enabled);


/////////////////////////////////////////////////////////////////
// NOTE - THESE FUNCTIONS ARE FOR LEGACY CODE SUPPORT ONLY.
// They need to be removed when the mage architecture is
// implemented in the main scan code
/////////////////////////////////////////////////////////////////
// reset the pic odma
void pic_odma_soft_reset(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(outputDmaData, output_dma_function_struct,
                                output_dma_handle, pic_instance);
    
    ft->pic_reset(pic_subblock_data);
}
EXPORT_SYMBOL(pic_odma_soft_reset);

// enable pic common interrupts
// caller sets the irqstruct field bit to true for each interrupt to enable
// irqstruct of NULL means to enable all interrupts
void pic_enable_pic_common_irqs(uint8_t pic_instance,
                                struct pic_common_ints *irqstruct)
{
    PIC_RETRIEVE_function_table(picCommonData, pic_common_function_struct,
                                pic_common_handle, pic_instance);

    ft->enable_pic_common_irqs(pic_subblock_data, irqstruct);
}
EXPORT_SYMBOL(pic_enable_pic_common_irqs);

// disable pic common interrupts
// caller sets the irqstruct field bit to true for each interrupt to disable
// irqstruct of NULL means to disable all interrupts
void pic_disable_pic_common_irqs(uint8_t pic_instance, struct pic_common_ints *irqstruct)
{
    PIC_RETRIEVE_function_table(picCommonData, pic_common_function_struct,
                                pic_common_handle, pic_instance);
    
    ft->disable_pic_common_irqs(pic_subblock_data, irqstruct);
}
EXPORT_SYMBOL(pic_disable_pic_common_irqs);

// enable requested ints for requested channel
// if irqstruct is NULL, enable all ints for the requested channel
void pic_enable_pic_output_dma_channel_irqs(uint8_t pic_instance,
                                            struct pic_output_dma_interrupt_info *irqstruct,
                                            uint8_t channel)
{
    bool enable;
    PIC_RETRIEVE_function_table(outputDmaData, output_dma_function_struct,
                                output_dma_handle, pic_instance);

    enable=true;
    ft->enable_output_dma_irqs(pic_subblock_data, irqstruct, channel, enable);
}
EXPORT_SYMBOL(pic_enable_pic_output_dma_channel_irqs);

// disable requested ints for requested channel
// if irqstruct is NULL, disable all ints for the requested channel
void pic_disable_pic_output_dma_channel_irqs(uint8_t pic_instance,
                                             struct pic_output_dma_interrupt_info *irqstruct,
                                             uint8_t channel)
{
    bool disable;
    PIC_RETRIEVE_function_table(outputDmaData, output_dma_function_struct,
                                output_dma_handle, pic_instance);

    disable = false;
    ft->enable_output_dma_irqs(pic_subblock_data, irqstruct, channel, disable);
}
EXPORT_SYMBOL(pic_disable_pic_output_dma_channel_irqs);

///////////////////////////////////////////////////////

///////////////////////////////////////////////////////
// only for use by the pic_common interrupt handler - not a public API
///////////////////////////////////////////////////////
void pic_output_dma_channel_handle_irqs(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(outputDmaData, output_dma_function_struct,
                                output_dma_handle, pic_instance);
    
    ft->handle_dma_channels_irqs(pic_subblock_data);
}

void pic_pd_idma2d_handle_irqs(uint8_t pic_instance)
{
    PIC_RETRIEVE_function_table(picIdma2dData, pic_idma2d_function_struct,
                                pic_idma2d_handle, pic_instance);
    
    ft->pic_idma2d_handle_irqs(pic_subblock_data);
}
///////////////////////////////////////////////////////
// only for use by the pic driverlib - not a public API
///////////////////////////////////////////////////////
// we allocate - caller must deallocate.
// we return size of table (in bytes) or error
int pic_get_subblock_sizes_array(uint32_t **subblock_parm)
{
    enum pic_full_subblock_list i;
    int x;
    int allocsize;
    uint32_t *subblock_array;
    
    // note that for PIC, we assume that the PIC0 array object is always present,
    // and that it will be the same as the PIC1 object (if it exists), so we always
    // use the pic_object's array entry for PIC0
    PIC_RETRIEVE_function_table_retval(outputDmaData, output_dma_function_struct,
                                       output_dma_handle, 0, -1);
    
    allocsize = max_subblock_pic * sizeof(uint32_t);
    subblock_array = vmalloc(allocsize);
    *subblock_parm = subblock_array;
    
    if (subblock_array == NULL)
        return -ENOMEM;
    
    debug_print("%s: max_subblock=%d\n", __func__, max_subblock_pic);
    
    // 0 out all subblocks as default
    for (i=common_index;i<max_subblock_pic;i++)
        subblock_array[i] = 0;
    
    if (pic_object[0].pic_common_handle != NULL)
        subblock_array[common_index] = sizeof(PIC_REGS_t);
    
    if (pic_object[0].pic_adcnorm_handle != NULL)
        subblock_array[adcnorm_index] = sizeof(ADCNORM_REGS_t);
    
    if (pic_object[0].pic_bulbmon_handle != NULL)
        subblock_array[bulbmon_index] = sizeof(BULBMON_REGS_t);
    
    if (pic_object[0].lrmargin_handle[0] != NULL)
        subblock_array[lrmargin0_index] = sizeof(LRMARGIN0_REGS_t);
    
    if (pic_object[0].pic_idma2d_handle != NULL)
        subblock_array[idma2d_index] = sizeof(IDMA_2D_REGS_t);
    
    if (pic_object[0].pic_pd_handle != NULL)
        subblock_array[pd_index] = sizeof(PRNUDSNU_REGS_t);
    
    if (pic_object[0].chipgap_handle != NULL)
        subblock_array[chipgap_index] = sizeof(NEWMAN_REGS_t);
    
    if (pic_object[0].ps_esd_handle != NULL)
        subblock_array[ps_esd_index] = sizeof(PS_ESD_REGS_t);
    
    if (pic_object[0].hfir_handle != NULL)
        subblock_array[hfir_index] = sizeof(HFIR_REGS_t);
    
    if (pic_object[0].hscale_handle != NULL)
        subblock_array[hscale_index] = sizeof(HSCALE_REGS_t);
    
    if (pic_object[0].lrmargin_handle[1] != NULL)
        subblock_array[lrmargin1_index] = sizeof(LRMARGIN1_REGS_t);
    
    if (pic_object[0].output_dma_handle != NULL)
        ft->pic_get_output_dma_sizes_array(subblock_array, pic_subblock_data);
    
    if (pic_object[0].skew_handle != NULL)
        subblock_array[skew_index] = sizeof(SKEWDET_REGS_t);
    
    if (pic_object[0].skew_odma_handle != NULL)
        subblock_array[skew_odma_index] = sizeof(SKEW_ODMA_REGS_t);
    
    if (pic_object[0].flare_handle != NULL)
        subblock_array[flare_index] = sizeof(ANTIFLIN_REGS_t);
    
    for (x=0;x<MAX_FLARE_DMAS;x++)
    {
        if (pic_object[0].flare_wdma_handle[x] != NULL)
            subblock_array[flare_dma0_index + x] = sizeof(ANTIFLIN_WDMA0_REGS_t);
    }
    
    if (pic_object[0].bdr_handle != NULL)
        subblock_array[bdr_index] = sizeof(BITREDUCT_REGS_t);
    return allocsize;
}
EXPORT_SYMBOL(pic_get_subblock_sizes_array);
