/*
 ***************************************************************************************
 * (c) Copyright 2015 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/module.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/of_device.h>
#include <linux/dma-mapping.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/string.h>  // for strcpy
#include <linux/vmalloc.h>

#include "pic_data.h"
#include "pic_pltfm.h"
#include "pic_full_subblock_list.h"

//#define DURING_DEBUG 1

#ifdef DURING_DEBUG
  #define debug_print(args...) printk(args)
#else
#define debug_print(args...)
#endif

struct device *global_dev;

// structure to store state of block during suspend/resume
struct device_suspend_context_struct
{
    int placeholder;  // what needs to be stored in suspend for this block?  FIXME TODO
};

extern void init_pic(int picinstance);  // pic.c needs to know about the starting of a probe.
extern void exit_pic(int picinstance);  // pic.c needs to know about the exit call
extern int pic_get_subblock_sizes_array(uint32_t **copybuf);
extern void pic_fixup_shadow_pointers(void *pic_handle);
extern int pic_do_revcheck(void *pic_handle);

static char *revcheck_status="unknown";   // revcheck hasn't run yet

module_param(revcheck_status, charp, S_IRUSR);

// binary sysfs entry to allow user space to give the pic handle to kernel space
// and have the kernel check the revision
static ssize_t write_pic_handle(struct file *filp, struct kobject *kobj,
                                    struct bin_attribute *attr,
                                    char *buf, loff_t offset, size_t count)
{
    static uint32_t *pic_handle_ptr;
    static int first_write = 1;
    static int pic_handle_size;
    uint8_t *curr_pic_handle_byte_ptr;
    uint32_t *tmp_ptr;
    
    if (first_write)
    {
        strcpy(revcheck_status, "pending");  // revcheck has started
        tmp_ptr = (uint32_t *) buf;
        pic_handle_size = tmp_ptr[0];

        // create a pic_handle to pass to the driver for a revision check
        pic_handle_ptr = (uint32_t *) vmalloc(pic_handle_size+1); // +1 for margin
        first_write = 0;
    }

    curr_pic_handle_byte_ptr = (uint8_t *) pic_handle_ptr;
    curr_pic_handle_byte_ptr += offset;  // move to the current location in our handle
    
    memcpy(curr_pic_handle_byte_ptr, buf, count);

    if (offset+count >= pic_handle_size)
    {
        // we're done copying in data from the user, set up for next time
        // the user writes the pic_handle
        first_write = 1;
        
        // when the pic handle is passed in, the pointers are invalid (user space),
        // so we fix them for kernel space
        pic_fixup_shadow_pointers(pic_handle_ptr);

        if (pic_do_revcheck(pic_handle_ptr) == -1)
        {
            printk("Error!!  revcheck failed\n");
            
            strcpy(revcheck_status, "error");
        }
        else
            strcpy(revcheck_status, "passed");
    }

    return count;
}

// binary sysfs entry to allow user space to give the pic handle to kernel space
static ssize_t read_pic_handle(struct file *filp, struct kobject *kobj,
                               struct bin_attribute *attr,
                               char *buf, loff_t offset, size_t count)
{
#if 0    
    static uint32_t *pic_handle_ptr = NULL;
    static int first_read = 1;
    static int all_data_returned = 0;
    static int pic_handle_size = 0;
    uint8_t *curr_pic_handle_byte_ptr;
    int i;
#endif    
    
#if 1
    printk("FIXME, need to have a function to call to retrieve the pic_handle\n");
    printk("How to figure out which pic_handle you want, and how to get the ptr\n");
    return 0;
#else
// commented out until we have a function to call to return the requested pic_handle
    if (all_data_returned)
    {
        all_data_returned = 0;
        first_read = 1; // we're done, set up for next time a read is called
        return 0;  // tell caller we finished the read
    }
    
    if (first_read)
    {
        pic_handle_ptr = get_pic_handle_function_must_be_written();
        pic_handle_size = pic_handle_ptr[0];
        first_read = 0;
    }
    // When we are called, the system tells us what offset from the start of our buffer we are at.
    // First time, offset=0, count=4096, next time, offset=4096, count 4096, next time offset=8192....
    curr_pic_handle_byte_ptr = (uint8_t *) pic_handle_ptr;
    curr_pic_handle_byte_ptr += offset;
    
    if (offset+count >= pic_handle_size)
    {
        int holdcount;
        // we've sent back all the data except this last buffer 
        all_data_returned = 1;

        // calculate how many bytes we're really returning this time
        holdcount = count;
        count = pic_handle_size % count;

        // handle the case where our pic_handle_size is the exact multiple of system count
        if (count == 0)
            count = holdcount;
    }

    for (i=0;i<count;i++)
    {
        // copy only as much as they have allocated
        // (or as much as we have left)
        memcpy(buf, curr_pic_handle_byte_ptr, count);
    }

    return count;  // tell caller we have return 
    
#endif    

}

// binary sysfs entry to allow user space to ask for the subblock size array
static ssize_t retrieve_subblock_sizes_array(struct file *filp, struct kobject *kobj,
                                             struct bin_attribute *attr,
                                             char *buf, loff_t offset, size_t size)
{
    uint32_t *copybuf;
    int count;

    if (offset >= max_subblock_pic * sizeof(uint32_t))
        return 0;

    // the size of the data to send to user space is well under the 4K page size,
    // so no worries about having to do multiple copies to user space
    count = pic_get_subblock_sizes_array(&copybuf);

    if (count < 0)
    {
        printk("ERROR, pic_get_subblock_sizes_array returned %d\n", count);
        return 0;
    }
    
    memcpy(buf, copybuf, count); 

    vfree(copybuf);  // pic_get_subblock_sizes_array allocates memory, we must deallocate
    return count;
}
// reading from pic_handle returns the pic handle (tbd)  writing to it sends down
// the completed pic_handle (and does a revcheck
BIN_ATTR(pic_handle_revcheck, S_IRUGO | S_IWUSR, read_pic_handle, write_pic_handle, 0);
BIN_ATTR(subblock_sizes_array, S_IRUGO, retrieve_subblock_sizes_array, NULL, max_subblock_pic * sizeof(uint32_t));

// functions/struct only used for debug and bringup - not needed in normal driver use
extern void pic_dump(uint8_t pic_instance);
ssize_t debug_test_write1(struct device *dev, struct device_attribute *attr,
                          const char *buf, size_t count)
{
    // dump pic0 unless specifically requested to dump pic1
    if (buf[0] == '1')
    {
        pic_dump(1);
    }
    else
    {
        pic_dump(0);
    }
    
    return count;
}

DEVICE_ATTR(pictest1, S_IWUSR | S_IRUGO, NULL, debug_test_write1);

const struct attribute *pictest_attrs[] = {
    &dev_attr_pictest1.attr,
    NULL,
};

// get the subblock data out of the device tree.  If there is no
// such subblock, just return NULL.
// allocate device data and put it on the linked list
// get the subblock data out of the device tree.  If there is no
// such subblock, just return NULL.
// allocate device data and put it on the linked list
static struct device_data *init_subblock(char *subblock_name,
                                         void (*init_func)(struct device_data *),
                                         void (*exit_func)(struct device_data *),
                                         int irq, int picinstance,
                                         struct device_node *parent_node,
                                         struct list_head *device_data_list, uint32_t asicrev)
{
    struct resource res;
    struct device_node *child_node;
    struct device_data *device_data;
    int submoduleinstance, retval;

    // now this is for each child
    child_node = of_get_child_by_name(parent_node, subblock_name);
    if (child_node == NULL)
    {
        debug_print("no child by name of %s\n", subblock_name);
        return NULL; // no child by that name, just return
    }

    retval = of_address_to_resource(child_node, 0, &res);
    if (retval < 0)
    {
        dev_err(global_dev, "%s can't get resource for node, returned 0x%X\n", subblock_name, retval);
        goto early_failure;
    }

    device_data = (struct device_data *)kzalloc(sizeof(struct device_data), GFP_KERNEL);
    if (!device_data)
    {
        dev_err(global_dev, "kzalloc failed\n");
        goto early_failure;
    }

    if (!request_mem_region(res.start, resource_size(&res), subblock_name))
    {
        dev_err(global_dev, "%s request_mem_region failed: %s\n", subblock_name, __func__);
        goto request_mem_region_failed;
    } 

    device_data->virt_addr = of_iomap(child_node, 0); // get virtual address
    if (device_data->virt_addr == NULL)
    {
        dev_err(global_dev, "%s of_iomap failed: %s\n", subblock_name, __func__);
        device_data->virt_addr = 0;
        goto iomap_failed;
    }

    device_data->asicrev = asicrev;
    device_data->submodinstance = -1;
    // get the various instance variables
    if (of_property_read_u32(child_node, "lrmargininstance", &submoduleinstance) == 0)
    {
        device_data->submodinstance = submoduleinstance;
    }
    if (of_property_read_u32(child_node, "wdmainstance", &submoduleinstance) == 0)
    {
        device_data->submodinstance = submoduleinstance;
    }
    if (of_property_read_u32(child_node, "aflarewdmainstance", &submoduleinstance) == 0)
    {
        device_data->submodinstance = submoduleinstance;
    }
    
    device_data->instance = picinstance;
    device_data->phys_addr = res.start;
    device_data->addr_size = resource_size(&res); 
    device_data->irq = irq;  
    device_data->exit_func = exit_func;
//    debug_print("%s: virt_addr=0x%p, phys_addr=0x%X, pic %d, submod=%d\n", subblock_name,
//                device_data->virt_addr, device_data->phys_addr,
//                device_data->instance, device_data->submodinstance);
    
    // if we are a subblock that has its own IRQ (the calling function knows this), 
    // then map the interrupt service routine
    if (irq > -1)
    {
        retval = request_threaded_irq(irq, NULL,
                                      pic_platform_irq,
                                      IRQF_ONESHOT, "PIC", device_data);
        if (retval)
        {
            dev_err(global_dev, "request_irq failed\n");
            goto request_irq_failed;
        }
    }

    // add the device_data to the linked list
    list_add((struct list_head *)&device_data->linked_list, device_data_list);
    
    init_func(device_data);

    of_node_put(child_node);  // of_get_child_by_name() requires of_node_put() when done
    return device_data;

request_irq_failed:
    device_data->irq = 0;

    iounmap(device_data->virt_addr); 
iomap_failed:
    release_mem_region(device_data->phys_addr, device_data->addr_size); 
request_mem_region_failed:
    kfree(device_data);
early_failure:
    of_node_put(child_node);  // of_get_child_by_name() requires of_node_put() when done
    return NULL;
}

// one probe received per pic for the entire structured device tree entry
static int pic_platform_probe(struct platform_device *pdev)
{
    int retval;
    int irq;
    struct device_node *parent_node = pdev->dev.of_node;
    uint32_t picinstance, asicrev;
    int ignore;
    struct device_data *device_data;
    struct list_head *device_data_list;

    // initialize the head ptr for the whole linked list of device_data
    global_dev = &pdev->dev; // for error print statements (dev_err)
    
    if (!parent_node)
    {
        dev_err(global_dev, "Could not find device node %s\n", "pic");
        return -ENODEV;
    }

    device_data_list = kmalloc(sizeof(struct list_head), GFP_KERNEL);
    if (device_data_list == NULL)
    {
        dev_err(global_dev, "Could not allocate list_head\n");
        return -ENOMEM;
    }

    INIT_LIST_HEAD(device_data_list);
    
    // get the irq from the parent node (each pic has 1 interrupt)
    irq = irq_of_parse_and_map(parent_node, 0);
    if (irq < 0)
    {
        dev_err(global_dev, "platform_get_irq failed\n");
        retval = -ENXIO;
        goto early_failure;
    }
    debug_print("Probe in %s: irq = %d\n", __func__, irq);

    of_property_read_u32(parent_node, "picinstance", &picinstance);
    if (of_property_read_u32(parent_node, "asicrev", &asicrev) != 0)
        asicrev = 0;  // rev A doesn't have a setting
    
    // tell pic.c we're starting up now.  It will get called 1 time for each pic, so
    // the init needs to be aware of that fact
    init_pic(picinstance);
    
    // pic has many subblocks - parse them all out of the device tree, allocate device_data
    // for each of them, put that data on the device_data_list, and call their
    // respective init functions in the main driver code

    device_data = init_subblock("pic-top", pic_common_init, pic_common_exit, irq,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-adcnorm", pic_adcnorm_init, pic_adcnorm_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-bulbmon", pic_bulbmon_init, pic_bulbmon_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-lrmargin0", pic_lrmargin_init, pic_lrmargin_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-idma-2d", pic_idma2d_init, pic_idma2d_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-prnudsnu", pic_pd_init, pic_pd_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-hscale", pic_hscale_init, pic_hscale_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-wdma-top", pic_wdma_init, pic_wdma_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-wdma-dma0", pic_wdma_channel_init, pic_wdma_channel_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-wdma-dma1", pic_wdma_channel_init, pic_wdma_channel_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-wdma-dma2", pic_wdma_channel_init, pic_wdma_channel_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-lrmargin1", pic_lrmargin_init, pic_lrmargin_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-newman", pic_chipgap_init, pic_chipgap_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    device_data = init_subblock("pic-bitreduct", pic_bdr_init, pic_bdr_exit, -1,
                                picinstance, parent_node, device_data_list, asicrev);
    
    platform_set_drvdata(pdev, device_data_list); // store our data in the device struct
    // for debugging and bringup to read registers- not needed for everyday driver work
    ignore = sysfs_create_files(&(pdev->dev.kobj), pictest_attrs);
    // allow userspace to access the pic subblock sizes array as a *binary* value
    ignore = sysfs_create_bin_file(&(pdev->dev.kobj), &bin_attr_pic_handle_revcheck);
    ignore = sysfs_create_bin_file(&(pdev->dev.kobj), &bin_attr_subblock_sizes_array);    
    
#if 0 //PD sys attributes
    {
    	extern void pd_sys_att_init(struct device *dev);
    	pd_sys_att_init(&(pdev->dev));
    }
#endif

    return 0;
    
early_failure:    
    return retval;
}


static int pic_platform_remove(struct platform_device *pdev)
{
    // get linked list of device_datas
    struct list_head *device_data_list_ptr = platform_get_drvdata(pdev); 
    struct device_data *device_data, *dd_next; // dd_next is used by list function/macro
    int instance = 0;

    list_for_each_entry_safe(device_data, dd_next, device_data_list_ptr, linked_list)
    {
        // items were added to the list LIFO, and we remove FIFO
        list_del(&device_data->linked_list);
        device_data->exit_func(device_data);  // call exit function for driver subblock
        iounmap(device_data->virt_addr);
        release_mem_region(device_data->phys_addr, device_data->addr_size);
        if (device_data->irq > 0)
            free_irq(device_data->irq, device_data);
        instance = device_data->instance; // all device_data for this pic will have same instance
        kfree(device_data);
    }
    // tell pic.c we're closing down now.  It will be called once for each pic
    exit_pic(instance);

    kfree(device_data_list_ptr);
    
    platform_set_drvdata(pdev, NULL);

#if 0 //PD sys attributes
    {
    	extern void pd_sys_att_remove(struct device *dev);
    	pd_sys_att_remove(&(pdev->dev));
    }
#endif

    sysfs_remove_files(&(pdev->dev.kobj), pictest_attrs);
    sysfs_remove_bin_file(&(pdev->dev.kobj), &bin_attr_pic_handle_revcheck);
    sysfs_remove_bin_file(&(pdev->dev.kobj), &bin_attr_subblock_sizes_array);    
    debug_print("%s: complete\n",__func__);
    return 0;
}

static int pic_platform_suspend(struct platform_device *pdev, pm_message_t state)
{
    struct device_data *device_data = platform_get_drvdata(pdev);
    printk(KERN_WARNING "%s event = %d\n",__func__, state.event);

    // allocate suspend context structure
    device_data->device_suspend_context = kmalloc(sizeof(struct device_suspend_context_struct), GFP_KERNEL);
    if (device_data->device_suspend_context == NULL)
    {
        return -1;
    }
    
    return (0);
}

static int pic_platform_resume(struct platform_device *pdev)
{
    struct device_data *device_data = platform_get_drvdata(pdev);
    printk(KERN_WARNING "%s\n",__func__);

    if (device_data->device_suspend_context == NULL)
    {
        printk(KERN_ERR "%s: invalid suspend context\n",__func__);
        return -1;
    }
    
    // restore block context here TODO FIXME what to do?

    // free the suspend context
    kfree(device_data->device_suspend_context);
    device_data->device_suspend_context = NULL;
    return(0);
}



#ifdef CONFIG_OF
static const struct of_device_id mrvl_pic_dt_match[] = {

    { .compatible = "g2-imgpipe-pic"}, // TODO(njones): add marvell to this
    { .compatible = "ipg2pic"},
    {}, 
};
MODULE_DEVICE_TABLE(of, mrvl_pic_dt_match);
#endif

static struct platform_driver pic_platform_driver =
{
    .probe   = pic_platform_probe,
    .remove  = pic_platform_remove,
    .suspend = pic_platform_suspend,
    .resume = pic_platform_resume,
//    .id_table = mrvl_pic_driver_ids, ???
    .driver  = {
        .name  = "pic_block",
        .owner = THIS_MODULE,
        .of_match_table = of_match_ptr(mrvl_pic_dt_match),
    }
};

module_platform_driver(pic_platform_driver);

MODULE_AUTHOR("Copyright (c) 2015 Marvell International Ltd.");
MODULE_DESCRIPTION("Marvell PIC block driver");

MODULE_LICENSE("GPL");
MODULE_VERSION("2014_July_8");

