/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/module.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/of_device.h>
#include <linux/dma-mapping.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>

//#include <linux/list.h>

#include "icetest_data.h"
#include "icetest_pltfm.h"

struct device *global_dev;

// structure to store state of block during suspend/resume
struct device_suspend_context_struct
{
    int placeholder;  // what needs to be stored in suspend for this block?  FIXME TODO
};

extern void icetest_dump_regs(void);
extern void icetest_dump_idma_regs(void);
// functions only used for debug and bringup - not needed in normal driver use
ssize_t debug_test_write1(struct device *dev, struct device_attribute *attr,
                          const char *buf, size_t count)
{
    icetest_dump_regs();
    icetest_dump_idma_regs();
    return count;
}

DEVICE_ATTR(icetest1, S_IWUSR | S_IRUGO, NULL, debug_test_write1);

const struct attribute *icetest_attrs[] = {
    &dev_attr_icetest1.attr,
    NULL,
};

// get the subblock data out of the device tree.  If there is no
// such subblock, just return NULL.
// allocate device data and put it on the linked list
static struct device_data *init_subblock(char *subblock_name,
                                         void (*init_func)(struct device_data *),
                                         void (*exit_func)(struct device_data *),
                                         int irq, struct device_node *parent_node,
                                         struct list_head *device_data_list)
{
    struct resource res;
    struct device_node *child_node;
    struct device_data *device_data;
    int instance, retval;

    // now this is for each child
    child_node = of_get_child_by_name(parent_node, subblock_name);
    if (child_node == NULL)
    {
        return NULL; // no child by that name, just return
    }

    retval = of_address_to_resource(child_node, 0, &res);
    if (retval < 0)
    {
        dev_err(global_dev, "%s can't get resource for node, returned 0x%X\n", subblock_name, retval);
        goto early_failure;
    }

    device_data = (struct device_data *)kzalloc(sizeof(struct device_data), GFP_KERNEL);
    if (!device_data)
    {
        dev_err(global_dev, "kzalloc failed\n");
        goto early_failure;
    }

    if (!request_mem_region(res.start, resource_size(&res), subblock_name))
    {
        dev_err(global_dev, "%s request_mem_region failed: %s\n", subblock_name, __func__);
        goto request_mem_region_failed;
    } 

    device_data->virt_addr = of_iomap(child_node, 0); // get virtual address
    if (device_data->virt_addr == NULL)
    {
        dev_err(global_dev, "%s of_iomap failed: %s\n", subblock_name, __func__);
        device_data->virt_addr = 0;
        goto iomap_failed;
    }

    // get the variable instance - so we know which block instance it is
    if (of_property_read_u32(child_node, "instance", &instance))
    {
        dev_err(global_dev, "reading instance failed\n");
        goto read_dtsi_failed;
    }

    device_data->instance = instance;
    device_data->phys_addr = res.start;
    device_data->addr_size = resource_size(&res); 
    device_data->irq = irq;  
    device_data->exit_func = exit_func;
    debug_print("%s: virt_addr=0x%p, phys_addr=0x%X\n", subblock_name,
                device_data->virt_addr, device_data->phys_addr);
    
    // if we are a subblock that has its own IRQ (the calling function knows this), 
    // then map the interrupt service routine
    if (irq > -1)
    {
        retval = request_threaded_irq(irq, NULL,
                                      icetest_platform_irq, IRQF_ONESHOT,
                                      "ICETEST", device_data);    
        if (retval)
        {
            dev_err(global_dev, "request_irq failed\n");
            goto request_irq_failed;
        }
    }

    // add the device_data to the linked list
    list_add(&device_data->linked_list, device_data_list);
    
    init_func(device_data);

    of_node_put(child_node);  // of_get_child_by_name() requires of_node_put() when done
    return device_data;

request_irq_failed:
    device_data->irq = 0;
read_dtsi_failed:    
    iounmap(device_data->virt_addr); 
iomap_failed:
    release_mem_region(device_data->phys_addr, device_data->addr_size); 
request_mem_region_failed:
    kfree(device_data);
early_failure:
    of_node_put(child_node);  // of_get_child_by_name() requires of_node_put() when done
    return NULL;
}

static LIST_HEAD(device_data_list);

// one probe received for the entire icetest structured device tree entry
static int icetest_platform_probe(struct platform_device *pdev)
{
    int retval;
    int irq;
    struct device_node *parent_node = pdev->dev.of_node;

    int ignore;
    struct device_data *device_data;
    // initialize the head ptr for the whole linked list of device_data
    global_dev = &pdev->dev; // for error print statements (dev_err)
    
    if (!parent_node)
    {
        dev_err(global_dev, "Could not find device node %s\n", "icetest");
        return -ENODEV;
    }
    else
    {
        debug_print("of_node name: %s 0x%p\n",parent_node->full_name, parent_node);
    }

    // get the irq from the parent node (icetest has 1 interrupt, icetest-common will store it)
    irq = irq_of_parse_and_map(parent_node, 0);
    if (irq < 0)
    {
        dev_err(global_dev, "platform_get_irq failed\n");
        retval = -ENXIO;
        goto early_failure;
    }
    debug_print("Probe in %s: irq = %d\n",__func__,irq);

    // icetest has 2 subblocks - parse them both out of the device tree, allocate device_data
    // for each of them, put that data on the device_data_list, and call their
    // respective init functions in the main driver code
    device_data = init_subblock("icetest-common", icetest_init, icetest_exit, irq,
                                 parent_node, &device_data_list);

    device_data = init_subblock("icetest-idma", icetest_idma_init, icetest_idma_exit, -1,
                                parent_node, &device_data_list);

    platform_set_drvdata(pdev, &device_data_list); // store our device_data linked list into the device struct

    // for debugging and bringup to read registers- not needed for everyday driver work
    ignore = sysfs_create_files(&(pdev->dev.kobj), icetest_attrs);

    return 0;
    
// probe_failed:
    dev_err(global_dev, "FAILURE IN PROBE - FREEING IRQ\n");
    free_irq(device_data->irq, device_data);

early_failure:    
    return retval;
}

// cleanup everything that probe did - run through the device data linked list
// and remove from list, unmap, free mem, etc.
static int icetest_platform_remove(struct platform_device *pdev)
{
    struct list_head *device_data_list_ptr = platform_get_drvdata(pdev); // get llist of device_datas
    struct device_data *device_data, *dd_next; // dd_next is used by list function/macro

    list_for_each_entry_safe(device_data, dd_next, device_data_list_ptr, linked_list)
    {
        // items were added to the list LIFO, and we remove FIFO
        list_del(&device_data->linked_list);
        device_data->exit_func(device_data);  // call exit function for driver subblock
        iounmap(device_data->virt_addr);
        release_mem_region(device_data->phys_addr, device_data->addr_size);
        if (device_data->irq > 0)
            free_irq(device_data->irq, device_data); 
        kfree(device_data);
    }
    platform_set_drvdata(pdev, NULL);
    sysfs_remove_files(&(pdev->dev.kobj), icetest_attrs);
    debug_print("%s: complete\n",__func__);
    return 0;
}

static int icetest_platform_suspend(struct platform_device *pdev, pm_message_t state)
{
    struct device_data *device_data = platform_get_drvdata(pdev);
    print(KERN_WARNING "%s event = %d\n",__func__, state.event);

    // allocate suspend context structure
    device_data->device_suspend_context = kmalloc(sizeof(struct device_suspend_context_struct), GFP_KERNEL);
    if (device_data->device_suspend_context == NULL)
    {
        return -1;
    }
    
    return (0);
}

static int icetest_platform_resume(struct platform_device *pdev)
{
    struct device_data *device_data = platform_get_drvdata(pdev);
    print(KERN_WARNING "%s\n",__func__);

    if (device_data->device_suspend_context == NULL)
    {
        error_print(KERN_ERR "%s: invalid suspend context\n",__func__);
        return -1;
    }
    
    // restore block context here TODO FIXME what to do?

    // free the suspend context
    kfree(device_data->device_suspend_context);
    device_data->device_suspend_context = NULL;
    return(0);
}

#ifdef CONFIG_OF
static const struct of_device_id mrvl_icetest_dt_match[] = {

    { .compatible = "icetest block"},  // IP Major Tag Rev 0, Rev 1
    {},
};
MODULE_DEVICE_TABLE(of, mrvl_icetest_dt_match);
#endif

static struct platform_driver icetest_platform_driver =
{
    .probe   = icetest_platform_probe,
    .remove  = icetest_platform_remove,
    .suspend = icetest_platform_suspend,
    .resume = icetest_platform_resume,
//    .id_table = mrvl_icetest_driver_ids, ???
    .driver  = {
        .name  = "icetest_block",
        .owner = THIS_MODULE,
        .of_match_table = of_match_ptr(mrvl_icetest_dt_match),
    }
};

module_platform_driver(icetest_platform_driver);

MODULE_AUTHOR("Copyright (c) 2014 Marvell International Ltd.");
MODULE_DESCRIPTION("Marvell ICETest subblock driver");

MODULE_LICENSE("GPL");
MODULE_VERSION("2014_Feb_7");

