/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/spinlock.h>   // for spinlock_t, printk, BUG_ON, and memset
#include <linux/io.h>
#include <linux/export.h>

#include "ICE_test_dma_regstructs.h"
#include "ICE_test_dma_regmasks.h"

#include "icetest_data.h"
#include "icetest_if.h"
#include "icetest_driver.h"
#include "icetest_idma.h"

static void set_all_irqstruct(struct device_data *unused,
                              struct icetest_idma_interrupt_info *irqstruct, bool setval)
{
    irqstruct->rresp_int = setval;
    irqstruct->bresp_int = setval;
    irqstruct->rst_int = setval;
    irqstruct->eoi_int = setval;
    irqstruct->fin_int = setval;
    irqstruct->who_int = setval;    
}

static uint32_t convert_irqstruct_to_update_uintarray(struct device_data *unused,
                                                      uint32_t ints,
                                                      struct icetest_idma_interrupt_info *irqstruct,
                                                      bool set_to_one,
                                                      enum mask_type operation)
{
    uint8_t set_val;

    set_val = set_to_one?1:0;

    // NOTE, if all of the IRQ registers were not using the same IRQ bit positions,
    // we would need to use the operation parameter - but we aren't currently, since
    // it isn't needed.  See icetest_common.c for an example of how it is used

    if (irqstruct->rresp_int)
        ints = TESTIDMA_INT_ST_RRESP_INT_ST_REPLACE_VAL(ints, set_val);
    if (irqstruct->bresp_int)
        ints = TESTIDMA_INT_ST_BRESP_INT_ST_REPLACE_VAL(ints, set_val);
    if (irqstruct->rst_int)
        ints = TESTIDMA_INT_ST_RST_INT_ST_REPLACE_VAL(ints, set_val);
    if (irqstruct->eoi_int)
        ints = TESTIDMA_INT_ST_EOI_INT_ST_REPLACE_VAL(ints, set_val);
    if (irqstruct->fin_int)
        ints = TESTIDMA_INT_ST_FIN_INT_ST_REPLACE_VAL(ints, set_val);
    if (irqstruct->who_int)
        ints = TESTIDMA_INT_ST_WHO_INT_ST_REPLACE_VAL(ints, set_val);
    return ints;
}

void convert_uintarray_to_irqstruct(struct device_data *unused, uint32_t ints,
                                    struct icetest_idma_interrupt_info *irqstruct,
                                    enum mask_type operation)
{
    // NOTE, if all of the IRQ registers were not using the same IRQ bit positions,
    // we would need to use the operation parameter - but we aren't currently, since
    // it isn't needed.  See icetest_common.c for an example of how it is used

    // if the given bit is set, the interrupt field gets a true, false otherwise
    irqstruct->rresp_int = (TESTIDMA_INT_ST_RRESP_INT_ST_MASK_SHIFT(ints) != 0);
    irqstruct->bresp_int= (TESTIDMA_INT_ST_BRESP_INT_ST_MASK_SHIFT(ints) != 0);
    irqstruct->rst_int = (TESTIDMA_INT_ST_RST_INT_ST_MASK_SHIFT(ints) != 0);
    irqstruct->eoi_int = (TESTIDMA_INT_ST_EOI_INT_ST_MASK_SHIFT(ints) != 0);
    irqstruct->fin_int = (TESTIDMA_INT_ST_FIN_INT_ST_MASK_SHIFT(ints) != 0);
    irqstruct->who_int = (TESTIDMA_INT_ST_WHO_INT_ST_MASK_SHIFT(ints) != 0);
    irqstruct->debug_intarray = ints;
}

// the callback to be called by the interrupt service routine
static void register_callback(struct device_data *device_data, void *callback_fcn)
{
    device_data->interrupt_callback = callback_fcn;
}

static uint32_t clear_icetest_idma_irqs_uint32(struct device_data *device_data,
                                               uint32_t int_val)
{
    unsigned long flags;  
    uint32_t reg;

    PROTECT_INTREG_ACCESS_IRQ;
    reg = iceTestIDMARead(int_st);
    iceTestIDMAWrite(int_cl, int_val);  // clear requested ints
    UNPROTECT_INTREG_ACCESS_IRQ;
    return reg;
}

// clear requested (or all set) interrupts - return the ones that
// were set if the pointer was non-NULL
// 
void clear_icetest_idma_irqs(struct device_data *device_data,
                             struct icetest_idma_interrupt_info *irqstruct)
{
    uint32_t reg, irqs_to_clear;

    if (irqstruct == NULL)
    {
        reg = clear_icetest_idma_irqs_uint32(device_data, 0xFFFFFFFF);  // just clear everything and get out
        return; // ignoring the reg return
    }
    
    irqs_to_clear = convert_irqstruct_to_update_uintarray(NULL, 0, irqstruct, true, ack);
    reg = clear_icetest_idma_irqs_uint32(device_data, irqs_to_clear);

    // set fields for all ints that were set
    convert_uintarray_to_irqstruct(NULL, reg, irqstruct, pending);
}

void enable_icetest_idma_irqs(struct device_data *device_data,
                              struct icetest_idma_interrupt_info *irqstruct)
{
    unsigned long flags;
    uint32_t reg;
    struct icetest_idma_interrupt_info irq;

    if (irqstruct == NULL)
    {
        set_all_irqstruct(NULL, &irq, true);
        irqstruct = &irq;
    }
    // enable interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = iceTestIDMARead(int_en);
    reg = convert_irqstruct_to_update_uintarray(NULL, reg, irqstruct, true, enable);
    iceTestIDMAWrite(int_en, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

void disable_icetest_idma_irqs(struct device_data *device_data,
                               struct icetest_idma_interrupt_info *irqstruct)
{
    unsigned long flags;
    uint32_t reg;
    struct icetest_idma_interrupt_info irq;

    if (irqstruct == NULL)
    {
        set_all_irqstruct(NULL, &irq, true);
        irqstruct = &irq;
    }
    // disable interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = iceTestIDMARead(int_en);
    reg = convert_irqstruct_to_update_uintarray(NULL, reg, irqstruct, false, enable);
    iceTestIDMAWrite(int_en, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

// force each field that has a field set
void force_icetest_idma_irqs(struct device_data *device_data, 
                             struct icetest_idma_interrupt_info *irqstruct)
{
    uint32_t reg;
    unsigned long flags;
    struct icetest_idma_interrupt_info irq;

    reg = 0;

    if (irqstruct == NULL)
    {
        set_all_irqstruct(NULL, &irq, true);
        irqstruct = &irq;
    }
    PROTECT_INTREG_ACCESS_IRQ;
    // This is a write-only register - no read-modify-write possible
    reg = convert_irqstruct_to_update_uintarray(NULL, reg, irqstruct, true, force);
    iceTestIDMAWrite(int_fo, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
    debug_print("icetest idma force ints on %d \n",device_data->instance);
}

/**
 * this function runs in interrupt context - no long operations.  Just
 * clear the ints, store the int data, and call the registered callback
 */
///  NOTE - This function runs in interrupt context - no long operations allowed
void handle_icetest_idma_irqs(struct device_data *device_data)
{
    uint32_t intval;
    struct icetest_idma_interrupt_info irqstruct;    

    intval = clear_icetest_idma_irqs_uint32(device_data, 0xFFFFFFFF); // clear all ints
    
    if (intval != 0)
    {
        convert_uintarray_to_irqstruct(NULL, intval, &irqstruct, pending);
        // if someone registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback(&irqstruct);
        }
    }
}

void dump_icetest_idma_regs(struct device_data *device_data)
{
    print("ICETEST IDMA Dump for %d\n", device_data->instance);
    print("cfg=0x%08X  ",iceTestIDMARead(cfg));
    print("status=0x%08X  ",iceTestIDMARead(status));
    print("line_width=0x%08X\n",iceTestIDMARead(line_width));
    print("int_en=0x%08X  ",iceTestIDMARead(int_en));
    print("int_st=0x%08X\n",iceTestIDMARead(int_st));
    print("desc_read=0x%08X  ",iceTestIDMARead(desc_read));
    print("xfer_length=0x%08X  ",iceTestIDMARead(xfer_length));
    print("xfer_addr=0x%08X  ",iceTestIDMARead(xfer_addr));
    print("ctrl_word=0x%08X\n",iceTestIDMARead(ctrl_word));
    print("REV0=0x%08X  ",iceTestIDMARead(REV0)); 
    print("REV1=0x%08X\n",iceTestIDMARead(REV1)); 
}

// Write-only register, and only 1 field to set
void start_icetest_idma(struct device_data *device_data, dma_addr_t phys_addr)
{
    iceTestIDMAWrite(desc_write, phys_addr);
}

void get_icetest_idma_desc_read(struct device_data *device_data, int *value)
{
    *value = iceTestIDMARead(desc_read);
}

// set the icetest idma config register
void set_icetest_idma_config(struct device_data *device_data, struct icetest_idma_config_info *info)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = iceTestIDMARead(cfg);
    if (info->replicate_valid)
        reg = TESTIDMA_CFG_REPLICATE_REPLACE_VAL(reg, info->replicate);
    if (info->enable_valid)
        reg = TESTIDMA_CFG_ENABLE_REPLACE_VAL(reg, info->enable);
    if (info->out_width_valid)
        reg = TESTIDMA_CFG_OUT_WIDTH_REPLACE_VAL(reg, info->out_width);
    if (info->handshake_enable_valid)
        reg = TESTIDMA_CFG_HANDSHAKEENABLE_REPLACE_VAL(reg, info->handshake_enable);
    if (info->own_write_disable_valid)
        reg = TESTIDMA_CFG_OWNWRITEDISABLE_REPLACE_VAL(reg, info->own_write_disable);
    if (info->own_polarity_valid)
        reg = TESTIDMA_CFG_OWNPOLARITY_REPLACE_VAL(reg, info->own_polarity);
    if (info->burst_len_valid)
        reg = TESTIDMA_CFG_BURST_LEN_REPLACE_VAL(reg, info->burst_len);
    iceTestIDMAWrite(cfg, reg);
    UNPROTECT_REG_ACCESS;
}

// get the icetest idma config register
void get_icetest_idma_config(struct device_data *device_data, struct icetest_idma_config_info *info)
{
    uint32_t reg;
    
    reg = iceTestIDMARead(cfg);
    info->replicate = TESTIDMA_CFG_REPLICATE_MASK_SHIFT(reg);
    info->enable = TESTIDMA_CFG_ENABLE_MASK_SHIFT(reg);
    info->out_width = TESTIDMA_CFG_OUT_WIDTH_MASK_SHIFT(reg);
    info->handshake_enable = TESTIDMA_CFG_HANDSHAKEENABLE_MASK_SHIFT(reg);
    info->own_write_disable = TESTIDMA_CFG_OWNWRITEDISABLE_MASK_SHIFT(reg);
    info->own_polarity = TESTIDMA_CFG_OWNPOLARITY_MASK_SHIFT(reg);
    info->burst_len = TESTIDMA_CFG_BURST_LEN_MASK_SHIFT(reg);
    info->debug_array = reg;
}

// get the icetest idma status register
void get_icetest_idma_status(struct device_data *device_data, struct icetest_idma_status_info *info)
{
    uint32_t reg;
    
    reg = iceTestIDMARead(status);
    info->dma_paused = TESTIDMA_STATUS_DMAPAUSED_MASK_SHIFT(reg);
    info->pause_after = TESTIDMA_STATUS_PAUSEAFTER_MASK_SHIFT(reg);
    info->pause_before = TESTIDMA_STATUS_PAUSEBEFORE_MASK_SHIFT(reg);
    info->soi = TESTIDMA_STATUS_SOI_MASK_SHIFT(reg);
    info->eoi = TESTIDMA_STATUS_EOI_MASK_SHIFT(reg);
    info->soft_reset = TESTIDMA_STATUS_SOFTRESET_MASK_SHIFT(reg);
    info->full_cbuf = TESTIDMA_STATUS_FULL_CBUF_MASK_SHIFT(reg);
    info->empty_cbuf = TESTIDMA_STATUS_EMPTY_CBUF_MASK_SHIFT(reg);        
    info->full_dbuf = TESTIDMA_STATUS_FULL_DBUF_MASK_SHIFT(reg);
    info->empty_dbuf = TESTIDMA_STATUS_EMPTY_DBUF_MASK_SHIFT(reg);
    info->packer_empty = TESTIDMA_STATUS_PACKER_EMPTY_MASK_SHIFT(reg);
    info->dma_busy = TESTIDMA_STATUS_DMA_BUSY_MASK_SHIFT(reg);
    info->debug_array = reg;
}

void get_icetest_idma_linewidth(struct device_data *device_data, int *value)
{
    *value = iceTestIDMARead(line_width);
}

void set_icetest_idma_linewidth(struct device_data *device_data, int value)
{
    iceTestIDMAWrite(line_width, value);
}

void get_icetest_idma_transfer_length(struct device_data *device_data, int *value)
{
    *value = iceTestIDMARead(xfer_length);
}

void get_icetest_idma_transfer_addr(struct device_data *device_data, int *value)
{
    *value = iceTestIDMARead(xfer_addr);
}

void get_icetest_idma_ctrl_word(struct device_data *device_data, int *value)
{
    *value = iceTestIDMARead(ctrl_word);
}

void set_icetest_idma_soft_reset(struct device_data *device_data, int value)
{
    iceTestIDMAWrite(reset, value);
}

struct icetest_idma_function_struct icetest_idma_functions =
{
    .version = 0,
    .dump_icetest_idma_regs = dump_icetest_idma_regs,
    .start_icetest_idma = start_icetest_idma,
    .convert_uintarray_to_irqstruct = convert_uintarray_to_irqstruct,
    .get_icetest_idma_desc_read = get_icetest_idma_desc_read,
    .set_icetest_idma_config = set_icetest_idma_config,
    .get_icetest_idma_config = get_icetest_idma_config,
    .get_icetest_idma_status = get_icetest_idma_status,
    .get_icetest_idma_linewidth = get_icetest_idma_linewidth,
    .set_icetest_idma_linewidth = set_icetest_idma_linewidth,
    .get_icetest_idma_transfer_length = get_icetest_idma_transfer_length,
    .get_icetest_idma_transfer_addr = get_icetest_idma_transfer_addr,
    .get_icetest_idma_ctrl_word = get_icetest_idma_ctrl_word,
    .set_icetest_idma_soft_reset = set_icetest_idma_soft_reset,
    .handle_icetest_idma_irqs = handle_icetest_idma_irqs,
    .clear_icetest_idma_irqs = clear_icetest_idma_irqs,
    .enable_icetest_idma_irqs = enable_icetest_idma_irqs,
    .disable_icetest_idma_irqs = disable_icetest_idma_irqs,
    .register_callback = register_callback,
};

void icetest_idma_init(struct device_data *device_data)
{
    // register with the parent
    device_data->fcn_tbl = &icetest_idma_functions;
    register_icetest_subblock(icetest_idma, device_data, device_data->instance);

    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses reg_spinlock
    spin_lock_init(&(device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(device_data->reg_spinlock));
    device_data->interrupt_callback = NULL;    
}
EXPORT_SYMBOL(icetest_idma_init);

void icetest_idma_exit(struct device_data *device_data)
{
    // unregister with the parent
    unregister_icetest_subblock(icetest_idma, device_data->instance);
}
EXPORT_SYMBOL(icetest_idma_exit);
