/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

// driver for the icetest common subblock

#include <linux/irqreturn.h> // needed for irqreturn_t and IRQ_HANDLED
#include <linux/spinlock.h>   // for spinlock_t, printk, BUG_ON, and memset
#include <linux/io.h> // for ioread/write32 (Read/Write macros)
#include <linux/export.h> // for EXPORT_SYMBOL

#include "ICE_test_dma_regstructs.h"
#include "ICE_test_dma_regmasks.h"

#include "icetest_if.h"
#include "icetest_driver.h"
#include "icetest_data.h"
#include "icetest_common.h"

//#define GET_VERSION(device_data) ( device_data->version )
#define GET_VERSION(device_data) ( ICE_TEST_DMA_REV0_MAJ_MASK_SHIFT(iceTestRead(REV0)) )

void set_all_irqstruct(struct device_data *unused, struct icetest_interrupt_info *irqstruct,
                       bool setval)
{
    irqstruct->overflow_int = setval;
    irqstruct->dma_int = setval;
    irqstruct->underflow_int = setval;
}

// pass in the current value for the interrupt register, and only the set fields
// in the irqstruct will update it.
// This function is structured for the case where the interrupt registers have different
// bit mappings for enable, pending, ack, or force.  It is required that the caller
// pass in the type of masks to use for conversion
static uint32_t convert_irqstruct_to_update_uintarray(struct device_data *device_data,
                                                      uint32_t ints,
                                                      struct icetest_interrupt_info *irqstruct,
                                                      bool set_to_one,
                                                      enum mask_type operation)
{
    uint8_t set_val;
    
    set_val = set_to_one?1:0;
    if (irqstruct->overflow_int)
    {
        switch(operation)
        {
        case pending:
            if (GET_VERSION(device_data) == REV_A_GR_ICETEST)
                ints = ICE_TEST_DMA_INTPEND_OVERINTPEND_REPLACE_VAL_REVA(ints, set_val);
            else
                ints = ICE_TEST_DMA_INTPEND_OVERINTPEND_REPLACE_VAL(ints, set_val);
            break;
        case enable:
            ints = ICE_TEST_DMA_INTEN_OVERFLOWINTEN_REPLACE_VAL(ints, set_val);
            break;
        case ack:
            ints = ICE_TEST_DMA_INTACK_OVERFLOWINTACK_REPLACE_VAL(ints, set_val);
            break;
        case force:
            ints = ICE_TEST_DMA_INTFORCE_OVERFLOWINTFORCE_REPLACE_VAL(ints, set_val);
            break;
        }
    }
    if (irqstruct->dma_int)
    {
        switch(operation)
        {
        case pending:
            if (GET_VERSION(device_data) == REV_A_GR_ICETEST)
                ints = ICE_TEST_DMA_INTPEND_DMAINTPEND_REPLACE_VAL_REVA(ints, set_val);
            else
                ints = ICE_TEST_DMA_INTPEND_DMAINTPEND_REPLACE_VAL(ints, set_val);
            break;
        case enable:
        case ack:
        case force:
            debug_print("no operation defined for dma_int other than pending, ignoring %d\n", operation);
            break;

        // no bits for dma_int in the other cases - don't error in case someone just set all
        // bits true
        }
    }
    if (irqstruct->underflow_int)
    {
        switch(operation)
        {
        case pending:
            ints = ICE_TEST_DMA_INTPEND_UNDERINTPEND_REPLACE_VAL(ints, set_val);
            break;
        case enable:
            ints = ICE_TEST_DMA_INTEN_UNDERINTEN_REPLACE_VAL(ints, set_val);
            break;
        case ack:
            ints = ICE_TEST_DMA_INTACK_UNDERINTACK_REPLACE_VAL(ints, set_val);
            break;
        case force:
            ints = ICE_TEST_DMA_INTFORCE_UNDERINTFORCE_REPLACE_VAL(ints, set_val);
            break;
        }
    }
    return ints;
}

// This function is structured for the case where the interrupt registers have different
// bit mappings for enable, pending, ack, or force.  It is required that the caller
// pass in the type of masks to use for conversion
static void convert_uintarray_to_irqstruct(struct device_data *device_data,
                                           uint32_t ints,
                                           struct icetest_interrupt_info *irqstruct,
                                           enum mask_type operation)
{
    switch(operation)
    {
    case pending:
        if (GET_VERSION(device_data) == REV_A_GR_ICETEST)
        {
            irqstruct->overflow_int = (ICE_TEST_DMA_INTPEND_OVERINTPEND_MASK_SHIFT_REVA(ints) != 0);
            irqstruct->dma_int = (ICE_TEST_DMA_INTPEND_DMAINTPEND_MASK_SHIFT_REVA(ints) != 0);
        }
        else
        {
            irqstruct->overflow_int = (ICE_TEST_DMA_INTPEND_OVERINTPEND_MASK_SHIFT(ints) != 0);
            irqstruct->dma_int = (ICE_TEST_DMA_INTPEND_DMAINTPEND_MASK_SHIFT(ints) != 0);
        }
        irqstruct->underflow_int = (ICE_TEST_DMA_INTPEND_UNDERINTPEND_MASK_SHIFT(ints) != 0);
        break;
    case enable:
        irqstruct->overflow_int = (ICE_TEST_DMA_INTEN_OVERFLOWINTEN_MASK_SHIFT(ints) != 0);
        irqstruct->dma_int = false; // no int bit for enable
        irqstruct->underflow_int = (ICE_TEST_DMA_INTEN_UNDERINTEN_MASK_SHIFT(ints) != 0);
        break;
    case ack:
        irqstruct->overflow_int = (ICE_TEST_DMA_INTACK_OVERFLOWINTACK_MASK_SHIFT(ints) != 0);
        irqstruct->dma_int = false; // no int bit for ack
        irqstruct->underflow_int = (ICE_TEST_DMA_INTACK_UNDERINTACK_MASK_SHIFT(ints) != 0);
        break;
    case force:
        irqstruct->overflow_int = (ICE_TEST_DMA_INTFORCE_OVERFLOWINTFORCE_MASK_SHIFT(ints) != 0);
        irqstruct->dma_int = false; // no int bit for force
        irqstruct->underflow_int = (ICE_TEST_DMA_INTFORCE_UNDERINTFORCE_MASK_SHIFT(ints) != 0);
        break;
    }
    irqstruct->debug_intarray = ints;
}

static void register_callback(struct device_data *device_data, void *callback_fcn)
{
    device_data->interrupt_callback = callback_fcn;
}

///  NOTE - This function can run in interrupt context - no long operations allowed
static uint32_t clear_icetest_irqs_uint32(struct device_data *device_data,
                                          uint32_t int_val)
{
    unsigned long flags; // for IRQ protection macro
    uint32_t reg;

    PROTECT_INTREG_ACCESS_IRQ;
    reg = iceTestRead(IntPend);
    iceTestWrite(IntAck,int_val);  // clear requested ints
    UNPROTECT_INTREG_ACCESS_IRQ;
    return reg;
}

void clear_icetest_irq(struct device_data *device_data,
                        struct icetest_interrupt_info *irqstruct)

{
    uint32_t reg, ints_to_clear;

    if (irqstruct == NULL)
    {
        reg = clear_icetest_irqs_uint32(device_data, 0xFFFFFFFF); // just clear everything
        return; // ignore the reg return
    }
    ints_to_clear = convert_irqstruct_to_update_uintarray(device_data, 0, irqstruct, true, ack);
    reg = clear_icetest_irqs_uint32(device_data, ints_to_clear);

    // set fields for all ints that were set
    convert_uintarray_to_irqstruct(device_data, reg, irqstruct, pending);
}

void enable_icetest_irq(struct device_data *device_data,
                        struct icetest_interrupt_info *irqstruct)
{
    unsigned long flags;
    uint32_t reg;
    struct icetest_interrupt_info irq;
    
    if (irqstruct == NULL)
    {
        set_all_irqstruct(NULL, &irq, true);
        irqstruct = &irq;
    }
    // enable interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = iceTestRead(IntEn);
    reg = convert_irqstruct_to_update_uintarray(device_data, reg, irqstruct, true, enable);
    iceTestWrite(IntEn, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

void disable_icetest_irq(struct device_data *device_data,
                         struct icetest_interrupt_info *irqstruct)
{
    unsigned long flags;
    uint32_t reg;
    struct icetest_interrupt_info irq;

    if (irqstruct == NULL)
    {
        set_all_irqstruct(NULL, &irq, true);
        irqstruct = &irq;
    }
    // disable interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = iceTestRead(IntEn);
    reg = convert_irqstruct_to_update_uintarray(device_data, reg, irqstruct, false, enable);
    iceTestWrite(IntEn, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

// force an interrupt for each field that is set to true
void force_icetest_irq(struct device_data *device_data,
                       struct icetest_interrupt_info *irqstruct)
{
    unsigned long flags;
    uint32_t reg;
    struct icetest_interrupt_info irq;

    reg = 0;

    if (irqstruct == NULL)
    {
        set_all_irqstruct(NULL, &irq, true);
        irqstruct = &irq;
    }
    PROTECT_INTREG_ACCESS_IRQ;
    // This is a write-only register - no read-modify-write possible
    reg = convert_irqstruct_to_update_uintarray(device_data, reg, irqstruct, true, force);
    iceTestWrite(IntForce, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

/*************************************
 * ice_platform_irq: interrupt handler
 * @irq:  irq number
 * @pdev: interrupt source
 *
 * This function returns IRQ_HANDLED if the IRQ has been handled
 * This is an ISR don't trace, use attribute interface instead
 **************************************/
///  NOTE - This function runs in interrupt context - no long operations allowed
irqreturn_t icetest_platform_irq(int irq, void *pdev)
{
    uint32_t pending_reg_val, ack_reg_val, intval;
    struct device_data *device_data;
    struct icetest_interrupt_info irqstruct;
    unsigned long flags;

    device_data = pdev;
    // check for the force register - testing uses this to cause an interrupt - clear it
    PROTECT_INTREG_ACCESS_IRQ;
    pending_reg_val = iceTestRead(IntPend);
    // force register is write only, and a 1 shot, so don't mess with it here
    
    // now clear the interrupts

    // NOTE THAT depending on the version of the icetest common block,
    // the Pending register may have a different bit order from the
    // ack register, so don't just assume it is ok to write the value
    // we read to clear only the ints we have stored away.  So, we
    // have to clear the bits individually by building up an ack
    // register write value interrupt bit by interrupt bit
    ack_reg_val = 0;
    if (GET_VERSION(device_data) == REV_A_GR_ICETEST)
    {
        // overflow int moved between revA and later
        if (ICE_TEST_DMA_INTPEND_OVERINTPEND_MASK_REVA & pending_reg_val)
            ack_reg_val = ICE_TEST_DMA_INTACK_OVERFLOWINTACK_MASK;
    }
    else
    {
        if (ICE_TEST_DMA_INTPEND_OVERINTPEND_MASK & pending_reg_val)
            ack_reg_val = ICE_TEST_DMA_INTACK_OVERFLOWINTACK_MASK;
    }

    // underflow int unmoved between revA and later
    if (ICE_TEST_DMA_INTPEND_UNDERINTPEND_MASK & pending_reg_val)
        ack_reg_val |= ICE_TEST_DMA_INTACK_UNDERINTACK_MASK;

    iceTestWrite(IntAck,ack_reg_val);
    UNPROTECT_INTREG_ACCESS_IRQ;
    // Check to see if DMA is causing the interrupt
    if (GET_VERSION(device_data) == REV_A_GR_ICETEST)
        intval = ICE_TEST_DMA_INTPEND_DMAINTPEND_MASK_SHIFT_REVA(pending_reg_val);
    else
        intval = ICE_TEST_DMA_INTPEND_DMAINTPEND_MASK_SHIFT(pending_reg_val);
    if (intval != 0)
    {
        // service any DMA interrupts
        icetest_handle_idma_irqs();
    }

    if (pending_reg_val != 0)
    {
        // unpack the interrupts into the interrupt info structure
        convert_uintarray_to_irqstruct(device_data, pending_reg_val, &irqstruct, pending);
        // now that we have a list of interrupts that were set,
        // execute the registered callback function
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback(&irqstruct);
        }
    }
    return IRQ_HANDLED;
}

void dump_icetest_regs(struct device_data *device_data)
{
    print("ICETEST reg dump\n");
    print("TestConfig=0x%08X  ",iceTestRead(TestConfig));
    print("IntEn=0x%08X  ",iceTestRead(IntEn));
    print("IntPend=0x%08X\n",iceTestRead(IntPend));
    print("EOLBlank=0x%08X  ",iceTestRead(EOLBlank));
    print("REV0=0x%08X  ",iceTestRead(REV0));
    print("REV1=0x%08X\n",iceTestRead(REV1));
}
EXPORT_SYMBOL(dump_icetest_regs);  // for debugging

uint32_t get_icetest_rev0(struct device_data *device_data)
{
    return iceTestRead(REV0);
}

uint32_t get_icetest_rev1(struct device_data *device_data)
{
    return iceTestRead(REV1);
}

// set the icetest config register
void set_icetest_config(struct device_data *device_data, struct icetest_testconfig_info *cfg)
{
    uint32_t reg;
    
    PROTECT_REG_ACCESS;
    reg = iceTestRead(TestConfig);
    if (cfg->pic_rate_valid)
        reg = ICE_TEST_DMA_TESTCONFIG_PICRATE_REPLACE_VAL(reg, cfg->pic_rate);
    if (cfg->disable_pacing_valid)
        reg = ICE_TEST_DMA_TESTCONFIG_DISABLEPACING_REPLACE_VAL(reg, cfg->disable_pacing);
    if (cfg->blank_en_valid)
        reg = ICE_TEST_DMA_TESTCONFIG_BLANKEN_REPLACE_VAL(reg, cfg->blank_en);
    if (cfg->softreset_active_valid)
        reg = ICE_TEST_DMA_TESTCONFIG_TESTSR_REPLACE_VAL(reg, cfg->softreset_active);
    if (cfg->icetest_enable_valid)
        reg = ICE_TEST_DMA_TESTCONFIG_TESTEN_REPLACE_VAL(reg, cfg->icetest_enable);
    if (cfg->icetest_mode_select_valid)
        reg = ICE_TEST_DMA_TESTCONFIG_TESTSEL_REPLACE_VAL(reg, cfg->icetest_mode_select);
    iceTestWrite(TestConfig, reg);
    UNPROTECT_REG_ACCESS;
}

// get the icetest config register
void get_icetest_config(struct device_data *device_data, struct icetest_testconfig_info *info)
{
    uint32_t reg;
    
    reg = iceTestRead(TestConfig);
    info->pic_rate = ICE_TEST_DMA_TESTCONFIG_PICRATE_MASK_SHIFT(reg);
    info->disable_pacing = ICE_TEST_DMA_TESTCONFIG_DISABLEPACING_MASK_SHIFT(reg);
    info->blank_en = ICE_TEST_DMA_TESTCONFIG_BLANKEN_MASK_SHIFT(reg);
    info->softreset_active = ICE_TEST_DMA_TESTCONFIG_TESTSR_MASK_SHIFT(reg);
    info->icetest_enable = ICE_TEST_DMA_TESTCONFIG_TESTEN_MASK_SHIFT(reg);
    info->icetest_mode_select = ICE_TEST_DMA_TESTCONFIG_TESTSEL_MASK_SHIFT(reg);
}

struct icetest_common_function_struct icetest_functions =
{
    .version = 0,
    .set_all_irqstruct = set_all_irqstruct,
    .clear_icetest_irq = clear_icetest_irq,
    .enable_icetest_irq = enable_icetest_irq,
    .disable_icetest_irq = disable_icetest_irq,        
    .force_icetest_irq = force_icetest_irq,
    .set_icetest_config = set_icetest_config,
    .get_icetest_config = get_icetest_config,
    .dump_icetest_regs = dump_icetest_regs,
    .get_icetest_rev0 = get_icetest_rev0,
    .get_icetest_rev1 = get_icetest_rev1,    
    .register_callback = register_callback,
};

void icetest_init(struct device_data *device_data)
{
    device_data->fcn_tbl = &icetest_functions;
    
    // register with the parent
    register_icetest_subblock(common, device_data, device_data->instance);
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses reg_spinlock
    spin_lock_init(&(device_data->int_spinlock));
    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(device_data->reg_spinlock));
    device_data->interrupt_callback = NULL;
}
EXPORT_SYMBOL(icetest_init);

void icetest_exit(struct device_data *device_data)
{
    // unregister with the parent
    unregister_icetest_subblock(common, device_data->instance);
}
EXPORT_SYMBOL(icetest_exit);
