/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

// top level driver for the icetest block
#include <linux/spinlock.h> // for spinlock_t, printk, BUG_ON
#include <linux/export.h>
#include <linux/string.h>  // for memset

#include "icetest_data.h"
#include "icetest_if.h"
#include "icetest_driver.h"

struct icetest_object_struct
{
    icetestDeviceHandle *icetest_handle;
    icetestIDMADeviceHandle *icetest_idma_handle;
};

struct icetest_object_struct icetest_object;

#define ICETEST_COMMON_RETRIEVE_handle_ft                               \
    icetestDeviceHandle *icetest_common_handle;                         \
    struct icetest_common_function_struct *ft;                          \
                                                                        \
    icetest_common_handle = icetest_object.icetest_handle;              \
    BUG_ON(icetest_common_handle == NULL);                              \
    ft = icetest_common_handle->fcn_tbl;

#define ICETEST_IDMA_RETRIEVE_handle_ft                                 \
    icetestIDMADeviceHandle *icetest_idma_handle;                       \
    struct icetest_idma_function_struct *ft;                            \
                                                                        \
    icetest_idma_handle = icetest_object.icetest_idma_handle;           \
    BUG_ON(icetest_idma_handle == NULL);                                \
    ft = icetest_idma_handle->fcn_tbl;


void init_icetest_object_struct(void)
{
    icetest_object.icetest_handle = NULL;
    icetest_object.icetest_idma_handle = NULL;
}

void register_icetest_subblock(enum icetestsubblock icetestsubblock,
                               void *icetest_subblock_data, int instance)
{
    static int initialized=0;
    
    if (initialized == 0)
    {
        init_icetest_object_struct();
        initialized = 1;
    }
    
    if (icetestsubblock == common)
    {
        debug_print("registered icetest_common, instance=%d with icetest.c\n", instance);
        icetest_object.icetest_handle = icetest_subblock_data;
    }
    else if (icetestsubblock == icetest_idma)
    {
        debug_print("registered icetest dma, instance=%d with icetest.c\n", instance);
        icetest_object.icetest_idma_handle = icetest_subblock_data;
    }
    else
    {
        error_print("Error!!! no support yet for blocks %d\n", icetestsubblock);
    }
}
EXPORT_SYMBOL(register_icetest_subblock);

void *unregister_icetest_subblock(enum icetestsubblock icetestsubblock, int instance)
{
    void *subblock_data;
    
    if (icetestsubblock == common)
    {
        subblock_data = icetest_object.icetest_handle;
        icetest_object.icetest_handle = NULL;
    }
    else if (icetestsubblock == icetest_idma)
    {
        subblock_data = icetest_object.icetest_idma_handle;
        icetest_object.icetest_idma_handle = NULL;
    }
    else
    {
        error_print("Error!!! no support yet for any blocks except common\n");
        return NULL;
    }
    return subblock_data;
}
EXPORT_SYMBOL(unregister_icetest_subblock);

//////////////////////// start of public API functions /////////////////////
void icetest_register_callback(void *callbackfcn)
{
    ICETEST_COMMON_RETRIEVE_handle_ft;
    
    ft->register_callback(icetest_common_handle, callbackfcn);
}
EXPORT_SYMBOL(icetest_register_callback);

void icetest_clear_irq(struct icetest_interrupt_info *irqstruct)
{
    ICETEST_COMMON_RETRIEVE_handle_ft;

    ft->clear_icetest_irq(icetest_common_handle, irqstruct);
}
EXPORT_SYMBOL(icetest_clear_irq);

// enable icetest common interrupts
// caller sets the irqstruct field bit to true for each interrupt to enable
// irqstruct of NULL means to enable all interrupts
void icetest_enable_irq(struct icetest_interrupt_info *irqstruct)
{
    ICETEST_COMMON_RETRIEVE_handle_ft;

    ft->enable_icetest_irq(icetest_common_handle, irqstruct);
}
EXPORT_SYMBOL(icetest_enable_irq);

// disable icetest common interrupts
// caller sets the irqstruct field bit to true for each interrupt to disable
// irqstruct of NULL means to disable all interrupts
void icetest_disable_irq(struct icetest_interrupt_info *irqstruct)
{
    ICETEST_COMMON_RETRIEVE_handle_ft;

    ft->disable_icetest_irq(icetest_common_handle, irqstruct);
}
EXPORT_SYMBOL(icetest_disable_irq);

void icetest_set_irq(struct icetest_interrupt_info *irqstruct)
{
    ICETEST_COMMON_RETRIEVE_handle_ft;
    
    ft->force_icetest_irq(icetest_common_handle, irqstruct);
}
EXPORT_SYMBOL(icetest_set_irq);

void icetest_set_config(struct icetest_testconfig_info *info)
{
    ICETEST_COMMON_RETRIEVE_handle_ft;

    ft->set_icetest_config(icetest_common_handle, info);
}
EXPORT_SYMBOL(icetest_set_config);

void icetest_get_config(struct icetest_testconfig_info *info)
{
    ICETEST_COMMON_RETRIEVE_handle_ft;

    ft->get_icetest_config(icetest_common_handle, info);
}
EXPORT_SYMBOL(icetest_get_config);

void icetest_dump_regs(void)
{
    ICETEST_COMMON_RETRIEVE_handle_ft;

    ft->dump_icetest_regs(icetest_common_handle);
}
EXPORT_SYMBOL(icetest_dump_regs);

void icetest_soft_reset(int resetval)
{
    struct icetest_testconfig_info full_struct;
    ICETEST_COMMON_RETRIEVE_handle_ft;

    memset(&full_struct, 0x0, sizeof(struct icetest_testconfig_info));
    full_struct.softreset_active_valid = true;
    full_struct.softreset_active = resetval;
    
    ft->set_icetest_config(icetest_common_handle, &full_struct);
}
EXPORT_SYMBOL(icetest_soft_reset);

uint32_t icetest_get_rev0(void)
{
    ICETEST_COMMON_RETRIEVE_handle_ft;

    return ft->get_icetest_rev0(icetest_common_handle);
}
EXPORT_SYMBOL(icetest_get_rev0);

uint32_t icetest_get_rev1(void)
{
    ICETEST_COMMON_RETRIEVE_handle_ft;

    return ft->get_icetest_rev1(icetest_common_handle);
}
EXPORT_SYMBOL(icetest_get_rev1);

// ice dma pubic functions
void icetest_register_callback_idma(void *callbackfcn)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;
    
    ft->register_callback(icetest_idma_handle, callbackfcn);
}
EXPORT_SYMBOL(icetest_register_callback_idma);

void icetest_handle_idma_irqs(void)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;
    
    ft->handle_icetest_idma_irqs(icetest_idma_handle);
}
EXPORT_SYMBOL(icetest_handle_idma_irqs);

void icetest_clear_idma_irqs(struct icetest_idma_interrupt_info *irqstruct)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;

    ft->clear_icetest_idma_irqs(icetest_idma_handle, irqstruct);
}
EXPORT_SYMBOL(icetest_clear_idma_irqs);

// enable icetest idma interrupts
// caller sets the irqstruct field bit to true for each interrupt to enable
// irqstruct of NULL means to enable all interrupts
void icetest_enable_idma_irqs(struct icetest_idma_interrupt_info *irqstruct)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;

    ft->enable_icetest_idma_irqs(icetest_idma_handle, irqstruct);
}
EXPORT_SYMBOL(icetest_enable_idma_irqs);

// disable icetest idma interrupts
// caller sets the irqstruct field bit to true for each interrupt to disable
// irqstruct of NULL means to disable all interrupts
void icetest_disable_idma_irqs(struct icetest_idma_interrupt_info *irqstruct)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;

    ft->disable_icetest_idma_irqs(icetest_idma_handle, irqstruct);
}
EXPORT_SYMBOL(icetest_disable_idma_irqs);

void icetest_idma_soft_reset(int resetval)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;
    
    ft->set_icetest_idma_soft_reset(icetest_idma_handle,
                                    resetval);
}
EXPORT_SYMBOL(icetest_idma_soft_reset);

void icetest_dump_idma_regs(void)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;
    
    ft->dump_icetest_idma_regs(icetest_idma_handle);
}
EXPORT_SYMBOL(icetest_dump_idma_regs);

void icetest_set_idma_config(struct icetest_idma_config_info *config)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;
    
    ft->set_icetest_idma_config(icetest_idma_handle, config);
}
EXPORT_SYMBOL(icetest_set_idma_config);

void icetest_set_idma_burstlen(int burstlen)
{
    struct icetest_idma_config_info full_struct;
    ICETEST_IDMA_RETRIEVE_handle_ft;

    memset(&full_struct, 0x0, sizeof(struct icetest_idma_config_info));
    full_struct.burst_len_valid = true;
    full_struct.burst_len = burstlen;
    ft->set_icetest_idma_config(icetest_idma_handle, &full_struct);
}
EXPORT_SYMBOL(icetest_set_idma_burstlen);

void icetest_set_idma_dma_width(int dma_width)
{
    struct icetest_idma_config_info full_struct;
    ICETEST_IDMA_RETRIEVE_handle_ft;
    
    memset(&full_struct, 0x0, sizeof(struct icetest_idma_config_info));
    full_struct.out_width_valid = true;
    full_struct.out_width = dma_width;
    ft->set_icetest_idma_config(icetest_idma_handle, &full_struct);
}
EXPORT_SYMBOL(icetest_set_idma_dma_width);

void icetest_set_idma_enable(int enable)
{
    struct icetest_idma_config_info full_struct;
    ICETEST_IDMA_RETRIEVE_handle_ft;
    
    memset(&full_struct, 0x0, sizeof(struct icetest_idma_config_info));
    full_struct.enable_valid = true;
    full_struct.enable = enable;
    ft->set_icetest_idma_config(icetest_idma_handle, &full_struct);

}
EXPORT_SYMBOL(icetest_set_idma_enable);

void icetest_set_idma_replicate(int replicate)
{
    struct icetest_idma_config_info full_struct;
    ICETEST_IDMA_RETRIEVE_handle_ft;
    
    memset(&full_struct, 0x0, sizeof(struct icetest_idma_config_info));
    full_struct.replicate_valid = true;
    full_struct.replicate = replicate;
    ft->set_icetest_idma_config(icetest_idma_handle, &full_struct);
}
EXPORT_SYMBOL(icetest_set_idma_replicate);

void icetest_get_idma_config(struct icetest_idma_config_info *config)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;
    
    ft->get_icetest_idma_config(icetest_idma_handle, config);
}
EXPORT_SYMBOL(icetest_get_idma_config);

void icetest_get_idma_status(struct icetest_idma_status_info *status)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;
    
    ft->get_icetest_idma_status(icetest_idma_handle, status);
}
EXPORT_SYMBOL(icetest_get_idma_status);

void icetest_set_idma_linewidth(int linewidth)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;

    ft->set_icetest_idma_linewidth(icetest_idma_handle, linewidth);
}
EXPORT_SYMBOL(icetest_set_idma_linewidth);

void icetest_get_idma_linewidth(int *linewidth)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;

    ft->get_icetest_idma_linewidth(icetest_idma_handle, linewidth);
}
EXPORT_SYMBOL(icetest_get_idma_linewidth);

void icetest_start_idma(dma_addr_t descriptor_phys_addr)
{
    ICETEST_IDMA_RETRIEVE_handle_ft;
    
    ft->start_icetest_idma(icetest_idma_handle, descriptor_phys_addr);
}
EXPORT_SYMBOL(icetest_start_idma);

