/*
 ***************************************************************************************
 * (c) Copyright 2013 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/export.h>
#include <linux/dma-mapping.h>
#include <linux/module.h>
#include "descriptor.h"

// ZX Descriptor Functions

static size_t zx_get_dma_alloc_size(struct zx_descriptor_info *info)
{
    // Allocate space for descriptors plus the command buffer
    return info->num_descriptors * sizeof(*info->descriptors) +
        info->num_command_words * 4;
}

int zx_descriptor_create_list(struct zx_descriptor_info *info,
                              struct device *dev,
                              unsigned int num_descriptors,
                              unsigned int num_command_words)
{
    if (!info)
    {
        printk(KERN_ERR "%s:info is NULL\n",__func__);
        return -EINVAL;
    }
    info->dev = dev;
    info->num_descriptors = num_descriptors;
    info->num_command_words = num_descriptors;
    info->descriptors = dma_alloc_coherent(info->dev,
                                           zx_get_dma_alloc_size(info),
                                           &info->base_phys_addr, GFP_KERNEL);
    if (!info->descriptors)
    {
        printk(KERN_ERR "%s:dma_alloc_coherent returned NULL\n",__func__);
        return -ENOMEM;
    }
    return 0;
}
EXPORT_SYMBOL(zx_descriptor_create_list);

void zx_descriptor_destroy_list(struct zx_descriptor_info *info)
{
    if (!info->descriptors)
        return;
    dma_free_coherent(info->dev, zx_get_dma_alloc_size(info),
                      info->descriptors, info->base_phys_addr);
    info->descriptors = NULL;
}
EXPORT_SYMBOL(zx_descriptor_destroy_list);

dma_addr_t zx_descriptor_get_phys_addr(struct zx_descriptor_info *info,
                                       int index)
{
    return info->base_phys_addr + index * sizeof(*info->descriptors);
}
EXPORT_SYMBOL(zx_descriptor_get_phys_addr);

uint32_t *zx_descriptor_get_command_buffer(struct zx_descriptor_info *info)
{
    // The space allocated past the last descriptor is the command buffer.
    return (uint32_t *) &info->descriptors[info->num_descriptors];
}
EXPORT_SYMBOL(zx_descriptor_get_command_buffer);

dma_addr_t zx_descriptor_get_command_buffer_phys_addr(struct zx_descriptor_info *info)
{
    return info->base_phys_addr + info->num_descriptors * sizeof(*info->descriptors);
}
EXPORT_SYMBOL(zx_descriptor_get_command_buffer_phys_addr);

void zx_descriptor_print_chain(struct zx_descriptor_info *info,
                               int num_command_words)
{
    int i;
    printk("===> zx descriptors\n");
    for (i = 0; i < info->num_descriptors; i++)
    {
        printk("  desc[%2d]: virt=0x%p phys=0x%08x ctrl=0x%08x source=0x%08x length=0x%08x next=0x%08x\n",
               i,
               &info->descriptors[i],
               zx_descriptor_get_phys_addr(info, i),
               info->descriptors[i].ctrl,
               info->descriptors[i].source,
               info->descriptors[i].length,
               info->descriptors[i].next);
    }
    printk("  zx commands: virt=0x%p phys=0x%08x\n",
           zx_descriptor_get_command_buffer(info),
           zx_descriptor_get_command_buffer_phys_addr(info));
    for (i = 0; i < num_command_words; i++)
    {
        uint32_t *cmd = zx_descriptor_get_command_buffer(info);
        printk("    0x%08X\n", cmd[i]);
    }
    printk("<=== zx descriptors\n");
}
EXPORT_SYMBOL(zx_descriptor_print_chain);


// ODMA AXI Descriptor Functions

static size_t odma_axi_get_dma_alloc_size(struct odma_axi_descriptor_info *info)
{
    return info->num_descriptors * sizeof(*info->descriptors);
}

int odma_axi_descriptor_create_list(struct odma_axi_descriptor_info *info,
                                    struct device *dev,
                                    unsigned int num_descriptors)
{
    if (!info)
    {
        printk(KERN_ERR "%s:info is NULL\n",__func__);
        return -EINVAL;
    }
    info->dev = dev;
    info->num_descriptors = num_descriptors;
    info->descriptors = dma_alloc_coherent(info->dev,
                                           odma_axi_get_dma_alloc_size(info),
                                           &info->base_phys_addr, GFP_KERNEL);
    if (!info->descriptors)
    {
        printk(KERN_ERR "%s:dma_alloc_coherent returned NULL\n",__func__);
        return -ENOMEM;
    }
    return 0;
}
EXPORT_SYMBOL(odma_axi_descriptor_create_list);

void odma_axi_descriptor_destroy_list(struct odma_axi_descriptor_info *info)
{
    if (!info->descriptors)
        return;
    dma_free_coherent(info->dev, odma_axi_get_dma_alloc_size(info),
                      info->descriptors, info->base_phys_addr);
    info->descriptors = NULL;
}
EXPORT_SYMBOL(odma_axi_descriptor_destroy_list);

dma_addr_t odma_axi_descriptor_get_phys_addr(struct odma_axi_descriptor_info *info,
                                             int index)
{
    return info->base_phys_addr + index * sizeof(*info->descriptors);
}
EXPORT_SYMBOL(odma_axi_descriptor_get_phys_addr);

void odma_axi_descriptor_print_chain(struct odma_axi_descriptor_info *info)
{
    int i;
    printk("===> odma axi descriptors\n");
    for (i = 0; i < info->num_descriptors; i++)
    {
        printk(
            "  desc[%2d]: virt=0x%p phys=0x%08x ctrl=0x%08x source=0x%08x length=0x%08x next=0x%08x\n",
            i,
            &info->descriptors[i],
            odma_axi_descriptor_get_phys_addr(info, i),
            info->descriptors[i].ctrl,
            info->descriptors[i].source,
            info->descriptors[i].length,
            info->descriptors[i].next);
        printk(
            "  desc[%2d]:     lines=0x%08x width=0x%08x last_line=0x%08x source2=0x%08x\n",
            i,
            info->descriptors[i].lines,
            info->descriptors[i].width,
            info->descriptors[i].last_line,
            info->descriptors[i].source2);
    }
    printk("<=== odma axi descriptors\n");
}
EXPORT_SYMBOL(odma_axi_descriptor_print_chain);



// Distort Descriptor Functions

dma_addr_t distort_descriptor_get_phys_addr(struct pie_distort_descriptor_info *info,
                                            int index)
{
    return info->base_phys_addr + index * sizeof(struct pie_distort_descriptor);
}
EXPORT_SYMBOL(distort_descriptor_get_phys_addr);

int distort_descriptor_set_prev_next(struct pie_distort_descriptor_info *info, int index,
                                     int prev_index, int next_index)
{
    info->descriptors[index].prev_phys_addr = distort_descriptor_get_phys_addr(info, prev_index);
    info->descriptors[index].next_phys_addr = distort_descriptor_get_phys_addr(info, next_index);
    return 0;
}
EXPORT_SYMBOL(distort_descriptor_set_prev_next);

void distort_descriptor_clear_chain(struct pie_distort_descriptor_info *info)
{
    int i;
    for (i = 0; i < info->count; i++)
    {
        info->descriptors[i].src = 1; // set ownership to cpu
        info->descriptors[i].line_range = 0;
    }
}
EXPORT_SYMBOL(distort_descriptor_clear_chain);

int distort_descriptor_set(struct pie_distort_descriptor_info *info, int index,
                           dma_addr_t databuf, unsigned int first_line_in_range,
                           unsigned int num_lines)
{
    if (databuf & 0x3)
    {
        printk(KERN_ERR "%s: databuf=%pa is not 4-byte aligned!\n", __func__, &databuf);
    }
    info->descriptors[index].src = databuf;
    info->descriptors[index].src &= ~0x1; // set ownership to dma
    info->descriptors[index].src &= ~0x2; // no interrupt on desc complete
    info->descriptors[index].line_range =
        (first_line_in_range << 16) | (first_line_in_range + num_lines - 1);
    return 0;
}
EXPORT_SYMBOL(distort_descriptor_set);

int distort_descriptor_create_list(struct pie_distort_descriptor_info *info,
                                   struct device *dev, unsigned int count)
{
    if (!info)
    {
        printk(KERN_ERR "%s:info is NULL\n",__func__);
        return -EINVAL;
    }
    info->dev = dev;
    info->count = count;
    info->descriptors = dma_alloc_coherent(info->dev,
                                           info->count * sizeof(*info->descriptors),
                                           &info->base_phys_addr, GFP_KERNEL);
    if (!info->descriptors)
    {
        printk(KERN_ERR "%s:dma_alloc_coherent returned NULL\n",__func__);
        return -ENOMEM;
    }
    return 0;
}
EXPORT_SYMBOL(distort_descriptor_create_list);

void distort_descriptor_destroy_list(struct pie_distort_descriptor_info *info)
{
    if (!info->descriptors)
        return;
    dma_free_coherent(info->dev,
                      info->count * sizeof(*info->descriptors),
                      info->descriptors, info->base_phys_addr);
    info->descriptors = NULL;
}
EXPORT_SYMBOL(distort_descriptor_destroy_list);

void distort_descriptor_print_chain(struct pie_distort_descriptor_info *info)
{
    int i;
    printk("===> descriptors\n");
    for (i = 0; i < info->count; i++)
    {
        printk("  desc[%2d]: addr=0x%08X src=0x%08X line_range=0x%08X prev=0x%08X next=0x%08X\n",
               i,
               distort_descriptor_get_phys_addr(info, i),
               info->descriptors[i].src,
               info->descriptors[i].line_range,
               info->descriptors[i].prev_phys_addr,
               info->descriptors[i].next_phys_addr);
    }
    printk("<=== descriptors\n");
}
EXPORT_SYMBOL(distort_descriptor_print_chain);



/////////////  private struct for the DMA
typedef struct DMA_descriptor_s
{
    // ctrl_word bits 24/16: ownership, 7: EOI, 6: SOI, 1: Int on end, 0: stop
    volatile uint32_t ctrl_word;
    // the next 2 fields are the transfer address and transfer length
    // transfer address = number of bytes to be transferred
    // transfer length = address of memory to be transferred, note, the DMA controller
    // insists this be a physical address, not virtual.
    // But, the 2 different descriptors swap these 2 fields, so we only know their
    // meaning by which version of the descriptor we want
    union
    {
        volatile uint32_t *xfer_addr_old;
        volatile uint32_t xfer_length_new;
    };
    union
    {
        volatile uint32_t xfer_length_old;
        volatile uint32_t *xfer_addr_new;
    };
    volatile uint32_t *next_desc; // note, DMA controller insists this be a physical address, not virtual
    /* These 4 fields used only with the 8-word descriptors */
    volatile uint32_t *SourceLo0;
    volatile uint32_t *SourceHi0;
    volatile uint32_t *SourceLo1;
    volatile uint32_t *SourceHi1;
    volatile int dtype;
}DMA_descriptor;

// descriptor - virtual pointer to the head of the descriptor list
// index - index of descriptor in list to set
// data_xfer_size - how much data to transfer for this descriptor
// databuf - *Physical* pointer to the data that the descriptors will be pointing to
// flags - flags to assign 
int set_descriptor(DMA_descriptor *descriptor, int index, int data_xfer_size, 
                    dma_addr_t databuf, uint32_t flags)
{
    uint8_t *databuf_byteptr;  // for indexing into the databuf by bytes
    uint32_t end_of_xfer;

    if (descriptor == NULL)
    {
        printk(KERN_ERR "%s: ERROR: return descriptor is NULL\n", __func__);
        return -EINVAL;
    }

    // data buffer must be 4 byte aligned
    BUG_ON((databuf & 0x3) != 0);
    // data size must be 4 byte aligned also
    BUG_ON((data_xfer_size & 0x3) != 0);

    databuf_byteptr = (uint8_t *) databuf;

    if (descriptor[index].dtype == old_4word_axi)
    {
        descriptor[index].xfer_addr_old = (uint32_t *) &databuf_byteptr[index * data_xfer_size];
        // write the number of bytes to be transferred
        descriptor[index].xfer_length_old = data_xfer_size;
    }
    else if (descriptor[index].dtype == new_8word_axi)
    {
        descriptor[index].xfer_addr_new = (uint32_t *) &databuf_byteptr[index * data_xfer_size];
        // write the number of bytes to be transferred
        descriptor[index].xfer_length_new = data_xfer_size;
    }
    else
    {
        printk("ERROR, bad descriptor_type %d for descriptor %d\n",
               descriptor[index].dtype, index);
    }

    // write control word
    if (descriptor[index].dtype == new_8word_axi)
       descriptor[index].ctrl_word |= DMA_NEW_OWNS;
    else
       descriptor[index].ctrl_word |= DMA_OLD_OWNS;
    descriptor[index].ctrl_word |= flags;

    if (descriptor[index].dtype == new_8word_axi)
    {
        // for 8 word descriptors, fill in the Lo0/1 and Hi0/1 - FIXME
        descriptor[index].SourceLo0 = descriptor[index].xfer_addr_new;
        descriptor[index].SourceLo1 = descriptor[index].SourceLo0;
        // end of xfer is the last byte to be transferred, so -1 to get there.
        end_of_xfer = (((uint32_t) descriptor[index].SourceLo0) + data_xfer_size) - 1;
        descriptor[index].SourceHi0 = ((uint32_t *) end_of_xfer); 
        descriptor[index].SourceHi1 = ((uint32_t *) end_of_xfer); // FIXME - do we need to have round up 1024?
    }

//    printk("%s debug, dumping 1 descriptor\n",__func__);
//    dump_descriptors(&descriptor[index], 1);
    
    return 0;
}
EXPORT_SYMBOL(set_descriptor);

// we allocate the descriptors in one big chunk - to avoid wasting memory since we use
// dma_alloc_coherent probably does things in page size, and descriptors are
// pretty small
//
// num_descriptors - the number of descriptors to create
// ownership_bit - DMA_AXI_NEW_OWNS - bit 24, DMA_AXI_OLD_OWNS - bit 16
// ret_descriptor - pointer to head DMA descriptor that will be created - this is the virtual addr
// ret_descriptor_phys -  pointer to physical address of the head DMA descriptor.  i.e. the physical address of "descriptor"
//
// returns 0 if successful
//
int create_descriptor_list(int num_descriptors, enum descriptor_type dtype,
                           DMA_descriptor **ret_descriptor, 
                           dma_addr_t *ret_descriptor_phys,
                           struct device *device_ptr)
{
    dma_addr_t dma_addr, first_dma_addr;
    int i;
    uint32_t *next_descriptor; // physical addr
    DMA_descriptor *descriptor;
    uint32_t ownership_bit;
    

    if (num_descriptors == 0)
    {
        printk(KERN_ERR "%s: ERROR: number of descriptors to create is 0\n", __func__);
        return -EINVAL;
    }
    if (ret_descriptor == NULL)
    {
        printk(KERN_ERR "%s: ERROR: return descriptor is NULL\n", __func__);
        return -EINVAL;
    }
    if (ret_descriptor_phys == NULL)
    {
        printk(KERN_ERR "%s: ERROR: return descriptor phys is NULL\n", __func__);
        return -EINVAL;
    }

    if (dtype == old_4word_axi)
        ownership_bit = DMA_OLD_OWNS;
    else if (dtype == new_8word_axi)
        ownership_bit = DMA_NEW_OWNS;
    else
    {
        printk(KERN_ERR "%s: ERROR: bad descriptor_type %d\n", __func__, dtype);
        return -EINVAL;
    }
    
    // allocate dma descriptors
    descriptor = dma_alloc_coherent(device_ptr, (num_descriptors*sizeof(DMA_descriptor)), &first_dma_addr,GFP_DMA);
    // Descriptors have to be a minimum of 16 byte aligned
    BUG_ON((first_dma_addr & 0xF) != 0);

    //printk("virt_addr of descriptor is 0x%p, phys_addr of descriptor is 0x%X\n",
    //             descriptor, first_dma_addr);
    if (!descriptor)
    {
        printk(KERN_ERR "%s:dma_alloc_coherent returned NULL\n",__func__);
        return -ENOMEM;
    }
    
    dma_addr = first_dma_addr;
    
    //printk("first descriptor - phys=0x%X, virt=0x%p\n",dma_addr, descriptor);
    // set up descriptors, iterate over each descriptor
    for (i=0;i<num_descriptors;i++)
    {
        if (i == (num_descriptors-1))
        {
            // make the last descriptor point to itself (DougK recommendation, 12-Sep-2013)
            next_descriptor = (uint32_t *) dma_addr; // physical addr
            //printk("last descriptor  phys=0x%X, virt=0x%p\n", 
            //             (unsigned int) next_descriptor, &descriptor[i]);
        }
        else
        {
            dma_addr += sizeof(DMA_descriptor);
            next_descriptor = (uint32_t *) dma_addr;  //physical addr
        }
        // write the physical pointer to the next descriptor        
        descriptor[i].next_desc = next_descriptor;
        // write control word
        // mark it as owned by the DMA h/w (it will err out if the f/w owns it)
        descriptor[i].ctrl_word = ownership_bit;
        // set the descriptor type
        descriptor[i].dtype = dtype;
    }
    *ret_descriptor = descriptor;
    *ret_descriptor_phys = first_dma_addr;

    // printk("%s, looking at %d descriptors before set\n",__func__, num_descriptors);
    // dump_descriptors(descriptor, num_descriptors);
    return 0;
}
EXPORT_SYMBOL(create_descriptor_list);

void dump_descriptors(DMA_descriptor *descriptor, int num_descriptors)
{
    int i, done;

    i=0;
    done = 0;
    printk("Descriptor Dump at address 0x%p\n",descriptor);
    printk("num_descriptors is set to %d\n",num_descriptors);

    do
    {
        printk("ctrlword=0x%X, ", descriptor[i].ctrl_word);
        if (descriptor[i].dtype == old_4word_axi)
        {
            printk("xfer_addro=0x%p, xfer_lengtho=0x%X, ",
                   descriptor[i].xfer_addr_old,
                   descriptor[i].xfer_length_old);
            if (descriptor[i].xfer_addr_old == descriptor[i].next_desc)
                // the only time the next descriptor should point at ourself is when
                // we are on the last descriptor
                done = 1;
        }
        else if (descriptor[i].dtype == new_8word_axi)
        {
            printk("xfer_addrn=0x%p, xfer_lengthn=0x%X, ",
                   descriptor[i].xfer_addr_new,
                   descriptor[i].xfer_length_new);
            if (descriptor[i].xfer_addr_new == descriptor[i].next_desc)
                // the only time the next descriptor should point at ourself is when
                // we are on the last descriptor
                done = 1;
        }

        printk("next=0x%p\n",descriptor[i].next_desc);
        if (descriptor[i].dtype == new_8word_axi)
        {
            printk("Lo0=0x%p, Hi0=0x%p, Lo1=0x%p, Hi1=0x%p\n",
                   descriptor[i].SourceLo0, descriptor[i].SourceHi0,
                   descriptor[i].SourceLo1, descriptor[i].SourceHi1);
        }
        
        i++;
        if (num_descriptors != 0)
        {
            if (i==num_descriptors)
                done = 1;
        }
            
    } while (done==0);
}
EXPORT_SYMBOL(dump_descriptors);

// num_descriptors is the same as in create_descriptor_list
// descriptor_addr is the virtual address of the descriptor
// descriptor_addr_phys is the physical address of the descriptor
// device_ptr is the same as in create_descriptor_list
void destroy_descriptors(int num_descriptors, DMA_descriptor *descriptor_addr,
                         dma_addr_t descriptor_addr_phys, struct device *device_ptr)
{
    dma_free_coherent(device_ptr, num_descriptors*sizeof(DMA_descriptor), descriptor_addr,
                      (dma_addr_t)descriptor_addr_phys);
}
EXPORT_SYMBOL(destroy_descriptors);







// descriptor - virtual pointer to the head of the descriptor list
// index - index of descriptor in list to set
// data_xfer_size - how much data to transfer for this descriptor
// databuf - *Physical* pointer to the data that the descriptors will be pointing to
// flags - flags to assign
int set_descriptor_direct(DMA_descriptor *descriptor, int index, int data_xfer_size,
                           dma_addr_t strip_databuf, uint32_t flags)
{
    return set_descriptor_direct_eightword(descriptor,index,data_xfer_size,strip_databuf,0,0,flags);
}
EXPORT_SYMBOL(set_descriptor_direct);

// descriptor - virtual pointer to the head of the descriptor list
// index - index of descriptor in list to set
// data_xfer_size - how much data to transfer for this descriptor
// databuf - *Physical* pointer to the data that the descriptors will be pointing to
// flags - flags to assign
int set_descriptor_direct_eightword(DMA_descriptor *descriptor, int index, int data_xfer_size,
                                    dma_addr_t strip_databuf, int data_xfer_size2,
                                    dma_addr_t strip_databuf2, uint32_t flags)
{
    uint8_t *databuf_byteptr;  // for indexing into the databuf by bytes
    uint32_t end_of_xfer;

    if (descriptor == NULL)
    {
        printk(KERN_ERR "%s: ERROR: return descriptor is NULL\n", __func__);
        return -EINVAL;
    }

    // data buffer must be 4 byte aligned
    BUG_ON((strip_databuf & 0x3) != 0);
    // data size must be 4 byte aligned also
    BUG_ON((data_xfer_size & 0x3) != 0);
    // data buffer must be 4 byte aligned
    BUG_ON((strip_databuf2 & 0x3) != 0);
    // data size must be 4 byte aligned also
    BUG_ON((data_xfer_size2 & 0x3) != 0);

    databuf_byteptr = (uint8_t *) strip_databuf;

    if (descriptor[index].dtype == old_4word_axi)
    {
        descriptor[index].xfer_addr_old = (uint32_t *) strip_databuf;
        // write the number of bytes to be transferred
        descriptor[index].xfer_length_old = data_xfer_size;
    }
    else if (descriptor[index].dtype == new_8word_axi)
    {
        descriptor[index].xfer_addr_new = (uint32_t *) strip_databuf;
        // write the number of bytes to be transferred
        descriptor[index].xfer_length_new = data_xfer_size + data_xfer_size2;
    }
    else
    {
        printk("ERROR, bad descriptor_type %d for descriptor %d\n",
               descriptor[index].dtype, index);
    }

    // write control word
    if (descriptor[index].dtype == new_8word_axi)
       descriptor[index].ctrl_word |= DMA_NEW_OWNS;
    else
       descriptor[index].ctrl_word |= DMA_OLD_OWNS;
    descriptor[index].ctrl_word |= flags;

    if (descriptor[index].dtype == new_8word_axi)
    {
        if (data_xfer_size2)
        {
            descriptor[index].SourceLo0 = (uint32_t *) strip_databuf;
            // end of xfer is the last byte to be transferred, so -1 to get there.
            end_of_xfer = (((uint32_t) descriptor[index].SourceLo0) + data_xfer_size) - 1;
            descriptor[index].SourceHi0 = ((uint32_t *) end_of_xfer);

            descriptor[index].SourceLo1 = (uint32_t *) strip_databuf2;
            // end of xfer is the last byte to be transferred, so -1 to get there.
            end_of_xfer = (((uint32_t) descriptor[index].SourceLo1) + data_xfer_size2) - 1;
            descriptor[index].SourceHi1 = ((uint32_t *) end_of_xfer);
        }
        else
        {
            // for 8 word descriptors, fill in the Lo0/1 and Hi0/1 - FIXME
            descriptor[index].SourceLo0 = descriptor[index].xfer_addr_new;
            descriptor[index].SourceLo1 = descriptor[index].SourceLo0;
            // end of xfer is the last byte to be transferred, so -1 to get there.
            end_of_xfer = (((uint32_t) descriptor[index].SourceLo0) + data_xfer_size) - 1;
            descriptor[index].SourceHi0 = ((uint32_t *) end_of_xfer);
            descriptor[index].SourceHi1 = ((uint32_t *) end_of_xfer); // FIXME - do we need to have round up 1024?
        }
    }

//    printk("%s debug, dumping 1 descriptor\n",__func__);
//    dump_descriptors(&descriptor[index], 1);

    return 0;
}
EXPORT_SYMBOL(set_descriptor_direct_eightword);

dma_addr_t get_descriptor_phys_address (dma_addr_t descriptor_addr_phys, int index)
{
   return descriptor_addr_phys + index * sizeof(DMA_descriptor);
}
EXPORT_SYMBOL(get_descriptor_phys_address);

MODULE_LICENSE("GPL");
