/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/spinlock.h>   // for spinlock_t, printk, BUG_ON, and memset
#include <linux/io.h>         // for ioread/write32 (Read/Write macros)
#include <linux/export.h>     // for EXPORT_SYMBOL

#include "CISX_ODMA_AXI_regstructs.h"
#include "CISX_ODMA_AXI_regmasks.h"

#include "cisx_if.h"
#include "cisx_data.h"
#include "cisx_driver.h"
#include "cisx_odma_if.h"
#include "cisx_odma.h"


enum cisx_odma_mask_type {enable, pending, ack, force};


//*****************************************************************************
// Debug Functions
//*****************************************************************************

void dump_cisx_odma_regs(cisx_odmaDeviceHandle *device_data)
{
    printk("CISX ODMA(%d) dump for cisx=%d\n", device_data->submodinstance, device_data->instance);
    printk("-------------------------------------\n");
    printk( "     cfg=0x%08x    status=0x%08x line_size=0x%08x\n", 
             cisx_odmaRead(cfg), cisx_odmaRead(status), cisx_odmaRead(line_size) );
    printk( "  int_en=0x%08x  int_pend=0x%08x desc_read=0x%08x\n", 
             cisx_odmaRead(int_en), cisx_odmaRead(int_pend), cisx_odmaRead(desc_read) );
    printk( "xfer_len=0x%08x  xferaddr=0x%08x  xferburs=0x%08x\n", 
             cisx_odmaRead(xfer_length), cisx_odmaRead(xfer_addr), cisx_odmaRead(xfer_burst) );
}

//*****************************************************************************
// IRQ Functions
//*****************************************************************************

static void set_all_irqstruct(struct cisx_odma_irqs *irqstruct, bool setval)
{
    irqstruct->rresp   = setval;
    irqstruct->bresp   = setval;
    irqstruct->rst     = setval;
    irqstruct->eoi     = setval;
    irqstruct->dir     = setval;
    irqstruct->cl_ali  = setval;
    irqstruct->eol_ali = setval;
    irqstruct->eoi_ali = setval;
    irqstruct->eoi_err = setval;
    irqstruct->fin     = setval;
    irqstruct->who     = setval;
}

static uint32_t convert_irqstruct_to_update_uintarray(
                    uint32_t curr_irqs,
                    struct   cisx_odma_irqs *irqstruct,
                    bool     set_to_one,
                    enum cisx_odma_mask_type operation )
{
    uint8_t set_val;

    // Are we enabling or disabling IRQs?
    set_val = set_to_one ? 1 : 0; 

    // Any IRQ's in irqstruct that are set to true will get set to set_val, other
    // bits will be left as-is.
    if (irqstruct->rresp)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_ODMA_INT_EN_RRESP_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_ODMA_INT_PEND_RRESP_INT_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_ODMA_INT_ACK_RRESP_INT_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_ODMA_INT_FO_RRESP_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->bresp)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_ODMA_INT_EN_BRESP_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_ODMA_INT_PEND_BRESP_INT_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_ODMA_INT_ACK_BRESP_INT_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_ODMA_INT_FO_BRESP_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->rst)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_ODMA_INT_EN_RST_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_ODMA_INT_PEND_RST_INT_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_ODMA_INT_ACK_RST_INT_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_ODMA_INT_FO_RST_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->eoi)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_ODMA_INT_EN_EOI_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_ODMA_INT_PEND_EOI_INT_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_ODMA_INT_ACK_EOI_INT_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_ODMA_INT_FO_EOI_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->dir)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_ODMA_INT_EN_DIR_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_ODMA_INT_PEND_DIR_INT_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_ODMA_INT_ACK_DIR_INT_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_ODMA_INT_FO_DIR_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->cl_ali)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_ODMA_INT_EN_CL_ALI_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_ODMA_INT_PEND_CL_ALI_INT_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_ODMA_INT_ACK_CL_ALI_INT_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_ODMA_INT_FO_CL_ALI_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->eol_ali)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_ODMA_INT_EN_EOL_ALI_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_ODMA_INT_PEND_EOL_ALI_INT_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_ODMA_INT_ACK_EOL_ALI_INT_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_ODMA_INT_FO_EOL_ALI_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->eoi_ali)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_ODMA_INT_EN_EOI_ALI_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_ODMA_INT_PEND_EOI_ALI_INT_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_ODMA_INT_ACK_EOI_ALI_INT_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_ODMA_INT_FO_EOI_ALI_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->eoi_err)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_ODMA_INT_EN_EOI_ERR_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_ODMA_INT_PEND_EOI_ERR_INT_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_ODMA_INT_ACK_EOI_ERR_INT_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_ODMA_INT_FO_EOI_ERR_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->fin)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_ODMA_INT_EN_FIN_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_ODMA_INT_PEND_FIN_INT_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_ODMA_INT_ACK_FIN_INT_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_ODMA_INT_FO_FIN_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }


    if (irqstruct->who)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_ODMA_INT_EN_WHO_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_ODMA_INT_PEND_WHO_INT_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_ODMA_INT_ACK_WHO_INT_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_ODMA_INT_FO_WHO_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    return curr_irqs;
}

static void convert_uintarray_to_irqstruct(
                     uint32_t curr_irqs,
                     struct cisx_odma_irqs *irqstruct,
                     enum cisx_odma_mask_type operation )
{
    // If the given bit is set the interrupt field gets a true, false otherwise
    switch (operation)
    {
        case enable:
            irqstruct->rresp   = (CISX_ODMA_INT_EN_RRESP_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->bresp   = (CISX_ODMA_INT_EN_BRESP_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->rst     = (CISX_ODMA_INT_EN_RST_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi     = (CISX_ODMA_INT_EN_EOI_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->dir     = (CISX_ODMA_INT_EN_DIR_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->cl_ali  = (CISX_ODMA_INT_EN_CL_ALI_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eol_ali = (CISX_ODMA_INT_EN_EOL_ALI_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi_ali = (CISX_ODMA_INT_EN_EOI_ALI_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi_err = (CISX_ODMA_INT_EN_EOI_ERR_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->fin     = (CISX_ODMA_INT_EN_FIN_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->who     = (CISX_ODMA_INT_EN_WHO_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            break;

        case pending:
            irqstruct->rresp   = (CISX_ODMA_INT_PEND_RRESP_INT_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->bresp   = (CISX_ODMA_INT_PEND_BRESP_INT_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->rst     = (CISX_ODMA_INT_PEND_RST_INT_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi     = (CISX_ODMA_INT_PEND_EOI_INT_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->dir     = (CISX_ODMA_INT_PEND_DIR_INT_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->cl_ali  = (CISX_ODMA_INT_PEND_CL_ALI_INT_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eol_ali = (CISX_ODMA_INT_PEND_EOL_ALI_INT_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi_ali = (CISX_ODMA_INT_PEND_EOI_ALI_INT_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi_err = (CISX_ODMA_INT_PEND_EOI_ERR_INT_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->fin     = (CISX_ODMA_INT_PEND_FIN_INT_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->who     = (CISX_ODMA_INT_PEND_WHO_INT_PEND_MASK_SHIFT(curr_irqs) != 0);
            break;

        case ack:
            irqstruct->rresp   = (CISX_ODMA_INT_ACK_RRESP_INT_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->bresp   = (CISX_ODMA_INT_ACK_BRESP_INT_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->rst     = (CISX_ODMA_INT_ACK_RST_INT_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi     = (CISX_ODMA_INT_ACK_EOI_INT_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->dir     = (CISX_ODMA_INT_ACK_DIR_INT_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->cl_ali  = (CISX_ODMA_INT_ACK_CL_ALI_INT_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eol_ali = (CISX_ODMA_INT_ACK_EOL_ALI_INT_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi_ali = (CISX_ODMA_INT_ACK_EOI_ALI_INT_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi_err = (CISX_ODMA_INT_ACK_EOI_ERR_INT_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->fin     = (CISX_ODMA_INT_ACK_FIN_INT_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->who     = (CISX_ODMA_INT_ACK_WHO_INT_ACK_MASK_SHIFT(curr_irqs) != 0);
            break;

        case force:
            irqstruct->rresp   = (CISX_ODMA_INT_FO_RRESP_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->bresp   = (CISX_ODMA_INT_FO_BRESP_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->rst     = (CISX_ODMA_INT_FO_RST_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi     = (CISX_ODMA_INT_FO_EOI_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->dir     = (CISX_ODMA_INT_FO_DIR_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->cl_ali  = (CISX_ODMA_INT_FO_CL_ALI_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eol_ali = (CISX_ODMA_INT_FO_EOL_ALI_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi_ali = (CISX_ODMA_INT_FO_EOI_ALI_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi_err = (CISX_ODMA_INT_FO_EOI_ERR_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->fin     = (CISX_ODMA_INT_FO_FIN_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->who     = (CISX_ODMA_INT_FO_WHO_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            break;
    }
}

void cisx_odma_register_callback_irq(cisx_odmaDeviceHandle *device_data, void *callbackfcn)
{
    device_data->interrupt_callback = callbackfcn;
}

static void set_irq_enable_reg(cisx_odmaDeviceHandle *device_data, 
                               struct cisx_odma_irqs *irqs, 
                               bool set_to_one)
{
    struct   cisx_odma_irqs irqstruct;
    uint32_t reg;
    unsigned long flags;

    // Passing a null irq struct?  The enable setting will be applied to all
    // interrupts.
    if (irqs == NULL)
    {
        // All interrupts need to be updated
        set_all_irqstruct(&irqstruct, true);
        irqs = &irqstruct;
    }

    // Modify interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = cisx_odmaRead(int_en);
    reg = convert_irqstruct_to_update_uintarray(reg, irqs, set_to_one, enable);
    cisx_odmaWrite(int_en, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

void cisx_odma_enable_irq(cisx_odmaDeviceHandle *device_data, struct cisx_odma_irqs *irqs)
{
    set_irq_enable_reg(device_data, irqs, true);
}

void cisx_odma_disable_irq(cisx_odmaDeviceHandle *device_data, struct cisx_odma_irqs *irqs)
{
    set_irq_enable_reg(device_data, irqs, false);
}

void cisx_odma_force_irq(cisx_odmaDeviceHandle *device_data, struct cisx_odma_irqs *irqs)
{
    struct   cisx_odma_irqs irqstruct;
    uint32_t reg = 0;
    unsigned long flags;

    // Passing a null irq struct?  The force setting will be applied to all
    // interrupts.
    if (irqs == NULL)
    {
        // Mark all interrupts to be forced
        set_all_irqstruct(&irqstruct, true);
        irqs = &irqstruct;
    }

    // Force interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = convert_irqstruct_to_update_uintarray(reg, irqs, true, force);
    cisx_odmaWrite(int_fo, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

static uint32_t clear_cisx_odma_irqs_uint32(struct device_data *device_data, uint32_t int_val)
{
    unsigned long flags;  
    uint32_t      reg;

    PROTECT_INTREG_ACCESS_IRQ;
    reg = cisx_odmaRead(int_pend);
    cisx_odmaWrite(int_ack, int_val);  // clear requested ints
    UNPROTECT_INTREG_ACCESS_IRQ;

    return reg;
}

void cisx_odma_clear_irq(cisx_odmaDeviceHandle *device_data, struct cisx_odma_irqs *irqs)
{
    struct   cisx_odma_irqs irqstruct;
    uint32_t reg = 0;

    // Passing a null irq struct?  The clear setting will be applied to all
    // interrupts.
    if (irqs == NULL)
    {
        // Mark all interrupts to be cleared
        set_all_irqstruct(&irqstruct, true);
        irqs = &irqstruct;
    }

    // Clear interrupts
    reg = convert_irqstruct_to_update_uintarray(reg, irqs, true, ack);
    clear_cisx_odma_irqs_uint32(device_data, reg);
}

// NOTE: This function runs in interrupt thread context - try to be brief
void cisx_odma_handle_irq(struct device_data *device_data)
{
    uint32_t int_val;
    struct   cisx_odma_irqs irqstruct;
        
    // Grab the pending irqs then ack them all
    int_val = clear_cisx_odma_irqs_uint32(device_data, 0xFFFFFFFF);

    // Anything pending?
    if (int_val != 0)
    {
        debug_print("%s: cisx=%d subbloc=%d IntPend=0x%x\n", 
               __FUNCTION__, device_data->instance, device_data->submodinstance, int_val);

        // If someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            convert_uintarray_to_irqstruct(int_val, &irqstruct, pending);
            irqstruct.cisx_instance = device_data->instance;
            irqstruct.dma_instance  = device_data->submodinstance;
            irqstruct.irq_array     = int_val;

            device_data->interrupt_callback(&irqstruct);
        }
    }
}


//*****************************************************************************
// Register Access Functions
//*****************************************************************************

void get_cisx_odma_cfg(cisx_odmaDeviceHandle *device_data, struct cisx_odma_cfg_reg *cfgr)
{
    uint32_t reg;

    reg = cisx_odmaRead(cfg);
    cfgr->serpentine = CISX_ODMA_CFG_SERPENTINE_MASK_SHIFT(reg);
    cfgr->line_rev = CISX_ODMA_CFG_LINE_REV_MASK_SHIFT(reg);
    cfgr->tran_rev = CISX_ODMA_CFG_TRAN_REV_MASK_SHIFT(reg);
    cfgr->upper_half = CISX_ODMA_CFG_UPPER_HALF_MASK_SHIFT(reg);
    cfgr->msb_in = CISX_ODMA_CFG_MSB_IN_MASK_SHIFT(reg);
    cfgr->enable = CISX_ODMA_CFG_ENABLE_MASK_SHIFT(reg);
    cfgr->in_width = CISX_ODMA_CFG_IN_WIDTH_MASK_SHIFT(reg);
    cfgr->burst_len = CISX_ODMA_CFG_BURST_LEN_MASK_SHIFT(reg);
}

void set_cisx_odma_cfg(cisx_odmaDeviceHandle *device_data, struct cisx_odma_cfg_reg *cfgr)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_odmaRead(cfg);
    if(cfgr->serpentine_valid)
        reg = CISX_ODMA_CFG_SERPENTINE_REPLACE_VAL(reg, cfgr->serpentine);
    if(cfgr->line_rev_valid)
        reg = CISX_ODMA_CFG_LINE_REV_REPLACE_VAL(reg, cfgr->line_rev);
    if(cfgr->tran_rev_valid)
        reg = CISX_ODMA_CFG_TRAN_REV_REPLACE_VAL(reg, cfgr->tran_rev);
    if(cfgr->upper_half_valid)
        reg = CISX_ODMA_CFG_UPPER_HALF_REPLACE_VAL(reg, cfgr->upper_half);
    if(cfgr->msb_in_valid)
        reg = CISX_ODMA_CFG_MSB_IN_REPLACE_VAL(reg, cfgr->msb_in);
    if(cfgr->enable_valid)
        reg = CISX_ODMA_CFG_ENABLE_REPLACE_VAL(reg, cfgr->enable);
    if(cfgr->in_width_valid)
        reg = CISX_ODMA_CFG_IN_WIDTH_REPLACE_VAL(reg, cfgr->in_width);
    if(cfgr->burst_len_valid)
        reg = CISX_ODMA_CFG_BURST_LEN_REPLACE_VAL(reg, cfgr->burst_len);
    cisx_odmaWrite(cfg, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_odma_status(cisx_odmaDeviceHandle *device_data, struct cisx_odma_status_reg *cfg)
{
    uint32_t reg;

    reg = cisx_odmaRead(status);
    cfg->desc_own = CISX_ODMA_STATUS_DESC_OWN_MASK_SHIFT(reg);
    cfg->desc_eoi = CISX_ODMA_STATUS_DESC_EOI_MASK_SHIFT(reg);
    cfg->desc_soi = CISX_ODMA_STATUS_DESC_SOI_MASK_SHIFT(reg);
    cfg->reverse = CISX_ODMA_STATUS_REVERSE_MASK_SHIFT(reg);
    cfg->softreset = CISX_ODMA_STATUS_SOFTRESET_MASK_SHIFT(reg);
    cfg->full_cbuf = CISX_ODMA_STATUS_FULL_CBUF_MASK_SHIFT(reg);
    cfg->empty_cbuf = CISX_ODMA_STATUS_EMPTY_CBUF_MASK_SHIFT(reg);
    cfg->full_dbuf = CISX_ODMA_STATUS_FULL_DBUF_MASK_SHIFT(reg);
    cfg->empty_dbuf = CISX_ODMA_STATUS_EMPTY_DBUF_MASK_SHIFT(reg);
    cfg->packer_empty = CISX_ODMA_STATUS_PACKER_EMPTY_MASK_SHIFT(reg);
    cfg->dma_busy = CISX_ODMA_STATUS_DMA_BUSY_MASK_SHIFT(reg);
}

void get_cisx_odma_line_size(cisx_odmaDeviceHandle *device_data, struct cisx_odma_line_size_reg *cfg)
{
    uint32_t reg;

    reg = cisx_odmaRead(line_size);
    cfg->line_size = CISX_ODMA_LINE_SIZE_LINE_SIZE_MASK_SHIFT(reg);
}

void set_cisx_odma_line_size(cisx_odmaDeviceHandle *device_data, struct cisx_odma_line_size_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_odmaRead(line_size);
    if(cfg->line_size_valid)
        reg = CISX_ODMA_LINE_SIZE_LINE_SIZE_REPLACE_VAL(reg, cfg->line_size);
    cisx_odmaWrite(line_size, reg);
    UNPROTECT_REG_ACCESS;
}

void set_cisx_odma_desc_write(cisx_odmaDeviceHandle *device_data, struct cisx_odma_desc_write_reg *cfg)
{
    // NOTE: this is a 32 bit write only register!

    //debug_print("%s: cisx=%d subbloc=%d desc=0x%x (%d)\n", 
    //       __FUNCTION__, device_data->instance, device_data->submodinstance, cfg->desc, cfg->desc_valid);

    PROTECT_REG_ACCESS;
    if(cfg->desc_valid)
    {
        cisx_odmaWrite(desc_write, cfg->desc);
    }
    UNPROTECT_REG_ACCESS;
}

void get_cisx_odma_desc_read(cisx_odmaDeviceHandle *device_data, struct cisx_odma_desc_read_reg *cfg)
{
    uint32_t reg;

    reg = cisx_odmaRead(desc_read);
    cfg->desc = CISX_ODMA_DESC_READ_DESC_MASK_SHIFT(reg);
}

void get_cisx_odma_xfer_length(cisx_odmaDeviceHandle *device_data, struct cisx_odma_xfer_length_reg *cfg)
{
    uint32_t reg;

    reg = cisx_odmaRead(xfer_length);
    cfg->len = CISX_ODMA_XFER_LENGTH_LEN_MASK_SHIFT(reg);
}

void get_cisx_odma_xfer_addr(cisx_odmaDeviceHandle *device_data, struct cisx_odma_xfer_addr_reg *cfg)
{
    uint32_t reg;

    reg = cisx_odmaRead(xfer_addr);
    cfg->addr = CISX_ODMA_XFER_ADDR_ADDR_MASK_SHIFT(reg);
}

void get_cisx_odma_xfer_burst(cisx_odmaDeviceHandle *device_data, struct cisx_odma_xfer_burst_reg *cfg)
{
    uint32_t reg;

    reg = cisx_odmaRead(xfer_burst);
    cfg->burst = CISX_ODMA_XFER_BURST_BURST_MASK_SHIFT(reg);
}

void set_cisx_odma_reset(cisx_odmaDeviceHandle *device_data, struct cisx_odma_reset_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_odmaRead(reset);
    if(cfg->soft_reset_valid)
        reg = CISX_ODMA_RESET_SOFT_RESET_REPLACE_VAL(reg, cfg->soft_reset);
    cisx_odmaWrite(reset, reg);
    UNPROTECT_REG_ACCESS;
}


//*****************************************************************************
// Driver Functions
//*****************************************************************************

struct cisx_odma_function_struct cisx_odma_functions =
{
    .version                   = 0,
    .dump_cisx_odma_regs       = dump_cisx_odma_regs,

    .get_cisx_odma_cfg         = get_cisx_odma_cfg,
    .set_cisx_odma_cfg         = set_cisx_odma_cfg,
    .get_cisx_odma_status      = get_cisx_odma_status,
    .get_cisx_odma_line_size   = get_cisx_odma_line_size,
    .set_cisx_odma_line_size   = set_cisx_odma_line_size,
    .set_cisx_odma_desc_write  = set_cisx_odma_desc_write,
    .get_cisx_odma_desc_read   = get_cisx_odma_desc_read,
    .get_cisx_odma_xfer_length = get_cisx_odma_xfer_length,
    .get_cisx_odma_xfer_addr   = get_cisx_odma_xfer_addr,
    .get_cisx_odma_xfer_burst  = get_cisx_odma_xfer_burst,
    .set_cisx_odma_reset       = set_cisx_odma_reset,

    .cisx_odma_register_callback_irq = cisx_odma_register_callback_irq,
    .cisx_odma_enable_irq      = cisx_odma_enable_irq,
    .cisx_odma_disable_irq     = cisx_odma_disable_irq,
    .cisx_odma_force_irq       = cisx_odma_force_irq,
    .cisx_odma_clear_irq       = cisx_odma_clear_irq,
    .cisx_odma_handle_irq      = cisx_odma_handle_irq,
};

void cisx_odma_init(struct device_data *device_data)
{
    // register with the parent
    device_data->fcn_tbl = &cisx_odma_functions;
    register_cisx_subblock(output_dma, 
                           device_data, 
                           device_data->instance, 
                           device_data->submodinstance);

    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(device_data->reg_spinlock));
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(device_data->int_spinlock));

    // no callbacks yet
    device_data->interrupt_callback = NULL;    
}
EXPORT_SYMBOL(cisx_odma_init);

void cisx_odma_exit(struct device_data *device_data)
{
    // unregister with the parent
    unregister_cisx_subblock(output_dma, 
                             device_data->instance,
                             device_data->submodinstance);
}
EXPORT_SYMBOL(cisx_odma_exit);

