/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/irqreturn.h>  // needed for irqreturn_t and IRQ_HANDLED
#include <linux/spinlock.h>   // for spinlock_t, printk, BUG_ON, and memset
#include <linux/io.h>         // for ioread/write32 (Read/Write macros)
#include <linux/export.h>     // for EXPORT_SYMBOL

#include "CISX_int_regstructs.h"
#include "CISX_int_regmasks.h"

#include "cisx_if.h"
#include "cisx_data.h"
#include "cisx_driver.h"
#include "cisx_int_if.h"
#include "cisx_int.h"

#define MAX_ODMA_TBL_LUT_IDX      64
#define MAX_COLOR_OUT_TBL_LUT_IDX 4

enum cisx_int_mask_type {enable, pending, ack, force};


//*****************************************************************************
// Private helpers
//*****************************************************************************
static void set_lut_cpu_access(cisx_intDeviceHandle *device_data, bool cpu_access)
{
    uint32_t reg;
    uint8_t  set_val;

    set_val = cpu_access ? 1 : 0; 

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(CisxCfg);
    reg = CISX_INT_CISXCFG_CPU_ACCESS_LUT_REPLACE_VAL(reg, set_val);
    cisx_intWrite(CisxCfg, reg);
    UNPROTECT_REG_ACCESS;
}


//*****************************************************************************
// Debug Functions
//*****************************************************************************

void dump_cisx_int_regs(cisx_intDeviceHandle *device_data)
{
    printk("CISX INT dump for cisx %d\n", device_data->instance);
    printk("-------------------------------------\n");
    printk( "     cfg=0x%08x     stat=0x%08x    inten=0x%08x intpend=0x%08x\n", 
             cisx_intRead(CisxCfg), cisx_intRead(CisxStat), cisx_intRead(IntEn), cisx_intRead(IntPend) );
    printk( "chan0pix=0x%08x chan1pix=0x%08x chan2pix=0x%08x\n", 
             cisx_intRead(Chan0Pix), cisx_intRead(Chan1Pix), cisx_intRead(Chan2Pix) );
    printk( "chan3pix=0x%08x chan4pix=0x%08x chan5pix=0x%08x\n", 
             cisx_intRead(Chan3Pix), cisx_intRead(Chan4Pix), cisx_intRead(Chan5Pix) );
    printk( "chan0shf=0x%08x chan1shf=0x%08x chan2shf=0x%08x\n", 
             cisx_intRead(Chan0Shuffle), cisx_intRead(Chan1Shuffle), cisx_intRead(Chan2Shuffle) );
    printk( "chan3shf=0x%08x chan4shf=0x%08x chan5shf=0x%08x\n", 
             cisx_intRead(Chan3Shuffle), cisx_intRead(Chan4Shuffle), cisx_intRead(Chan5Shuffle) );
    printk( "  tblcnt=0x%08x     rev0=0x%08x     rev1=0x%08x\n", 
             cisx_intRead(TBL_CNT), cisx_intRead(REV0), cisx_intRead(REV1) );

    set_lut_cpu_access(device_data, true);

    printk( "odmalut0=0x%08x odmalut1=0x%08x odmalut2=0x%08x\n", 
             cisx_intReadIdx(ODMA_TBL_LUT, 0), 
             cisx_intReadIdx(ODMA_TBL_LUT, 1), 
             cisx_intReadIdx(ODMA_TBL_LUT, 2) );
    printk( " clrlut0=0x%08x  clrlut1=0x%08x  clrlut2=0x%08x\n", 
              cisx_intReadIdx(COLOR_OUT_TBL_LUT, 0), 
              cisx_intReadIdx(COLOR_OUT_TBL_LUT, 1), 
              cisx_intReadIdx(COLOR_OUT_TBL_LUT, 2)); 

    set_lut_cpu_access(device_data, false);
}

//*****************************************************************************
// Platform IRQ Functions
//*****************************************************************************

irqreturn_t cisx_platform_irq(int irq, void *pdev)
{
    uint8_t  cisx_instance;
    uint32_t reg;
    cisx_intDeviceHandle *device_data;
    unsigned long flags;  // for IRQ protection macro

    // NOTE - This function runs in interrupt context - no long operations allowed

    // Both CISX blocks currently share the same IRQ, so don't trust the passed
    // pdev to contain valid device data information.  Need to loop through all
    // the CISX blocks to find out who yelled ...
    for (cisx_instance = 0; cisx_instance < MAX_NUM_CISXBLOCKS; cisx_instance++)
    {
        device_data = get_cisx_int_device(cisx_instance);
        if (device_data != NULL)
        {
            // Grab the cisx int irq pending register (uses device_data)
            PROTECT_INTREG_ACCESS_IRQ;
            reg = cisx_intRead(IntPend);
            UNPROTECT_INTREG_ACCESS_IRQ;

            debug_print("%s: cisx=%d IntPend=0x%x\n", __FUNCTION__, cisx_instance, reg);
            
            // Test for any pending IDMA interrupts
            if (reg & CISX_INT_INTPEND_IDMA1_IRQ_PEND_MASK)
                cisx_idma_handle_irqs(cisx_instance, 1);
            if (reg & CISX_INT_INTPEND_IDMA0_IRQ_PEND_MASK)
                cisx_idma_handle_irqs(cisx_instance, 0);

            // Test for any pending ODMA interrupts
            if (reg & CISX_INT_INTPEND_ODMA5_IRQ_PEND_MASK)
                cisx_odma_handle_irqs(cisx_instance, 5);
            if (reg & CISX_INT_INTPEND_ODMA4_IRQ_PEND_MASK)
                cisx_odma_handle_irqs(cisx_instance, 4);
            if (reg & CISX_INT_INTPEND_ODMA3_IRQ_PEND_MASK)
                cisx_odma_handle_irqs(cisx_instance, 3);
            if (reg & CISX_INT_INTPEND_ODMA2_IRQ_PEND_MASK)
                cisx_odma_handle_irqs(cisx_instance, 2);
            if (reg & CISX_INT_INTPEND_ODMA1_IRQ_PEND_MASK)
                cisx_odma_handle_irqs(cisx_instance, 1);
            if (reg & CISX_INT_INTPEND_ODMA0_IRQ_PEND_MASK)
                cisx_odma_handle_irqs(cisx_instance, 0);

            // Now handle the CISX INT irq (which will also ack them)
            cisx_int_handle_irqs(cisx_instance);
        }
    }

    return IRQ_HANDLED;
}


//*****************************************************************************
// CISX INT IRQ Functions
//*****************************************************************************

static void set_all_irqstruct(struct cisx_int_irqs *irqstruct, bool setval)
{
    irqstruct->imagedone = setval;
    irqstruct->odma5     = setval;
    irqstruct->odma4     = setval;
    irqstruct->odma3     = setval;
    irqstruct->odma2     = setval;
    irqstruct->odma1     = setval;
    irqstruct->odma0     = setval;
    irqstruct->idma1     = setval;
    irqstruct->idma0     = setval;
    irqstruct->dmabuserr = setval;
}

static uint32_t convert_irqstruct_to_update_uintarray(
                    uint32_t curr_irqs,
                    struct   cisx_int_irqs *irqstruct,
                    bool     set_to_one,
                    enum cisx_int_mask_type operation )
{
    uint8_t set_val;

    // Are we enabling or disabling IRQs?
    set_val = set_to_one ? 1 : 0; 

    // Any IRQ's in irqstruct that are set to true will get set to set_val, other
    // bits will be left as-is.
    if (irqstruct->imagedone)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_INT_INTEN_IMAGEDONE_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_INT_INTPEND_IMAGEDONE_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_INT_INTACK_IMAGEDONE_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_INT_INTFORCE_IMAGEDONE_FORCE_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->odma5)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_INT_INTEN_ODMA5_IRQ_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_INT_INTPEND_ODMA5_IRQ_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_INT_INTACK_ODMA5_IRQ_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_INT_INTFORCE_ODMA5_IRQ_FORCE_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->odma4)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_INT_INTEN_ODMA4_IRQ_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_INT_INTPEND_ODMA4_IRQ_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_INT_INTACK_ODMA4_IRQ_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_INT_INTFORCE_ODMA4_IRQ_FORCE_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->odma3)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_INT_INTEN_ODMA3_IRQ_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_INT_INTPEND_ODMA3_IRQ_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_INT_INTACK_ODMA3_IRQ_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_INT_INTFORCE_ODMA3_IRQ_FORCE_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->odma2)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_INT_INTEN_ODMA2_IRQ_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_INT_INTPEND_ODMA2_IRQ_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_INT_INTACK_ODMA2_IRQ_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_INT_INTFORCE_ODMA2_IRQ_FORCE_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->odma1)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_INT_INTEN_ODMA1_IRQ_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_INT_INTPEND_ODMA1_IRQ_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_INT_INTACK_ODMA1_IRQ_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_INT_INTFORCE_ODMA1_IRQ_FORCE_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->odma0)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_INT_INTEN_ODMA0_IRQ_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_INT_INTPEND_ODMA0_IRQ_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_INT_INTACK_ODMA0_IRQ_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_INT_INTFORCE_ODMA0_IRQ_FORCE_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->idma1)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_INT_INTEN_IDMA1_IRQ_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_INT_INTPEND_IDMA1_IRQ_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_INT_INTACK_IDMA1_IRQ_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_INT_INTFORCE_IDMA1_IRQ_FORCE_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->dmabuserr)
    {
       switch (operation)
        {
            case enable:
                curr_irqs = CISX_INT_INTEN_DMABUSERR_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_INT_INTPEND_DMABUSERR_PEND_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_INT_INTACK_DMABUSERR_ACK_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_INT_INTFORCE_DMABUSERR_FORCE_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    return curr_irqs;
}

static void convert_uintarray_to_irqstruct(
                     uint32_t curr_irqs,
                     struct cisx_int_irqs *irqstruct,
                     enum cisx_int_mask_type operation )
{
    // If the given bit is set the interrupt field gets a true, false otherwise
    switch (operation)
    {
        case enable:
            irqstruct->imagedone = (CISX_INT_INTEN_IMAGEDONE_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma5     = (CISX_INT_INTEN_ODMA5_IRQ_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma4     = (CISX_INT_INTEN_ODMA4_IRQ_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma3     = (CISX_INT_INTEN_ODMA3_IRQ_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma2     = (CISX_INT_INTEN_ODMA2_IRQ_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma1     = (CISX_INT_INTEN_ODMA1_IRQ_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma0     = (CISX_INT_INTEN_ODMA0_IRQ_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->idma1     = (CISX_INT_INTEN_IDMA1_IRQ_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->idma0     = (CISX_INT_INTEN_IDMA0_IRQ_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->dmabuserr = (CISX_INT_INTEN_DMABUSERR_EN_MASK_SHIFT(curr_irqs) != 0);
            break;

        case pending:
            irqstruct->imagedone = (CISX_INT_INTPEND_IMAGEDONE_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma5     = (CISX_INT_INTPEND_ODMA5_IRQ_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma4     = (CISX_INT_INTPEND_ODMA4_IRQ_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma3     = (CISX_INT_INTPEND_ODMA3_IRQ_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma2     = (CISX_INT_INTPEND_ODMA2_IRQ_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma1     = (CISX_INT_INTPEND_ODMA1_IRQ_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma0     = (CISX_INT_INTPEND_ODMA0_IRQ_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->idma1     = (CISX_INT_INTPEND_IDMA1_IRQ_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->idma0     = (CISX_INT_INTPEND_IDMA0_IRQ_PEND_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->dmabuserr = (CISX_INT_INTPEND_DMABUSERR_PEND_MASK_SHIFT(curr_irqs) != 0);
            break;

        case ack:
            irqstruct->imagedone = (CISX_INT_INTACK_IMAGEDONE_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma5     = (CISX_INT_INTACK_ODMA5_IRQ_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma4     = (CISX_INT_INTACK_ODMA4_IRQ_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma3     = (CISX_INT_INTACK_ODMA3_IRQ_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma2     = (CISX_INT_INTACK_ODMA2_IRQ_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma1     = (CISX_INT_INTACK_ODMA1_IRQ_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma0     = (CISX_INT_INTACK_ODMA0_IRQ_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->idma1     = (CISX_INT_INTACK_IDMA1_IRQ_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->idma0     = (CISX_INT_INTACK_IDMA0_IRQ_ACK_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->dmabuserr = (CISX_INT_INTACK_DMABUSERR_ACK_MASK_SHIFT(curr_irqs) != 0);
            break;

        case force:
            irqstruct->imagedone = (CISX_INT_INTFORCE_IMAGEDONE_FORCE_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma5     = (CISX_INT_INTFORCE_ODMA5_IRQ_FORCE_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma4     = (CISX_INT_INTFORCE_ODMA4_IRQ_FORCE_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma3     = (CISX_INT_INTFORCE_ODMA3_IRQ_FORCE_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma2     = (CISX_INT_INTFORCE_ODMA2_IRQ_FORCE_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma1     = (CISX_INT_INTFORCE_ODMA1_IRQ_FORCE_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->odma0     = (CISX_INT_INTFORCE_ODMA0_IRQ_FORCE_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->idma1     = (CISX_INT_INTFORCE_IDMA1_IRQ_FORCE_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->idma0     = (CISX_INT_INTFORCE_IDMA0_IRQ_FORCE_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->dmabuserr = (CISX_INT_INTFORCE_DMABUSERR_FORCE_MASK_SHIFT(curr_irqs) != 0);
            break;
    }
}

void cisx_int_register_callback_irq(cisx_intDeviceHandle *device_data, void *callbackfcn)
{
    device_data->interrupt_callback = callbackfcn;
}

static void set_irq_enable_reg(cisx_intDeviceHandle *device_data, 
                               struct cisx_int_irqs *irqs, 
                               bool set_to_one)
{
    struct   cisx_int_irqs irqstruct;
    uint32_t reg;
    unsigned long flags;

    // Passing a null irq struct?  The enable setting will be applied to all
    // interrupts.
    if (irqs == NULL)
    {
        // All interrupts need to be updated
        set_all_irqstruct(&irqstruct, true);
        irqs = &irqstruct;
    }

    // Modify interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = cisx_intRead(IntEn);
    reg = convert_irqstruct_to_update_uintarray(reg, irqs, set_to_one, enable);
    cisx_intWrite(IntEn, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}


void cisx_int_enable_irq(cisx_intDeviceHandle *device_data, struct cisx_int_irqs *irqs)
{
    set_irq_enable_reg(device_data, irqs, true);
}

void cisx_int_disable_irq(cisx_intDeviceHandle *device_data, struct cisx_int_irqs *irqs)
{
    set_irq_enable_reg(device_data, irqs, false);
}

void cisx_int_force_irq(cisx_intDeviceHandle *device_data, struct cisx_int_irqs *irqs)
{
    struct   cisx_int_irqs irqstruct;
    uint32_t reg = 0;
    unsigned long flags;

    // Passing a null irq struct?  The force setting will be applied to all
    // interrupts.
    if (irqs == NULL)
    {
        // Mark all interrupts to be forced
        set_all_irqstruct(&irqstruct, true);
        irqs = &irqstruct;
    }

    // Force interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = convert_irqstruct_to_update_uintarray(reg, irqs, true, force);
    cisx_intWrite(IntForce, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

static uint32_t clear_cisx_irqs_uint32(struct device_data *device_data, uint32_t int_val)
{
    unsigned long flags;  
    uint32_t      reg;

    PROTECT_INTREG_ACCESS_IRQ;
    reg = cisx_intRead(IntPend);
    cisx_intWrite(IntAck, int_val);  // clear requested ints
    UNPROTECT_INTREG_ACCESS_IRQ;

    return reg;
}

void cisx_int_clear_irq(cisx_intDeviceHandle *device_data, struct cisx_int_irqs *irqs)
{
    struct   cisx_int_irqs irqstruct;
    uint32_t reg = 0;

    // Passing a null irq struct?  The clear setting will be applied to all
    // interrupts.
    if (irqs == NULL)
    {
        // Mark all interrupts to be cleared
        set_all_irqstruct(&irqstruct, true);
        irqs = &irqstruct;
    }

    // Clear interrupts
    reg = convert_irqstruct_to_update_uintarray(reg, irqs, true, ack);
    clear_cisx_irqs_uint32(device_data, reg);
}

// NOTE: This function runs in interrupt thread context - try to be brief
void cisx_int_handle_irq(struct device_data *device_data)
{
    uint32_t int_val;
    struct   cisx_int_irqs irqstruct;
        
    // Grab the pending irqs then ack them all
    int_val = clear_cisx_irqs_uint32(device_data, 0xFFFFFFFF);

    // Anything pending?
    if (int_val != 0)
    {
        convert_uintarray_to_irqstruct(int_val, &irqstruct, pending);
        irqstruct.cisx_instance = device_data->instance;
        irqstruct.irq_array     = int_val;

        // If someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            device_data->interrupt_callback(&irqstruct);
        }
    }
}


//*****************************************************************************
// Driver Register Access Functions
//*****************************************************************************

void get_cisx_int_CisxCfg(cisx_intDeviceHandle *device_data, struct cisx_int_CisxCfg_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(CisxCfg);
    cfg->enable = CISX_INT_CISXCFG_ENABLE_MASK_SHIFT(reg);
    cfg->slave_inp_sel = CISX_INT_CISXCFG_SLAVE_INP_SEL_MASK_SHIFT(reg);
    cfg->bypass = CISX_INT_CISXCFG_BYPASS_MASK_SHIFT(reg);
    cfg->mode = CISX_INT_CISXCFG_MODE_MASK_SHIFT(reg);
    cfg->cbiout_pace = CISX_INT_CISXCFG_CBIOUT_PACE_MASK_SHIFT(reg);
    cfg->cpu_access_lut = CISX_INT_CISXCFG_CPU_ACCESS_LUT_MASK_SHIFT(reg);
    cfg->numchans = CISX_INT_CISXCFG_NUMCHANS_MASK_SHIFT(reg);
    cfg->ccd_order = CISX_INT_CISXCFG_CCD_ORDER_MASK_SHIFT(reg);
    cfg->color_out_tbl_en = CISX_INT_CISXCFG_COLOR_OUT_TBL_EN_MASK_SHIFT(reg);
    cfg->appchan_idata = CISX_INT_CISXCFG_APPCHAN_IDATA_MASK_SHIFT(reg);
    cfg->appchan_ext = CISX_INT_CISXCFG_APPCHAN_EXT_MASK_SHIFT(reg);
    cfg->appchan5 = CISX_INT_CISXCFG_APPCHAN5_MASK_SHIFT(reg);
    cfg->appchan4 = CISX_INT_CISXCFG_APPCHAN4_MASK_SHIFT(reg);
    cfg->appchan3 = CISX_INT_CISXCFG_APPCHAN3_MASK_SHIFT(reg);
    cfg->appchan2 = CISX_INT_CISXCFG_APPCHAN2_MASK_SHIFT(reg);
    cfg->appchan1 = CISX_INT_CISXCFG_APPCHAN1_MASK_SHIFT(reg);
    cfg->appchan0 = CISX_INT_CISXCFG_APPCHAN0_MASK_SHIFT(reg);
}

void set_cisx_int_CisxCfg(cisx_intDeviceHandle *device_data, struct cisx_int_CisxCfg_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(CisxCfg);
    if(cfg->enable_valid)
        reg = CISX_INT_CISXCFG_ENABLE_REPLACE_VAL(reg, cfg->enable);
    if(cfg->slave_inp_sel_valid)
        reg = CISX_INT_CISXCFG_SLAVE_INP_SEL_REPLACE_VAL(reg, cfg->slave_inp_sel);
    if(cfg->bypass_valid)
        reg = CISX_INT_CISXCFG_BYPASS_REPLACE_VAL(reg, cfg->bypass);
    if(cfg->mode_valid)
        reg = CISX_INT_CISXCFG_MODE_REPLACE_VAL(reg, cfg->mode);
    if(cfg->cbiout_pace_valid)
        reg = CISX_INT_CISXCFG_CBIOUT_PACE_REPLACE_VAL(reg, cfg->cbiout_pace);
    if(cfg->cpu_access_lut_valid)
        reg = CISX_INT_CISXCFG_CPU_ACCESS_LUT_REPLACE_VAL(reg, cfg->cpu_access_lut);
    if(cfg->numchans_valid)
        reg = CISX_INT_CISXCFG_NUMCHANS_REPLACE_VAL(reg, cfg->numchans);
    if(cfg->ccd_order_valid)
        reg = CISX_INT_CISXCFG_CCD_ORDER_REPLACE_VAL(reg, cfg->ccd_order);
    if(cfg->color_out_tbl_en_valid)
        reg = CISX_INT_CISXCFG_COLOR_OUT_TBL_EN_REPLACE_VAL(reg, cfg->color_out_tbl_en);
    if(cfg->appchan_idata_valid)
        reg = CISX_INT_CISXCFG_APPCHAN_IDATA_REPLACE_VAL(reg, cfg->appchan_idata);
    if(cfg->appchan_ext_valid)
        reg = CISX_INT_CISXCFG_APPCHAN_EXT_REPLACE_VAL(reg, cfg->appchan_ext);
    if(cfg->appchan5_valid)
        reg = CISX_INT_CISXCFG_APPCHAN5_REPLACE_VAL(reg, cfg->appchan5);
    if(cfg->appchan4_valid)
        reg = CISX_INT_CISXCFG_APPCHAN4_REPLACE_VAL(reg, cfg->appchan4);
    if(cfg->appchan3_valid)
        reg = CISX_INT_CISXCFG_APPCHAN3_REPLACE_VAL(reg, cfg->appchan3);
    if(cfg->appchan2_valid)
        reg = CISX_INT_CISXCFG_APPCHAN2_REPLACE_VAL(reg, cfg->appchan2);
    if(cfg->appchan1_valid)
        reg = CISX_INT_CISXCFG_APPCHAN1_REPLACE_VAL(reg, cfg->appchan1);
    if(cfg->appchan0_valid)
        reg = CISX_INT_CISXCFG_APPCHAN0_REPLACE_VAL(reg, cfg->appchan0);
    cisx_intWrite(CisxCfg, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_CisxStat(cisx_intDeviceHandle *device_data, struct cisx_int_CisxStat_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(CisxStat);
    cfg->line_sm = CISX_INT_CISXSTAT_LINE_SM_MASK_SHIFT(reg);
    cfg->odma_busy = CISX_INT_CISXSTAT_ODMA_BUSY_MASK_SHIFT(reg);
    cfg->idma_busy = CISX_INT_CISXSTAT_IDMA_BUSY_MASK_SHIFT(reg);
}

void get_cisx_int_Chan0Pix(cisx_intDeviceHandle *device_data, struct cisx_int_Chan0Pix_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(Chan0Pix);
    cfg->chan0dum = CISX_INT_CHAN0PIX_CHAN0DUM_MASK_SHIFT(reg);
    cfg->chan0dat = CISX_INT_CHAN0PIX_CHAN0DAT_MASK_SHIFT(reg);
}

void set_cisx_int_Chan0Pix(cisx_intDeviceHandle *device_data, struct cisx_int_Chan0Pix_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(Chan0Pix);
    if(cfg->chan0dum_valid)
        reg = CISX_INT_CHAN0PIX_CHAN0DUM_REPLACE_VAL(reg, cfg->chan0dum);
    if(cfg->chan0dat_valid)
        reg = CISX_INT_CHAN0PIX_CHAN0DAT_REPLACE_VAL(reg, cfg->chan0dat);
    cisx_intWrite(Chan0Pix, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_Chan1Pix(cisx_intDeviceHandle *device_data, struct cisx_int_Chan1Pix_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(Chan1Pix);
    cfg->chan1dum = CISX_INT_CHAN1PIX_CHAN1DUM_MASK_SHIFT(reg);
    cfg->chan1dat = CISX_INT_CHAN1PIX_CHAN1DAT_MASK_SHIFT(reg);
}

void set_cisx_int_Chan1Pix(cisx_intDeviceHandle *device_data, struct cisx_int_Chan1Pix_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(Chan1Pix);
    if(cfg->chan1dum_valid)
        reg = CISX_INT_CHAN1PIX_CHAN1DUM_REPLACE_VAL(reg, cfg->chan1dum);
    if(cfg->chan1dat_valid)
        reg = CISX_INT_CHAN1PIX_CHAN1DAT_REPLACE_VAL(reg, cfg->chan1dat);
    cisx_intWrite(Chan1Pix, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_Chan2Pix(cisx_intDeviceHandle *device_data, struct cisx_int_Chan2Pix_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(Chan2Pix);
    cfg->chan2dum = CISX_INT_CHAN2PIX_CHAN2DUM_MASK_SHIFT(reg);
    cfg->chan2dat = CISX_INT_CHAN2PIX_CHAN2DAT_MASK_SHIFT(reg);
}

void set_cisx_int_Chan2Pix(cisx_intDeviceHandle *device_data, struct cisx_int_Chan2Pix_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(Chan2Pix);
    if(cfg->chan2dum_valid)
        reg = CISX_INT_CHAN2PIX_CHAN2DUM_REPLACE_VAL(reg, cfg->chan2dum);
    if(cfg->chan2dat_valid)
        reg = CISX_INT_CHAN2PIX_CHAN2DAT_REPLACE_VAL(reg, cfg->chan2dat);
    cisx_intWrite(Chan2Pix, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_Chan3Pix(cisx_intDeviceHandle *device_data, struct cisx_int_Chan3Pix_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(Chan3Pix);
    cfg->chan3dum = CISX_INT_CHAN3PIX_CHAN3DUM_MASK_SHIFT(reg);
    cfg->chan3dat = CISX_INT_CHAN3PIX_CHAN3DAT_MASK_SHIFT(reg);
}

void set_cisx_int_Chan3Pix(cisx_intDeviceHandle *device_data, struct cisx_int_Chan3Pix_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(Chan3Pix);
    if(cfg->chan3dum_valid)
        reg = CISX_INT_CHAN3PIX_CHAN3DUM_REPLACE_VAL(reg, cfg->chan3dum);
    if(cfg->chan3dat_valid)
        reg = CISX_INT_CHAN3PIX_CHAN3DAT_REPLACE_VAL(reg, cfg->chan3dat);
    cisx_intWrite(Chan3Pix, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_Chan4Pix(cisx_intDeviceHandle *device_data, struct cisx_int_Chan4Pix_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(Chan4Pix);
    cfg->chan4dum = CISX_INT_CHAN4PIX_CHAN4DUM_MASK_SHIFT(reg);
    cfg->chan4dat = CISX_INT_CHAN4PIX_CHAN4DAT_MASK_SHIFT(reg);
}

void set_cisx_int_Chan4Pix(cisx_intDeviceHandle *device_data, struct cisx_int_Chan4Pix_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(Chan4Pix);
    if(cfg->chan4dum_valid)
        reg = CISX_INT_CHAN4PIX_CHAN4DUM_REPLACE_VAL(reg, cfg->chan4dum);
    if(cfg->chan4dat_valid)
        reg = CISX_INT_CHAN4PIX_CHAN4DAT_REPLACE_VAL(reg, cfg->chan4dat);
    cisx_intWrite(Chan4Pix, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_Chan5Pix(cisx_intDeviceHandle *device_data, struct cisx_int_Chan5Pix_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(Chan5Pix);
    cfg->chan5dum = CISX_INT_CHAN5PIX_CHAN5DUM_MASK_SHIFT(reg);
    cfg->chan5dat = CISX_INT_CHAN5PIX_CHAN5DAT_MASK_SHIFT(reg);
}

void set_cisx_int_Chan5Pix(cisx_intDeviceHandle *device_data, struct cisx_int_Chan5Pix_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(Chan5Pix);
    if(cfg->chan5dum_valid)
        reg = CISX_INT_CHAN5PIX_CHAN5DUM_REPLACE_VAL(reg, cfg->chan5dum);
    if(cfg->chan5dat_valid)
        reg = CISX_INT_CHAN5PIX_CHAN5DAT_REPLACE_VAL(reg, cfg->chan5dat);
    cisx_intWrite(Chan5Pix, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_Chan0Shuffle(cisx_intDeviceHandle *device_data, struct cisx_int_Chan0Shuffle_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(Chan0Shuffle);
    cfg->chan0sep = CISX_INT_CHAN0SHUFFLE_CHAN0SEP_MASK_SHIFT(reg);
}

void set_cisx_int_Chan0Shuffle(cisx_intDeviceHandle *device_data, struct cisx_int_Chan0Shuffle_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(Chan0Shuffle);
    if(cfg->chan0sep_valid)
        reg = CISX_INT_CHAN0SHUFFLE_CHAN0SEP_REPLACE_VAL(reg, cfg->chan0sep);
    cisx_intWrite(Chan0Shuffle, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_Chan1Shuffle(cisx_intDeviceHandle *device_data, struct cisx_int_Chan1Shuffle_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(Chan1Shuffle);
    cfg->chan1sep = CISX_INT_CHAN1SHUFFLE_CHAN1SEP_MASK_SHIFT(reg);
}

void set_cisx_int_Chan1Shuffle(cisx_intDeviceHandle *device_data, struct cisx_int_Chan1Shuffle_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(Chan1Shuffle);
    if(cfg->chan1sep_valid)
        reg = CISX_INT_CHAN1SHUFFLE_CHAN1SEP_REPLACE_VAL(reg, cfg->chan1sep);
    cisx_intWrite(Chan1Shuffle, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_Chan2Shuffle(cisx_intDeviceHandle *device_data, struct cisx_int_Chan2Shuffle_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(Chan2Shuffle);
    cfg->chan2sep = CISX_INT_CHAN2SHUFFLE_CHAN2SEP_MASK_SHIFT(reg);
}

void set_cisx_int_Chan2Shuffle(cisx_intDeviceHandle *device_data, struct cisx_int_Chan2Shuffle_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(Chan2Shuffle);
    if(cfg->chan2sep_valid)
        reg = CISX_INT_CHAN2SHUFFLE_CHAN2SEP_REPLACE_VAL(reg, cfg->chan2sep);
    cisx_intWrite(Chan2Shuffle, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_Chan3Shuffle(cisx_intDeviceHandle *device_data, struct cisx_int_Chan3Shuffle_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(Chan3Shuffle);
    cfg->chan3sep = CISX_INT_CHAN3SHUFFLE_CHAN3SEP_MASK_SHIFT(reg);
}

void set_cisx_int_Chan3Shuffle(cisx_intDeviceHandle *device_data, struct cisx_int_Chan3Shuffle_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(Chan3Shuffle);
    if(cfg->chan3sep_valid)
        reg = CISX_INT_CHAN3SHUFFLE_CHAN3SEP_REPLACE_VAL(reg, cfg->chan3sep);
    cisx_intWrite(Chan3Shuffle, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_Chan4Shuffle(cisx_intDeviceHandle *device_data, struct cisx_int_Chan4Shuffle_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(Chan4Shuffle);
    cfg->chan4sep = CISX_INT_CHAN4SHUFFLE_CHAN4SEP_MASK_SHIFT(reg);
}

void set_cisx_int_Chan4Shuffle(cisx_intDeviceHandle *device_data, struct cisx_int_Chan4Shuffle_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(Chan4Shuffle);
    if(cfg->chan4sep_valid)
        reg = CISX_INT_CHAN4SHUFFLE_CHAN4SEP_REPLACE_VAL(reg, cfg->chan4sep);
    cisx_intWrite(Chan4Shuffle, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_Chan5Shuffle(cisx_intDeviceHandle *device_data, struct cisx_int_Chan5Shuffle_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(Chan5Shuffle);
    cfg->chan5sep = CISX_INT_CHAN5SHUFFLE_CHAN5SEP_MASK_SHIFT(reg);
}

void set_cisx_int_Chan5Shuffle(cisx_intDeviceHandle *device_data, struct cisx_int_Chan5Shuffle_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(Chan5Shuffle);
    if(cfg->chan5sep_valid)
        reg = CISX_INT_CHAN5SHUFFLE_CHAN5SEP_REPLACE_VAL(reg, cfg->chan5sep);
    cisx_intWrite(Chan5Shuffle, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_TBL_CNT(cisx_intDeviceHandle *device_data, struct cisx_int_TBL_CNT_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(TBL_CNT);
    cfg->color_out_tbl_cnt = CISX_INT_TBL_CNT_COLOR_OUT_TBL_CNT_MASK_SHIFT(reg);
    cfg->odma_tbl_cnt = CISX_INT_TBL_CNT_ODMA_TBL_CNT_MASK_SHIFT(reg);
}

void set_cisx_int_TBL_CNT(cisx_intDeviceHandle *device_data, struct cisx_int_TBL_CNT_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_intRead(TBL_CNT);
    if(cfg->color_out_tbl_cnt_valid)
        reg = CISX_INT_TBL_CNT_COLOR_OUT_TBL_CNT_REPLACE_VAL(reg, cfg->color_out_tbl_cnt);
    if(cfg->odma_tbl_cnt_valid)
        reg = CISX_INT_TBL_CNT_ODMA_TBL_CNT_REPLACE_VAL(reg, cfg->odma_tbl_cnt);
    cisx_intWrite(TBL_CNT, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_int_REV0(cisx_intDeviceHandle *device_data, struct cisx_int_REV0_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(REV0);
    cfg->maj = CISX_INT_REV0_MAJ_MASK_SHIFT(reg);
    cfg->mid = CISX_INT_REV0_MID_MASK_SHIFT(reg);
}

void get_cisx_int_REV1(cisx_intDeviceHandle *device_data, struct cisx_int_REV1_reg *cfg)
{
    uint32_t reg;

    reg = cisx_intRead(REV1);
    cfg->cfg = CISX_INT_REV1_CFG_MASK_SHIFT(reg);
}

void get_cisx_int_ODMA_TBL_LUT(cisx_intDeviceHandle *device_data, 
                               struct cisx_int_ODMA_TBL_LUT_reg *cfg,
                               uint8_t index)
{
    uint32_t reg;

    if (index >= MAX_ODMA_TBL_LUT_IDX)
    {
        debug_print("%s: index out of range (idx=%d), ignoring\n", __FUNCTION__, index);
        return;
    }

    set_lut_cpu_access(device_data, true);
    reg = cisx_intReadIdx(ODMA_TBL_LUT, index);
    cfg->d = CISX_INT_ODMA_TBL_LUT_D_MASK_SHIFT(reg);
    set_lut_cpu_access(device_data, false);
}

void set_cisx_int_ODMA_TBL_LUT(cisx_intDeviceHandle *device_data,
                               struct cisx_int_ODMA_TBL_LUT_reg *cfg,
                               uint8_t index)
{
    uint32_t reg;

    if (index >= MAX_ODMA_TBL_LUT_IDX)
    {
        debug_print("%s: index out of range (idx=%d), ignoring\n", __FUNCTION__, index);
        return;
    }

    set_lut_cpu_access(device_data, true);

    PROTECT_REG_ACCESS;
    reg = cisx_intReadIdx(ODMA_TBL_LUT, index);
    if(cfg->d_valid)
        reg = CISX_INT_ODMA_TBL_LUT_D_REPLACE_VAL(reg, cfg->d);
    cisx_intWriteIdx(ODMA_TBL_LUT, index, reg);
    UNPROTECT_REG_ACCESS;

    set_lut_cpu_access(device_data, false);
}

void get_cisx_int_COLOR_OUT_TBL_LUT(cisx_intDeviceHandle *device_data, 
                                    struct cisx_int_COLOR_OUT_TBL_LUT_reg *cfg,
                                    uint8_t index)
{
    uint32_t reg;

    if (index >= MAX_COLOR_OUT_TBL_LUT_IDX)
    {
        debug_print("%s: index out of range (idx=%d), ignoring\n", __FUNCTION__, index);
        return;
    }

    set_lut_cpu_access(device_data, true);
    reg = cisx_intReadIdx(COLOR_OUT_TBL_LUT, index);
    cfg->d = CISX_INT_COLOR_OUT_TBL_LUT_D_MASK_SHIFT(reg);
    set_lut_cpu_access(device_data, false);
}

void set_cisx_int_COLOR_OUT_TBL_LUT(cisx_intDeviceHandle *device_data, 
                                    struct cisx_int_COLOR_OUT_TBL_LUT_reg *cfg,
                                    uint8_t index)
{
    uint32_t reg;

    if (index >= MAX_COLOR_OUT_TBL_LUT_IDX)
    {
        debug_print("%s: index out of range (idx=%d), ignoring\n", __FUNCTION__, index);
        return;
    }

    set_lut_cpu_access(device_data, true);

    PROTECT_REG_ACCESS;
    reg = cisx_intReadIdx(COLOR_OUT_TBL_LUT, index);
    if(cfg->d_valid)
        reg = CISX_INT_COLOR_OUT_TBL_LUT_D_REPLACE_VAL(reg, cfg->d);
    cisx_intWriteIdx(COLOR_OUT_TBL_LUT, index, reg);
    UNPROTECT_REG_ACCESS;

    set_lut_cpu_access(device_data, false);
}


//*****************************************************************************
// Driver Functions
//*****************************************************************************

struct cisx_int_function_struct cisx_int_functions =
{
    .version                   = 0,
    .dump_cisx_int_regs        = dump_cisx_int_regs,
    .get_cisx_int_CisxCfg      = get_cisx_int_CisxCfg,
    .set_cisx_int_CisxCfg      = set_cisx_int_CisxCfg,
    .get_cisx_int_CisxStat     = get_cisx_int_CisxStat,
    .get_cisx_int_Chan0Pix     = get_cisx_int_Chan0Pix,
    .set_cisx_int_Chan0Pix     = set_cisx_int_Chan0Pix,
    .get_cisx_int_Chan1Pix     = get_cisx_int_Chan1Pix,
    .set_cisx_int_Chan1Pix     = set_cisx_int_Chan1Pix,
    .get_cisx_int_Chan2Pix     = get_cisx_int_Chan2Pix,
    .set_cisx_int_Chan2Pix     = set_cisx_int_Chan2Pix,
    .get_cisx_int_Chan3Pix     = get_cisx_int_Chan3Pix,
    .set_cisx_int_Chan3Pix     = set_cisx_int_Chan3Pix,
    .get_cisx_int_Chan4Pix     = get_cisx_int_Chan4Pix,
    .set_cisx_int_Chan4Pix     = set_cisx_int_Chan4Pix,
    .get_cisx_int_Chan5Pix     = get_cisx_int_Chan5Pix,
    .set_cisx_int_Chan5Pix     = set_cisx_int_Chan5Pix,
    .get_cisx_int_Chan0Shuffle = get_cisx_int_Chan0Shuffle,
    .set_cisx_int_Chan0Shuffle = set_cisx_int_Chan0Shuffle,
    .get_cisx_int_Chan1Shuffle = get_cisx_int_Chan1Shuffle,
    .set_cisx_int_Chan1Shuffle = set_cisx_int_Chan1Shuffle,
    .get_cisx_int_Chan2Shuffle = get_cisx_int_Chan2Shuffle,
    .set_cisx_int_Chan2Shuffle = set_cisx_int_Chan2Shuffle,
    .get_cisx_int_Chan3Shuffle = get_cisx_int_Chan3Shuffle,
    .set_cisx_int_Chan3Shuffle = set_cisx_int_Chan3Shuffle,
    .get_cisx_int_Chan4Shuffle = get_cisx_int_Chan4Shuffle,
    .set_cisx_int_Chan4Shuffle = set_cisx_int_Chan4Shuffle,
    .get_cisx_int_Chan5Shuffle = get_cisx_int_Chan5Shuffle,
    .set_cisx_int_Chan5Shuffle = set_cisx_int_Chan5Shuffle,
    .get_cisx_int_TBL_CNT      = get_cisx_int_TBL_CNT,
    .set_cisx_int_TBL_CNT      = set_cisx_int_TBL_CNT,
    .get_cisx_int_REV0         = get_cisx_int_REV0,
    .get_cisx_int_REV1         = get_cisx_int_REV1,
    .get_cisx_int_ODMA_TBL_LUT = get_cisx_int_ODMA_TBL_LUT,
    .set_cisx_int_ODMA_TBL_LUT = set_cisx_int_ODMA_TBL_LUT,
    .get_cisx_int_COLOR_OUT_TBL_LUT = get_cisx_int_COLOR_OUT_TBL_LUT,
    .set_cisx_int_COLOR_OUT_TBL_LUT = set_cisx_int_COLOR_OUT_TBL_LUT,

    .cisx_int_register_callback_irq = cisx_int_register_callback_irq,
    .cisx_int_enable_irq       = cisx_int_enable_irq,
    .cisx_int_disable_irq      = cisx_int_disable_irq,
    .cisx_int_force_irq        = cisx_int_force_irq,
    .cisx_int_clear_irq        = cisx_int_clear_irq,
    .cisx_int_handle_irq       = cisx_int_handle_irq,
};

void cisx_int_init(struct device_data *device_data)
{
    // register with the parent
    device_data->fcn_tbl = &cisx_int_functions;
    register_cisx_subblock(cisx_int, device_data, device_data->instance, 0);

    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(device_data->reg_spinlock));
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(device_data->int_spinlock));

    // no callbacks yet
    device_data->interrupt_callback = NULL;    
}
EXPORT_SYMBOL(cisx_int_init);

void cisx_int_exit(struct device_data *device_data)
{
    // unregister with the parent
    unregister_cisx_subblock(cisx_int, device_data->instance, 0);
}
EXPORT_SYMBOL(cisx_int_exit);

