/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/spinlock.h>   // for spinlock_t, printk, BUG_ON, and memset
#include <linux/io.h>         // for ioread/write32 (Read/Write macros)
#include <linux/export.h>     // for EXPORT_SYMBOL

#include "CISX_IDMA_AXI_regstructs.h"
#include "CISX_IDMA_AXI_regmasks.h"

#include "cisx_if.h"
#include "cisx_data.h"
#include "cisx_driver.h"
#include "cisx_idma_if.h"
#include "cisx_idma.h"


enum cisx_idma_mask_type {enable, pending, ack, force};


//*****************************************************************************
// Debug Functions
//*****************************************************************************

void dump_cisx_idma_regs(cisx_idmaDeviceHandle *device_data)
{
    printk("CISX IDMA(%d) dump for cisx=%d\n", device_data->submodinstance, device_data->instance);
    printk("-------------------------------------\n");
    printk( "     cfg=0x%08x   status=0x%08x line_width=0x%08x\n", 
             cisx_idmaRead(cfg), cisx_idmaRead(status), cisx_idmaRead(line_width) );
    printk( "  int_en=0x%08x   int_st=0x%08x   descread=0x%08x\n", 
             cisx_idmaRead(int_en), cisx_idmaRead(int_st), cisx_idmaRead(desc_read) );
    printk( "xfer_len=0x%08x xferaddr=0x%08x  ctrl_word=0x%08x\n", 
             cisx_idmaRead(xfer_length), cisx_idmaRead(xfer_addr), cisx_idmaRead(ctrl_word) );
}

//*****************************************************************************
// IRQ Functions
//*****************************************************************************

static void set_all_irqstruct(struct cisx_idma_irqs *irqstruct, bool setval)
{
    irqstruct->rresp = setval;
    irqstruct->bresp = setval;
    irqstruct->rst   = setval;
    irqstruct->eoi   = setval;
    irqstruct->fin   = setval;
    irqstruct->who   = setval;
}

static uint32_t convert_irqstruct_to_update_uintarray(
                    uint32_t curr_irqs,
                    struct   cisx_idma_irqs *irqstruct,
                    bool     set_to_one,
                    enum cisx_idma_mask_type operation )
{
    uint8_t set_val;

    // Are we enabling or disabling IRQs?
    set_val = set_to_one ? 1 : 0; 

    // Any IRQ's in irqstruct that are set to true will get set to set_val, other
    // bits will be left as-is.
    if (irqstruct->rresp)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_IDMA_INT_EN_RRESP_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_IDMA_INT_ST_RRESP_INT_ST_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_IDMA_INT_CL_RRESP_INT_CL_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_IDMA_INT_FO_RRESP_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->bresp)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_IDMA_INT_EN_BRESP_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_IDMA_INT_ST_BRESP_INT_ST_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_IDMA_INT_CL_BRESP_INT_CL_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_IDMA_INT_FO_BRESP_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->rst)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_IDMA_INT_EN_RST_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_IDMA_INT_ST_RST_INT_ST_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_IDMA_INT_CL_RST_INT_CL_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_IDMA_INT_FO_RST_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->eoi)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_IDMA_INT_EN_EOI_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_IDMA_INT_ST_EOI_INT_ST_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_IDMA_INT_CL_EOI_INT_CL_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_IDMA_INT_FO_EOI_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    if (irqstruct->fin)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_IDMA_INT_EN_FIN_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_IDMA_INT_ST_FIN_INT_ST_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_IDMA_INT_CL_FIN_INT_CL_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_IDMA_INT_FO_FIN_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }


    if (irqstruct->who)
    {
        switch (operation)
        {
            case enable:
                curr_irqs = CISX_IDMA_INT_EN_WHO_INT_EN_REPLACE_VAL(curr_irqs, set_val);
                break;
            case pending:
                curr_irqs = CISX_IDMA_INT_ST_WHO_INT_ST_REPLACE_VAL(curr_irqs, set_val);
                break;
            case ack:
                curr_irqs = CISX_IDMA_INT_CL_WHO_INT_CL_REPLACE_VAL(curr_irqs, set_val);
                break;
            case force:
                curr_irqs = CISX_IDMA_INT_FO_WHO_INT_FO_REPLACE_VAL(curr_irqs, set_val);
                break;
        }
    }

    return curr_irqs;
}

static void convert_uintarray_to_irqstruct(
                     uint32_t curr_irqs,
                     struct cisx_idma_irqs *irqstruct,
                     enum cisx_idma_mask_type operation )
{
    // If the given bit is set the interrupt field gets a true, false otherwise
    switch (operation)
    {
        case enable:
            irqstruct->rresp = (CISX_IDMA_INT_EN_RRESP_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->bresp = (CISX_IDMA_INT_EN_BRESP_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->rst   = (CISX_IDMA_INT_EN_RST_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi   = (CISX_IDMA_INT_EN_EOI_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->fin   = (CISX_IDMA_INT_EN_FIN_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->who   = (CISX_IDMA_INT_EN_WHO_INT_EN_MASK_SHIFT(curr_irqs) != 0);
            break;

        case pending:
            irqstruct->rresp = (CISX_IDMA_INT_ST_RRESP_INT_ST_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->bresp = (CISX_IDMA_INT_ST_BRESP_INT_ST_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->rst   = (CISX_IDMA_INT_ST_RST_INT_ST_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi   = (CISX_IDMA_INT_ST_EOI_INT_ST_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->fin   = (CISX_IDMA_INT_ST_FIN_INT_ST_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->who   = (CISX_IDMA_INT_ST_WHO_INT_ST_MASK_SHIFT(curr_irqs) != 0);
            break;

        case ack:
            irqstruct->rresp = (CISX_IDMA_INT_CL_RRESP_INT_CL_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->bresp = (CISX_IDMA_INT_CL_BRESP_INT_CL_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->rst   = (CISX_IDMA_INT_CL_RST_INT_CL_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi   = (CISX_IDMA_INT_CL_EOI_INT_CL_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->fin   = (CISX_IDMA_INT_CL_FIN_INT_CL_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->who   = (CISX_IDMA_INT_CL_WHO_INT_CL_MASK_SHIFT(curr_irqs) != 0);
            break;

        case force:
            irqstruct->rresp = (CISX_IDMA_INT_FO_RRESP_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->bresp = (CISX_IDMA_INT_FO_BRESP_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->rst   = (CISX_IDMA_INT_FO_RST_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->eoi   = (CISX_IDMA_INT_FO_EOI_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->fin   = (CISX_IDMA_INT_FO_FIN_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            irqstruct->who   = (CISX_IDMA_INT_FO_WHO_INT_FO_MASK_SHIFT(curr_irqs) != 0);
            break;
    }
}

void cisx_idma_register_callback_irq(cisx_idmaDeviceHandle *device_data, void *callbackfcn)
{
    device_data->interrupt_callback = callbackfcn;
}

static void set_irq_enable_reg(cisx_idmaDeviceHandle *device_data, 
                               struct cisx_idma_irqs *irqs, 
                               bool set_to_one)
{
    struct   cisx_idma_irqs irqstruct;
    uint32_t reg;
    unsigned long flags;

    // Passing a null irq struct?  The enable setting will be applied to all
    // interrupts.
    if (irqs == NULL)
    {
        // All interrupts need to be updated
        set_all_irqstruct(&irqstruct, true);
        irqs = &irqstruct;
    }

    // Modify interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = cisx_idmaRead(int_en);
    reg = convert_irqstruct_to_update_uintarray(reg, irqs, set_to_one, enable);
    cisx_idmaWrite(int_en, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

void cisx_idma_enable_irq(cisx_idmaDeviceHandle *device_data, struct cisx_idma_irqs *irqs)
{
    set_irq_enable_reg(device_data, irqs, true);
}

void cisx_idma_disable_irq(cisx_idmaDeviceHandle *device_data, struct cisx_idma_irqs *irqs)
{
    set_irq_enable_reg(device_data, irqs, false);
}

void cisx_idma_force_irq(cisx_idmaDeviceHandle *device_data, struct cisx_idma_irqs *irqs)
{
    struct   cisx_idma_irqs irqstruct;
    uint32_t reg = 0;
    unsigned long flags;

    // Passing a null irq struct?  The force setting will be applied to all
    // interrupts.
    if (irqs == NULL)
    {
        // Mark all interrupts to be forced
        set_all_irqstruct(&irqstruct, true);
        irqs = &irqstruct;
    }

    // Force interrupts
    PROTECT_INTREG_ACCESS_IRQ;
    reg = convert_irqstruct_to_update_uintarray(reg, irqs, true, force);
    cisx_idmaWrite(int_fo, reg);
    UNPROTECT_INTREG_ACCESS_IRQ;
}

static uint32_t clear_cisx_idma_irqs_uint32(struct device_data *device_data, uint32_t int_val)
{
    unsigned long flags;  
    uint32_t      reg;

    PROTECT_INTREG_ACCESS_IRQ;
    reg = cisx_idmaRead(int_st);
    cisx_idmaWrite(int_cl, int_val);  // clear requested ints
    UNPROTECT_INTREG_ACCESS_IRQ;

    return reg;
}

void cisx_idma_clear_irq(cisx_idmaDeviceHandle *device_data, struct cisx_idma_irqs *irqs)
{
    struct   cisx_idma_irqs irqstruct;
    uint32_t reg = 0;

    // Passing a null irq struct?  The clear setting will be applied to all
    // interrupts.
    if (irqs == NULL)
    {
        // Mark all interrupts to be cleared
        set_all_irqstruct(&irqstruct, true);
        irqs = &irqstruct;
    }

    // Clear interrupts
    reg = convert_irqstruct_to_update_uintarray(reg, irqs, true, ack);
    clear_cisx_idma_irqs_uint32(device_data, reg);
}

// NOTE: This function runs in interrupt thread context - try to be brief
void cisx_idma_handle_irq(struct device_data *device_data)
{
    uint32_t int_val;
    struct   cisx_idma_irqs irqstruct;
        
    // Grab the pending irqs then ack them all
    int_val = clear_cisx_idma_irqs_uint32(device_data, 0xFFFFFFFF);

    // Anything pending?
    if (int_val != 0)
    {
        debug_print("%s: cisx=%d subblock=%d IntPend=0x%x\n", 
               __FUNCTION__, device_data->instance, device_data->submodinstance, int_val);

        // If someone higher up registered a callback function, execute it
        if (device_data->interrupt_callback != NULL)
        {
            convert_uintarray_to_irqstruct(int_val, &irqstruct, pending);
            irqstruct.cisx_instance = device_data->instance;
            irqstruct.dma_instance  = device_data->submodinstance;
            irqstruct.irq_array     = int_val;

            device_data->interrupt_callback(&irqstruct);
        }
    }
}


//*****************************************************************************
// Driver Register Access Functions
//*****************************************************************************

void get_cisx_idma_cfg(cisx_idmaDeviceHandle *device_data, struct cisx_idma_cfg_reg *cfgr)
{
    uint32_t reg;

    reg = cisx_idmaRead(cfg);
    cfgr->replicate = CISX_IDMA_CFG_REPLICATE_MASK_SHIFT(reg);
    cfgr->enable = CISX_IDMA_CFG_ENABLE_MASK_SHIFT(reg);
    cfgr->out_width = CISX_IDMA_CFG_OUT_WIDTH_MASK_SHIFT(reg);
    cfgr->burst_len = CISX_IDMA_CFG_BURST_LEN_MASK_SHIFT(reg);
}

void set_cisx_idma_cfg(cisx_idmaDeviceHandle *device_data, struct cisx_idma_cfg_reg *cfgr)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_idmaRead(cfg);
    if(cfgr->replicate_valid)
        reg = CISX_IDMA_CFG_REPLICATE_REPLACE_VAL(reg, cfgr->replicate);
    if(cfgr->enable_valid)
        reg = CISX_IDMA_CFG_ENABLE_REPLACE_VAL(reg, cfgr->enable);
    if(cfgr->out_width_valid)
        reg = CISX_IDMA_CFG_OUT_WIDTH_REPLACE_VAL(reg, cfgr->out_width);
    if(cfgr->burst_len_valid)
        reg = CISX_IDMA_CFG_BURST_LEN_REPLACE_VAL(reg, cfgr->burst_len);
    cisx_idmaWrite(cfg, reg);
    UNPROTECT_REG_ACCESS;
}

void get_cisx_idma_status(cisx_idmaDeviceHandle *device_data, struct cisx_idma_status_reg *cfg)
{
    uint32_t reg;

    reg = cisx_idmaRead(status);
    cfg->soi = CISX_IDMA_STATUS_SOI_MASK_SHIFT(reg);
    cfg->eoi = CISX_IDMA_STATUS_EOI_MASK_SHIFT(reg);
    cfg->softreset = CISX_IDMA_STATUS_SOFTRESET_MASK_SHIFT(reg);
    cfg->full_cbuf = CISX_IDMA_STATUS_FULL_CBUF_MASK_SHIFT(reg);
    cfg->empty_cbuf = CISX_IDMA_STATUS_EMPTY_CBUF_MASK_SHIFT(reg);
    cfg->full_dbuf = CISX_IDMA_STATUS_FULL_DBUF_MASK_SHIFT(reg);
    cfg->empty_dbuf = CISX_IDMA_STATUS_EMPTY_DBUF_MASK_SHIFT(reg);
    cfg->packer_empty = CISX_IDMA_STATUS_PACKER_EMPTY_MASK_SHIFT(reg);
    cfg->dma_busy = CISX_IDMA_STATUS_DMA_BUSY_MASK_SHIFT(reg);
}

void get_cisx_idma_line_width(cisx_idmaDeviceHandle *device_data, struct cisx_idma_line_width_reg *cfg)
{
    uint32_t reg;

    reg = cisx_idmaRead(line_width);
    cfg->line_width = CISX_IDMA_LINE_WIDTH_LINE_WIDTH_MASK_SHIFT(reg);
}

void set_cisx_idma_line_width(cisx_idmaDeviceHandle *device_data, struct cisx_idma_line_width_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_idmaRead(line_width);
    if(cfg->line_width_valid)
        reg = CISX_IDMA_LINE_WIDTH_LINE_WIDTH_REPLACE_VAL(reg, cfg->line_width);
    cisx_idmaWrite(line_width, reg);
    UNPROTECT_REG_ACCESS;
}

void set_cisx_idma_desc_write(cisx_idmaDeviceHandle *device_data, struct cisx_idma_desc_write_reg *cfg)
{
    // NOTE: this is a 32 bit write only register!

    //debug_print("%s: cisx=%d subbloc=%d desc=0x%x (%d)\n", 
    //       __FUNCTION__, device_data->instance, device_data->submodinstance, cfg->desc, cfg->desc_valid);

    PROTECT_REG_ACCESS;
    if (cfg->desc_valid)
    {
        cisx_idmaWrite(desc_write, cfg->desc);
    }
    UNPROTECT_REG_ACCESS;
}

void get_cisx_idma_desc_read(cisx_idmaDeviceHandle *device_data, struct cisx_idma_desc_read_reg *cfg)
{
    uint32_t reg;

    reg = cisx_idmaRead(desc_read);
    cfg->desc = CISX_IDMA_DESC_READ_DESC_MASK_SHIFT(reg);
}

void get_cisx_idma_xfer_length(cisx_idmaDeviceHandle *device_data, struct cisx_idma_xfer_length_reg *cfg)
{
    uint32_t reg;

    reg = cisx_idmaRead(xfer_length);
    cfg->len = CISX_IDMA_XFER_LENGTH_LEN_MASK_SHIFT(reg);
}

void get_cisx_idma_xfer_addr(cisx_idmaDeviceHandle *device_data, struct cisx_idma_xfer_addr_reg *cfg)
{
    uint32_t reg;

    reg = cisx_idmaRead(xfer_addr);
    cfg->addr = CISX_IDMA_XFER_ADDR_ADDR_MASK_SHIFT(reg);
}

void get_cisx_idma_ctrl_word(cisx_idmaDeviceHandle *device_data, struct cisx_idma_ctrl_word_reg *cfg)
{
    uint32_t reg;

    reg = cisx_idmaRead(ctrl_word);
    cfg->ctrl_word = CISX_IDMA_CTRL_WORD_CTRL_WORD_MASK_SHIFT(reg);
}

void set_cisx_idma_reset(cisx_idmaDeviceHandle *device_data, struct cisx_idma_reset_reg *cfg)
{
    uint32_t reg;

    PROTECT_REG_ACCESS;
    reg = cisx_idmaRead(reset);
    if(cfg->soft_reset_valid)
        reg = CISX_IDMA_RESET_SOFT_RESET_REPLACE_VAL(reg, cfg->soft_reset);
    cisx_idmaWrite(reset, reg);
    UNPROTECT_REG_ACCESS;
}


//*****************************************************************************
// Driver Functions
//*****************************************************************************

struct cisx_idma_function_struct cisx_idma_functions =
{
    .version                   = 0,
    .dump_cisx_idma_regs       = dump_cisx_idma_regs,

    .get_cisx_idma_cfg         = get_cisx_idma_cfg,
    .set_cisx_idma_cfg         = set_cisx_idma_cfg,
    .get_cisx_idma_status      = get_cisx_idma_status,
    .get_cisx_idma_line_width  = get_cisx_idma_line_width,
    .set_cisx_idma_line_width  = set_cisx_idma_line_width,
    .set_cisx_idma_desc_write  = set_cisx_idma_desc_write,
    .get_cisx_idma_desc_read   = get_cisx_idma_desc_read,
    .get_cisx_idma_xfer_length = get_cisx_idma_xfer_length,
    .get_cisx_idma_xfer_addr   = get_cisx_idma_xfer_addr,
    .get_cisx_idma_ctrl_word   = get_cisx_idma_ctrl_word,
    .set_cisx_idma_reset       = set_cisx_idma_reset,

    .cisx_idma_register_callback_irq = cisx_idma_register_callback_irq,
    .cisx_idma_enable_irq      = cisx_idma_enable_irq,
    .cisx_idma_disable_irq     = cisx_idma_disable_irq,
    .cisx_idma_force_irq       = cisx_idma_force_irq,
    .cisx_idma_clear_irq       = cisx_idma_clear_irq,
    .cisx_idma_handle_irq      = cisx_idma_handle_irq,
};

void cisx_idma_init(struct device_data *device_data)
{
    // register with the parent
    device_data->fcn_tbl = &cisx_idma_functions;
    register_cisx_subblock(input_dma, 
                           device_data, 
                           device_data->instance, 
                           device_data->submodinstance);

    // NOTE that macro PROTECT_REG_ACCESS uses reg_spinlock
    spin_lock_init(&(device_data->reg_spinlock));
    // NOTE that macro PROTECT_INTREG_ACCESS_IRQ uses int_spinlock
    spin_lock_init(&(device_data->int_spinlock));

    // no callbacks yet
    device_data->interrupt_callback = NULL;    
}
EXPORT_SYMBOL(cisx_idma_init);

void cisx_idma_exit(struct device_data *device_data)
{
    // unregister with the parent
    unregister_cisx_subblock(input_dma, 
                             device_data->instance, 
                             device_data->submodinstance);
}
EXPORT_SYMBOL(cisx_idma_exit);

