/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

#include <linux/spinlock.h>   // for spinlock_t, printk, BUG_ON, and memset
#include <linux/export.h>     // for EXPORT_SYMBOL

#include "cisx_data.h"
#include "cisx_driver.h"
#include "cisx_if.h"
#include "cisx_int_if.h"
#include "cisx_odma_if.h"
#include "cisx_idma_if.h"

struct cisx_object_struct
{
    cisx_intDeviceHandle   *int_handle[MAX_NUM_CISXBLOCKS]; 
    cisx_idmaDeviceHandle  *idma_handle[MAX_NUM_CISXBLOCKS][MAX_INPUT_DMAS];
    cisx_odmaDeviceHandle  *odma_handle[MAX_NUM_CISXBLOCKS][MAX_OUTPUT_DMAS];
};

struct cisx_object_struct cisx_object;


// NOTE: This is only for use by cisx_int.c when handling the CISX platform 
//       interrupt.  Yup, they only gave us one interrupt for both CISX
//       blocks.
cisx_intDeviceHandle *get_cisx_int_device(uint8_t cisx_instance)
{
    if (cisx_instance >= MAX_NUM_CISXBLOCKS)
    {
        return NULL;
    }
    else
    {
        return cisx_object.int_handle[cisx_instance];
    }
}
EXPORT_SYMBOL(get_cisx_int_device);


//*****************************************************************************
//  CISX Debug Functions
//*****************************************************************************

void cisx_drv_dump(uint8_t cisx_instance)
{
    cisx_int_dump_regs(cisx_instance);

    // TODO: what else should we dump from the top level?

}
EXPORT_SYMBOL(cisx_drv_dump);


//*****************************************************************************
//  CISX Sub-Block Access Macros
//*****************************************************************************

#define CISX_INT_RETRIEVE_handle_ft                            \
    cisx_intDeviceHandle *int_handle;                          \
    struct cisx_int_function_struct *ft;                       \
    uint8_t ucisx_instance=(uint8_t) cisx_instance;            \
                                                               \
    BUG_ON(ucisx_instance >= MAX_NUM_CISXBLOCKS);              \
    int_handle = cisx_object.int_handle[ucisx_instance];       \
    BUG_ON(int_handle == NULL);                                \
    ft = int_handle->fcn_tbl;

#define CISX_ODMA_RETRIEVE_handle_ft                                      \
    cisx_odmaDeviceHandle *odma_handle;                                   \
    struct cisx_odma_function_struct *ft;                                 \
    uint8_t ucisx_instance=(uint8_t) cisx_instance;                       \
                                                                          \
    BUG_ON(ucisx_instance >= MAX_NUM_CISXBLOCKS);                         \
    BUG_ON(odma_instance  >= MAX_OUTPUT_DMAS);                            \
    odma_handle = cisx_object.odma_handle[ucisx_instance][odma_instance]; \
    BUG_ON(odma_handle == NULL);                                          \
    ft = odma_handle->fcn_tbl;

#define CISX_IDMA_RETRIEVE_handle_ft                                      \
    cisx_idmaDeviceHandle *idma_handle;                                   \
    struct cisx_idma_function_struct *ft;                                 \
    uint8_t ucisx_instance=(uint8_t) cisx_instance;                       \
                                                                          \
    BUG_ON(ucisx_instance >= MAX_NUM_CISXBLOCKS);                         \
    BUG_ON(idma_instance  >= MAX_INPUT_DMAS);                             \
    idma_handle = cisx_object.idma_handle[ucisx_instance][idma_instance]; \
    BUG_ON(idma_handle == NULL);                                          \
    ft = idma_handle->fcn_tbl;


//*****************************************************************************
//  CISX Sub-Block Registration Functions
//*****************************************************************************

void init_cisx(int cisx_instance)
{
    int j;

    debug_print("%s: initializing cisx %d\n", __FUNCTION__, cisx_instance); 

    cisx_object.int_handle[cisx_instance] = NULL;

    for (j = 0; j < MAX_INPUT_DMAS; j++)
    {
        cisx_object.idma_handle[cisx_instance][j] = NULL;
    }

    for (j = 0; j < MAX_OUTPUT_DMAS; j++)
    {
        cisx_object.odma_handle[cisx_instance][j] = NULL;
    }
}

void exit_cisx(int cisx_instance)
{
    // We don't need to NULL out the cisx_object, since the unregister takes care of that.
    debug_print("%s: exiting cisx %d\n", __FUNCTION__, cisx_instance); 
}

void register_cisx_subblock(
                    enum  cisxsubblock subblock, 
                    void *cisx_subblock_data, 
                    int   cisx_instance,
                    int   subblock_instance)
{
    if (cisx_instance >= MAX_NUM_CISXBLOCKS)
    {
        error_print("%s: cisx_instance=%d out of range, ignoring\n", __FUNCTION__, cisx_instance);
        return;
    }

    if (subblock == cisx_int)
    {
        debug_print("%s: cisx_int, cisx_instance=%d\n", __FUNCTION__, cisx_instance);
        cisx_object.int_handle[cisx_instance] = cisx_subblock_data;
    }
    else if (subblock == output_dma)
    {
        if (subblock_instance >= MAX_OUTPUT_DMAS)
        {
            error_print("%s: odma instance=%d out of range, ignoring\n", __FUNCTION__, subblock_instance);
        }
        else
        {
            debug_print("%s: odma instance=%d, cisx_instance=%d\n", __FUNCTION__, subblock_instance, cisx_instance);
            cisx_object.odma_handle[cisx_instance][subblock_instance] = cisx_subblock_data;
        }
    }
    else if (subblock == input_dma)
    {
        if (subblock_instance >= MAX_INPUT_DMAS)
        {
            error_print("%s: idma instance=%d out of range, ignoring\n", __FUNCTION__, subblock_instance);
        }
        else
        {
            debug_print("%s: idma instance=%d, cisx_instance=%d\n", __FUNCTION__, subblock_instance, cisx_instance);
            cisx_object.idma_handle[cisx_instance][subblock_instance] = cisx_subblock_data;
        }
    }
    else
    {
        error_print("%s: no support for block=%d\n", __FUNCTION__, subblock);
    }
}
EXPORT_SYMBOL(register_cisx_subblock);


void *unregister_cisx_subblock(enum cisxsubblock subblock, 
                               int  cisx_instance, 
                               int  subblock_instance)
{
    void *subblock_data = NULL;
    
    if (cisx_instance >= MAX_NUM_CISXBLOCKS)
    {
        error_print("%s: cisx_instance=%d out of range, ignoring\n", __FUNCTION__, cisx_instance);
        return NULL;
    }

    if (subblock == cisx_int)
    {
        debug_print("%s: cisx_int, cisx_instance=%d\n", __FUNCTION__, cisx_instance);
        subblock_data = cisx_object.int_handle[cisx_instance];
        cisx_object.int_handle[cisx_instance] = NULL;
    }
    else if (subblock == output_dma)
    {
        if (subblock_instance >= MAX_OUTPUT_DMAS)
        {
            error_print("%s: odma instance=%d out of range, ignoring\n", __FUNCTION__, subblock_instance);
        }
        else
        {
            debug_print("%s: odma instance=%d, cisx_instance=%d\n", __FUNCTION__, subblock_instance, cisx_instance);
            subblock_data = cisx_object.odma_handle[cisx_instance][subblock_instance];
            cisx_object.odma_handle[cisx_instance][subblock_instance] = NULL;
        }
    }
    else if (subblock == input_dma)
    {
        if (subblock_instance >= MAX_INPUT_DMAS)
        {
            error_print("%s: idma instance=%d out of range, ignoring\n", __FUNCTION__, subblock_instance);
        }
        else
        {
            debug_print("%s: idma instance=%d, cisx_instance=%d\n", __FUNCTION__, subblock_instance, cisx_instance);
            subblock_data = cisx_object.idma_handle[cisx_instance][subblock_instance];
            cisx_object.idma_handle[cisx_instance][subblock_instance] = NULL;
        }
    }
    else
    {
        error_print("%s: no support for block=%d\n", __FUNCTION__, subblock);
    }

    return subblock_data;
}
EXPORT_SYMBOL(unregister_cisx_subblock);



//*****************************************************************************
//  Public API functions: CISX INT Registers
//*****************************************************************************

void cisx_int_get_CisxCfg(uint8_t cisx_instance, struct cisx_int_CisxCfg_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_CisxCfg(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_CisxCfg);

void cisx_int_set_CisxCfg(uint8_t cisx_instance, struct cisx_int_CisxCfg_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_CisxCfg(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_CisxCfg);

void cisx_int_get_CisxStat(uint8_t cisx_instance, struct cisx_int_CisxStat_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_CisxStat(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_CisxStat);

void cisx_int_get_Chan0Pix(uint8_t cisx_instance, struct cisx_int_Chan0Pix_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_Chan0Pix(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_Chan0Pix);

void cisx_int_set_Chan0Pix(uint8_t cisx_instance, struct cisx_int_Chan0Pix_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_Chan0Pix(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_Chan0Pix);

void cisx_int_get_Chan1Pix(uint8_t cisx_instance, struct cisx_int_Chan1Pix_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_Chan1Pix(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_Chan1Pix);

void cisx_int_set_Chan1Pix(uint8_t cisx_instance, struct cisx_int_Chan1Pix_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_Chan1Pix(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_Chan1Pix);

void cisx_int_get_Chan2Pix(uint8_t cisx_instance, struct cisx_int_Chan2Pix_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_Chan2Pix(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_Chan2Pix);

void cisx_int_set_Chan2Pix(uint8_t cisx_instance, struct cisx_int_Chan2Pix_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_Chan2Pix(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_Chan2Pix);

void cisx_int_get_Chan3Pix(uint8_t cisx_instance, struct cisx_int_Chan3Pix_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_Chan3Pix(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_Chan3Pix);

void cisx_int_set_Chan3Pix(uint8_t cisx_instance, struct cisx_int_Chan3Pix_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_Chan3Pix(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_Chan3Pix);

void cisx_int_get_Chan4Pix(uint8_t cisx_instance, struct cisx_int_Chan4Pix_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_Chan4Pix(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_Chan4Pix);

void cisx_int_set_Chan4Pix(uint8_t cisx_instance, struct cisx_int_Chan4Pix_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_Chan4Pix(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_Chan4Pix);

void cisx_int_get_Chan5Pix(uint8_t cisx_instance, struct cisx_int_Chan5Pix_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_Chan5Pix(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_Chan5Pix);

void cisx_int_set_Chan5Pix(uint8_t cisx_instance, struct cisx_int_Chan5Pix_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_Chan5Pix(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_Chan5Pix);

void cisx_int_get_Chan0Shuffle(uint8_t cisx_instance, struct cisx_int_Chan0Shuffle_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_Chan0Shuffle(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_Chan0Shuffle);

void cisx_int_set_Chan0Shuffle(uint8_t cisx_instance, struct cisx_int_Chan0Shuffle_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_Chan0Shuffle(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_Chan0Shuffle);

void cisx_int_get_Chan1Shuffle(uint8_t cisx_instance, struct cisx_int_Chan1Shuffle_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_Chan1Shuffle(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_Chan1Shuffle);

void cisx_int_set_Chan1Shuffle(uint8_t cisx_instance, struct cisx_int_Chan1Shuffle_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_Chan1Shuffle(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_Chan1Shuffle);

void cisx_int_get_Chan2Shuffle(uint8_t cisx_instance, struct cisx_int_Chan2Shuffle_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_Chan2Shuffle(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_Chan2Shuffle);

void cisx_int_set_Chan2Shuffle(uint8_t cisx_instance, struct cisx_int_Chan2Shuffle_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_Chan2Shuffle(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_Chan2Shuffle);

void cisx_int_get_Chan3Shuffle(uint8_t cisx_instance, struct cisx_int_Chan3Shuffle_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_Chan3Shuffle(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_Chan3Shuffle);

void cisx_int_set_Chan3Shuffle(uint8_t cisx_instance, struct cisx_int_Chan3Shuffle_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_Chan3Shuffle(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_Chan3Shuffle);

void cisx_int_get_Chan4Shuffle(uint8_t cisx_instance, struct cisx_int_Chan4Shuffle_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_Chan4Shuffle(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_Chan4Shuffle);

void cisx_int_set_Chan4Shuffle(uint8_t cisx_instance, struct cisx_int_Chan4Shuffle_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_Chan4Shuffle(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_Chan4Shuffle);

void cisx_int_get_Chan5Shuffle(uint8_t cisx_instance, struct cisx_int_Chan5Shuffle_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_Chan5Shuffle(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_Chan5Shuffle);

void cisx_int_set_Chan5Shuffle(uint8_t cisx_instance, struct cisx_int_Chan5Shuffle_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_Chan5Shuffle(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_Chan5Shuffle);

void cisx_int_get_TBL_CNT(uint8_t cisx_instance, struct cisx_int_TBL_CNT_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_TBL_CNT(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_TBL_CNT);

void cisx_int_set_TBL_CNT(uint8_t cisx_instance, struct cisx_int_TBL_CNT_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_TBL_CNT(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_set_TBL_CNT);

void cisx_int_get_REV0(uint8_t cisx_instance, struct cisx_int_REV0_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_REV0(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_REV0);

void cisx_int_get_REV1(uint8_t cisx_instance, struct cisx_int_REV1_reg *cfg)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_REV1(int_handle, cfg);
}
EXPORT_SYMBOL(cisx_int_get_REV1);

void cisx_int_get_ODMA_TBL_LUT(uint8_t cisx_instance, struct cisx_int_ODMA_TBL_LUT_reg *cfg, uint8_t index)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_ODMA_TBL_LUT(int_handle, cfg, index);
}
EXPORT_SYMBOL(cisx_int_get_ODMA_TBL_LUT);

void cisx_int_set_ODMA_TBL_LUT(uint8_t cisx_instance, struct cisx_int_ODMA_TBL_LUT_reg *cfg, uint8_t index)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_ODMA_TBL_LUT(int_handle, cfg, index);
}
EXPORT_SYMBOL(cisx_int_set_ODMA_TBL_LUT);

void cisx_int_get_COLOR_OUT_TBL_LUT(uint8_t cisx_instance, struct cisx_int_COLOR_OUT_TBL_LUT_reg *cfg, uint8_t index)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->get_cisx_int_COLOR_OUT_TBL_LUT(int_handle, cfg, index);
}
EXPORT_SYMBOL(cisx_int_get_COLOR_OUT_TBL_LUT);

void cisx_int_set_COLOR_OUT_TBL_LUT(uint8_t cisx_instance, struct cisx_int_COLOR_OUT_TBL_LUT_reg *cfg, uint8_t index)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->set_cisx_int_COLOR_OUT_TBL_LUT(int_handle, cfg, index);
}
EXPORT_SYMBOL(cisx_int_set_COLOR_OUT_TBL_LUT);


//*****************************************************************************
//  Public API functions: CISX INT Interrupts
//*****************************************************************************

void cisx_int_register_irq_callback(uint8_t cisx_instance, void *callbackfcn)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->cisx_int_register_callback_irq(int_handle, callbackfcn);
}
EXPORT_SYMBOL(cisx_int_register_irq_callback);

void cisx_int_enable_irqs(uint8_t cisx_instance, struct cisx_int_irqs *irqs)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->cisx_int_enable_irq(int_handle, irqs);
}
EXPORT_SYMBOL(cisx_int_enable_irqs);

void cisx_int_disable_irqs(uint8_t cisx_instance, struct cisx_int_irqs *irqs)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->cisx_int_disable_irq(int_handle, irqs);
}
EXPORT_SYMBOL(cisx_int_disable_irqs);

void cisx_int_handle_irqs(uint8_t cisx_instance)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->cisx_int_handle_irq(int_handle);
}
EXPORT_SYMBOL(cisx_int_handle_irqs);

void cisx_int_force_irqs(uint8_t cisx_instance, struct cisx_int_irqs *irqs)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->cisx_int_force_irq(int_handle, irqs);
}
EXPORT_SYMBOL(cisx_int_force_irqs);

void cisx_int_clear_irqs(uint8_t cisx_instance, struct cisx_int_irqs *irqs)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->cisx_int_clear_irq(int_handle, irqs);
}
EXPORT_SYMBOL(cisx_int_clear_irqs);


//*****************************************************************************
//  Public API functions: CISX INT Legacy Helpers
//*****************************************************************************

void cisx_int_dump_regs(uint8_t cisx_instance)
{
    CISX_INT_RETRIEVE_handle_ft;

    ft->dump_cisx_int_regs(int_handle);
}
EXPORT_SYMBOL(cisx_int_dump_regs);


//*****************************************************************************
//  Public API functions: CISX IDMA Registers
//*****************************************************************************

void cisx_idma_get_cfg(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_cfg_reg *cfg)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->get_cisx_idma_cfg(idma_handle, cfg);
}
EXPORT_SYMBOL(cisx_idma_get_cfg);

void cisx_idma_set_cfg(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_cfg_reg *cfg)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->set_cisx_idma_cfg(idma_handle, cfg);
}
EXPORT_SYMBOL(cisx_idma_set_cfg);

void cisx_idma_get_status(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_status_reg *cfg)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->get_cisx_idma_status(idma_handle, cfg);
}
EXPORT_SYMBOL(cisx_idma_get_status);

void cisx_idma_get_line_width(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_line_width_reg *cfg)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->get_cisx_idma_line_width(idma_handle, cfg);
}
EXPORT_SYMBOL(cisx_idma_get_line_width);

void cisx_idma_set_line_width(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_line_width_reg *cfg)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->set_cisx_idma_line_width(idma_handle, cfg);
}
EXPORT_SYMBOL(cisx_idma_set_line_width);

void cisx_idma_set_desc_write(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_desc_write_reg *cfg)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->set_cisx_idma_desc_write(idma_handle, cfg);
}
EXPORT_SYMBOL(cisx_idma_set_desc_write);

void cisx_idma_get_desc_read(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_desc_read_reg *cfg)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->get_cisx_idma_desc_read(idma_handle, cfg);
}
EXPORT_SYMBOL(cisx_idma_get_desc_read);

void cisx_idma_get_xfer_length(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_xfer_length_reg *cfg)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->get_cisx_idma_xfer_length(idma_handle, cfg);
}
EXPORT_SYMBOL(cisx_idma_get_xfer_length);

void cisx_idma_get_xfer_addr(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_xfer_addr_reg *cfg)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->get_cisx_idma_xfer_addr(idma_handle, cfg);
}
EXPORT_SYMBOL(cisx_idma_get_xfer_addr);

void cisx_idma_get_ctrl_word(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_ctrl_word_reg *cfg)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->get_cisx_idma_ctrl_word(idma_handle, cfg);
}
EXPORT_SYMBOL(cisx_idma_get_ctrl_word);

void cisx_idma_set_reset(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_reset_reg *cfg)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->set_cisx_idma_reset(idma_handle, cfg);
}
EXPORT_SYMBOL(cisx_idma_set_reset);


//*****************************************************************************
//  Public API functions: CISX IDMA Interrupts
//*****************************************************************************

void cisx_idma_register_irq_callback(uint8_t cisx_instance, uint8_t idma_instance, void *callbackfcn)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->cisx_idma_register_callback_irq(idma_handle, callbackfcn);
}
EXPORT_SYMBOL(cisx_idma_register_irq_callback);

void cisx_idma_enable_irqs(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_irqs *irqs)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->cisx_idma_enable_irq(idma_handle, irqs);
}
EXPORT_SYMBOL(cisx_idma_enable_irqs);

void cisx_idma_disable_irqs(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_irqs *irqs)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->cisx_idma_disable_irq(idma_handle, irqs);
}
EXPORT_SYMBOL(cisx_idma_disable_irqs);

void cisx_idma_handle_irqs(uint8_t cisx_instance, uint8_t idma_instance)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->cisx_idma_handle_irq(idma_handle);
}
EXPORT_SYMBOL(cisx_idma_handle_irqs);

void cisx_idma_force_irqs(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_irqs *irqs)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->cisx_idma_force_irq(idma_handle, irqs);
}
EXPORT_SYMBOL(cisx_idma_force_irqs);

void cisx_idma_clear_irqs(uint8_t cisx_instance, uint8_t idma_instance, struct cisx_idma_irqs *irqs)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->cisx_idma_clear_irq(idma_handle, irqs);
}
EXPORT_SYMBOL(cisx_idma_clear_irqs);


//*****************************************************************************
//  Public API functions: CISX IDMA Legacy Helpers
//*****************************************************************************

void cisx_idma_dump_regs(uint8_t cisx_instance, uint8_t idma_instance)
{
    CISX_IDMA_RETRIEVE_handle_ft;

    ft->dump_cisx_idma_regs(idma_handle);
}
EXPORT_SYMBOL(cisx_idma_dump_regs);


//*****************************************************************************
//  Public API functions: CISX ODMA Registers
//*****************************************************************************

void cisx_odma_get_cfg(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_cfg_reg *cfg)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->get_cisx_odma_cfg(odma_handle, cfg);
}
EXPORT_SYMBOL(cisx_odma_get_cfg);

void cisx_odma_set_cfg(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_cfg_reg *cfg)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->set_cisx_odma_cfg(odma_handle, cfg);
}
EXPORT_SYMBOL(cisx_odma_set_cfg);

void cisx_odma_get_status(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_status_reg *cfg)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->get_cisx_odma_status(odma_handle, cfg);
}
EXPORT_SYMBOL(cisx_odma_get_status);

void cisx_odma_get_line_size(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_line_size_reg *cfg)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->get_cisx_odma_line_size(odma_handle, cfg);
}
EXPORT_SYMBOL(cisx_odma_get_line_size);

void cisx_odma_set_line_size(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_line_size_reg *cfg)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->set_cisx_odma_line_size(odma_handle, cfg);
}
EXPORT_SYMBOL(cisx_odma_set_line_size);

void cisx_odma_set_desc_write(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_desc_write_reg *cfg)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->set_cisx_odma_desc_write(odma_handle, cfg);
}
EXPORT_SYMBOL(cisx_odma_set_desc_write);

void cisx_odma_get_desc_read(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_desc_read_reg *cfg)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->get_cisx_odma_desc_read(odma_handle, cfg);
}
EXPORT_SYMBOL(cisx_odma_get_desc_read);

void cisx_odma_get_xfer_length(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_xfer_length_reg *cfg)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->get_cisx_odma_xfer_length(odma_handle, cfg);
}
EXPORT_SYMBOL(cisx_odma_get_xfer_length);

void cisx_odma_get_xfer_addr(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_xfer_addr_reg *cfg)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->get_cisx_odma_xfer_addr(odma_handle, cfg);
}
EXPORT_SYMBOL(cisx_odma_get_xfer_addr);

void cisx_odma_get_xfer_burst(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_xfer_burst_reg *cfg)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->get_cisx_odma_xfer_burst(odma_handle, cfg);
}
EXPORT_SYMBOL(cisx_odma_get_xfer_burst);

void cisx_odma_set_reset(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_reset_reg *cfg)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->set_cisx_odma_reset(odma_handle, cfg);
}
EXPORT_SYMBOL(cisx_odma_set_reset);


//*****************************************************************************
//  Public API functions: CISX ODMA Interrupts
//*****************************************************************************

void cisx_odma_register_irq_callback(uint8_t cisx_instance, uint8_t odma_instance, void *callbackfcn)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->cisx_odma_register_callback_irq(odma_handle, callbackfcn);
}
EXPORT_SYMBOL(cisx_odma_register_irq_callback);

void cisx_odma_enable_irqs(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_irqs *irqs)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->cisx_odma_enable_irq(odma_handle, irqs);
}
EXPORT_SYMBOL(cisx_odma_enable_irqs);

void cisx_odma_disable_irqs(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_irqs *irqs)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->cisx_odma_disable_irq(odma_handle, irqs);
}
EXPORT_SYMBOL(cisx_odma_disable_irqs);

void cisx_odma_handle_irqs(uint8_t cisx_instance, uint8_t odma_instance)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->cisx_odma_handle_irq(odma_handle);
}
EXPORT_SYMBOL(cisx_odma_handle_irqs);

void cisx_odma_force_irqs(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_irqs *irqs)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->cisx_odma_force_irq(odma_handle, irqs);
}
EXPORT_SYMBOL(cisx_odma_force_irqs);

void cisx_odma_clear_irqs(uint8_t cisx_instance, uint8_t odma_instance, struct cisx_odma_irqs *irqs)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->cisx_odma_clear_irq(odma_handle, irqs);
}
EXPORT_SYMBOL(cisx_odma_clear_irqs);


//*****************************************************************************
//  Public API functions: CISX ODMA Legacy Helpers
//*****************************************************************************

void cisx_odma_dump_regs(uint8_t cisx_instance, uint8_t odma_instance)
{
    CISX_ODMA_RETRIEVE_handle_ft;

    ft->dump_cisx_odma_regs(odma_handle);
}
EXPORT_SYMBOL(cisx_odma_dump_regs);

