/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

/** 
* \file afe_linux.c
* 
* \brief Linux wrapper for AFE APIs
* 
* Provides a linux interface to the AFE APIs
* 
**/
#include <linux/init.h>
#include <linux/module.h>
#include <linux/of_device.h>
#include <linux/fs.h>
#include <linux/slab.h>

#include "afe_if.h"
#include "afe_driver.h"

#define AFE_NAME "afe"

static ssize_t afe_dev_get_bpp(struct device *dev, struct device_attribute *attr,char *buf);
static ssize_t afe_dev_get_index(struct device *dev, struct device_attribute *attr,char *buf);
static ssize_t afe_dev_get_cmd(struct device *dev, struct device_attribute *attr,char *buf);
static ssize_t afe_dev_set_cmd(struct device *dev, struct device_attribute *attr,const char *buf, size_t count);
static ssize_t afe_dev_set_gains(struct device *dev, struct device_attribute *attr,const char *buf, size_t count);
static ssize_t afe_dev_set_offsets(struct device *dev, struct device_attribute *attr,const char *buf, size_t count);
static ssize_t afe_dev_get_regval(struct device *dev, struct device_attribute *attr,char *buf);
static ssize_t afe_dev_set_regval(struct device *dev, struct device_attribute *attr,const char *buf, size_t count);
static ssize_t afe_dev_get_regsel(struct device *dev, struct device_attribute *attr,char *buf);
static ssize_t afe_dev_set_regsel(struct device *dev, struct device_attribute *attr,const char *buf, size_t count);


static DEVICE_ATTR(bpp,     S_IRUSR,  afe_dev_get_bpp, NULL);
static DEVICE_ATTR(index,   S_IRUSR,  afe_dev_get_index, NULL);
static DEVICE_ATTR(cmd,     S_IRUSR | S_IWUSR, afe_dev_get_cmd, afe_dev_set_cmd);
static DEVICE_ATTR(gains,   S_IWUSR,  NULL, afe_dev_set_gains);
static DEVICE_ATTR(offsets, S_IWUSR,  NULL, afe_dev_set_offsets);
static DEVICE_ATTR(regsel,  S_IRUSR | S_IWUSR, afe_dev_get_regsel, afe_dev_set_regsel);
static DEVICE_ATTR(regval,  S_IRUSR | S_IWUSR, afe_dev_get_regval, afe_dev_set_regval);

static struct attribute *afe_dev_attrs[] =
{
    &dev_attr_bpp.attr,
    &dev_attr_index.attr,
    &dev_attr_cmd.attr,
    &dev_attr_gains.attr,
    &dev_attr_offsets.attr,
    &dev_attr_regsel.attr,
    &dev_attr_regval.attr,
    NULL,
};

static struct attribute_group afe_dev_attr_group = {
    .attrs = afe_dev_attrs
};

static const struct attribute_group *afe_dev_attr_groups[] = {
    &afe_dev_attr_group,
    NULL
};

static struct class afe_class =
{
    .name        = AFE_NAME,
    .owner       = THIS_MODULE,
    .dev_groups  = afe_dev_attr_groups,
};

typedef struct
{
    uint32_t afe_index;
    uint32_t afe_selected_reg;
} afe_dev_info_t;


static void afe_device_added(uint32_t index)
{
    char afe_name[21];
    afe_dev_info_t *afe_info;

    afe_get_name( index, afe_name, sizeof(afe_name) );

    afe_info = kmalloc( sizeof( afe_dev_info_t ), GFP_KERNEL );
    afe_info->afe_index = index;
    afe_info->afe_selected_reg = 0;

    device_create( &afe_class, NULL, MKDEV(0, 0), afe_info, afe_name);
}

static void afe_device_removed(uint32_t index)
{
    struct class_dev_iter iter;
    struct device *dev;

    class_dev_iter_init( &iter, &afe_class, NULL, NULL );
    while ( ( dev = class_dev_iter_next( &iter ) ) )
    {
        afe_dev_info_t *afe_info = dev_get_drvdata( dev );
        if ( (afe_info != NULL) && (afe_info->afe_index == index ) )
        {
            device_unregister(dev);
            put_device(dev);
            kfree( afe_info );
            break;
        }
    }
    class_dev_iter_exit(&iter);
}

static ssize_t afe_dev_get_bpp(struct device *dev, struct device_attribute *attr,char *buf)
{
    afe_dev_info_t *afe_info = (afe_dev_info_t *)dev_get_drvdata( dev );
    uint32_t afe_index = afe_info->afe_index;

    sprintf( buf, "%d\n", afe_get_bits_per_pixel( afe_index ) );
    return strlen( buf );
}

static ssize_t afe_dev_get_index(struct device *dev, struct device_attribute *attr,char *buf)
{
    afe_dev_info_t *afe_info = (afe_dev_info_t *)dev_get_drvdata( dev );
    uint32_t afe_index = afe_info->afe_index;

    sprintf( buf, "%d\n", afe_index );
    return strlen( buf );
}

static const char *cmd_help = \
"commands:\n" \
"    reset\n" \
"    dump_regs\n";

static ssize_t afe_dev_get_cmd(struct device *dev, struct device_attribute *attr,char *buf)
{
    strcpy(buf, cmd_help);
    return strlen(buf);
}

static ssize_t afe_dev_set_cmd(struct device *dev, struct device_attribute *attr,const char *buf, size_t count)
{
    afe_dev_info_t *afe_info = (afe_dev_info_t *)dev_get_drvdata( dev );
    uint32_t afe_index = afe_info->afe_index;

    if ( strncmp(buf, "reset", 5) == 0 )
    {
        afe_reset(afe_index);
    }
    else if ( strncmp(buf, "dump_regs", 9) == 0 )
    {
        afe_reg_dump(afe_index);
    }
    return count;
}

static ssize_t afe_dev_set_gains(struct device *dev, struct device_attribute *attr,const char *buf, size_t count)
{
    afe_dev_info_t *afe_info = (afe_dev_info_t *)dev_get_drvdata( dev );
    uint32_t afe_index = afe_info->afe_index;
    uint32_t red;
    uint32_t green;
    uint32_t blue;

    sscanf(buf, "%i,%i,%i", &red, &green, &blue);

    afe_set_gains( afe_index, red, green, blue);

    return count;
}

static ssize_t afe_dev_set_offsets(struct device *dev, struct device_attribute *attr,const char *buf, size_t count)
{
    afe_dev_info_t *afe_info = (afe_dev_info_t *)dev_get_drvdata( dev );
    uint32_t afe_index = afe_info->afe_index;
    int32_t red;
    int32_t green;
    int32_t blue;

    sscanf(buf, "%i,%i,%i", &red, &green, &blue);

    afe_set_offsets( afe_index, red, green, blue);

    return count;
}

static ssize_t afe_dev_set_regsel(struct device *dev, struct device_attribute *attr,const char *buf, size_t count)
{
    afe_dev_info_t *afe_info = (afe_dev_info_t *)dev_get_drvdata( dev );

    sscanf(buf, "%i", &afe_info->afe_selected_reg);

    return count;
}

static ssize_t afe_dev_get_regsel(struct device *dev, struct device_attribute *attr,char *buf)
{
    afe_dev_info_t *afe_info = (afe_dev_info_t *)dev_get_drvdata( dev );

    sprintf(buf, "%d\n", afe_info->afe_selected_reg );

    return strlen(buf);
}

static ssize_t afe_dev_set_regval(struct device *dev, struct device_attribute *attr,const char *buf, size_t count)
{
    afe_dev_info_t *afe_info = (afe_dev_info_t *)dev_get_drvdata( dev );
    uint32_t afe_index = afe_info->afe_index;
    uint32_t val;

    sscanf(buf, "%i", &val);
    afe_reg_write( afe_index, afe_info->afe_selected_reg, val );

    return count;
}

static ssize_t afe_dev_get_regval(struct device *dev, struct device_attribute *attr,char *buf)
{
    afe_dev_info_t *afe_info = (afe_dev_info_t *)dev_get_drvdata( dev );
    uint32_t afe_index = afe_info->afe_index;
    uint32_t val;

    val = afe_reg_read( afe_index, afe_info->afe_selected_reg );

    sprintf(buf, "0x%x\n", val );

    return strlen(buf);
}

afe_os_functions_t os_functions = 
{
    .dev_added   = afe_device_added,
    .dev_removed = afe_device_removed,
};

static int afe_linux_init(void)
{
    class_register( &afe_class );

    afe_system_init( &os_functions );
    return  0;
}
  
static void afe_linux_exit(void)
{
    afe_system_unload();

    class_unregister( &afe_class );

    return;
}


module_init(afe_linux_init);
module_exit(afe_linux_exit);

EXPORT_SYMBOL(afe_register_device);
EXPORT_SYMBOL(afe_remove_device);

EXPORT_SYMBOL(afe_get_num_afes);

EXPORT_SYMBOL(afe_reset);
EXPORT_SYMBOL(afe_soft_setup);

EXPORT_SYMBOL(afe_set_gains);
EXPORT_SYMBOL(afe_set_offsets);

EXPORT_SYMBOL(afe_get_bits_per_pixel);

EXPORT_SYMBOL(afe_get_name);
EXPORT_SYMBOL(afe_reg_read);
EXPORT_SYMBOL(afe_reg_write);
EXPORT_SYMBOL(afe_reg_dump);

MODULE_AUTHOR("Copyright (c) 2014 Marvell International Ltd.");
MODULE_DESCRIPTION("Marvell 62xx series AFE Driver");

MODULE_LICENSE("GPL");
MODULE_VERSION("2014_JUL_08");

