/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

/** 
* \file afe_sample.c
* 
* \brief Sample AFE driver (not connected to any actual 
*        hardware)
* 
* Starting point for future AFEs. 
* 
**/
#include <linux/init.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/slab.h>

#include "afe_driver.h"

#define NUM_AFES_TO_SIMULATE 2

typedef struct 
{
    uint32_t afe_index;
    char     afe_name[21];

    uint32_t gain_red;
    uint32_t gain_green;
    uint32_t gain_blue;

    int32_t  offset_red;
    int32_t  offset_green;
    int32_t  offset_blue;

} sample_afe_info_t;

static sample_afe_info_t **sample_afes;

static int reset(void *dev)
{
    sample_afe_info_t *dev_info = (sample_afe_info_t *)dev;
    int err = 0;

    if (dev_info != NULL)
    {
        dev_info->gain_red   = 100;
        dev_info->gain_green = 100;
        dev_info->gain_blue  = 100;

        dev_info->offset_red   = 0;
        dev_info->offset_green = 0;
        dev_info->offset_blue  = 0;
    }
    else
    {
        err = -EINVAL;
    }

    return err;
}

static int soft_setup(void *dev, void *config)
{
    sample_afe_info_t *dev_info = (sample_afe_info_t *)dev;
    int err = 0;

    if (dev_info != NULL)
    {
        // Nothing to do here for simulated AFE
    }
    else
    {
        err = -EINVAL;
    }

    return err;
}

static int set_gains(void *dev, uint32_t red, uint32_t green, uint32_t blue)
{
    sample_afe_info_t *dev_info = (sample_afe_info_t *)dev;
    int err = 0;

    if (dev_info != NULL)
    {
        dev_info->gain_red   = red;
        dev_info->gain_green = green;
        dev_info->gain_blue  = blue;
    }
    else
    {
        err = -EINVAL;
    }

    return err;
}

static int set_offsets(void *dev, int32_t red, int32_t green, int32_t blue)
{
    sample_afe_info_t *dev_info = (sample_afe_info_t *)dev;
    int err = 0;

    if (dev_info != NULL)
    {
        dev_info->offset_red   = red;
        dev_info->offset_green = green;
        dev_info->offset_blue  = blue;
    }
    else
    {
        err = -EINVAL;
    }

    return err;
}

static int get_bits_per_pixel(void *dev)
{
    sample_afe_info_t *dev_info = (sample_afe_info_t *)dev;
    int err = 0;

    if (dev_info != NULL)
    {
        return 16;
    }
    else
    {
        err = -EINVAL;
    }

    return err;
}

static int get_name(void *dev, char *buffer, int buf_len)
{
    sample_afe_info_t *dev_info = (sample_afe_info_t *)dev;
    int err = 0;

    if (dev_info != NULL)
    {
        strncpy( buffer, dev_info->afe_name, buf_len );
        buffer[buf_len - 1] = '\0';
    }
    else
    {
        err = -EINVAL;
    }

    return err;
}

static int reg_dump(void *dev)
{
    sample_afe_info_t *dev_info = (sample_afe_info_t *)dev;
    int err = 0;

    if (dev_info != NULL)
    {
        pr_alert( "%s:\n", dev_info->afe_name );
        pr_alert( "    gains  : %6d, %6d, %6d\n", dev_info->gain_red, dev_info->gain_green, dev_info->gain_blue);
        pr_alert( "    offsets: %6d, %6d, %6d\n", dev_info->offset_red, dev_info->offset_green, dev_info->offset_blue);
    }
    else
    {
        err = -EINVAL;
    }

    return err;
}

static afe_driver_functions_t driver_functions =
{
    .reset              = reset,
    .soft_setup         = soft_setup,

    .set_gains          = set_gains,
    .set_offsets        = set_offsets,

    .get_bits_per_pixel = get_bits_per_pixel,

    .get_name           = get_name,
    .reg_read           = NULL,
    .reg_write          = NULL,
    .reg_dump           = reg_dump,
};

static int afe_sample_init(void)
{
    int i;

    sample_afes = (sample_afe_info_t **)kmalloc( sizeof(sample_afe_info_t *) * NUM_AFES_TO_SIMULATE, GFP_KERNEL );

    if (sample_afes != NULL)
    {
        for (i = 0; i < NUM_AFES_TO_SIMULATE; i++)
        {
            sample_afe_info_t *dev_info;

            dev_info = kmalloc( sizeof(sample_afe_info_t), GFP_KERNEL );

            if (dev_info != NULL)
            {
                sample_afes[i] = dev_info;
                sprintf( dev_info->afe_name, "afe_sample_%d", i + 1 );

                dev_info->afe_index = afe_register_device( &driver_functions, dev_info );
            }
        }
        return  0;
    }
    else
    {
        return -1;
    }
}

static void afe_sample_exit(void)
{
    int i;
    for ( i = 0; i < NUM_AFES_TO_SIMULATE; i++ )
    {
        afe_remove_device( & driver_functions, sample_afes[i] );
        kfree( sample_afes[i] );
    }
    return;
}

module_init(afe_sample_init);
module_exit(afe_sample_exit);

MODULE_AUTHOR("Copyright (c) 2014 Marvell International Ltd.");
MODULE_DESCRIPTION("Marvell 62xx Series Sample AFE Device Driver");

MODULE_LICENSE("GPL");
MODULE_VERSION("2014_JUL_11");

