/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

/** 
* \file afe_driver.h
* 
* \brief Interface routines for individual AFE drivers to use 
*        when registering with the system.
* 
* The system may contain multiple different types of AFEs. 
* These routines allow each AFE implementation to register with 
* the top level AFE APIs so they can be accessed in a unified 
* way. 
* 
**/

#ifndef INC_AFE_DRIVER_H
#define INC_AFE_DRIVER_H

typedef struct
{
    int      (*reset)(void *dev);                                               ///< Reset to power-on values
    int      (*soft_setup)(void *dev, void *config);                            ///< Set to known state prior to each new use

    int      (*set_gains)(void *dev, uint32_t red, uint32_t green, uint32_t blue);  ///< Set gains for red, green and blue (val of 100 = gain of 1.0)
    int      (*set_offsets)(void *dev, int32_t red, int32_t green, int32_t blue);   ///< Set offsets for red, green and blue (offset of 1 should give shift of 1 in final output)

    int      (*get_bits_per_pixel)(void *dev);                                  ///< Bits per pixel for the AFE output

    int      (*get_name)(void *dev, char *buffer, int buf_len);                 ///< (optional) A string name to represent this AFE
    uint32_t (*reg_read)(void *dev, uint32_t reg_index);                        ///< (optional) Read any AFE register
    int      (*reg_write)(void *dev, uint32_t reg_index, uint32_t reg_value);   ///< (optional) Write any AFE register
    int      (*reg_dump)(void *dev);                                            ///< (optional) Dump all AFE registers

} afe_driver_functions_t;

/** 
* \brief Called by lower level driver to register itself with 
*        the upper level AFE APIs.
* 
* \param callbacks table providing function pointers to each AFE 
*        routine the driver implements.
* 
* \param dev Driver-selected value that will be passed back with 
*        each call.
* 
* \return numeric index of the AFE in list of registered drivers
* 
* All AFEs in the system should be registered with the upper 
* layer through this call.  There may be multiple different 
* types of AFEs (with different callbacks) and there may also be 
* multiple instances of the same AFE with the same callbacks, 
* but a different *dev for each instance. 
* 
**/
int afe_register_device(afe_driver_functions_t *callbacks, void *dev);

/** 
* \brief Called by lower level driver to remove itself from the 
*        upper level code's list of devices.
* 
* \param callbacks table matching the one that was passed to 
*        afe_register_device.
* 
* \param dev Driver-selected value matching the one that was 
*        passed to afe_register_device.
* 
* \return 0 on success 
* 
**/
int afe_remove_device(afe_driver_functions_t *callbacks, void *dev);

#endif // INC_AFE_DRIVER_H
