/*
 ***************************************************************************************
 * (c) Copyright 2014 Marvell International Ltd.
 **************************************************************************************
 *
 * Marvell Commercial License Option
 *
 * If you received this File from Marvell as part of a proprietary software release,
 * the File is considered Marvell Proprietary and Confidential Information, and is
 * licensed to you under the terms of the applicable Commercial License.
 *
 **************************************************************************************
 *
 * Marvell GPL License Option
 *
 * If you received this File from Marvell as part of a Linux distribution, this File
 * is licensed to you in accordance with the terms and conditions of the General Public
 * License Version 2, June 1991 (the "GPL License").  You can redistribute it and/or
 * modify it under the terms of the GPL License; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GPL License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see http://www.gnu.org/licenses/old-licenses/gpl-2.0.html.
 *
 **************************************************************************************
 */

/** 
* \file afe.c
* 
* Provides the top level APIs for the AFEs.  AFE drivers call 
* into this code to register themselves, then high-level code 
* can talk to each of the drivers in a uniform way. 
* 
**/

#include <linux/types.h>
#include <linux/semaphore.h>
#include <linux/slab.h>

#include "afe_if.h"
#include "afe_driver.h"

static struct semaphore afe_list_sem;
static struct list_head afe_list;
static uint32_t num_afe_drivers;

typedef struct
{
    struct list_head        list;

    uint32_t                index;
    afe_driver_functions_t *functions;
    void                   *dev;
} afe_driver_config_t;

static afe_os_functions_t *os_functions;

#define ENTER() pr_debug(__FILE__ "ENTER %s\n", __FUNCTION__)
#define EXIT()  pr_debug(__FILE__ "EXIT  %s:%d\n", __FUNCTION__, __LINE__)

void     afe_system_init( afe_os_functions_t *functions )
{
    ENTER();

    os_functions = functions;

    num_afe_drivers = 0;
    INIT_LIST_HEAD( &afe_list );
    sema_init( &afe_list_sem, 1 );

    EXIT();
}

void     afe_system_unload(void)
{
    afe_driver_config_t *temp = NULL;

    ENTER();

    down( &afe_list_sem );

    list_for_each_entry( temp, &afe_list, list )
    {
        pr_err(" Error - can't unload with devices still registered\n");
        BUG();
    }

    EXIT();
}

int afe_register_device(afe_driver_functions_t *callbacks, void *dev)
{
    afe_driver_config_t *afe_config;
    int err = 0;

    ENTER();

    afe_config = kmalloc( sizeof( afe_driver_config_t ), GFP_KERNEL );
    if ( afe_config != NULL)
    {
        afe_config->dev       = dev;
        afe_config->functions = callbacks;

        down( &afe_list_sem );
            afe_config->index = num_afe_drivers++;
            list_add_tail( &afe_config->list, &afe_list );
        up( &afe_list_sem );
    }
    else
    {
        err = -ENOMEM;
    }

    EXIT();

    if (err != 0)
    {
        return err;
    }
    else
    {
        if ( ( os_functions != NULL ) && ( os_functions->dev_added != NULL ) )
        {
            os_functions->dev_added( afe_config->index );
        }

        return afe_config->index;
    }
}

int afe_remove_device(afe_driver_functions_t *callbacks, void *dev)
{
    int err = -EINVAL;
    afe_driver_config_t *temp = NULL;
    afe_driver_config_t *next = NULL;
    uint32_t afe_index;

    ENTER();

    down( &afe_list_sem );

    list_for_each_entry_safe( temp, next, &afe_list, list )
    {
        if ( ( temp->functions == callbacks ) && ( temp->dev == dev ) )
        {
            afe_index = temp->index;
            list_del( &temp->list );
            kfree( temp );
            err = 0;
            break;
        }
    }

    up( &afe_list_sem );

    if ( err == 0 )
    {
        if ( ( os_functions != NULL ) && ( os_functions->dev_removed != NULL ) )
        {
            os_functions->dev_removed( afe_index );
        }
    }

    EXIT();

    return err;
}

uint32_t afe_get_num_afes(void)
{
    ENTER();
    EXIT();
    return num_afe_drivers;
}

static afe_driver_config_t *get_afe_config(int afe_index)
{
    afe_driver_config_t *temp = NULL;
    afe_driver_config_t *requested_afe = NULL;

    ENTER();

    down( &afe_list_sem );

    list_for_each_entry( temp, &afe_list, list )
    {
        if ( temp->index == afe_index )
        {
            requested_afe = temp;
            break;
        }
    }

    up( &afe_list_sem );

    EXIT();

    return requested_afe;
}

#define CALL_AFE_DRIVER_ROUTINE( func_name, ... )                       \
    int err = 0;                                                        \
    afe_driver_config_t *afe = NULL;                                    \
                                                                        \
    ENTER();                                                            \
                                                                        \
    afe = get_afe_config( afe_index );                                  \
    if ( afe != NULL )                                                  \
    {                                                                   \
        if (afe->functions->func_name != NULL)                          \
        {                                                               \
            err = afe->functions->func_name( afe->dev, ##__VA_ARGS__ ); \
        }                                                               \
    }                                                                   \
    else                                                                \
    {                                                                   \
        err = -EINVAL;                                                  \
    }                                                                   \
                                                                        \
    EXIT();                                                             \
                                                                        \
    return err;                                                         \


int      afe_reset(int afe_index)
{
    CALL_AFE_DRIVER_ROUTINE( reset );
}

int      afe_soft_setup(int afe_index, void *config)
{
    CALL_AFE_DRIVER_ROUTINE( soft_setup, config );
}

int      afe_set_gains(int afe_index, uint32_t red, uint32_t green, uint32_t blue)
{
    CALL_AFE_DRIVER_ROUTINE( set_gains, red, green, blue );
}

int      afe_set_offsets(int afe_index, int32_t red, int32_t green, int32_t blue)
{
    CALL_AFE_DRIVER_ROUTINE( set_offsets, red, green, blue );
}

int      afe_get_bits_per_pixel(int afe_index)
{
    CALL_AFE_DRIVER_ROUTINE( get_bits_per_pixel );
}

int      afe_get_name(int afe_index, char *buffer, int buf_len)
{
    CALL_AFE_DRIVER_ROUTINE( get_name, buffer, buf_len );
}

uint32_t afe_reg_read(int afe_index, uint32_t reg_index)
{
    uint32_t reg_val = 0;
    afe_driver_config_t *afe = NULL;

    ENTER();

    afe = get_afe_config( afe_index );
    if ( afe != NULL )
    {
        if (afe->functions->reg_read != NULL)
        {
            reg_val = afe->functions->reg_read( afe->dev, reg_index );
        }
    }
    else
    {
        reg_val = 0xBAD00AFE;
    }

    EXIT();

    return reg_val;
}

int      afe_reg_write(int afe_index, uint32_t reg_index, uint32_t reg_value)
{
    CALL_AFE_DRIVER_ROUTINE( reg_write, reg_index, reg_value );
}

int      afe_reg_dump(int afe_index)
{
    CALL_AFE_DRIVER_ROUTINE( reg_dump );
}

