/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2011-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#ifndef MODULE_TEST
#include "scos.h"
#endif

#include "memAPI.h"
#include "lassert.h"

#include "scantypes.h"
#include "scancore.h"
#include "scandbg.h"
#include "scanplat.h"
#include "scanplat_sb.h"
#include "scansen.h"
#include "scanvars.h"
#include "scanmech.h"
#include "scanmech_null.h"
#include "scancap.h"
#include "siqdata.h"
#include "scanvarsiq.h"
#include "scanmech_granum2.h"
#include "pie_handle_if.h"
#include "pie_if.h"
#include "scands.h"
#ifdef SCAN_MECH_LINUX_APP
    #include "scanmech_app.h"
#endif

#include "afe_wm8213.h"

#ifdef MODULE_TEST
static const struct scan_platform_capabilities g_caps;
#else
static const struct scan_platform_capabilities g_caps = {
    .has_adf = true,
    .has_flatbed = true,
    .has_duplex = true,
    .has_nsensor = true,

    .num_sensors = 2,

    /* zero terminate these lists */
    .fb_supported_horiz_dpi = { 300, 600, 0 },
    .fb_supported_vert_dpi = { 300, 600, 0 },
    .adf_supported_horiz_dpi = { 300, 600, 0 },
    .adf_supported_vert_dpi =  { 300, 600, 0 },

    /* null terminate this list, too */
    .scan_data_type_list = {SCAN_DATA_TYPE_XRGB, 
                            SCAN_DATA_TYPE_MONO, SCAN_DATA_TYPE_RGBX, 
                            SCAN_DATA_TYPE_RGB, SCAN_DATA_TYPE_PLANE, 
                            SCAN_DATA_TYPE_BILEVEL,
                            SCAN_DATA_NULL_TYPE }
};
#endif

static t_ScanMech scan_mech; 

void scanplat_init_mech(void)
{
    memset( &scan_mech, 0, sizeof( t_ScanMech ) );
#ifdef NULL_MECH_DRIVER
    smech_null_init( &scan_mech );
#elif defined (SCAN_MECH_GRANUM2)
    smech_granum2_init( &scan_mech );
#elif defined SCAN_MECH_LINUX_APP
    smech_app_init( &scan_mech );
#else
    #error Unknown/unsupported scan mech
#endif

    /* davep 25-Jun-2013 ; bump way up the number of buffers because this is
     * such a fast platform.
     *
     */
//    scands_set_integer( "pipe_pic_pie_buffers", 40 );
    
    scands_set_integer( "pipe_pic_pie_buffers", 30 );   
}

void scanplat_get_capabilities( struct scan_platform_capabilities *caps )
{
    memcpy( caps, &g_caps, sizeof(struct scan_platform_capabilities));
}

uint32_t scanplat_get_max_scan_length( scan_document_source_t docsrc  )
{
    uint32_t max_scan_length;

    // NOTE: the scan length is in thinches (1/1000")

    max_scan_length = 0;

    switch( docsrc ) {
        case SCAN_DOCUMENT_SOURCE_FLATBED:
        case SCAN_DOCUMENT_SOURCE_FLATBED_NSENSOR:
            max_scan_length = 11700*2; /* ~11" */
            break;

        case SCAN_DOCUMENT_SOURCE_ADF:
        case SCAN_DOCUMENT_SOURCE_ADF_DUPLEX:
        case SCAN_DOCUMENT_SOURCE_ADF_NSENSOR:
            max_scan_length = 15000;
            break;

        default :
            /* say what? */
            XASSERT(0,docsrc);
            break;
    }
    
    return max_scan_length;
}

void scanplat_init_scan_gpio( void )
{
    scanplat_sb_init_scan_gpio();
}


//*********************************************************************************
// Notes on scanvar constructors:
//
// Core scanvars.c requires the platform to provide scanvar constructors for 
// several copy and scan combinations (see scanvar_constructor_list in scanvars.c).
//
// In our Linux implementation a pickled scanvar struct comes down from user space
// (in a SIQ buffer) letting us know what to do here in kernel space. The user
// space scanvar is pickled by scanvar_to_siqfile, which copies everything specified
// by g_siq_names (in siqnames.c).  So in kernel space, there is no point setting
// anything named in g_siq_names because it will simply be overwritten when merge
// gets called (which happens at least once at the end of scanvar_from_siqfile).
//*********************************************************************************



struct scanvars *scanplat_sv_mono_scan_new( void )
{
    struct scanvars *sv;

    // Call the core constructor in scanvars.  This should get us almost everything
    // we need (some stuff that won't get stomped by the SIQ merge, plus some stuff
    // that will).
    sv = scanvar_mono_scan_new();
    if( sv==NULL ) {
        dbg1( "no memory for scanvar!\n" );
        return NULL;
    }

    // Now tweak any settings for our platform.  Again, no point setting stuff the
    // the SIQ merge will stomp.
    scanvar_set_pic_bpp( sv, 8 );
    
    // Merge to pick up any changes from a siqfile.  Note that if this scanvar
    // constructor is being called in the normal manner, this merge won't do anything
    // (siqfile_to_scanvar hasn't set the SIQ data yet).  This is mostly here for
    // debug.
    scanvar_siq_merge( sv );

    return sv;
}


struct scanvars *scanplat_sv_color_scan_new( void )
{
    struct scanvars *sv;

    // Call the core constructor in scanvars.  This should get us almost everything
    // we need (some stuff that won't get stomped by the SIQ merge, plus some stuff
    // that will).
    sv = scanvar_color_scan_new();
    if( sv==NULL ) {
        return NULL;
    }

    // Now tweak any settings for our platform.  Again, no point setting stuff the
    // the SIQ merge will stomp.
    scanvar_set_pic_bpp( sv, 8 );

    // Merge to pick up any changes from a siqfile.  Note that if this scanvar
    // constructor is being called in the normal manner, this merge won't do anything
    // (siqfile_to_scanvar hasn't set the SIQ data yet).  This is mostly here for
    // debug.
    scanvar_siq_merge( sv );

    return sv;
}

static void init_copy_settings( struct scanvars *sv )
{
    /* common settings for all copy scanvars 
     */
    scanvar_set_pic_bpp( sv, 8 );
}

struct scanvars *scanplat_sv_draft_copy_new( void )
{
    struct scanvars *sv;
    sv = scanvar_draft_copy_new();
    init_copy_settings( sv );
    return sv;
}

struct scanvars *scanplat_sv_text_copy_new( void )
{ 
    struct scanvars *sv;
    sv = scanvar_text_copy_new(); 
    init_copy_settings( sv );
    return sv;
}

struct scanvars *scanplat_sv_mixed_copy_new( void )
{ 
    struct scanvars *sv;
    sv = scanvar_mixed_copy_new(); 
    init_copy_settings( sv );
    return sv;
}

struct scanvars *scanplat_sv_filmphoto_copy_new( void )
{ 
    struct scanvars *sv;
    sv = scanvar_filmphoto_copy_new(); 
    init_copy_settings( sv );
    return sv;
}

struct scanvars *scanplat_sv_picture_copy_new( void )
{ 
    struct scanvars *sv;
    sv = scanvar_picture_copy_new(); 
    init_copy_settings( sv );
    return sv;
}

struct scanvars *scanplat_sv_draft_color_copy_new( void )
{ 
    struct scanvars *sv;
    sv = scanvar_draft_color_copy_new(); 
    init_copy_settings( sv );
    return sv;
}

struct scanvars *scanplat_sv_text_color_copy_new( void )
{ 
    struct scanvars *sv;
    sv = scanvar_text_color_copy_new(); 
    init_copy_settings( sv );
    return sv;
}

struct scanvars *scanplat_sv_mixed_color_copy_new( void )
{ 
    struct scanvars *sv;
    sv = scanvar_mixed_color_copy_new(); 
    init_copy_settings( sv );
    return sv;
}

struct scanvars *scanplat_sv_filmphoto_color_copy_new( void )
{ 
    struct scanvars *sv;
    sv = scanvar_filmphoto_color_copy_new(); 
    init_copy_settings( sv );
    return sv;
}

struct scanvars *scanplat_sv_picture_color_copy_new( void )
{ 
    struct scanvars *sv;
    sv = scanvar_picture_color_copy_new(); 
    init_copy_settings( sv );
    return sv;
}

struct scanvars *scanplat_sv_text_fax_new( void )
{ 
    return scanvar_text_fax_new(); 
}

struct scanvars *scanplat_sv_mixed_fax_new( void )
{ 
    return scanvar_mixed_fax_new(); 
}

struct scanvars *scanplat_sv_mono_cal_new( void )
{ 
    return scanvar_mono_cal_new(); 
}

struct scanvars *scanplat_sv_color_cal_new( void )
{ 

    return scanvar_color_cal_new(); 
}

// What is sclkper, and why do we care. First, a quick tutorial on scanline time:
//
// The way to calculate how much time between start pulses (or linestarts) for a scanline is:
//     Tscanline = Tpixel * PixelsPerScanline  (where Tpixel is the time for a pixel)
// But since we don't get to tell the ASIC the time for a pixel, but rather we tell it
// PIXPER, which is in units of Tbusclock, then our equation becomes:
//     Tscanline = Tbusclk * (PIXPER + 1) * (PixelsPerScanline)  (+1 since the ASIC stores it -1)
// But the ASIC doesn't let us tell it PixelsPerScanline, but rather we can tell it SCLKPER and
// SCYCLE.  In essense (SCLKPER+1) * (SCYCLE+1) replaces PixelsPerScanLine so that the equation is:
//     Tscanline = Tbusclk * (PIXPER + 1) * (SCLKPER+1) * (SCYCLE+1)
//
// So that's where sclkper comes from.  But, sclkper is only used by the ASIC while we are in
// SYNC_FIRST or SYNC_NONE mode, and it is only used to compute simulated linestarts (since when in
// SYNC_NONE mode there are no linestarts at all, and SYNC_FIRST only uses the very first one).
// Otherwise, SYNC_EVERY is the mode, and all line starts come from the motor, and sclkper
// is not needed. 
//
// So when you are calculating the value of sclkper to return to the caller, the above formula can
// be solved for SCLKPER:
// SCLKPER = (Tscanline/(Tbusclk * (PIXPER+1) * (SCYCLE+1))) - 1
// But actually, the code that calls this function is expecting that it will -1 from SCLKPER
// before writing it to the ASIC, so don't do the -1 you see in that equation!
//
// So if we want a 750usec line time, and we have a 250MHz clock (4ns), a PIXPER of 55, and an
// SCYCLE of 31 then SCLKPER = (750usec/(4ns * 56 * 32)) = 105 (rounded up)
// and if we want a 1500usec line time, and we have a 250MHz clock (4ns), a PIXPER of 55, and an
// SCYCLE of 31 then SCLKPER = (1500usec/(4ns * 56 * 32)) = 210 (rounded up)
//
// NOTE: just because the sensor can do 1 line in 750usec, doesn't mean that's the line time 
// to pick.  If the sensor can capture data faster than the motor can move, it is necessary to
// adjust the line time to account for the slowness of the motor.  But if the motor can move faster
// than the sensor line time, it inadvisable to shorten the line time, since the sensor might be
// unhappy about receiving start pulses closer than the spec allows.
uint32_t scanplat_calc_sclkper(scan_cmode_t cmode, uint32_t dpi, uint32_t pixeltimes_per_line,
                               scan_document_source_t doc_src)
{
    // this mech doesn't have a speed difference between adf and fb, so we ignore the doc_src parm
    if (cmode == SCAN_CMODE_MONO) {
        if (dpi == 300) {
            return (33);  // 227.6 usec line time
        } else if (dpi == 600) {
            return (66);  // 455.2 usec line time
        } else if (dpi == 1200) {
            XASSERT(0, dpi); //not support "now"
            return (-1);
        } else {
            XASSERT(0, dpi);
            return (-1);
        }
    } else if (cmode == SCAN_CMODE_COLOR) {
        if (dpi == 300) {
            return (33);  // 227.6 usec line time
        } else if (dpi == 600) { 
            return (66);  // 455.2 usec line time
        } else if (dpi == 1200) {
            XASSERT(0, dpi); //not support "now"
            return (-1);
        } else {
            XASSERT(0, dpi);
            return (-1);
        }
    } else {
        XASSERT(0, cmode);
        return (-1);
    }
}

// Note - these are old notes previously inside the scanplat_calc_sclkper function
    //MRVL-Harris notes here
    //for stepper
    //200MHz/pwm_m/ls_incr ~ 250MHz/pixer/SCAN_SCYCLE_DEFAULT/sclkper
    //sclkper ~ 250MHz/pixer/SCAN_SCYCLE_DEFAULT*ls_incr*pwm_m/200MHz
//   300 dpi   =250MHz/60/32*1200*55/200MHz
//   600 dpi   =250MHz/60/32*200*500/200MHz
// End old notes


#ifdef MODULE_TEST
afe_wm8213_config_t plat_afe_configs[] = {0};
#else
afe_wm8213_config_t plat_afe_configs[] = 
{
    {
        .setup1 = 0x35, // pgafs=positive video, 1 channel mode, no CDS, enable AFE
        .setup2 = 0x20, // RLCDARANGE to set VRLCtop to 2.05V, (change to 0x21 to debug 8 bit output)
        .setup3 = 0x10, // set RLCDAC to 0 value for VRLCbot of .4 V 
        .setup4 = 0x00,
        .setup5 = 0x00,
        .setup6 = 0x00, // disable Reset Level Clamping

        .dac_red   = 0x80,
        .dac_green = 0x80,
        .dac_blue  = 0x80,

        .pga_lsb_red   = 0x00,
        .pga_lsb_green = 0x00,
        .pga_lsb_blue  = 0x00,
        .pga_msb_red   = 0x0C,
        .pga_msb_green = 0x0C,
        .pga_msb_blue  = 0x0C,
    }
};
#endif

void *scanplat_afe_get_softsetup_param(int afe)
{
    if ( afe >= sizeof(plat_afe_configs) / sizeof(plat_afe_configs[0]) )
    {
        return NULL;
    }
    return &plat_afe_configs[afe];
}

#ifdef MODULE_TEST
static scanplat_cal_config_t plat_cal_configs[] = {0}; 
#else
static scanplat_cal_config_t plat_cal_configs[] = 
{
    {
        .analog_cal_default_gain_not_null = false,
        .analog_cal_default_gain          = 1.0,

        .illum_start_val                  = 80,
        .illum_reduction_percent          = 5,
        .expected_output_reduction        = 5,
    }
};
#endif

scanplat_cal_config_t *scanplat_get_cal_params(int sensor)
{
    if ( sensor >= sizeof(plat_cal_configs) / sizeof(plat_cal_configs[0]) )
    {
        return NULL;
    }
    return &plat_cal_configs[sensor];
}

