/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * Description:
 *   Stepper motor configuration for the Granum2 mech.
 *
 **/

#ifndef SCANMECH_STEP_CFG_GRANUM2_H
#define SCANMECH_STEP_CFG_GRANUM2_H

// Motor profile tables must be ordered to match this enum, the logic that
// searches the tables may depend on it.
typedef enum {
   DPI_SLOW,
   DPI_300M,
   DPI_300C,
   DPI_600M,
   DPI_600C,
} motor_profile_index_t;

// The Granum2 mech has two motors (flatbed and ADF)
#define NUM_MOTOR     2

// Stepper motor block select
 enum {
    STEP_MTR_BLK_0 = 0,              // ASIC stepper motor block 0
    STEP_MTR_BLK_1,                  // ASIC stepper motor block 1
    STEP_MTR_BLK_2,                  // ASIC stepper motor block 2
    STEP_MTR_BLK_3,                  // ASIC stepper motor block 3
    STEP_MTR_BLK_4,                  // ASIC stepper motor block 4
    STEP_MTR_BLK_5,                  // ASIC stepper motor block 5
};


//*****************************************************************************
// Flatbed
//*****************************************************************************

// Note that FB_NRAMP_VALS and ADF_NRAMP_VALS are constants defining the size of their
// respective ramp tables.  When tuning for a particular mech, it may be that the 
// steady state (post ramp) speed needs to be slower for that mech.  In that case,
// the profiles defined below (stmotor_move_param_t stmotor_fb/adf_dpi_profiles[]) will have
// decreased the number of ramp steps (chopping off the steps at the end of the table)
// by subtracting a constant value from one or more of the accel_table_entries fields.
#define FB_NRAMP_VALS       200

static stmotor_TVT_t fb_300M_tvt = {{1000,1000,1000,1000,1000,1000,1000,1000,1000}};
static stmotor_TVT_t fb_300C_tvt = {{666,666,666,666,666,666,666,666,666}};
static stmotor_TVT_t fb_600M_tvt = {{400,400,400,400,400,400,400,400,400}};
static stmotor_TVT_t fb_600C_tvt = {{500,500,500,500,500,500,500,500,500}};
static stmotor_TVT_t fb_SLOW_tvt = {{500,500,500,500,500,500,500,500,500}};

static stmotor_pin_sequence_t stmotor_fb_sequence = {
   {
    0xCCCCCCCC,     // STEP, simple 4-bit pattern
    0x0,            // DIRECTION - see REG_01, bit 0
    0x0,            // nENABLE - see REG_01, bit 1
    0x0,            // VREF PWM
    0xFFFFFFFF,     // MS1
    0xFFFFFFFF      // MS2
   }
};

static  uint32_t fb_accel8u_300M[FB_NRAMP_VALS] = {
  270,267,265,262,260,257,255,253,250,248,245,243,241,238,236,234,232,230,227,225,
  223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,190,188,186,
  184,182,181,179,177,176,174,172,171,169,167,166,164,163,161,160,158,157,155,154,
  152,151,149,148,146,145,144,142,141,140,138,137,136,134,133,132,131,129,128,127,
  126,125,123,122,121,120,119,118,116,115,114,113,112,111,110,109,108,107,106,105,
  104,103,102,101,100,99,98,97,96,95,94,93,93,92,91,90,89,88,87,87,
  86,85,84,83,83,82,81,80,79,79,78,77,76,76,75,74,74,73,72,72,
  71,70,70,69,68,68,67,66,66,65,64,64,63,63,62,61,61,60,60,59,
  59,58,57,57,56,56,55,55,54,54,53,53,52,52,51,51,50,50,49,49,
  48,48,47,47,47,46,46,45,45,44,44,44,43,43,42,42,42,41,41,40}; //1.0096
static  uint32_t fb_accel8u_300C[FB_NRAMP_VALS] = {
  200,199,198,197,196,195,195,194,193,192,191,190,189,188,188,187,186,185,184,183,
  182,182,181,180,179,178,178,177,176,175,174,173,173,172,171,170,170,169,168,167,
  166,166,165,164,163,163,162,161,160,160,159,158,158,157,156,155,155,154,153,153,
  152,151,150,150,149,148,148,147,146,146,145,144,144,143,142,142,141,140,140,139,
  139,138,137,137,136,135,135,134,134,133,132,132,131,131,130,129,129,128,128,127,
  126,126,125,125,124,124,123,122,122,121,121,120,120,119,119,118,117,117,116,116,
  115,115,114,114,113,113,112,112,111,111,110,110,109,109,108,108,107,107,106,106,
  105,105,104,104,103,103,102,102,101,101,100,100,100,99,99,98,98,97,97,96,
  96,96,95,95,94,94,93,93,93,92,92,91,91,90,90,90,89,89,88,88,
  88,87,87,86,86,86,85,85,84,84,84,83,83,82,82,82,81,81,81,80}; //1.0046
static  uint32_t fb_accel8u_600M[FB_NRAMP_VALS] = {
  144,144,143,143,142,142,141,141,141,140,140,139,139,139,138,138,137,137,136,136,
  136,135,135,134,134,134,133,133,132,132,132,131,131,130,130,130,129,129,129,128,
  128,127,127,127,126,126,125,125,125,124,124,124,123,123,122,122,122,121,121,121,
  120,120,120,119,119,119,118,118,117,117,117,116,116,116,115,115,115,114,114,114,
  113,113,113,112,112,112,111,111,111,110,110,110,109,109,109,108,108,108,107,107,
  107,106,106,106,105,105,105,105,104,104,104,103,103,103,102,102,102,101,101,101,
  101,100,100,100,99,99,99,98,98,98,98,97,97,97,96,96,96,96,95,95,
  95,94,94,94,94,93,93,93,92,92,92,92,91,91,91,91,90,90,90,89,
  89,89,89,88,88,88,88,87,87,87,87,86,86,86,86,85,85,85,84,84,
  84,84,83,83,83,83,82,82,82,82,82,81,81,81,81,80,80,80,80,79}; //1.003
static  uint32_t fb_accel8u_600C[FB_NRAMP_VALS] = {
  81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,
  81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,
  81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,
  81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,
  81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,
  81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,
  81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,
  81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,
  81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,
  81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81,81};
static  uint32_t fb_accel8u_SLOW[FB_NRAMP_VALS] = {
  240,240,239,239,239,238,238,238,237,237,237,236,236,236,235,235,235,234,234,234,
  233,233,233,232,232,232,231,231,231,230,230,230,229,229,229,229,228,228,228,227,
  227,227,226,226,226,225,225,225,224,224,224,223,223,223,223,222,222,222,221,221,
  221,220,220,220,219,219,219,219,218,218,218,217,217,217,216,216,216,215,215,215,
  215,214,214,214,213,213,213,212,212,212,212,211,211,211,210,210,210,210,209,209,
  209,208,208,208,208,207,207,207,206,206,206,205,205,205,205,204,204,204,203,203,
  203,203,202,202,202,201,201,201,201,200,200,200,200,199,199,199,198,198,198,198,
  197,197,197,196,196,196,196,195,195,195,195,194,194,194,193,193,193,193,192,192,
  192,192,191,191,191,191,190,190,190,189,189,189,189,188,188,188,188,187,187,187, 
  187,186,186,186,186,185,185,185,184,184,184,184,183,183,183,183,182,182,182,180};


// WARNING: profile order must match enum motor_profile_index_t
static stmotor_move_param_t stmotor_fb_dpi_profiles[] =
{
    //pwm_m/p, pwm_const, preholding_steps, postholding_steps, tvt_table, accel_tableeentries, accel_table
    {ACCEL_PWM_M, 120,   0,  0, &fb_SLOW_tvt, FB_NRAMP_VALS - 1,    fb_accel8u_SLOW },  // slow-move profile
    {ACCEL_PWM_M,  58,  64, 64, &fb_300M_tvt, FB_NRAMP_VALS - 1,    fb_accel8u_300M },  // 300dpi  mono
    {ACCEL_PWM_M,  87,  64, 64, &fb_300C_tvt, FB_NRAMP_VALS - 1,    fb_accel8u_300C },  // 300dpi  color
    {ACCEL_PWM_M, 115,  64, 64, &fb_600M_tvt, FB_NRAMP_VALS - 1 - 6, fb_accel8u_600M }, // 600dpi  mono - tuned speed, final speed is 6 less than max ramp value
    {ACCEL_PWM_M, 345,  64, 64, &fb_600C_tvt, FB_NRAMP_VALS - 1,    fb_accel8u_600C },  // 600dpi  color
};

// FB motor connections
static stmotor_pin_connect_t stmotor_fb_pin_connects = {
    STEP_MTR_BLK_1,
    STEP_OUTPUT_MODE_REG1,        // nENBL source
    {
        STEP_OUTPUT_MODE_SEQ,     // step0   STEP
        STEP_OUTPUT_MODE_REG0,    // step1   DIR 
        STEP_OUTPUT_MODE_REG1,    // step2   nENBL
        STEP_OUTPUT_MODE_PWM1,    // step3   VREF PWM
        STEP_OUTPUT_MODE_SEQ,     // step4   MS1
        STEP_OUTPUT_MODE_SEQ      // step5   MS2 
    }
};


//*****************************************************************************
// ADF
//*****************************************************************************

// see comment above for FB_NRAMP_VALS
#define ADF_NRAMP_VALS  200

/* TVT table used to set duty-cycle stepper-driver VREF.  Period is set in
 * stepper.c and is currently hardcoded to 5556 
 */
static stmotor_TVT_t adf_300M_tvt = {{1600,1600,1600,1600,1600,1600,1600,1600,1600}};
static stmotor_TVT_t adf_300C_tvt = {{750,750,750,750,750,750,750,750,750}};
static stmotor_TVT_t adf_SLOW_tvt = {{500,500,500,500,500,500,500,500,500}};

static stmotor_pin_sequence_t stmotor_adf_sequence = {
   {
    0x33333333,         // STEP, simple 4-bit pattern
    0x0,                // DIRECTION - see REG_01, bit 0
    0x0,                // nENABLE - see REG_01, bit 1
    0x0,                // VREF PWM
    0xFFFFFFFF,         // MS1
    0xFFFFFFFF          // MS2
   }
};

static uint32_t adf_accel8u_300M[ADF_NRAMP_VALS] = {
  270,267,265,262,260,257,255,253,250,248,245,243,241,238,236,234,232,230,227,225,
  223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,190,188,186,
  184,182,181,179,177,176,174,172,171,169,167,166,164,163,161,160,158,157,155,154,
  152,151,149,148,146,145,144,142,141,140,138,137,136,134,133,132,131,129,128,127,
  126,125,123,122,121,120,119,118,116,115,114,113,112,111,110,109,108,107,106,105,
  104,103,102,101,100,99,98,97,96,95,94,93,93,92,91,90,89,88,87,87,
  86,85,84,83,83,82,81,80,79,79,78,77,76,76,75,74,74,73,72,72,
  71,70,70,69,68,68,67,66,66,65,64,64,63,63,62,61,61,60,60,59,
  59,58,57,57,56,56,55,55,54,54,53,53,52,52,51,51,50,50,49,49,
  48,48,47,47,47,46,46,45,45,44,44,44,43,43,42,42,42,41,41,40}; //1.0096
static uint32_t adf_accel8u_300C[ADF_NRAMP_VALS] = {
  200,199,198,197,196,195,195,194,193,192,191,190,189,188,188,187,186,185,184,183,
  182,182,181,180,179,178,178,177,176,175,174,173,173,172,171,170,170,169,168,167,
  166,166,165,164,163,163,162,161,160,160,159,158,158,157,156,155,155,154,153,153,
  152,151,150,150,149,148,148,147,146,146,145,144,144,143,142,142,141,140,140,139,
  139,138,137,137,136,135,135,134,134,133,132,132,131,131,130,129,129,128,128,127,
  126,126,125,125,124,124,123,122,122,121,121,120,120,119,119,118,117,117,116,116,
  115,115,114,114,113,113,112,112,111,111,110,110,109,109,108,108,107,107,106,106,
  105,105,104,104,103,103,102,102,101,101,100,100,100,99,99,98,98,97,97,96,
  96,96,95,95,94,94,93,93,93,92,92,91,91,90,90,90,89,89,88,88,
  87,87,86,86,85,85,84,84,83,83,82,82,81,81,80,80,79,79,79,78};
static uint32_t adf_accel8u_SLOW[ADF_NRAMP_VALS] = {
  270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,
  270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,
  270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,
  270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,
  270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,
  270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,
  270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,
  270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,
  270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,
  270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270,270};

// WARNING: profile order must match enum motor_profile_index_t
static stmotor_move_param_t stmotor_adf_dpi_profiles[] =
{
    {ACCEL_PWM_M, 135,   0,  0, &adf_SLOW_tvt, ADF_NRAMP_VALS - 1,      adf_accel8u_SLOW  }, // slow-move profile
    {ACCEL_PWM_M,  58,  64, 64, &adf_300M_tvt, ADF_NRAMP_VALS - 1 - 73, adf_accel8u_300M  }, // 300dpi  mono - tuned speed, final speed is 73 less than max ramp value
    {ACCEL_PWM_M, 174,  64, 64, &adf_300C_tvt, ADF_NRAMP_VALS - 1 - 6,  adf_accel8u_300C  }, // 300dpi  color - tuned speed, final speed is 6 less than max ramp value
};

// ADF motor connections
static stmotor_pin_connect_t stmotor_adf_pin_connects = {
    STEP_MTR_BLK_2,
    STEP_OUTPUT_MODE_REG1,       // nENBL source
    {
        STEP_OUTPUT_MODE_SEQ,    // step0  STEP
        STEP_OUTPUT_MODE_REG0,   // step1  DIR 
        STEP_OUTPUT_MODE_REG1,   // step2  nENBL
        STEP_OUTPUT_MODE_PWM1,   // step3  VREF PWM
        STEP_OUTPUT_MODE_SEQ ,   // step4  MS1
        STEP_OUTPUT_MODE_SEQ     // step5  MS2 
    }
};

//*****************************************************************************
// Platform line start increments (only used by SYNC_EVERY mode)
// Notes: 
//   - these are tuned using the pwm_m period (LS_SRC_PWM_M)
//   - changing motor speed or sensor settings can invalidate these values
//   - color motor speeds should be 1/3 the speed of mono, but the
//     pwm_m values don't always follow the same pattern.  This can make
//     the ls_incr values look odd.
//*****************************************************************************

#define FB_LSINCR_MONO_300     3712     // LS @  742 usec
#define FB_LSINCR_COLO_300     1856     // LS @  742 usec
#define FB_LSINCR_MONO_600     3680     // LS @ 1490 usec
#define FB_LSINCR_COLO_600     3680     // LS @ 1490 usec

#define ADF_LSINCR_MONO_300    1856     // LS @  742 usec
#define ADF_LSINCR_COLO_300    1856     // LS @  742 usec

#endif // SCANMECH_STEP_CFG_GRANUM2_H

