/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2006-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * Description:
 *   Marvell Granum2 mech driver.  This driver has rough support for two physical
 *   motors, but:
 *     - only one motor may run at a time
 *     - the mechType global must always indicate the running mech/motor
 *     - pause/resume is only supported on the flatbed
 *
 * Notes:
 *   - the mech does not use a physical 'hardstop', but has a home position sensor
 *   - the mech does not contain home position 'notches'
 *   - only 300 and 600dpi are supported on flatbed
 *   - only 300dpi is supported on ADF
 *   - a hinch is a hundredth of an inch
 *   - a thinch is a thousandth of an inch
 *   - during a scan operation, scanlib will send events to the mech in the following
 *     order: prepare, setup, run, done
 *   - work in units of motor steps where possible, it improves accuracy
 *   - many functions are just minimal stubs, add product specific behavior as needed
 *     (maintain mech state, signal control panel, etc.)
 *
 * TODO:
 *   - convert internal mech units from hinches to thinches
 **/

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/io.h>
#include <linux/semaphore.h>
#include <asm/gpio.h>
#include "lassert.h"
#include "scancore.h"
#include "scantypes.h"
#include "scandbg.h"
#include "scanhwerr.h"
#include "scan.h"
#include "scansen.h"
#include "scanvars.h"
#include "scanmech.h"
#include "scancmdq.h"
#include "scanalyzer.h"
#include "scanblk_if.h"
#include "scanif.h"
#include "scanlib.h"
#include "debug.h"
#include "utils.h"
#include "scanmargin.h"
#include "scands.h"
#include "cal.h"                       // For scan power-on calibration
#include "cal_common.h"                // For scan power-on calibration
#include "adfpath.h"
#include "adfgapper.h"
#include "scanmech_granum2.h"
#include "adfsensor.h"
#include "adfsensor_granum2.h"
#include "scanplat_sb.h"
#include "scos.h"
#include "scantask.h"
#include "cpu_api.h"
#include "scos_cmd_proc.h"

#include "stepper-api.h"               // stepper-mod: driver interface
#include "scanmech_step_cfg_granum2.h" // Mech specific motor tables

#include "regAddrs.h"
#include "ips_apb_config_regstructs.h"
#include "ips_apb_config_regmasks.h"

/*
 * Structure that has a motor handle for each motor and associated profile handle
 */
typedef struct motor_param_s {
   stmotor_t             *motor_handle;
   stmotor_move_param_t  *motor_profile;
}motor_param_t;

#define smech_func_enter()              dbg2("SMECH@%s: enter\n", __FUNCTION__)
#define smech_func_leave()              dbg2("SMECH@%s: leave\n", __FUNCTION__)
#define smech_func_infor(msg, args...)  dbg1("SMECH@%s: " msg, __FUNCTION__, ##args)
#define smech_func_debug(msg, args...)  dbg2("SMECH@%s: " msg, __FUNCTION__, ##args)
#define smech_func_error(msg, args...)  dbg1("SMECH@%s.%d: " msg, __FUNCTION__, __LINE__, ##args)
#define smech_func_helpu(msg, args...)  dbg1(msg, ##args)

// Enable to turn on IRQ debugging (don't leave enabled!)
//#define SMECH_IRQ_DEBUG 

#ifdef SMECH_IRQ_DEBUG
    #define smech_irq_debug(msg, args...)  printk("SMECHIRQ@%s: " msg, __FUNCTION__, ##args)
#else
    #define smech_irq_debug(msg, args...)
#endif


/*
 * Mech driver capabilities
 */

// Some mechs have a calibration strip that does not have guaranteed placement.  In that
// case, it is neccesary to scan a larger area, and search for the cal strip (overscan)
// This mech does not need this - for mechs that do, uncomment this line
//#define OVERSCAN_CAL_REQUIRED

static const struct scanmech_capabilities scanmech_granum2_capabilities =
{ 
    .version = sizeof(struct scanmech_capabilities), 
    
    .supported_target_types =
    {
        SCAN_TARGET_TYPE_DOC, 
        SCAN_TARGET_TYPE_STATIONARY_CAL,
#ifdef OVERSCAN_CAL_REQUIRED
        SCAN_TARGET_TYPE_OVERSCAN_CAL,
#else
        SCAN_TARGET_TYPE_MOVING_CAL,
#endif
        
        /* zero terminate the list */
        SCAN_TARGET_TYPE_NONE
    },
};


/* Line start constants */
#define LS_SRC_STEP         0     // PWM_P * PWM_M sysclks
#define LS_SRC_PWM_M        1     // PWM_M sysclks
#define LS_ENABLED          true
#define LS_DISABLED         false
#define NO_LS_STEPNUM       -1


/*
 * Flatbed platform margins
 *  (Tuned for 1.5mm top and left, using home sensor location)
 *  Note that the scan app must get these margins using SCANMECH_IOCTL_GET_FLATBED_MARGINS
 *  and adjust the user scan area accordingly before starting a scan.
 */
static struct scan_flatbed_margins g_flatbed_margins[] = {
    /* dpi, cmode, left_mar_thinches_x, top_mar_thinches_y, right_mar_thinches_x, bottom_mar_thinches_y, left_mar_thinches_backside */
    { 300,  SCAN_CMODE_MONO,   70,  0, 87500, 11700*2, 0},
    { 300,  SCAN_CMODE_COLOR,  70,  0, 87500, 11700*2, 0},
    { 600,  SCAN_CMODE_MONO,   70,  0, 87500, 11700, 0},
    { 600,  SCAN_CMODE_COLOR,  70,  0, 87500, 11700, 0},

    /* end of table marker */
    { 0, SCAN_CMODE_MONO, 0, 0 },
};


/*
 * ADF platform margins
 */
static struct scan_adf_margins g_adf_margins[] = {
    /* dpi, cmode, tof, bof, left_margin_thinches, rows_to_tof_backside, rows_to_bof_backside, left_margin_thinches_backside */
    {  .dpi=300,
       .cmode=SCAN_CMODE_MONO,
       .rows_to_tof=0,
       .rows_to_bof=332,
       .left_margin_thinches=0,
       .rows_to_tof_backside=150,
       .rows_to_bof_backside=0,
       .left_margin_thinches_backside=0,
    },
    {  .dpi=300,
       .cmode=SCAN_CMODE_COLOR,
       .rows_to_tof=0,
       .rows_to_bof=166,
       .left_margin_thinches=0,
       .rows_to_tof_backside=120,
       .rows_to_bof_backside=0,
       .left_margin_thinches_backside=0,
    },
    {  .dpi=600,
       .cmode=SCAN_CMODE_MONO,
       .rows_to_tof=0,
       .rows_to_bof=332,
       .left_margin_thinches=0,
       .rows_to_tof_backside=300,
       .rows_to_bof_backside=0,
       .left_margin_thinches_backside=0,
    },
    {  .dpi=600,
       .cmode=SCAN_CMODE_COLOR,
       .rows_to_tof=0,
       .rows_to_bof=332,
       .left_margin_thinches=0,
       .rows_to_tof_backside=240,
       .rows_to_bof_backside=0,
       .left_margin_thinches_backside=0,
    },

    /* end of table marker */
    { .dpi=0, 
      .cmode=SCAN_CMODE_MONO
    },
};


/*
 * Granum2 FB mech definitions 
 */
#define FB_STEPS_PER_HINCH    192    // Steps per hinch
#define FB_MAX_CMD_STEPS       32

// NOTE that the constant for FB_NRAMP_VALS may not be the actual number of ramp values
// see comment above the definition of FB_NRAMP_VALS for details
#define FB_RAMP_STEPS         (FB_NRAMP_VALS * FB_MAX_CMD_STEPS)
#define FB_EXTRA_STEPS_FOR_SYNC_PULSES 34  // tunable parm - extra steps in case we need more for SYNC_EVERY mode to complete

#define FB_MAX_DIST           (1320 * FB_STEPS_PER_HINCH)  // Home sensor to far wall distance (steps)

#define FB_HOME_POSITION      (0    * FB_STEPS_PER_HINCH)  // Scanbar home position (steps)
#define FB_DFLT_ADF_POSITION  (9    * FB_STEPS_PER_HINCH)  // FB ADF scanning position (steps)
#define FB_CAL_POSITION       (45   * FB_STEPS_PER_HINCH)  // Cal strip center pos (steps)
#define FB_MV_CAL_POSITION    (40   * FB_STEPS_PER_HINCH)  // Cal strip moving start pos (steps)
#define FB_OVERSCAN_CAL_POSITION (30  * FB_STEPS_PER_HINCH) // Cal strip overscan start pos (steps)  
#define FB_TOP_BED_POSITION   (75   * FB_STEPS_PER_HINCH)  // Start of flatbed glass (steps)
#define FB_BOT_BED_POSITION   (1265 * FB_STEPS_PER_HINCH)  // End of flatbed glass (steps)

#define SHORT_MOVE_STEPS      (200  * FB_STEPS_PER_HINCH)  // 2 inch threshold for short/slow moves


/*
 * Granum2 ADF mech definitions 
 */
#define ADF_STEPS_PER_HINCH          96  // Steps per hinch
#define ADF_MAX_CMD_STEPS            32

// NOTE that the constant for ADF_NRAMP_VALS may not be the actual number of ramp values
// see comment above the definition of ADF_NRAMP_VALS for details
#define ADF_RAMP_STEPS  (ADF_NRAMP_VALS * ADF_MAX_CMD_STEPS)

#define ADF_PAGE_LENGTH       (1500 * ADF_STEPS_PER_HINCH) // Distance to clear a single LGL page (steps)
#define ADF_STAGE_DIST        (800  * ADF_STEPS_PER_HINCH) // ADF stage distance (steps)
#define ADF_EMPTY_DIST        (2000 * ADF_STEPS_PER_HINCH) // ADF paper flush distance (steps)
#define ADF_EMPTY_THRESH      (850  * ADF_STEPS_PER_HINCH) // Threshold to declare no more paper in tray (steps)
#define ADF_LIFT_PICK_ROLLER  (25   * ADF_STEPS_PER_HINCH) // Distance to lift the pick roller (steps)

#define ADF_MAX_PAGES_TO_CLEAR       10  // Max pages to flush from input tray
#define ADF_MAX_PAGES_TO_SCAN        25  // Adjust for ADF tray capacity

#define ADF_SCAN_DIST   (ADF_PAGE_LENGTH * ADF_MAX_PAGES_TO_SCAN)

/* 
 * Commonly used motor profiles (must be supported by FB and ADF)
 */
#define MOVE_SPD_AUTO     -1
#define MOVE_SPD_SLOW     DPI_300C
#define MOVE_SPD_QUIET    DPI_SLOW
#define MOVE_SPD_NORMAL   DPI_300M


/*
 * Support for writing the IPS APB MFC register (directly, without a driver)
 */
#define IPS_APB_CFG_LEN   0x18

#define ips_apb_cfgWrite(rname, val) \
    iowrite32(val, &((IPS_APB_CONFIG_REGS_t *)(virt_addr))->rname)

#define ips_apb_cfgRead(rname)  \
    ioread32(&((IPS_APB_CONFIG_REGS_t *)(virt_addr))->rname)

#define IPS_APB_CFG_MFC_RS_ST6   14
#define IPS_APB_CFG_MFC_RS_ST5   13
#define IPS_APB_CFG_MFC_RS_ST4   12
#define IPS_APB_CFG_MFC_RS_ST3   11
#define IPS_APB_CFG_MFC_RS_ST2   10
#define IPS_APB_CFG_MFC_RS_ST1    9
#define IPS_APB_CFG_MFC_RS_ST0    8

/*
 * Internal globals
 */
static t_ScanMechType      mechType;
static struct pt_mech      granum2_mech;
static scan_target_type_t  current_scan_target_type;
static MOTOR_SYNC          smech_motor_sync;
static uint32_t            smech_sync_start_step;
static uint32_t            smech_profile_index;
static bool                smech_pause_rewind_enable;
static int                 smech_pause_mtr_loc;
static bool                home_find_enable;
static struct semaphore    mech_sem;
static motor_param_t motor_param[NUM_MOTOR] = {{0,0},{0,0}};
static stmotor_t          *fb_motor  = NULL;
static stmotor_t          *adf_motor = NULL;


/*
 * Internal function prototypes
 */
static bool         isDocSourceADF(void);
static stmotor_t *  get_motor_handle_from_mech(t_ScanMechType mech_type);
static uint32_t     get_stmotor_profile_index(scan_cmode_t cmode, uint32_t dpi);
static uint32_t     thinches_to_steps(t_ScanMechType mech_type, uint32_t thinches);
static uint32_t     hinches_to_steps(t_ScanMechType mech_type, uint32_t hinches);
static int          steps_to_hinches(t_ScanMechType mech_type, int steps);

static MOTOR_SEL    get_stmotor_scif(t_ScanMechType mech_mtr);

static scan_err_t smech_wait_for_stop(stmotor_t *motor_handle);
static scan_err_t smech_move_rel_nosyncs(stmotor_t *motor_handle, int speed, uint32_t steps, uint8_t dir, bool wait);
static scan_err_t smech_move_to_position(int target, bool wait);
static scan_err_t smech_move_to_home(void);
static scan_err_t smech_find_home_position(void);

static scan_err_t smech_adf_clear_paper(void);

static scan_err_t smech_fb_pause_rewind(void);

static void smech_trig_mtr_stopped(stmotor_t *motor_handle, void *cb_user_data);
static void smech_trig_scan_stopped(stmotor_t *motor_handle, void *cb_user_data);
static void smech_trig_paused_stopped(stmotor_t *motor_handle, void *cb_user_data);

static void smech_cmdline_init( void );

static bool flatbed_sensor_home_is_found(void);
static void home_sensor_find_disable(void);
static void home_sensor_find_enable(void);

#if 0 //todo for power-on cal
static scan_err_t smech_poweron_cal_run(struct scanvars **psv);
static scan_err_t smech_poweron_cal(scan_cmode_t cmode, int dpi);
#endif


/**
 *  \brief request motor profile handle to be returned  
 *
 *  helper function to return handle to the structure that defines 
 *  motor parameters for upcoming move
 *
 *  \param[in]  (stmotor_t *)  motor handle 
 *  \param[in]  index          index into motor profile index must be a valid and inbound
 *
 *  \return     (stmotor_move_param_t *)
 *  \retval     NULL, did not find profile handle
 *  \retval     valid handle to the motor move profile
 **/
static stmotor_move_param_t * get_smot_move_profile(stmotor_t *motor_handle, uint8_t profile_index)
{
    uint8_t i;
    
    for (i = 0; i < NUM_MOTOR; i++)
    {
        if (motor_param[i].motor_handle == motor_handle)
        {
            return(&motor_param[i].motor_profile[profile_index]);
        }
    }
    XASSERT(NULL, (uint32_t) NULL);
    return NULL; 
}


/**
 *  \brief Ask if document source is ADF
 *
 *  The scan client chooses the document source and provides it in the scan vars, the
 *  mech code should honor this request whenever possible (situations like paper not
 *  present may make it impossible). This helper routine checks to see if the source
 *  is the ADF.
 *
 *  \return bool
 *  \retval true  Document source is ADF
 *  \retval false Document source is not ADF
 **/
static bool isDocSourceADF(void)
{
    bool is_adf = false;
    const struct scanvars *sv = scanvar_peek();

    if (sv != NULL)
    {
        switch (sv->doc_src)
        {
            case SCAN_DOCUMENT_SOURCE_FLATBED:
            case SCAN_DOCUMENT_SOURCE_FLATBED_NSENSOR:
                is_adf = false;
                break;
            case SCAN_DOCUMENT_SOURCE_ADF:
            case SCAN_DOCUMENT_SOURCE_ADF_NSENSOR:
                is_adf = true;
                break;
             default:
                smech_func_error("unknown document source %d\n", sv->doc_src);
                XASSERT(0, sv->doc_src);
                break;
        }
    }
    else
    {
        smech_func_debug("warning: called when no valid scanvars\n");
    }

    smech_func_debug("return %s (%d)\n", (is_adf == 0) ? "FB" : "ADF", is_adf);
    return (is_adf);
}

/**
 *  \brief Drain the mech semaphore
 *
 **/
static void drain_mech_sem(void)
{
    while(down_trylock(&mech_sem) == 0)
    {
        // Nothing to do here ...
    }
}

/**
 *  \brief Is flatbed home found?
 *
 *  \return bool
 *  \retval true  Home is found
 *  \retval false Home is not found
 **/
static bool flatbed_sensor_home_is_found(void)
{
    bool     home_found  = false;
    uint32_t logic_level = 0xFFFF;

    logic_level = scanplat_kernel_gpio_get_value(granum2_mech.scan_home_sensor_gpio);
    if (logic_level != 0)
    {
        home_found = true;
    }
    else
    {
        home_found = false;
    }
    return (home_found);

}

/**
 *  \brief Enable/disable home sensor isr actions
 *
 **/
static void home_sensor_find_disable(void)
{
    smech_func_enter();

    home_find_enable = false;
}

static void home_sensor_find_enable(void)
{
    smech_func_enter();

    home_find_enable = true;
}


/**
 *  \brief Home sensor gpio isr
 *
 *  Called by the GPIO code whenever the home sensor changes state.
 **/
bool home_sensor_isr(void)
{
    static struct timespec prev_isr_time = {0, 0};
    struct   timespec cur_isr_time;
    bool     bouncing = true;
    bool     home_found;

    // WARNING: This function is likely called from gpio interrupt context!

    getrawmonotonic(&cur_isr_time);
    
    if (cur_isr_time.tv_sec - prev_isr_time.tv_sec >= 2)
    {
        bouncing = false;
    }
    else
    {
        long   ms_diff;
        ms_diff  = (cur_isr_time.tv_sec - prev_isr_time.tv_sec) * 1000;
        ms_diff += (cur_isr_time.tv_nsec  / 1000000);
        ms_diff -= (prev_isr_time.tv_nsec / 1000000);

        if (ms_diff > 30)
        {
            bouncing = false;
        }
    }

    // Grab the state of the home sensor
    home_found = flatbed_sensor_home_is_found();

    smech_irq_debug("bouncing=%d home=%d home_find_enable=%d\n", bouncing, home_found, home_find_enable);

    // Only take action if someone is interested
    if (home_find_enable && (!bouncing))
    {
        if (home_found)
        {
            // Found the home sensor.
            // Emergency halt will stop the motor without ramping down, motor position
            // could be compromised (which doesn't matter here, we need to stop ASAP).
#if 0            
            // smot_step_emergency_halt(FB_MOTOR);
#endif            

            // NOTE: do not call up(&mech_sem) here, wait for the motor stop trigger
            //       to fire, see smech_trig_mtr_stopped. You did set a stop trigger ;)
        }
    }

    prev_isr_time = cur_isr_time;
    return true;
}


/**
 *  \brief Motor stopped trigger
 *
 *  Motor stopped trigger callback, used to block on motor move complete
 **/
static void smech_trig_mtr_stopped(stmotor_t *motor_handle, void *cb_user_data)
{
    // WARNING: This function is likely called from motor interrupt context!

    smech_irq_debug("motor_handle=%d\n", (uint32_t) motor_handle);

    // Unblock anyone waiting for the motor to stop
    up(&mech_sem);
}


/**
 *  \brief Scan stopped trigger
 *
 *  Trigger used when async scan stop is requested
 **/
static void smech_trig_scan_stopped(stmotor_t *motor_handle, void *cb_user_data)
{
    // WARNING: This function is likely called from motor interrupt context!
    //          (smech_set_status is IRQ safe, it simply sends a message)
    smech_set_status(SCAN_MECH_READY);
}

/**
 *  \brief Scan paused trigger
 *
 *  Trigger used when the motor stops after a pause request
 **/
static void smech_trig_paused_stopped(stmotor_t *motor_handle, void *cb_user_data)
{
    // WARNING: This function is likely called from motor interrupt context!

    smech_irq_debug("motor_handle=%d\n", (uint32_t) motor_handle);

    // The system requested a mech pause, and now the motor is stopped (we are
    // paused). We can't rewind in IRQ context, so set flag to handle the resume
    // later (in thread context).
    smech_pause_rewind_enable = true;

    // Set the mech back to ready so the core scan code can continue (even though
    // we may have deferred the rewind).
    smech_set_status(SCAN_MECH_READY);
}


/**
 *  \brief Get the motor id from the mech type
 *
 *  Given the scan mech type, get the associated motor id.  This is
 *  the 'normal' way lookups should be done.
 **/
static stmotor_t * get_motor_handle_from_mech(t_ScanMechType mech_type)
{
    stmotor_t *motor_handle = NULL;

    switch(mech_type)
    {
        case SCAN_MECH_FLATBED:
            motor_handle = fb_motor;
            break;
        case SCAN_MECH_ADF:
            motor_handle = adf_motor;
            break;
        default:
            XASSERT(0, mech_type);
            break;
    }
    XASSERT(motor_handle != NULL, (uint32_t) motor_handle);
//    smech_func_debug("return %d[%d=fb_motor,%d=adf_motor]\n", (uint32_t) motor_handle, (uint32_t) fb_motor,(uint32_t) adf_motor);

    return motor_handle;
}


/**
 *  \brief Get a profile ID for stepper motor
 *
 *  Apart from a few required profiles, the scan mech may not support all color/dpi
 *  combinations. Use this helper to trap wonky combinations.
 **/
static uint32_t get_stmotor_profile_index(scan_cmode_t cmode, uint32_t dpi)
{
    motor_profile_index_t index = DPI_300M;

    smech_func_infor("cmode=%d[%d=MONO,%d=COLOR], dpi=%d, mechType=%s\n", 
             cmode, SCAN_CMODE_MONO, SCAN_CMODE_COLOR, dpi,
             mechType == SCAN_MECH_FLATBED ? "FB" : "ADF");

    if (mechType == SCAN_MECH_FLATBED)
    {
        // The flatbed does not support 1200dpi
        switch (dpi)
        {
            case 300:
                index = (cmode == SCAN_CMODE_COLOR) ? DPI_300C : DPI_300M;
                break;
            case 600:
                index = (cmode == SCAN_CMODE_COLOR) ? DPI_600C : DPI_600M;
                break;
            default:
                smech_func_error("unsupported DPI\n");
                XASSERT(0, dpi);
                break;
        } 
    }
    else
    {
        // The adf only supports 300dpi
        switch (dpi)
        {
            case 300:
                index = (cmode == SCAN_CMODE_COLOR) ? DPI_300C : DPI_300M;
                break;
            case 600:
                index = (cmode == SCAN_CMODE_COLOR) ? DPI_600C : DPI_600M;
                break;				
            default:
                smech_func_error("unsupported DPI\n");
                XASSERT(0, dpi);
                break;
        } 
    }

    return (index);
}


/**
 *  \brief Convert thinches to steps
 *
 **/
static uint32_t thinches_to_steps(t_ScanMechType mech_type, uint32_t thinches)
{
    uint32_t steps = 0;

    // TODO: at some point the mech code needs to move from hinches to
    //       thinches.  This routine has the potential to have some loss
    //       of accuracy.

    if (mech_type == SCAN_MECH_FLATBED)
    {
        steps = (thinches * FB_STEPS_PER_HINCH) / 10;
    }
    else if (mech_type == SCAN_MECH_ADF)
    {
        steps = (thinches * ADF_STEPS_PER_HINCH) / 10;
    }
    else
    {
        smech_func_error("unknown ScanMechType\n");
        XASSERT(0, mech_type);
    }

    return (steps);
}


/**
 *  \brief Convert hinches to steps
 *
 **/
static uint32_t hinches_to_steps(t_ScanMechType mech_type, uint32_t hinches)
{
    uint32_t steps = 0;

    if (mech_type == SCAN_MECH_FLATBED)
    {
        steps = hinches * FB_STEPS_PER_HINCH;
    }
    else if (mech_type == SCAN_MECH_ADF)
    {
        steps = hinches * ADF_STEPS_PER_HINCH;
    }
    else
    {
        smech_func_error("unknown ScanMechType\n");
        XASSERT(0, mech_type);
    }

    return (steps);
}


/**
 *  \brief Convert steps to hinches
 *
 **/
static int steps_to_hinches(t_ScanMechType mech_type, int steps)
{
    int hinches = 0;

    if (mech_type == SCAN_MECH_FLATBED)
    {
        hinches = steps / FB_STEPS_PER_HINCH;        
    }
    else if (mech_type == SCAN_MECH_ADF)
    {
        hinches = steps / ADF_STEPS_PER_HINCH;
    }
    else
    {
        smech_func_error("unknown ScanMechType\n");
        XASSERT(0, mech_type);
    }

    return (hinches);
}


/**
 *  \brief Set LS of stepper motor
 **/
static void set_stmotor_ls_source(t_ScanMechType mech_mtr)
{
    stmotor_block_id_t  block_num;
    void  __iomem       *virt_addr;
    uint32_t            reg;
    uint8_t             rs_val;

    // Need to poke at the MFC (Misc Feature Control) register to route the motor
    // LS signals. No driver for this so we need to do some direct fiddling, which
    // is OS and ASIC specific.

#ifndef __KERNEL__
    #error linux kernel only
#endif

    // Get the motor block
    switch(mech_mtr)
    {
        case SCAN_MECH_FLATBED:
//            block_num = stmotor_fb_pin_connects.block_num;
			block_num =	STEP_MTR_BLK_0;
            break;
        case SCAN_MECH_ADF:
            block_num = stmotor_adf_pin_connects.block_num;
            break;
        default:
            block_num = stmotor_fb_pin_connects.block_num;
            smech_func_error("unknown ScanMechType\n");
            XASSERT(0, mech_mtr);
            break;
    }

    // Select the appropriate MFC RS setting for the motor block
    switch(block_num)
    {
        case STEP_MTR_BLK_0:
            rs_val= IPS_APB_CFG_MFC_RS_ST0;
            break;
        case STEP_MTR_BLK_1:
            rs_val= IPS_APB_CFG_MFC_RS_ST1;
            break;
        case STEP_MTR_BLK_2:
            rs_val= IPS_APB_CFG_MFC_RS_ST2;
            break;
        case STEP_MTR_BLK_3:
            rs_val= IPS_APB_CFG_MFC_RS_ST3;
            break;
        case STEP_MTR_BLK_4:
            rs_val= IPS_APB_CFG_MFC_RS_ST4;
            break;
        case STEP_MTR_BLK_5:
            rs_val= IPS_APB_CFG_MFC_RS_ST5;
            break;
        default:
            rs_val= IPS_APB_CFG_MFC_RS_ST0;
            smech_func_error("unknown motor block number\n");
            XASSERT(0, block_num);
            break;
    }

    //smech_func_debug("block_num=%d rs_val=0x%x\n", block_num, rs_val);

    // Now poke the register MFC value
    virt_addr = ioremap((uint32_t)IPS_IPS_APB_CONFIG_BASE, IPS_APB_CFG_LEN);

    reg = ips_apb_cfgRead(MFC);
    reg = IPS_APB_CONFIG_MFC_RS_REPLACE_VAL(reg, rs_val);
    ips_apb_cfgWrite(MFC, reg);

    iounmap(virt_addr);
}


/**
 *  \brief Get the SCIF motor number
 *
 *  Figure out the mapping between our motor block and the scif motor
 *  input: we need to do this to properly route the row sync signal
 *  between the blocks.
 **/
static MOTOR_SEL get_stmotor_scif(t_ScanMechType mech_mtr)
{
    MOTOR_SEL           scif_mtr;
    stmotor_block_id_t  block_num;

    switch(mech_mtr)
    {
        case SCAN_MECH_FLATBED:
            block_num = stmotor_fb_pin_connects.block_num;
            block_num = STEP_MTR_BLK_0;
            break;
        case SCAN_MECH_ADF:
            block_num = stmotor_adf_pin_connects.block_num;
            block_num = STEP_MTR_BLK_0;
            break;
        default:
            block_num = stmotor_fb_pin_connects.block_num;
            smech_func_error("unknown ScanMechType\n");
            XASSERT(0, mech_mtr);
            break;
    }

    switch(block_num)
    {
        case STEP_MTR_BLK_0:
        case STEP_MTR_BLK_1:
            scif_mtr = MOTOR1;
            break;
        case STEP_MTR_BLK_2:
        case STEP_MTR_BLK_3:
        case STEP_MTR_BLK_4:
            scif_mtr = MOTOR2;
            break;
        default:
            scif_mtr = MOTOR1;
            smech_func_error("unknown motor block number\n");
            XASSERT(0, block_num);
            break;
    }
    
    scif_mtr = MOTOR1;
    return scif_mtr;
}


/**
 *  \brief Block until motor move completes
 *
 *  In some cases the mech code will completely handle a motor move (no calls
 *  back out to scanlib).  In these cases the mech code must use this routine
 *  to block until the move is done.
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE   Move complete, no error
 *  \retval SCANERR_*      Move failure, failure specific error code
 **/
static scan_err_t smech_wait_for_stop(stmotor_t *motor_handle)
{
    scan_err_t scerr = SCANERR_NONE;
#if 0
    // NOTE: spin loops like this may not be friendly to the rest of the system,
    //       so try to limit using this function.  We mostly use it as a safety 
    //       check between moves when the motor should already be stopped and we 
    //       can't use our mech semaphore.

    while (!smot_step_motor_is_idle(motor_handle));
    {
        msleep_interruptible(10);
    }
#endif
    return scerr;
}


/**
 * \brief  Relative motor move, no line syncs
 *
 *  A relative motor move, with or without waiting for completion.
 *
 *  \param[in] motor_handle Motor to move
 *  \param[in] speed        Move speed (profile index)
 *  \param[in] steps        Steps to move
 *  \param[in] dir          Move direction
 *  \param[in] wait         Wait for move to complete
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE   Move complete, no error
 *  \retval SCANERR_*      Move failure,  specific error code
 **/
static scan_err_t smech_move_rel_nosyncs(stmotor_t *motor_handle, int speed, uint32_t steps, uint8_t dir, bool wait)
{
    scan_err_t              scerr = SCANERR_NONE;
#if 0
    uint32_t                profile_index;
    stmotor_move_param_t    *move_params;

    smech_func_enter();

    // Wait for any previous movement to stop before proceeding
    scerr = smech_wait_for_stop(motor_handle);
    if (scerr != SCANERR_NONE)
    {
        smech_func_error("smech_wait_for_stop failed, scerr = %d\n", scerr);
        return scerr;
    }
    if (speed == MOVE_SPD_AUTO)
    {
        // Use slower/quieter profile for short moves
        if (steps <= SHORT_MOVE_STEPS)
        {
            profile_index = MOVE_SPD_SLOW;
        }
        else 
        {
            profile_index = MOVE_SPD_NORMAL;
        }
    }
    else
    {
        // Use the provided profile index
        profile_index = (uint32_t)speed;
    }

    if (wait == true)
    {
        // Flush out our semaphore just in case it was left dirty
        drain_mech_sem();

        // Set up a motor trigger to fire when the move is done
        smot_step_add_trigger(motor_handle, TRIG_MOTOR_STOP, 0, smech_trig_mtr_stopped, NULL);
    }

    move_params = get_smot_move_profile(motor_handle,profile_index);
    smot_step_set_motor_move_params(motor_handle,move_params);

    // Start the motor move
    smot_step_move_rel(motor_handle,
                       steps,
                       dir,
                       LS_DISABLED,
                       NO_LS_STEPNUM);
    if (wait == true)
    {
        // Sleep until move done (motor stopped trigger will fire)
        down(&mech_sem);
    }
#endif
    return scerr;
}


/**
 * \brief  Move the scan bar to an absolute position
 *
 *  A flatbed move operation to position the scan bar at a specific position:
 *  the mech code will be in control of blocking until the move is complete.
 *
 *  \param[in] target_pos Absolute target position (in steps)
 *  \param[in] wait       Wait for move to complete
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE   Move complete, no error
 *  \retval SCANERR_*      Move failure,  specific error code
 **/
static scan_err_t smech_move_to_position(int target_pos, bool wait)
{

    scan_err_t   scerr = SCANERR_NONE;
#if 0	    
    int          current_pos;
    uint32_t     move_steps;
    uint8_t      move_direction;

    smech_func_enter(); 

    // Flatbed check: this operation only makes sense on flatbed
    if (mechType != SCAN_MECH_FLATBED)
    {
        smech_func_error("MechType is not flatbed\n");
        return SCANERR_INVALID_PARAM;
    }

    // Wait for any previous movement to stop before proceeding
    // (should never be moving when we get here, but play it safe)
    scerr = smech_wait_for_stop(fb_motor);
    if (scerr != SCANERR_NONE)
    {
        smech_func_error("smech_wait_for_stop failed, scerr = %d\n", scerr);
        return scerr;
    }

    current_pos = smot_step_get_location(fb_motor);

    // Figure out which way we need to move
    if (current_pos > target_pos)
    {
        move_steps = (uint32_t)(current_pos - target_pos);
        move_direction = MOVE_REVERSE;
    }
    else if (current_pos < target_pos)
    {
        move_steps = (uint32_t)(target_pos - current_pos);
        move_direction = MOVE_FORWARD;
    }
    else
    {
        // We are already there!
        return SCANERR_NONE;
    }

    smech_func_infor("current_pos=%d, target_pos=%d, move_steps=%d\n", current_pos, target_pos, move_steps);
    scerr = smech_move_rel_nosyncs(
                    fb_motor,
                    MOVE_SPD_AUTO,
                    move_steps,
                    move_direction,
                    wait);

    if (scerr != SCANERR_NONE)
    {
        smech_func_error("ErrCode = %d\n", scerr);
    }
#endif    
    return scerr;
}


/**
 * \brief  Move the scan bar to the Cal Strip position
 *
 *  A flatbed move operation to position the scan bar for a calibration scan: the mech
 *  code will be in control of blocking until the move is complete.
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE   Move complete, no error
 *  \retval SCANERR_*      Move failure,  specific error code
 **/
static scan_err_t smech_move_to_calibration(void)
{
    scan_err_t scerr = SCANERR_NONE;

    smech_func_enter();

    // Flatbed check
    if (mechType != SCAN_MECH_FLATBED)
    {
        smech_func_error("MechType is not flatbed\n");
        scerr = SCANERR_INVALID_PARAM;
        return scerr;
    }

    scerr = smech_move_to_position(FB_CAL_POSITION, true);
    if (scerr != SCANERR_NONE)
    {
        smech_func_error("ErrCode = %d\n", scerr);
    }

    smech_func_leave(); 
    return scerr;
}


/**
 * \brief  Move the scan bar to the home position
 *
 *  A flatbed move operation to position the scan bar at the home position: the mech
 *  code will be in control of blocking until the move is complete.
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE   Move complete, no error
 *  \retval SCANERR_*      Move failure,  specific error code
 **/
static scan_err_t smech_move_to_home(void)
{
    scan_err_t scerr = SCANERR_NONE;

    smech_func_enter();

    // Flatbed check
    if (mechType != SCAN_MECH_FLATBED)
    {
        smech_func_error("MechType is not flatbed\n");
        scerr = SCANERR_INVALID_PARAM;
        return scerr;
    }

    scerr = smech_move_to_position(FB_HOME_POSITION, true);
    if (scerr != SCANERR_NONE)
    {
        smech_func_error("ErrCode = %d\n", scerr);
    }

    smech_func_leave();
    return scerr;
}


/**
 * \brief  Find the home position by locating the home sensor
 *
 *  A flatbed move operation to position the scan bar at the home position: the mech
 *  code will be in control of blocking until the move is complete.  This routine is
 *  used when we don't know exactly where we are ...
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE   Move complete, no error
 *  \retval SCANERR_*      Move failure,  specific error code
 **/
static scan_err_t smech_find_home_position(void)
{
    scan_err_t  scerr = SCANERR_NONE;
#if 0    
    bool        home_sense;
    uint32_t    ErrorCode    = 0x0000;

    smech_func_enter();

    // Flatbed check
    if (mechType != SCAN_MECH_FLATBED)
    {
        ErrorCode = 0xE100;
        scerr = SCANERR_INVALID_PARAM;
        goto smech_find_home_position_exit;
    }

    // Make sure we are stopped before proceeding
    scerr = smech_wait_for_stop(fb_motor);
    if (scerr != SCANERR_NONE)
    {
        ErrorCode = 0xE101;
        goto smech_find_home_position_exit;
    }

    // Grab the current state of the home sensor
    home_sense = flatbed_sensor_home_is_found();

    // We can skip the fast home find if we are already sitting on the sensor.
    if (!home_sense)
    {
        // Stage 1: fast home finding (fast, but probably not accurate)

        // Enable home sensor isr actions (stops motor when we hit home)
        home_sensor_find_enable();

        // Move to the home sensor at a faster pace.  This will find the sensor,
        // but may not be very accurate.
        smech_func_debug("stage 1: long move back\n");
        smech_move_rel_nosyncs(
                    fb_motor,
                    MOVE_SPD_NORMAL,
                    FB_MAX_DIST,
                    MOVE_REVERSE,
                    true);

        // Turn off home find isr actions
        home_sensor_find_disable();

        // Check home sensor status
        home_sense = flatbed_sensor_home_is_found();
        smech_func_infor("FB home sensor %s found\n", 
                  (home_sense == true) ? "WAS" : "WAS NOT");

        if (home_sense == false)
        {
            ErrorCode = 0xE400;
            scerr = SCANERR_HARDWARE_FAIL;
            goto smech_find_home_position_exit;
        }
    }
    else
    {
        smech_func_debug("stage 1: skipped, already at home sensor\n");
    }


    // Stage 2: slow home finding

    // Bump out from home position
    smech_func_debug("stage 2: bump out\n");
    scerr = smech_move_rel_nosyncs(
                    fb_motor,
                    MOVE_SPD_AUTO,
                    hinches_to_steps(SCAN_MECH_FLATBED, 100),
                    MOVE_FORWARD,
                    true);
    if (scerr != SCANERR_NONE)
    {
        ErrorCode = 0xE500;
        goto smech_find_home_position_exit;
    }

    // We should NOT be on the home sensor (just moved out)
    home_sense = flatbed_sensor_home_is_found();
    XASSERT(home_sense != true, (uint32_t)home_sense);

    // Enable home sensor isr actions (stops motor when we hit home)
    home_sensor_find_enable();

    // Slow find home sensor
    smech_func_debug("stage 2: move back\n");
    smech_move_rel_nosyncs(
                    fb_motor,
                    MOVE_SPD_SLOW,
                    hinches_to_steps(SCAN_MECH_FLATBED, 200),  // 2"
                    MOVE_REVERSE,
                    true);

    // Check home sensor status
    home_sense = flatbed_sensor_home_is_found();
    smech_func_infor("FB home sensor %s found\n", (home_sense == true) ? "WAS" : "WAS NOT" );

    if (home_sense == false)
    {
        ErrorCode = 0xE700;
        scerr = SCANERR_HARDWARE_FAIL;
        goto smech_find_home_position_exit;
    }

    // Set the home position
    smech_func_debug("set FB home location, steps=%d\n", FB_HOME_POSITION);
    smot_step_set_location(fb_motor, FB_HOME_POSITION);


smech_find_home_position_exit:    

    // Make sure our home sensor isr actions are disabled
    home_sensor_find_disable();

    // Display error if things went badly
    if (ErrorCode != 0x0000)
    {
        smech_func_error("ErrCode=0x%X, scerr=%d\n", ErrorCode, scerr);
    }

    smech_func_leave();
#endif
    return scerr;
}


/**
 *  \brief ADF Clear paper path
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE   Empty input tray operation successful
 *  \retval SCANERR_*      Move failure, failure specific error code
 **/
static scan_err_t smech_adf_clear_paper(void)
{
    scan_err_t scerr = SCANERR_NONE;
#if 0    
    int        firstEmptyPos = 0;
    int        currEmptyPos  = 0;
    int        deltaPos;
    stmotor_move_param_t    *move_params;

    smech_func_enter();

    // ADF check
    if (mechType != SCAN_MECH_ADF)
    {
        smech_func_error("MechType is not ADF\n");
        return SCANERR_INVALID_PARAM;
    }

    // Make sure motor is stopped
    scerr = smech_wait_for_stop(adf_motor);
    if (scerr != SCANERR_NONE)
    {
        smech_func_error("Wait for ADF stop failed, scerr=%d\n", scerr);
        return scerr;
    }

    // Path already clear?  Bail-out!
    if( !adf_sensor_paper_in_path() && !adf_sensor_paper_present())
    {
        smech_func_infor("paper path already clear, bail\n");
        return SCANERR_NONE;
    }

    // Set ADF motor location to a known position
    smot_step_set_location(adf_motor, 0);

    // We typically limit the number of pages to flush, but could run the motor
    // 'forever' if dumping huge stacks of paper is needed.  Sometimes we just
    // clear the first sheet 'in path' and leave remaining sheets staged.

    move_params = get_smot_move_profile(adf_motor,MOVE_SPD_NORMAL);
    smot_step_set_motor_move_params(adf_motor, move_params);

    smot_step_move_rel(adf_motor,
                       (ADF_PAGE_LENGTH * ADF_MAX_PAGES_TO_CLEAR),
                       MOVE_FORWARD,
                       LS_DISABLED,
                       NO_LS_STEPNUM);
    do
    {
        if (adf_sensor_paper_in_path())
        {
            /* Well, things aren't empty yet */
            firstEmptyPos = 0;
            currEmptyPos  = 0;
        }
        else
        {
            currEmptyPos = smot_step_get_location(adf_motor);
            if (firstEmptyPos == 0)
            {
                // We saw the leading edge
                firstEmptyPos = currEmptyPos;
            }

            if (currEmptyPos >= firstEmptyPos)
            {
                deltaPos = currEmptyPos - firstEmptyPos;
            }
            else
            {
                deltaPos = 0;
            }
 
            if (deltaPos >= ADF_EMPTY_THRESH)
            {
                smot_step_request_motor_stop(adf_motor);
                smech_func_debug("path clear at deltaPos=%d\n", deltaPos);
                break;
            }
        }

        msleep_interruptible(10);

    } while (!smot_step_motor_is_idle(adf_motor));


    // Make sure motor is stopped
    scerr = smech_wait_for_stop(adf_motor);

    // NOTE: this simple algorithm does not recover from leaving a page in the
    // path.  Is ADF_MAX_PAGES_TO_CLEAR set properly for your mech?
    if (adf_sensor_paper_in_path())
    {
        smech_func_error("paper left in ADF path\n");
        scerr = SCANERR_PAPER_JAM;
    }
#endif
    return scerr;
}


#if 0 //todo for power-on cal
scan_err_t smech_poweron_cal_run(struct scanvars **psv)
{
    scan_err_t cal_scerr, final_scerr;
    struct scanvars *sv;
    struct scanvars *cal_sv;
    struct scanvars *tmp_sv;

    PTR_ASSIGN(sv, *psv);

    scanvar_push(sv);

    sv = scanvar_get();

    scanvar_calc_pixel_area(sv);

    cal_sv = scanvar_cal_newcopy(sv);

    scanvar_push(cal_sv);


    // hardcode sensor 0 (sensor_bitmap) to do poweron cal
    cal_scerr = calibrate(0x01, cal_sv->hw_dpi_horiz, cal_sv->cmode, true);

    tmp_sv = scanvar_pop();
    XASSERT( tmp_sv==cal_sv, tmp_sv->id );
    tmp_sv = NULL;

    scanvar_delete(&cal_sv);

    if (cal_scerr != SCANERR_NONE)
    {
        final_scerr = cal_scerr;
        goto smech_poweron_cal_run_exit;
    }

    final_scerr = SCANERR_NONE;

smech_poweron_cal_run_exit:

    if (scanvar_peek())
    {
        tmp_sv = scanvar_pop();
        XASSERT( tmp_sv==sv, tmp_sv->id );
        tmp_sv = NULL;
        PTR_ASSIGN( *psv, sv );
    }

    XASSERT( scanvar_peek()==NULL, scanvar_peek()->id );
    return final_scerr;
}


scan_err_t smech_poweron_cal(scan_cmode_t cmode, int dpi)
{
    struct scanvars *sv;
    scan_err_t scerr;

    if (cmode==SCAN_CMODE_MONO)
    {
        sv = scanplat_sv_mono_scan_new();
    }
    else
    {
        sv = scanplat_sv_color_scan_new();
    }
    if (sv==NULL)
    {
        scerr = SCANERR_OUT_OF_MEMORY;
        goto smech_poweron_cal_exit;
    }

    scerr = scanvar_set_dpi(sv, dpi);
    if (scerr != SCANERR_NONE)
    {
        goto smech_poweron_cal_exit;
    }

    scerr = smech_poweron_cal_run(&sv);

    ASSERT(sv!=NULL);

smech_poweron_cal_exit:

    scanvar_delete(&sv);
    return scerr;
}
#endif


/**
 *  \brief Get Line Start Incr (Helper)
 *
 **/
static uint16_t get_ls_incr(t_ScanMechType mech_type, scan_cmode_t cmode, uint32_t hw_dpi_vert)
{
    uint16_t ls_incr = 0;

    if (mechType == SCAN_MECH_FLATBED)
    {
        // The flatbed only supports 300 and 600 dpi
        switch (hw_dpi_vert)
        {
            case 300:
                ls_incr = (cmode == SCAN_CMODE_COLOR) ? FB_LSINCR_COLO_300 : FB_LSINCR_MONO_300;
                break;
            case 600:
                ls_incr = (cmode == SCAN_CMODE_COLOR) ? FB_LSINCR_COLO_600 : FB_LSINCR_MONO_600;
                break;
            default:
                XASSERT(0, hw_dpi_vert); // dpi not supported
                break;
        }

    }
    else
    {
        // The ADF only supports 300dpi
        switch (hw_dpi_vert)
        {
            case 300:
                ls_incr = (cmode == SCAN_CMODE_COLOR) ? ADF_LSINCR_COLO_300 : ADF_LSINCR_MONO_300;
                break;
            default:
                XASSERT(0, hw_dpi_vert); // dpi not supported
                break;
        }
    }

    return ls_incr;
}


/**
 *  \brief SCIF motor setup (Helper)
 *
 *  Configure SCIF motor settings
 *
 **/
static void smech_scif_mtr_setup(t_ScanMechType mech_type, MOTOR_SYNC motor_sync)
{
    const struct scanvars *scanvar;
    uint16_t     ls_incr;

    if (motor_sync != SYNC_DISABLE)
    {
        scif_motor_setup(get_stmotor_scif(mech_type), motor_sync, MOT_EXTERNAL_SYNC, MOT_EXTERNAL_SYNC);
        
        scanvar  = scanvar_peek();
        XASSERT(scanvar != NULL, (uint32_t)scanvar);

        ls_incr  = get_ls_incr(mech_type, scanvar->cmode, scanvar->hw_dpi_vert);
    }
    else
    {
        scif_motor_setup(get_stmotor_scif(mech_type), motor_sync, MOT_INTERNAL_SYNC, MOT_INTERNAL_SYNC);

        ls_incr = 0;
    }

    // Setup the line starts
    set_stmotor_ls_source(mech_type);
//    smot_step_set_line_start(get_motor_id_from_mech(mech_type), LS_SRC_PWM_M, ls_incr);
}


/**
 * \brief  Move the scan bar to the scan start position
 *
 *  A flatbed move operation to position the scan bar for normal scan: the mech
 *  code will be in control of blocking until the move is complete.
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE   Move complete, no error
 *  \retval SCANERR_*      Move failure,  specific error code
 **/
static scan_err_t smech_fb_move_to_start(void)
{
	
    scan_err_t scerr = SCANERR_NONE;
    
#if 0    
    long int x_thinch, y_thinch, width, height;
    const struct scanvars *sv;
    int scan_start_steps;

    smech_func_enter();

    // Flatbed check: this operation only makes sense on flatbed
    if (mechType != SCAN_MECH_FLATBED)
    {
        smech_func_error("MechType is not flatbed\n");
        return SCANERR_INVALID_PARAM;
    }

    // Special case: start scan from home position
    if ( scan_get_config() & SCAN_CONFIG_SCAN_FROM_HOME )
    {
        scerr = smech_move_to_position(FB_HOME_POSITION, true);
        return scerr;
    }

    // Ask system for starting location of the scan (in 1/1000s of an inch)
    sv = scanvar_peek();
    scanvar_get_area(sv, &x_thinch, &y_thinch, &width, &height);

    // The scan start will be: fb start of glass, minus ramp distance, plus offset
    scan_start_steps = FB_TOP_BED_POSITION - 
                       FB_RAMP_STEPS + 
                       thinches_to_steps(mechType, y_thinch);

    // Sanity check to make sure we haven't defined a negative starting point
    XASSERT(scan_start_steps >= FB_HOME_POSITION, scan_start_steps);

    scerr = smech_move_to_position(scan_start_steps, true);

    // Debug: where are we?
    smech_func_debug("scan_start_steps = %d\n", scan_start_steps);
    smech_func_debug("actual start pos = %d\n", smot_step_get_location(fb_motor));

    smech_func_leave();
#endif    
    return scerr;
}


/**
 * \brief  Rewind the scan bar after a pause
 *
 *  A flatbed move operation to position the scan bar 'ramp steps' before
 *  the point the last scan line was captured.
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE   Move complete, no error
 *  \retval SCANERR_*      Move failure,  specific error code
 **/
static scan_err_t smech_fb_pause_rewind(void)
{
    scan_err_t scerr = SCANERR_NONE;
#if 0
    int        targetstep;

    /*
     * There are two easy ways to calculate the needed rewind position:
     *  - use the motor location marker set in smech_granum2_start_pause.  This
     *    should be very close to right, but may have some slop.
     *
     *  - use the number of rows actually scanned and calculate the position from
     *    the start of the scan. This will only work if your motor speed is 100%
     *    correct (no stretch/shrink, even a 1% error will cause trouble). 
     *
     *    Getting the steps/rows scanned (avoiding rounding errors):
     *      int  scale_factor = 1000;
     *      smech_scan_cmdq_row_status = scan_cmdq_get_row_status();
     *      smech_completed_rows = smech_scan_cmdq_row_status.rs_completed_rows;
     *      scanned_steps = hinches_to_steps(SCAN_MECH_FLATBED, ((smech_completed_rows * 100 * scale_factor) / sv->hw_dpi_vert));
     *      scanned_steps /= scale_factor;
     *
     *    Remembering the start of scan was:
     *      sv = scanvar_peek();
     *      scanvar_get_area(sv, &x_thinch, &y_thinch, &width, &height);
     *      scan_start_steps = FB_TOP_BED_POSITION - 
     *                         FB_RAMP_STEPS + 
     *                         thinches_to_steps(mechType, y_thinch);
     *
     *    Then you would start at:
     *      targetstep = scan_start_steps + scanned_steps;
     *
     */

    smech_func_enter();

    // Flatbed check: this operation only makes sense on flatbed
    if (mechType != SCAN_MECH_FLATBED)
    {
        smech_func_error("MechType is not flatbed\n");
        return SCANERR_INVALID_PARAM;
    }

    // Sanity check
    XASSERT(smech_pause_rewind_enable, smech_pause_rewind_enable);

    // Use the motor location marker method, back up smech_sync_start_step (ramp)
    targetstep = smech_pause_mtr_loc - smech_sync_start_step;

    // Some debug output
    smech_func_debug("smech_pause_mtr_loc  = %d\n", smech_pause_mtr_loc);
    smech_func_debug("smech_sync_start_step= %d\n", smech_sync_start_step);
    smech_func_debug("currentstep          = %d\n", smot_step_get_location(fb_motor));
    smech_func_debug("targetstep           = %d\n", targetstep);

    // Do the rewind, block until it completes
    scerr = smech_move_to_position(targetstep, true);
#endif
    return scerr;
}


/**
 *  \brief Lift ADF Pick Roller Helper
 *
 *  Helper function to lift the ADF pick roller.
 */ 
static scan_err_t smech_adf_lift_pick_roller(void)
{
    scan_err_t scerr;

    // Lift roller, wait for move to complete
    scerr = smech_move_rel_nosyncs(
                    adf_motor,
                    MOVE_SPD_NORMAL,
                    ADF_LIFT_PICK_ROLLER,
                    MOVE_REVERSE,
                    true);

    return scerr;
}


/**
 *  \brief ADF Page Prepare Helper
 *
 *  Helper function to prepare the mechanism for an ADF scan.
 *
 *  Note that we have both hard and soft failures here.  Soft failures can
 *  be recovered (ADF mispick), while hard failures indicate a serious
 *  hardware problem.
 *
 *  \param[in] scan_mech    Pointer to mech data structure
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE           ADF page prepare passed
 *  \retval SCANERR_HARDWARE_FAIL  Motor move failed, scanner is stuck/dead
 *  \retval SCANERR_SCAN_CANCELLED Jam/mispick, cancel
 *  \retval SCANERR_NO_PAPER_EDGE  Never saw the edge
 **/
static scan_err_t smech_adf_page_prepare(t_ScanMech *scan_mech)
{
    scan_err_t scerr = SCANERR_NONE;
	mechType = SCAN_MECH_ADF;
	dbg1("RICOH:smech_adf_page_prepare \n");


#if 0
    // Bail out if ADF cover is open
    if (adf_sensor_cover_is_open())
    {
        smech_func_error("ADF cover is open\n");
        return SCANERR_SCAN_CANCELLED;
    }

    // Bail out if paper already in path
    if (adf_sensor_paper_in_path())
    {
        smech_func_error("paper was already in ADF path\n");
        return SCANERR_PAPER_JAM;
    }

    // Bail out if ADF doesn't have paper loaded
    if (!adf_sensor_paper_present())
    {
        smech_func_error("please load paper in ADF first\n");
        return SCANERR_SCAN_CANCELLED;
    }

    // Move the scanbar under the ADF scanning area.
    mechType = SCAN_MECH_FLATBED;
    scerr = smech_move_to_position(FB_DFLT_ADF_POSITION, true);
    if (scerr != SCANERR_NONE)
    {
        smech_func_error("FB move to ADF position failed\n");
        return SCANERR_SCAN_CANCELLED;
    }

    // Switch to ADF.  
    // NOTE: do NOT stage the first sheet here, the core scan code will assert
    //       if you do ...
#endif
 
    mechType = SCAN_MECH_ADF;

    return scerr;
}
 

/**
 *  \brief Page prepare (Mech API)
 *
 *  Mechanism API function to prepare a page for scanning.  This step must
 *  put the mech in a state to do the requested scan.  For example:
 *   - a calibration scan needs to be positioned under the cal target
 *   - a flatbed scan needs the scanbar to be staged in the proper position
 *   - an ADF scan needs the paper to be positioned at Top Of Form
 *
 *  If we can't get the mech ready an error must be returned so we can unwind.
 *
 *  \warning
 *  Preparing for one type of scan may require us to do another type of scan, 
 *  which will effectively cause this routine to be re-entered.
 *
 *  \param[in] scan_mech         Pointer to mech data structure
 *  \param[in] scan_target_type  Type of scan to prepare for
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE          Page prepare passed
 *  \retval SCANERR_HARDWARE_FAIL Motor move failed, scanner is stuck/dead
 **/
static scan_err_t smech_granum2_page_prepare(t_ScanMech* scan_motor, scan_target_type_t scan_target_type)
{
    scan_err_t scerr = SCANERR_NONE;
    int scan_start_steps;

	dbg1("RICOH:smech_granum2_page_prepare \n");
    smech_func_enter();

    current_scan_target_type  = scan_target_type;
    smech_pause_rewind_enable = false;

    // TODO: if your mech code maintains error state (motor failure, jam, etc),
    //       make sure to check it someplace in this prepare function and return
    //       failure as needed.

    switch(scan_target_type)
    {
        case SCAN_TARGET_TYPE_STATIONARY_CAL:
            // Stationary cal needs the scanbar right under the cal target.
            mechType = SCAN_MECH_FLATBED;
            scerr = smech_move_to_position(FB_CAL_POSITION, true);
            break;

#ifdef OVERSCAN_CAL_REQUIRED
            // variable location for the calibration strip - do overscan to find it
        case SCAN_TARGET_TYPE_OVERSCAN_CAL:
            // Moving cal needs the scanbar ramp steps away from where we start scanning
            mechType = SCAN_MECH_FLATBED;
 
            // don't worry about backing up the starting position based on the ramp distance
            // this is overscan cal - it's permitted
            scan_start_steps = FB_OVERSCAN_CAL_POSITION;
            // Sanity check to make sure we haven't defined a negative starting point
            XASSERT(scan_start_steps >= FB_HOME_POSITION, scan_start_steps);

            scerr = smech_move_to_position(scan_start_steps, true);
            break;
#else
        case SCAN_TARGET_TYPE_MOVING_CAL:
            // Moving cal needs the scanbar ramp steps away from the cal target.
            mechType = SCAN_MECH_FLATBED;
 
            // Back up the starting position based on the ramp distance
            scan_start_steps = FB_MV_CAL_POSITION - FB_RAMP_STEPS;

            // Sanity check to make sure we haven't defined a negative starting point
            XASSERT(scan_start_steps >= FB_HOME_POSITION, scan_start_steps);

            scerr = smech_move_to_position(scan_start_steps, true);
            break;

#endif            
            
        case SCAN_TARGET_TYPE_DOC:
            if (isDocSourceADF())
            {
                // Call a helper to prepare the ADF (move bar to adf, stage paper, etc).
                // Leading edge of the paper should be at the TOF sensor ...
                scerr = smech_adf_page_prepare(scan_motor);
            }
            else
            {
                // Get the scanbar in position (which depends on the scan area).
                // Should be ramp steps away from start of scan area ...
                mechType = SCAN_MECH_FLATBED;
//                scerr = smech_fb_move_to_start();
            }
            break;

        default:
            // Note: This mech does not support notch find or the legacy cal type
            XASSERT(0, scan_target_type);
            break;
    }

    if (scerr != SCANERR_NONE)
    {
        smech_func_error("ErrCode = %d\n", scerr);
        current_scan_target_type = SCAN_TARGET_TYPE_NONE;
    }

    smech_func_leave();
    scerr = SCANERR_NONE;
    return scerr;
}


/**
 *  \brief Page setup (Mech API)
 *
 *  Mechanism API function to setup for scanning. At this point the page prepare
 *  operation has completed and we need to set the motor parameters and row sync
 *  start position for the actual scan.
 *
 *  \param[in] scan_mech         Pointer to mech data structure
 *
 **/
static void smech_granum2_scan_setup(t_ScanMech* scan_motor)
{
    const struct scanvars *scanvar;

    smech_func_enter();

    XASSERT(current_scan_target_type!=SCAN_TARGET_TYPE_NONE,
            current_scan_target_type);

    // Default settings
    smech_profile_index   = 0;
    smech_motor_sync      = SYNC_DISABLE;
    smech_sync_start_step = 0;

    scanvar = scanvar_peek();

    
    dbg1("RICOH:smech_granum2_scan_setup \n");

    switch (current_scan_target_type)
    {
        case SCAN_TARGET_TYPE_STATIONARY_CAL:
                smech_motor_sync      = SYNC_DISABLE;
                smech_sync_start_step = 0;
                smech_profile_index   = get_stmotor_profile_index(scanvar->cmode, scanvar->hw_dpi_vert);
                smech_scif_mtr_setup(mechType, smech_motor_sync);
            break;

#ifdef OVERSCAN_CAL_REQUIRED
            // variable location for the calibration strip - do overscan to find it
       case SCAN_TARGET_TYPE_OVERSCAN_CAL:
                smech_motor_sync       = SYNC_FIRST;
                smech_sync_start_step  = 0; // allow scanning while ramping for overscan
                smech_profile_index    = get_stmotor_profile_index(scanvar->cmode, scanvar->hw_dpi_vert);

                smech_scif_mtr_setup(mechType, smech_motor_sync);
            break;
#else
        case SCAN_TARGET_TYPE_MOVING_CAL:
                smech_motor_sync       = SYNC_FIRST;
                smech_sync_start_step  = FB_RAMP_STEPS;
                smech_profile_index    = get_stmotor_profile_index(scanvar->cmode, scanvar->hw_dpi_vert);

                smech_scif_mtr_setup(mechType, smech_motor_sync);
            break;

#endif
            
        case SCAN_TARGET_TYPE_DOC:
            if (mechType == SCAN_MECH_ADF)
            {
                scanvar_set_flip_horizontal(scanvar_get(), 0, true);   // flip front

                // This mech is not dual sensor, but if it was you could set the 
                // back side flip here ...
                scanvar_set_flip_horizontal(scanvar_get(), 1, false);  // don't flip back

                // Set ADF motor location to a known position
#if 0
                // smot_step_set_location(get_motor_id_from_mech(mechType), hinches_to_steps(mechType, 0));
#endif                

                smech_motor_sync       = SYNC_DISABLE;
                smech_sync_start_step  = ADF_RAMP_STEPS;
                smech_profile_index    = get_stmotor_profile_index(scanvar->cmode, scanvar->hw_dpi_vert);

                smech_scif_mtr_setup(mechType, smech_motor_sync);
            }
            else
            {
                // No flip on flatbed sensor
                scanvar_set_flip_horizontal(scanvar_get(), 0, false);

                smech_motor_sync       = SYNC_FIRST;
//				smech_motor_sync       = SYNC_DISABLE;	
                smech_sync_start_step  = FB_RAMP_STEPS;
                smech_profile_index    = get_stmotor_profile_index(scanvar->cmode, scanvar->hw_dpi_vert);

                if ( scan_get_config() & SCAN_CONFIG_SCAN_FROM_HOME )
                {
                    smech_sync_start_step = 0;
                }

                smech_scif_mtr_setup(mechType, smech_motor_sync);
            }
            break;

        default:
            // Note: This mech does not support notch find or the legacy cal type
            XASSERT(0, current_scan_target_type);
            break;
    }

	switch ( scanvar->user_area_thinch.height ) {
		/* read AGC, and shading (black) data */
	case 80:
	case 130:
	case 160:
	case 260:
		smech_scif_mtr_setup(SCAN_MECH_FLATBED, SYNC_DISABLE);
		break;
	case 81:
	case 161:  /* read shading white data (moving cal) */
		// No flip on flatbed sensor
		scanvar_set_flip_horizontal(scanvar_get(), 0, false);
		
		smech_motor_sync       = SYNC_FIRST;
		//				smech_motor_sync       = SYNC_DISABLE;	
		smech_sync_start_step  = FB_RAMP_STEPS;
		smech_profile_index    = get_stmotor_profile_index(scanvar->cmode, scanvar->hw_dpi_vert);
		
		if ( scan_get_config() & SCAN_CONFIG_SCAN_FROM_HOME )
			{
				smech_sync_start_step = 0;
			}
		smech_scif_mtr_setup(mechType, smech_motor_sync);
		break;
	}

    smech_func_debug("smech_motor_sync = %d, smech_sync_start_step = %d\n", smech_motor_sync, smech_sync_start_step);
}


/**
 *  \brief Page run (Mech API)
 *
 *  Mechanism API function to run a scan. At this point the page prepare
 *  and setup operations have completed, now we need to start the motors.
 *
 *  \param[in] scan_mech         Pointer to mech data structure
 *
 **/
static void smech_granum2_scan_start(t_ScanMech* scan_motor)
{
    bool ls_enable;
    stmotor_move_param_t *move_params;

    smech_func_enter();

	dbg1("RICOH:smech_granum2_scan_start \n");

    // Wait for any previous move to complete
//    smech_wait_for_stop(get_motor_handle_from_mech(mechType));

    smech_set_status(SCAN_MECH_RUNNING);

    // Do we need line syncs?
    ls_enable = (smech_motor_sync == SYNC_DISABLE) ? LS_DISABLED : LS_ENABLED;
#if 0
    switch (current_scan_target_type)
    {
        case SCAN_TARGET_TYPE_STATIONARY_CAL:
            // This "move" is stationary. Stay put.
            break;

#ifdef OVERSCAN_CAL_REQUIRED
            // variable location for the calibration strip - do overscan to find it
        case SCAN_TARGET_TYPE_OVERSCAN_CAL:

            move_params = get_smot_move_profile(fb_motor, smech_profile_index);
            smot_step_set_motor_move_params(fb_motor, move_params );

            smot_step_move_abs(fb_motor,
                               FB_TOP_BED_POSITION + FB_EXTRA_STEPS_FOR_SYNC_PULSES,
                               ls_enable,
                               smech_sync_start_step);
            break;
#else
        case SCAN_TARGET_TYPE_MOVING_CAL:

            move_params = get_smot_move_profile(fb_motor, smech_profile_index);
            smot_step_set_motor_move_params(fb_motor, move_params );

            smot_step_move_abs(fb_motor,
                               FB_TOP_BED_POSITION + FB_EXTRA_STEPS_FOR_SYNC_PULSES,
                               ls_enable,
                               smech_sync_start_step);
            break;
#endif
            
        case SCAN_TARGET_TYPE_DOC:
            if (mechType == SCAN_MECH_ADF)
            {
                if (smech_pause_rewind_enable)
                {
                    // The ADF doesn't support pause during a page, only in the
                    // page gaps.  Nothing special to do if we are resuming ...
                    smech_pause_rewind_enable = false;
                }

                adf_sensor_pip_callback_enable();
                adf_sensor_cio_callback_enable();

                move_params = get_smot_move_profile(adf_motor,smech_profile_index);
                smot_step_set_motor_move_params(adf_motor, move_params );

                smot_step_move_rel(adf_motor,
                                   ADF_SCAN_DIST,
                                   MOVE_FORWARD,
                                   ls_enable,
                                   smech_sync_start_step);
            }
            else
            {
                if (smech_pause_rewind_enable)
                {
                    // We are resuming a flatbed pause, so we need to rewind the
                    // scan bar to 'ramp steps' before the point we stopped 
                    // capturing scan data.
                    smech_fb_pause_rewind();
                    smech_pause_rewind_enable = false;
                }
                move_params = get_smot_move_profile(fb_motor, smech_profile_index);
                smot_step_set_motor_move_params(fb_motor, move_params );

                smot_step_move_abs(fb_motor,
                                   FB_BOT_BED_POSITION + FB_EXTRA_STEPS_FOR_SYNC_PULSES,
                                   ls_enable,
                                   smech_sync_start_step);
 
            }
            break;

        default:
            // Note: This mech does not support notch find or the legacy cal type
            XASSERT(0, current_scan_target_type);
            break;
    }
#endif    
}


/**
 *  \brief Page done (Mech API)
 *
 *  Mechanism API function to do page done processing.  In general this
 *  routine should leave the scanbar where it started before the scan, which
 *  is usually the PP position.  Multi-page ADF scans are an execption as we
 *  need to scan additional pages.
 *
 *  \param[in] scan_mech         Pointer to mech data structure
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE           ADF page prepare passed
 *  \retval SCANERR_HARDWARE_FAIL  Motor move failed, scanner is stuck/dead
 *  \retval SCANERR_SCAN_CANCELLED Jam/mispick, cancel
 **/
static scan_err_t smech_granum2_page_done(t_ScanMech* scan_motor, smech_done_flags_t flags)
{
    bool                   AreWeCancelling;
    scan_err_t             scerr = SCANERR_NONE;

		dbg1("RICOH:smech_granum2_page_done \n");

    smech_func_enter();

#if 0
    AreWeCancelling = false;
    if (flags == SMECH_DONE_FLAG_CANCEL)
    {
        smech_func_infor("flags == SMECH_DONE_FLAG_CANCEL\n");
        AreWeCancelling = true;
    }

    if (AreWeCancelling && (current_scan_target_type == SCAN_TARGET_TYPE_NONE))
    {
        return scerr;
    }

    // Wait for any previous move to complete
    smech_wait_for_stop(get_motor_handle_from_mech(mechType));

    switch(current_scan_target_type)
    {
        case SCAN_TARGET_TYPE_STATIONARY_CAL:
            // This scan didn't move, so just leave the scanbar in the cal position.
            // Will save a lot of moving during cal ... the next prepare
            // operation may move elsewhere.
            // Unless we are cancelling, then we might as well move home as the cal
            // operation is toast.
            if (AreWeCancelling)
            {
                // Rewind to home
                scerr = smech_move_to_home();
            }
            break;

#ifdef OVERSCAN_CAL_REQUIRED
        case SCAN_TARGET_TYPE_OVERSCAN_CAL:
            // Rewind to home
            scerr = smech_move_to_home();
            break;

#else
        case SCAN_TARGET_TYPE_MOVING_CAL:
            // Rewind to home
            scerr = smech_move_to_home();
            break;
#endif            
        case SCAN_TARGET_TYPE_DOC:
            if (mechType == SCAN_MECH_ADF)
            {
                if (AreWeCancelling)
                {
                    // Clear out the paper path
                    scerr = smech_adf_clear_paper();
                }

                // Make sure the pick roller is up
                smech_adf_lift_pick_roller();

                // Set ADF motor location to a known position
                smot_step_set_location(adf_motor, hinches_to_steps(SCAN_MECH_ADF, 0));

                adf_sensor_cio_callback_disable();
                adf_sensor_pip_callback_disable();
            }

            // The flatbed always goes back to home
            mechType = SCAN_MECH_FLATBED;
            scerr = smech_move_to_home();
            break;

        default:
            // Note: This mech does not support notch find or the legacy cal type
            smech_func_error("unexpected scan target %d\n", current_scan_target_type);
            XASSERT(0, current_scan_target_type);
            break;
    }

    current_scan_target_type = SCAN_TARGET_TYPE_NONE;
#endif

    smech_set_status(SCAN_MECH_READY);

    if (scerr != SCANERR_NONE)
    {
        smech_func_error("ErrCode = %d\n", scerr);
    }

    return scerr;
}


/**
 *  \brief Start a blocking motor stop (Mech API)
 *
 *  Mechanism API function to start a motor stop, the caller will be blocked
 *  until the motor comes to a complete stop. Must not be called from IRQ
 *  context!
 *
 *  \param[in] scan_mech         Pointer to mech data structure
 **/
static void smech_granum2_blocking_stop(t_ScanMech* scan_motor)
{

	dbg1("RICOH:smech_granum2_blocking_stop \n");
    smech_func_enter();
#if 0    
	stmotor_t *motor_handle = get_motor_handle_from_mech(mechType);
    smot_step_request_motor_stop(motor_handle);
    smech_wait_for_stop(motor_handle);
    
#endif    
    smech_set_status(SCAN_MECH_READY);
}


/**
 *  \brief Start an async motor stop (Mech API)
 *
 *  Mechanism API function to start a motor stop, the caller will not hang
 *  around and wait for the stop to complete (they will watch for the mech
 *  status to go to ready).
 *
 *  \param[in] scan_mech         Pointer to mech data structure
 **/
static void smech_granum2_async_stop(t_ScanMech* scan_motor)
{
		dbg1("RICOH:smech_granum2_async_stop \n");
		
		smech_set_status(SCAN_MECH_READY);
#if 0	
    stmotor_t *motor_handle = get_motor_handle_from_mech(mechType);

    smech_func_enter();

    // Tell the motor to stop: trigger will fire when motor is actually stopped
    smot_step_add_trigger(motor_handle, TRIG_MOTOR_STOP, 0, smech_trig_scan_stopped, NULL);
    smot_step_request_motor_stop(motor_handle);
#endif
}


/**
 *  \brief Emergency motor stop (Mech API)
 *
 *  Mechanism API function to do an emergency stop.  The motor driver will NOT
 *  allow any motor moves after this call, the system must restart.
 *
 *  \param[in] scan_mech         Pointer to mech data structure
 **/
static void smech_granum2_halt(t_ScanMech* scan_motor)
{
	dbg1("RICOH:smech_granum2_halt \n");	
#if 0	
    smech_func_enter();

    scanplat_kernel_gpio_set_value(granum2_mech.scan_fb_motor_driver_nslp,  0);
    scanplat_kernel_gpio_set_value(granum2_mech.scan_adf_motor_driver_nslp, 0);
    smot_step_emergency_halt(fb_motor);
    smot_step_emergency_halt(adf_motor);
#endif
}


/**
 *  \brief Start a motor pause (Mech API)
 *
 *  Mechanism API function to start a motor pause.  Note that while pause/resume
 *  is only supported on the flatbed, this routine is also used to stop the motor
 *  for our ADF rest stops (done to prevent encoder rollover or motor overheat).
 *
 *  \param[in] scan_mech         Pointer to mech data structure
 **/
static void smech_granum2_start_pause(t_ScanMech* scan_motor)
{
		dbg1("RICOH:smech_granum2_start_pause \n");	
	
#if 0	
    stmotor_t *motor_handle = get_motor_handle_from_mech(mechType);

    /*
     * Warning!
     *
     * Can be called from interrupt context.
     */
    smech_func_enter();

    // Grab the current location
    smech_pause_mtr_loc = smot_step_get_location(motor_handle);

    // Tell the motor to stop: trigger will fire when motor is actually stopped
    smot_step_add_trigger(motor_handle, TRIG_MOTOR_STOP, 0, smech_trig_paused_stopped, NULL);
    smot_step_request_motor_stop(motor_handle);
#endif
}

/**
 *  \brief Scan mech self test
 *
 *  Do a quick test of the mech motors and sensors.
 **/
static scan_err_t smech_granum2_selftest(t_ScanMech* scan_motor)
{
    scan_err_t scerr = SCANERR_NONE;
#if 0    
    #if 0 //todo for power-on cal
    bool       scan_poweron_cal_enable = false;
    #endif

    smech_func_enter();

    // Find home
    mechType = SCAN_MECH_FLATBED;
    scerr = smech_find_home_position();
    if (scerr != SCANERR_NONE)
    {
        smech_func_error("ErrCode = %d\n", scerr);
        return scerr;
    }

    #if 0 //todo for power-on cal
    //scan_poweron_cal_enable = true;
    if (scan_poweron_cal_enable)
    {
        scerr = smech_poweron_cal(SCAN_CMODE_COLOR, 300);
        if (scerr != SCANERR_NONE)
        {
            return scerr;
        }

        scerr = smech_poweron_cal(SCAN_CMODE_MONO, 300);
        if (scerr != SCANERR_NONE)
        {
            return scerr;
        }

        scerr = smech_move_to_home();
        if (scerr != SCANERR_NONE)
        {
            return scerr;
        }
    }
    #endif

    // Make sure the pick roller is up
    smech_adf_lift_pick_roller();

    // Set the ADF motor location to a known position
    smot_step_set_location(adf_motor, 0);
#endif
    return scerr;
}


/**
 * \brief Called by the core code when a SMSG_SCAN_MECH_FAULT message is
 *        received.
 *
 * This function and SMSG_SCAN_MECH_FAULT are only used by the mech driver
 * to transition from the mech driver's interrupt context to a thread
 * context. The mech code simply can't do all needed fault processing in
 * interrupt context (bad things will happen).
 *
 * mech driver                      scanpipe
 * -----------                      --------
 *  isr: detect fault!
 *  send SMSG_SCAN_MECH_FAULT --->  message queued (with fault codes)
 *
 *                                  thread: message loop gets SMSG_SCAN_MECH_FAULT  
 *              <----------------   call smech_fault_msg()
 *  thread: handle fault and
 *    return error code---------->  cancel scan if error return code
 *
 * Core code has no idea what's in the message or what it means. It's a
 * message from the mech driver to the mech driver. Passed directly down to the
 * mech driver once received by the core message processing loop.
 *
 * If this routine returns an error, the core scan code will cancel the current
 * scan.
 *  
 **/
static scan_err_t smech_granum2_fault_msg(t_ScanMech *scan_mech, scan_msg_t *msg)
{
    scan_err_t      scerr = SCANERR_NONE;
    t_ScanMechType  MechType;
    uint16_t        ErrorCode;

    // NOTE: we are now running in THREAD space.

    smech_func_enter();

    XASSERT(scan_mech != NULL, (uint32_t)scan_mech);
    XASSERT(msg != NULL,       (uint32_t)msg);

    // Grab our fault parameters from the scan message (remember that we sent
    // them to ourself from interrupt context)
    ErrorCode = msg->param1;
    MechType  = msg->param2;

    smech_func_infor("ErrorCode=%d, MechType=%d\n", ErrorCode, MechType);

    // NOTE: The code in this function is only a stub, you must implement the
    // desired behavior for your product.

    // Any emergency actions (like stopping motors) should have already been 
    // handled in interrupt context.

    if (MechType == SCAN_MECH_ADF)
    {
        if (ErrorCode == SMECH_ADF_ERROR_CIO)
        {
            // Cover is open, return error to cancel scan
            scerr = SCANERR_PAPER_JAM;

            // TODO: add product specific actions here. For example: set mech
            //       code internal error state (cio/jam flag), notify control
            //       panel, etc.
        }
        else if (ErrorCode == SMECH_ADF_ERROR_JAM)
        {
            // ADF jam, return error to cancel scan
            scerr = SCANERR_PAPER_JAM;

            // TODO: add product specific actions here. For example: set mech
            //       code internal error state (jam flag), notify control
            //       panel, etc.
        }
        else
        {
            // Unsupported error code, consider asserting here.
            smech_func_error("Unsupported ADF error code (%d)\n", ErrorCode);
        }
    }
    else
    {
        // Unsupported error code, consider asserting here.
        smech_func_error("Unsupported flatbed error code (%d)\n", ErrorCode);
    }

    return scerr;
}


/**
 * \brief Called by the core code into the mech driver to force a mech fault.
 * 
 *  If you get here, the common scan code has detected a problem with the current
 *  scan and is notifying the mech code (in thread context) so it can clean up
 *  and complete any needed product specific actions.
 *
 *  No matter what you do here (or what error code you return), the common
 *  scan code will cancel the current scan as soon as this function returns.
 *
 *  Think of this as a top down scan failure, detected by the core code.  
 *  The logic in smech_granum2_fault_msg is meant to handle bottom up scan
 *  failures, which are failures detected by the mech code.
 *
 **/
static scan_err_t smech_granum2_force_failure(t_ScanMech *scan_mech, smech_failure_t fail_code)
{
    smech_func_enter();
#if 0

    // As of this writing, the only error forced by the core scan code is an
    // ADF jam.  The code in this function is only a stub, you must implement the
    // desired behavior for your product.

    if (fail_code == SMECH_FAILURE_ADF_PAPER_JAM)
    {
        // Something has gone wrong with the ADF scan. This usually happens when the
        // core code is in a page (it has seen TOF), but the BOF has not arrived in
        // a pre-defined number of scan rows. It assumes a jam condition ...
        
        // Make sure the ADF motor is stopped
        smot_step_request_motor_stop(adf_motor);
        smech_wait_for_stop(adf_motor);

        // TODO: add product specific actions here. For example: set mech code
        //       internal error state (jam flag), notify control panel, etc.
    }
    else
    {
        // This isn't good.  What is the core code complaining about and how do we
        // handle it? Consider asserting here.
        smech_func_error("Unknown fail_code (%d)\n", fail_code);
    }
#endif
    return SCANERR_NONE;
}


/**
 *  \brief Set mech sleep mode
 *
 *  Scan power may be going away or coming back on: make sure the mech is
 *  ready.
 *
 *  \param sleep_mode - true: enter sleep mode, false: exit sleep mode
 *
 *  \return scan_err_t
 *  \retval SCANERR_NONE   Sleep mode set, no errors
 **/
static scan_err_t smech_granum2_sleep(t_ScanMech* scan_motor, bool sleep_mode)
{
    return SCANERR_NONE;
}


/**
 * \brief Return true/false if the motor is overheating.
 *
 * Only used during ADF scans. When we scan ADF, we start pulling paper and
 * don't pause between pages (see the pipecutter). Consequently, the motor will
 * run for a large duty cycle. 
 *
 * If the motor is overheating, the core code will pause between pages to
 * reduce the duty cycle.
 *
 * Usual method is to track duty cycle, return TRUE when duty cycle exceeds a
 * certain level.
 *  
 **/
static bool smech_granum2_is_motor_overheating(t_ScanMech *scan_motor)
{
    // Just report not overheating.  Implement if needed ...
    return false;
}


static scan_err_t smech_granum2_ioctl(t_ScanMech *scan_mech, uint32_t request, int arg, void *ptr)
{
    scan_err_t scerr;
    struct     scan_adf_margins        *adf_margins = NULL;
    struct     scan_flatbed_margins    *fb_margins;
    struct     scanmech_capabilities   *mech_cap;

    /*
     *   Note! Note! Note! 
     *
     *   This function can be called from any thread context!  Do not assume
     *   the caller is running in scantask thread context.
     *
     *   Dangerous or long running operations (such as motor moves) will be
     *   triggered via a SMIRB (see scanman_smirb.c). Other operations (such as
     *   getting/setting margins) can be called from any thread context.
     *
     */

    smech_func_enter();

    switch( request ) 
    {
    case SCANMECH_IOCTL_GET_MECH_CAPABILITIES:
        dbg2("%s: SCANMECH_IOCTL_GET_MECH_CAPABILITIES\n", __FUNCTION__);
        if(!ptr)
        {
            return SCANERR_INVALID_PARAM;
        }
        mech_cap = (struct scanmech_capabilities *)ptr;
        if( mech_cap->version != sizeof(struct scanmech_capabilities) )
        {
            /* I don't know how to handle this structure's version */
            return SCANERR_INVALID_PARAM;
        }

        memcpy( mech_cap, &scanmech_granum2_capabilities, 
                sizeof(struct scanmech_capabilities)  );
        break;
        
    case SCANMECH_IOCTL_MOVE_TO_CAL :
        return smech_move_to_position(FB_MV_CAL_POSITION, true);
        break;
        
    case SCANMECH_IOCTL_MOVE_TO_HOME:
        return smech_move_to_home();
        break;
        
    case SCANMECH_IOCTL_MOVE_TO_ADF:
        break;
        
    case SCANMECH_IOCTL_NOTCHFIND:
        // Notchfind not supported by this mech
        return SCANERR_NOT_IMPLEMENTED;
        break;
        
    case SCANMECH_IOCTL_HARDSTOP:
        return smech_find_home_position();
        break;
        
    case SCANMECH_IOCTL_GET_FLATBED_MARGINS:
        dbg2("%s: SCANMECH_IOCTL_GET_FLATBED_MARGINS\n", __FUNCTION__);
        scerr = SCANERR_NONE;
        
        if( arg != sizeof(struct scan_flatbed_margins) ) {
            return SCANERR_INVALID_PARAM;
        }
        
        fb_margins = (struct scan_flatbed_margins *)ptr;
        scerr = scanmargin_get_flatbed_margins( g_flatbed_margins, fb_margins->dpi, fb_margins->cmode, fb_margins );
        return scerr;
        break;
        
    case SCANMECH_IOCTL_GET_ADF_MARGINS:
        dbg2("%s: SCANMECH_IOCTL_GET_ADF_MARGINS\n", __FUNCTION__);
        scerr = SCANERR_NONE;
        
        if( arg != sizeof(struct scan_adf_margins) ) {
            return SCANERR_INVALID_PARAM;
        }
        
        adf_margins = (struct scan_adf_margins *)ptr;
        
        scerr = scanmargin_get_adf_margins( g_adf_margins, adf_margins->dpi, adf_margins->cmode, adf_margins );
        return scerr;
        break;
        
    case SCANMECH_IOCTL_DEBUG_SET_ADF_MARGINS:
        scerr = SCANERR_NONE;
        
        if (arg != sizeof(struct scan_adf_margins))
        {
            return SCANERR_INVALID_PARAM;
        }
        
        adf_margins = (struct scan_adf_margins *)ptr;
        
        scerr = scanmargin_debug_set_adf_margins( g_adf_margins, adf_margins );
        return scerr;
        break;
        
    case SCANMECH_IOCTL_DEBUG_SET_FLATBED_MARGINS:
        scerr = SCANERR_NONE;
        
        if (arg != sizeof(struct scan_flatbed_margins))
        {
            return SCANERR_INVALID_PARAM;
        }
        
        fb_margins = (struct scan_flatbed_margins *)ptr;
        
        scerr = scanmargin_debug_set_fb_margins( g_flatbed_margins, fb_margins );
        return scerr;
        break;
        
    case SCANMECH_IOCTL_PAPER_JAM_RECOVER:
        //under building
        scerr = SCANERR_NONE;
        return scerr;
        break;
        
    case SCANMECH_IOCTL_ADF_CLIENT_PAUSE:
        /* Client pause/resume, a feature where downstream code can request
         * the ADF stop at the next page gap.
         * The client code (e.g., copyapp or scanapp) is responsible for
         * clearing the pause condition so the ADF scan can resume.
         */
        if (isDocSourceADF())
        {
            if(arg==1)
            {
                scerr = adf_page_gapper_client_pause();
            }
            else if( arg==0 )
            {
                scerr = adf_page_gapper_client_resume();
            }
            else { 
                scerr = SCANERR_INVALID_PARAM;
                }
            }
            else
            {
                scerr = SCANERR_INVALID_PARAM;
            }
            return scerr;
            break;

        case SCANMECH_IOCTL_ADF_PICK_AND_KICK:
            mechType = SCAN_MECH_ADF;
            smech_adf_clear_paper();
            break;

        default:
            dbg2("[%s:%d] ioctl (%d) not implmented\n", __FILE__, __LINE__, request);
            return SCANERR_NOT_IMPLEMENTED;
            break;
    }

    return SCANERR_NONE;
}


/**
 *  \brief mech driver power on init
 *
 *  must be called once at startup before using any other mech driver interfaces.
 *
 *  \param[in] scan_mech         pointer to mech data structure
 **/
void smech_granum2_init(t_ScanMech *scan_mech)
{
    int err;
    printk("RICOH SCAN DRIVER Ver.0.1.1\n");	

#ifndef __KERNEL__
    #error linux kernel only
#endif

    smech_func_enter();

    memset( scan_mech, 0, sizeof(t_ScanMech) );

    // In an error state until init completes
    scan_mech->mech_status = SCAN_MECH_ERROR;

    /* pending_high_water needed by scan_cmdq_ready_for_restart() */
    scan_mech->pending_high_water = 1500;

    /* pending_low_water needed by scan_cmdq_isr() */
    scan_mech->pending_low_water = 240;

    scan_mech->scan_page_prepare_fcn          = smech_granum2_page_prepare;
    scan_mech->scan_setup_fcn                 = smech_granum2_scan_setup;
    scan_mech->smot_scan_start_fcn            = smech_granum2_scan_start;
    scan_mech->scan_page_done_fcn             = smech_granum2_page_done;
    scan_mech->smot_scan_blocking_stop_fcn    = smech_granum2_blocking_stop;
    scan_mech->smot_scan_async_stop_fcn       = smech_granum2_async_stop;
    scan_mech->smot_scan_halt_fcn             = smech_granum2_halt;
    scan_mech->smot_scan_start_pause_fcn      = smech_granum2_start_pause;
    scan_mech->smech_selftest_fcn             = smech_granum2_selftest;
    scan_mech->smech_notchfind_fcn            = NULL;
    scan_mech->smech_poll_fcn                 = NULL;
    scan_mech->smech_fault_msg_fcn            = smech_granum2_fault_msg;
    scan_mech->smech_force_failure_fcn        = smech_granum2_force_failure;
    scan_mech->smech_sleep_fcn                = smech_granum2_sleep;
    scan_mech->smech_is_motor_overheating_fcn = smech_granum2_is_motor_overheating;
    scan_mech->smech_ioctl_fcn                = smech_granum2_ioctl;
    scan_mech->smech_adf_duplex_flip_page_fcn = NULL;
    scan_mech->smech_adf_duplex_eject_fcn     = NULL;
    scan_mech->smech_adf_duplex_stage_input_sheet_fcn = NULL;

    scan_set_scan_mech(scan_mech);

#if 0	
    // Create flatbed motor (setup for 8 usteps)
    fb_motor = smot_step_create_motor(&stmotor_fb_pin_connects,
                                      &stmotor_fb_sequence,
                                      8);
    XASSERT(fb_motor != NULL, (uint32_t) fb_motor);
    motor_param[0].motor_handle  = fb_motor;
    motor_param[0].motor_profile = stmotor_fb_dpi_profiles;
    
    smech_func_infor("fb_motor handle = 0x%p\n", fb_motor);

    // Create adf motor (setup for 8 usteps)
    adf_motor = smot_step_create_motor(&stmotor_adf_pin_connects,
                                       &stmotor_adf_sequence,
                                       8);
    XASSERT(adf_motor != NULL, (uint32_t) adf_motor);    
    motor_param[1].motor_handle  = adf_motor;
    motor_param[1].motor_profile = stmotor_adf_dpi_profiles;
    
    smech_func_infor("adf_motor handle = 0x%p\n", adf_motor);

#endif
    // Mech semaphore   
    sema_init( &mech_sem, 0 );

    // Our home sensor find actions are disabled by default
    home_find_enable = false;

    /* GPIO and DMA init function calls are in a GPL module */
    memset( &granum2_mech, 0, sizeof(struct pt_mech) );
    err = smech_kernel_init(&granum2_mech, home_sensor_isr);
    XASSERT(err==0,err);

    // Turn on the FB/ADF sensor power (3.3V)
//    scanplat_kernel_gpio_set_value(granum2_mech.scan_sensor_pwr_en_n, 0);

    adf_sensor_onetime_init();

    // We are not in a pause/resume condition
    smech_pause_rewind_enable = false;

    // Register a scan mech debug command line
    smech_cmdline_init();

    smech_set_status(SCAN_MECH_READY);
}



//*****************************************************************************
// Command line debug functions
//*****************************************************************************

static const char *smech_cmd_sensors_desc  = "Display the state of all mech sensors";
static const char *smech_cmd_sensors_usage = NULL;
static const char *smech_cmd_sensors_notes = NULL;
static int smech_cmd_sensors_cb( int argc, char *argv[] )
{
    if ( argc != 1 )
    {
        return CMD_USAGE_ERROR;
    }

    cmd_printf("ADF paper in path : %d\n", adf_sensor_paper_in_path());
    cmd_printf("ADF paper present : %d\n", adf_sensor_paper_present());
    cmd_printf("ADF cover open    : %d\n", adf_sensor_cover_is_open());
    cmd_printf("FB  home          : %d\n", flatbed_sensor_home_is_found());

    return CMD_OK;
}

static const char *smech_cmd_fb_home_desc  = "Find the flatbed home position";
static const char *smech_cmd_fb_home_usage = NULL;
static const char *smech_cmd_fb_home_notes = NULL;
static int smech_cmd_fb_home_cb( int argc, char *argv[] )
{
    if ( argc != 1 )
    {
        return CMD_USAGE_ERROR;
    }

    cmd_printf("FB finding home ...\n");
    mechType = SCAN_MECH_FLATBED;
    smech_find_home_position();
    cmd_printf("FB home complete\n");

    return CMD_OK;
}

static const char *smech_cmd_fb_tohome_desc  = "Move flatbed to the home position";
static const char *smech_cmd_fb_tohome_usage = NULL;
static const char *smech_cmd_fb_tohome_notes = NULL;
static int smech_cmd_fb_tohome_cb( int argc, char *argv[] )
{
    scan_err_t scerr;

    if ( argc != 1 )
    {
        return CMD_USAGE_ERROR;
    }

    cmd_printf("FB moving to home position ...\n");
    mechType = SCAN_MECH_FLATBED;
    scerr = smech_move_to_home();
    cmd_printf("FB to home position complete, scerr=%x\n", scerr);

    return CMD_OK;
}

static const char *smech_cmd_fb_tocal_desc  = "Move flatbed to the cal position";
static const char *smech_cmd_fb_tocal_usage = NULL;
static const char *smech_cmd_fb_tocal_notes = NULL;
static int smech_cmd_fb_tocal_cb( int argc, char *argv[] )
{
    scan_err_t scerr;

    if ( argc != 1 )
    {
        return CMD_USAGE_ERROR;
    }

    cmd_printf("FB moving to cal position ...\n");
    mechType = SCAN_MECH_FLATBED;
    scerr = smech_move_to_position(FB_MV_CAL_POSITION, true);
    cmd_printf("FB to cal position complete, scerr=%x\n", scerr);

    return CMD_OK;
}

static const char *smech_cmd_fb_move_rel_desc  = "Relative move flatbed";
static const char *smech_cmd_fb_move_rel_usage = "mvrel table_idx dir hinches";
static const char *smech_cmd_fb_move_rel_notes = NULL;
static int smech_cmd_fb_move_rel_cb( int argc, char *argv[] )
{
#if 0	
    uint32_t num32;
    uint32_t profile_index;
    uint32_t move_steps;
    uint8_t  move_direction;
    stmotor_t   *motor_handle;
    stmotor_move_param_t *move_params;

    if ( argc != 4 )
    {
        return CMD_USAGE_ERROR;
    }

    if( str_mkint( argv[1], &num32 ) != 0 )
    {
        cmd_printf("bad integer \"%s\" for profile index\n", argv[1] );
        return CMD_USAGE_ERROR;
    }
    profile_index = num32;
 
    if( str_mkint( argv[2], &num32 ) != 0 )
    {
        cmd_printf("bad integer \"%s\" for direction\n", argv[1] );
        return CMD_USAGE_ERROR;
    }
    move_direction = num32 ? MOVE_FORWARD : MOVE_REVERSE;

    if( str_mkint( argv[3], &num32 ) != 0 )
    {
        cmd_printf("bad integer \"%s\" for hinches\n", argv[1] );
        return CMD_USAGE_ERROR;
    }
    move_steps = hinches_to_steps(SCAN_MECH_FLATBED, num32);

    cmd_printf("FB relative move: profile=%d steps=%d dir=%d\n",
                 profile_index, move_steps, move_direction);

    motor_handle = get_motor_handle_from_mech(SCAN_MECH_FLATBED);
    move_params  = get_smot_move_profile(motor_handle,profile_index);
    smot_step_set_motor_move_params(motor_handle, move_params);

    smot_step_move_rel(motor_handle,
                       move_steps,
                       move_direction,
                       LS_DISABLED,
                       NO_LS_STEPNUM);
    cmd_printf("FB relative move complete\n");

    cmd_printf("FB relative move complete\n");
#endif
    return CMD_OK;
}

static const char *smech_cmd_fb_move_abs_desc  = "Move flatbed to absolute position";
static const char *smech_cmd_fb_move_abs_usage = "mvabs pos_hinches";
static const char *smech_cmd_fb_move_abs_notes = NULL;
static int smech_cmd_fb_move_abs_cb( int argc, char *argv[] )
{
    uint32_t move_pos, move_steps;
 
    if ( argc != 2 )
    {
        return CMD_USAGE_ERROR;
    }

    if( str_mkint( argv[1], &move_pos ) != 0 )
    {
        cmd_printf("bad integer \"%s\" for position hinches\n", argv[1] );
        return CMD_USAGE_ERROR;
    }
 
    move_steps = hinches_to_steps(SCAN_MECH_FLATBED, move_pos);

    cmd_printf("FB abs move: pos=%d, steps=%d\n", move_pos, move_steps);

    mechType = SCAN_MECH_FLATBED;
    smech_move_to_position(move_steps, true);

    cmd_printf("FB abs move complete\n");

    return CMD_OK;
}

static const char *smech_cmd_fb_get_pos_desc  = "Get flatbed position/location";
static const char *smech_cmd_fb_get_pos_usage = NULL;
static const char *smech_cmd_fb_get_pos_notes = NULL;
static int smech_cmd_fb_get_pos_cb( int argc, char *argv[] )
{
#if 0	
    int fb_pos;

    if ( argc != 1 )
    {
        return CMD_USAGE_ERROR;
    }

    fb_pos = smot_step_get_location(get_motor_handle_from_mech(SCAN_MECH_FLATBED));

    cmd_printf("FB position: steps=%d hinches=%d\n", 
                fb_pos, steps_to_hinches(SCAN_MECH_FLATBED, fb_pos));

#endif
    return CMD_OK;
}

static const char *smech_cmd_fb_set_pos_desc  = "Set flatbed position/location";
static const char *smech_cmd_fb_set_pos_usage = "setpos hinches";
static const char *smech_cmd_fb_set_pos_notes = NULL;
static int smech_cmd_fb_set_pos_cb( int argc, char *argv[] )
{
    int hinches;
    int steps;

    if ( argc != 2 )
    {
        return CMD_USAGE_ERROR;
    }

    if( str_mkint( argv[1], &hinches ) != 0 )
    {
        cmd_printf("bad integer \"%s\" for hinches\n", argv[1] );
        return CMD_USAGE_ERROR;
    }

    steps = hinches_to_steps(SCAN_MECH_FLATBED, hinches);

    cmd_printf("FB set position: steps=%d hinches=%d\n", steps, hinches);

//    smot_step_set_location(get_motor_handle_from_mech(SCAN_MECH_FLATBED), steps);

    return CMD_OK;
}

static const char *smech_cmd_fb_test_desc  = "Flatbed position test";
static const char *smech_cmd_fb_test_usage = NULL;
static const char *smech_cmd_fb_test_notes = NULL;
static int smech_cmd_fb_test_cb( int argc, char *argv[] )
{
    int fb_pos, tgt_pos;

#if 0
    if ( argc != 1 )
    {
        return CMD_USAGE_ERROR;
    }

    mechType = SCAN_MECH_FLATBED;
    smech_move_to_position(FB_HOME_POSITION, true);

    fb_pos = smot_step_get_location(fb_motor);
    cmd_printf("Starting at home position = %d\n", fb_pos);

    smech_move_to_position(FB_BOT_BED_POSITION, false);

    tgt_pos = FB_BOT_BED_POSITION - hinches_to_steps(SCAN_MECH_FLATBED, 150);

    while (tgt_pos > smot_step_get_location(fb_motor));
    {
        msleep_interruptible(10);
    }

    smot_step_request_motor_stop(fb_motor);
    smech_wait_for_stop(fb_motor);

    fb_pos = smot_step_get_location(fb_motor);
    cmd_printf("FB bottom stop position = %d\n", fb_pos);

    smech_move_to_position(FB_HOME_POSITION, true);

    fb_pos = smot_step_get_location(fb_motor);
    cmd_printf("Ending position = %d\n", fb_pos);
#endif
    return CMD_OK;
}



static const char *smech_cmd_fb_speed_desc  = "Adjust flatbed speed";
static const char *smech_cmd_fb_speed_usage = "speed table_idx delta_spd_idx";
static const char *smech_cmd_fb_speed_notes = NULL;
static int smech_cmd_fb_speed_cb( int argc, char *argv[] )
{
#if 0	
    uint32_t num32;
    uint32_t profile_index;
    uint32_t delta_idx;

    if ( argc != 3 )
    {
        return CMD_USAGE_ERROR;
    }

    if( str_mkint( argv[1], &num32 ) != 0 )
    {
        cmd_printf("bad integer \"%s\" for profile index\n", argv[1] );
        return CMD_USAGE_ERROR;
    }
    profile_index = num32;
 
    if( str_mkint( argv[2], &num32 ) != 0 )
    {
        cmd_printf("bad integer \"%s\" for delta_idx\n", argv[1] );
        return CMD_USAGE_ERROR;
    }
    delta_idx = num32;

    cmd_printf("FB speed: profile=%d delta_idx=%d (idx=%d)\n",
                 profile_index, delta_idx, (FB_NRAMP_VALS - delta_idx));

    smot_step_set_speed( fb_motor,
                             (FB_NRAMP_VALS - delta_idx) );

#endif
    return CMD_OK;
}



static const char *smech_cmd_adf_move_rel_desc  = "Relative move adf";
static const char *smech_cmd_adf_move_rel_usage = "mvrel table_idx dir hinches";
static const char *smech_cmd_adf_move_rel_notes = NULL;
static int smech_cmd_adf_move_rel_cb( int argc, char *argv[] )
{
#if 0	
    uint32_t    num32;
    uint32_t    profile_index;
    uint32_t    move_steps;
    uint8_t     move_direction;
    stmotor_t   *motor_handle;
    stmotor_move_param_t *move_params;
    if ( argc != 4 )
    {
        return CMD_USAGE_ERROR;
    }

    if( str_mkint( argv[1], &num32 ) != 0 )
    {
        cmd_printf("bad integer \"%s\" for profile index\n", argv[1] );
        return CMD_USAGE_ERROR;
    }
    profile_index = num32;
 
    if( str_mkint( argv[2], &num32 ) != 0 )
    {
        cmd_printf("bad integer \"%s\" for direction\n", argv[1] );
        return CMD_USAGE_ERROR;
    }
    move_direction = num32 ? MOVE_FORWARD : MOVE_REVERSE;

    if( str_mkint( argv[3], &num32 ) != 0 )
    {
        cmd_printf("bad integer \"%s\" for hinches\n", argv[1] );
        return CMD_USAGE_ERROR;
    }
    move_steps = hinches_to_steps(SCAN_MECH_ADF, num32);

    cmd_printf("ADF relative move: profile=%d steps=%d dir=%d\n",
                 profile_index, move_steps, move_direction);

    motor_handle = get_motor_handle_from_mech(SCAN_MECH_ADF);
    move_params  = get_smot_move_profile(motor_handle,profile_index);
    smot_step_set_motor_move_params(motor_handle,move_params);

    smot_step_move_rel(motor_handle,
                       move_steps,
                       move_direction,
                       LS_DISABLED,
                       NO_LS_STEPNUM);

    cmd_printf("ADF relative move complete\n");
#endif
    return CMD_OK;
}

static const char *smech_cmd_adf_test_desc  = "ADF pick and kick";
static const char *smech_cmd_adf_test_usage = NULL;
static const char *smech_cmd_adf_test_notes = NULL;
static int smech_cmd_adf_test_cb( int argc, char *argv[] )
{
    if ( argc != 1 )
    {
        return CMD_USAGE_ERROR;
    }

    mechType = SCAN_MECH_ADF;

    cmd_printf("ADF clear start ...\n");
    smech_adf_clear_paper();
    cmd_printf("ADF clear complete\n");

    cmd_printf("ADF lift pick roller ...\n");
    smech_adf_lift_pick_roller();
    cmd_printf("ADF lift pick roller complete\n");

    return CMD_OK;
}

static void smech_cmdline_init( void )
{
    int retcode;

    retcode = cmd_register_cmd( "smech",
                                NULL,
                                NULL,
                                NULL,
                                NULL,
                                NULL );
    XASSERT( retcode==CMD_OK, retcode );

    retcode = cmd_register_subcmd( "smech",
                                   "sensors",
                                   smech_cmd_sensors_desc,
                                   smech_cmd_sensors_usage,
                                   smech_cmd_sensors_notes,
                                   smech_cmd_sensors_cb );
    XASSERT( retcode==CMD_OK, retcode );

    // FB commands
    retcode = cmd_register_subcmd( "smech",
                                   "fb",
                                   NULL,
                                   NULL,
                                   NULL,
                                   NULL );
    XASSERT( retcode==CMD_OK, retcode );

    retcode = cmd_register_subcmd( "smech fb",
                                   "home",
                                   smech_cmd_fb_home_desc,
                                   smech_cmd_fb_home_usage,
                                   smech_cmd_fb_home_notes,
                                   smech_cmd_fb_home_cb );
    XASSERT( retcode==CMD_OK, retcode );

    retcode = cmd_register_subcmd( "smech fb",
                                   "tohome",
                                   smech_cmd_fb_tohome_desc,
                                   smech_cmd_fb_tohome_usage,
                                   smech_cmd_fb_tohome_notes,
                                   smech_cmd_fb_tohome_cb );
    XASSERT( retcode==CMD_OK, retcode );

    retcode = cmd_register_subcmd( "smech fb",
                                   "tocal",
                                   smech_cmd_fb_tocal_desc,
                                   smech_cmd_fb_tocal_usage,
                                   smech_cmd_fb_tocal_notes,
                                   smech_cmd_fb_tocal_cb );
    XASSERT( retcode==CMD_OK, retcode );

    retcode = cmd_register_subcmd( "smech fb",
                                   "mvrel",
                                   smech_cmd_fb_move_rel_desc,
                                   smech_cmd_fb_move_rel_usage,
                                   smech_cmd_fb_move_rel_notes,
                                   smech_cmd_fb_move_rel_cb );
    XASSERT( retcode==CMD_OK, retcode );

    retcode = cmd_register_subcmd( "smech fb",
                                   "mvabs",
                                   smech_cmd_fb_move_abs_desc,
                                   smech_cmd_fb_move_abs_usage,
                                   smech_cmd_fb_move_abs_notes,
                                   smech_cmd_fb_move_abs_cb );
    XASSERT( retcode==CMD_OK, retcode );

    retcode = cmd_register_subcmd( "smech fb",
                                   "getpos",
                                   smech_cmd_fb_get_pos_desc,
                                   smech_cmd_fb_get_pos_usage,
                                   smech_cmd_fb_get_pos_notes,
                                   smech_cmd_fb_get_pos_cb );
    XASSERT( retcode==CMD_OK, retcode );

    retcode = cmd_register_subcmd( "smech fb",
                                   "setpos",
                                   smech_cmd_fb_set_pos_desc,
                                   smech_cmd_fb_set_pos_usage,
                                   smech_cmd_fb_set_pos_notes,
                                   smech_cmd_fb_set_pos_cb );
    XASSERT( retcode==CMD_OK, retcode );

    retcode = cmd_register_subcmd( "smech fb",
                                   "test",
                                   smech_cmd_fb_test_desc,
                                   smech_cmd_fb_test_usage,
                                   smech_cmd_fb_test_notes,
                                   smech_cmd_fb_test_cb );
    XASSERT( retcode==CMD_OK, retcode );

    retcode = cmd_register_subcmd( "smech fb",
                                   "speed",
                                   smech_cmd_fb_speed_desc,
                                   smech_cmd_fb_speed_usage,
                                   smech_cmd_fb_speed_notes,
                                   smech_cmd_fb_speed_cb );
    XASSERT( retcode==CMD_OK, retcode );

    // ADF commands
    retcode = cmd_register_subcmd( "smech",
                                   "adf",
                                   NULL,
                                   NULL,
                                   NULL,
                                   NULL );
    XASSERT( retcode==CMD_OK, retcode );

    retcode = cmd_register_subcmd( "smech adf",
                                   "mvrel",
                                   smech_cmd_adf_move_rel_desc,
                                   smech_cmd_adf_move_rel_usage,
                                   smech_cmd_adf_move_rel_notes,
                                   smech_cmd_adf_move_rel_cb );
    XASSERT( retcode==CMD_OK, retcode );

    retcode = cmd_register_subcmd( "smech adf",
                                   "test",
                                   smech_cmd_adf_test_desc,
                                   smech_cmd_adf_test_usage,
                                   smech_cmd_adf_test_notes,
                                   smech_cmd_adf_test_cb );
    XASSERT( retcode==CMD_OK, retcode );

}

