/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2009-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



/** 
 * \file adfsensor_granum2.c
 *
 * \brief Marvell Granum2 ADF sensor functions
 *
 *  Implementations for the Marvell Granum2 ADF sensors. 
 *    Paper In Path (PIP)
 *    Paper Present (PP)
 *    Cover Is Open (CIO)
 *
 **/

#include "stdint.h"
#include "stdbool.h"
#include "lassert.h"
#include "scos.h"  
#include "scancore.h"
#include "scantypes.h"
#include "scansen.h"
#include "scanvars.h"
#include "scanmech.h"
#include "scandbg.h"
#include "scantask.h"      
#include "adfsensor.h"
#include "adfsensor_granum2.h"
#include "scanmech_granum2.h"
#include "gpio_api.h"
#include "gpio_config_api.h"
#include "scanplat_sb.h"
#include "pic_handle_if.h"
#include "pic_if.h"
#include <asm/gpio.h>
#include <linux/timer.h>
#include "interrupt_api.h"
#include <linux/ioport.h>

#define INTRL_GIC_OFFSET						(32)
#define INTNUM_CDMA_IRQ                     	(189 + INTRL_GIC_OFFSET)		//CDMA_IRQ

typedef struct CDMA_REGS_s
{
  volatile uint32_t CFG;  ///< 0x0 [R/W]: DMA Configuration Register
  volatile uint32_t Control;  ///< 0x4 [R/W]: DMA Control Register
  volatile uint32_t Status;  ///< 0x8 [R]: DMA Status Register
  volatile uint32_t reserved0;
  volatile uint32_t CPR;  ///< 0x10 [R]: CDMA Parameter Register
  volatile uint32_t CDR;  ///< 0x14 [R/W]: CDMA Descriptor Register
  volatile uint32_t CNDAR;  ///< 0x18 [R]: CDMA Next Descriptor Address Register
  volatile uint32_t FillValue;  ///< 0x1c [R/W]: Fill Value Register
  volatile uint32_t intEn;  ///< 0x20 [R/W]: CDMA Interrupt Enable Register
  volatile uint32_t intPend;  ///< 0x24 [R]: CDMA Interrupt Pending Register
  volatile uint32_t intAck;  ///< 0x28 [W]: CDMA Interrupt Acknowledge Register
  volatile uint32_t intForce;  ///< 0x2c [W]: CDMA Interrupt Force Register
  volatile uint32_t TimerControl;  ///< 0x30 [R/W]: CDMA Timer Control Register
  volatile uint32_t TimeOutStat;  ///< 0x34 [R]: CDMA Timeout Count Status Register
  volatile uint32_t CRBAR;  ///< 0x38 [R]: CDMA Read Burst Address Register
  volatile uint32_t CRBLR;  ///< 0x3c [R]: CDMA Read Burst Length Register
  volatile uint32_t CWBAR;  ///< 0x40 [R]: CDMA Write Burst Address Register
  volatile uint32_t CWBLR;  ///< 0x44 [R]: CDMA Write Burst Length Register
  volatile uint32_t CWBRR;  ///< 0x48 [R]: CDMA Write Bytes Remain Register
  volatile uint32_t CSRR;  ///< 0x4c [R/W]: DMA Save/Restore Control Register
  volatile uint32_t CSRLI;  ///< 0x50 [R/W]: Save/Restore Lower DMA ID Register
  volatile uint32_t CSRUI;  ///< 0x54 [R/W]: Save/Restore Upper DMA ID Register
  volatile uint32_t CRSL;  ///< 0x58 [R]: DMA Lower Request Status Register
  volatile uint32_t CRSU;  ///< 0x5c [R]: DMA Upper Request Stautus Register
  volatile uint32_t CAFR;  ///< 0x60 [W]: DMA Ack Force Register
} CDMA_REGS_t;


#define adfsen_func_enter()              dbg2("ADFSEN@%s: enter\n", __FUNCTION__)
#define adfsen_func_leave()              dbg2("ADFSEN@%s: leave\n", __FUNCTION__)
#define adfsen_func_infor(msg, args...)  dbg1("ADFSEN@%s: " msg, __FUNCTION__, ##args)
#define adfsen_func_debug(msg, args...)  dbg2("ADFSEN@%s: " msg, __FUNCTION__, ##args)
#define adfsen_func_error(msg, args...)  dbg1("ADFSEN@%s.%d: " msg, __FUNCTION__, __LINE__, ##args)
#define adfsen_func_helpu(msg, args...)  dbg1(msg, ##args)


/*
 * Internal globals
 */
static struct pt_adf granum2_adf;
static bool          pip_callback_enable;
static bool          pp_callback_enable;
static bool          cio_callback_enable;

static CDMA_REGS_t* cdma_addr_ADF_TOP;
static CDMA_REGS_t* cdma_addr_ADF_END;


/*
 * ADF Paper In Path functions
 */
void adf_sensor_pip_callback_disable(void)
{
    /* Stop callback to core scan code */
    pip_callback_enable = false;
}

void adf_sensor_pip_callback_enable(void)
{
    /* Enable callback to core scan code */
    pip_callback_enable = true;
}

bool adf_sensor_paper_in_path(void)
{
    bool      flag = false;
    uint32_t logic_level = 0xFFFF;

    logic_level = scanplat_kernel_gpio_get_value(granum2_adf.adf_pip_sensor_gpio);
    if (logic_level != 0x00)
    {
        flag = false;
    }
    else    
    {
        flag = true;
    }
     flag = false;
    return (flag);
}

bool pip_isr(void)
{
    static struct timespec prev_isr_time = {0, 0};
    struct   timespec cur_isr_time;
    bool     bouncing = true;
    uint32_t trans_len = 0;
    uint8_t  pic_instance = 0;
    uint8_t  wdma_channel = 0;

    // WARNING: you are in GPIO ISR context here!

    getrawmonotonic(&cur_isr_time);
    
    if (cur_isr_time.tv_sec - prev_isr_time.tv_sec >= 2)
    {
        bouncing = false;
    }
    else
    {
        long   ms_diff;
        ms_diff  = (cur_isr_time.tv_sec - prev_isr_time.tv_sec) * 1000;
        ms_diff += (cur_isr_time.tv_nsec  / 1000000);
        ms_diff -= (prev_isr_time.tv_nsec / 1000000);

        if (ms_diff > 30)
        {
            bouncing = false;
        }
    }

    // Warning: debug output in isr context!  Don't leave this enabled.
    //printk("%s: bouncing=%d pip=%d\n", __FUNCTION__, bouncing, adf_sensor_paper_in_path());

    if (pip_callback_enable && (!bouncing))
    {
        // Peek at the remaining transfer length (in bytes) of a running PIC WDMA
        // descriptor.  Scan code will use this to fine tune where the pipe cutter
        // will cut the stream (as dma buffers get bigger, can have lots of slop simply
        // cutting at buffer boundaries).  NOTE: don't really care which PIC instance 
        // or DMA channel ...
        pic_get_trans_len_output_dma_channel(pic_instance, &trans_len, wdma_channel);

        smech_adf_paper_event_callback(SCAN_ADF_PAPER_TOF, adf_sensor_paper_in_path() ? 1 : 0, trans_len);
    }

    prev_isr_time = cur_isr_time;
    return true;
}


/*
 * ADF Cover Is Open functions
 */
void adf_sensor_cio_callback_disable(void)
{
    /* Stop callback to core scan code */
    cio_callback_enable = false;
}

void adf_sensor_cio_callback_enable(void)
{
    /* Enable callback to core scan code */
    cio_callback_enable = true;
}

bool adf_sensor_cover_is_open(void)
{
    bool     flag = false;
    uint32_t logic_level = 0xFFFF;

    logic_level = scanplat_kernel_gpio_get_value(granum2_adf.adf_cio_sensor_gpio);

    if (logic_level != 0x00)
    {
        flag = false;
    }
    else    
    {
        flag = true;
    }
    return (flag);
}

bool cio_isr(void)
{
    static struct timespec prev_isr_time = {0, 0};
    struct        timespec cur_isr_time;
    bool          bouncing = true;
    scan_msg_t    msg;
    scan_err_t    scerr;

    // WARNING: you are in GPIO ISR context here!

    getrawmonotonic(&cur_isr_time);
    
    if (cur_isr_time.tv_sec - prev_isr_time.tv_sec >= 2)
    {
        bouncing = false;
    }
    else
    {
        long   ms_diff;
        ms_diff  = (cur_isr_time.tv_sec - prev_isr_time.tv_sec) * 1000;
        ms_diff += (cur_isr_time.tv_nsec  / 1000000);
        ms_diff -= (prev_isr_time.tv_nsec / 1000000);

        if (ms_diff > 30)
        {
            bouncing = false;
        }
    }

    // Warning: debug output in isr context!  Don't leave this enabled.
    //printk("%s: bouncing=%d cio=%d\n", __FUNCTION__, bouncing, adf_sensor_cover_is_open());

    if (cio_callback_enable && (!bouncing))
    {
        if (adf_sensor_cover_is_open())
        {
            // The cover is open.

            // TODO: do any needed emergency processing here (stopping motors for example).
            //   Something like smot_step_emergency_halt(adf_motor) would probably be
            //   an appropriate action, have to make a function in the mech code to do it.

            // We are in isr context, so send a message to the mech code in thread space to 
            // complete error handling (message passed through core scan message queue).
            msg.msgType = SMSG_SCAN_MECH_FAULT;
            msg.param1  = SMECH_ADF_ERROR_CIO;
            msg.param2  = SCAN_MECH_ADF;
            msg.param3  = 0;
            scerr = scantask_msg_send_nowait(&msg);
            XASSERT(scerr == 0, scerr);
        }
    }

    prev_isr_time = cur_isr_time;
    return true;
}


/*
 * ADF Paper Present functions
 */
bool adf_sensor_paper_present(void)
{
    bool     flag = false;
    uint32_t logic_level = 0xFFFF;
    
    logic_level = scanplat_kernel_gpio_get_value(granum2_adf.adf_pp_sensor_gpio);
    if (logic_level != 0x00)
    {
        flag = false;
    }
    else    
    {
        flag = true;
    }
    return (flag);
}

static irqreturn_t cdmaISR(uint32_t Num)
{
    //割り込みハンドラ.
    uint32_t trans_len = 0;
    uint8_t  pic_instance = 0;
    uint8_t  wdma_channel = 0;
    // Peek at the remaining transfer length (in bytes) of a running PIC WDMA
    // descriptor.  Scan code will use this to fine tune where the pipe cutter
    // will cut the stream (as dma buffers get bigger, can have lots of slop simply
    // cutting at buffer boundaries).  NOTE: don't really care which PIC instance 
    // or DMA channel ...
    
    //TIME OUT割り込みだったら割り込み処理を行う.
    if ( ioread32(&cdma_addr_ADF_TOP->intPend) & BIT(1) )
    {
		pic_get_trans_len_output_dma_channel(pic_instance, &trans_len, wdma_channel);
		
        printk("ADF TOP pend:%d\n",cdma_addr_ADF_TOP->intPend);
        smech_adf_paper_event_callback(SCAN_ADF_PAPER_TOF, 1, trans_len);
        
        //割り込みフラグをクリアする.
        iowrite32(BIT(1),&cdma_addr_ADF_TOP->intAck);
    }
    else if ( ioread32(&cdma_addr_ADF_END->intPend) & BIT(1) )
    {
		pic_get_trans_len_output_dma_channel(pic_instance, &trans_len, wdma_channel);
		
        printk("ADF END pend:%d\n",cdma_addr_ADF_END->intPend);
        smech_adf_paper_event_callback(SCAN_ADF_PAPER_TOF, 0, trans_len);

        //割り込みフラグをクリアする.
        iowrite32(BIT(1),&cdma_addr_ADF_END->intAck);
    }
    else {}

	return IRQ_HANDLED;
}

/**
 * \brief Initialize the ADF sensors
 *
 *  See documentation in adfsensor.h
 */
void adf_sensor_onetime_init( void )
{
#ifndef __KERNEL__
    #error linux kernel only
#endif

    int err;
    static const uint32_t CDMA_ADDR_CH5 = 0xF9060600;
    static const uint32_t CDMA_ADDR_CH6 = 0xF9060700;


    /* GPIO function calls are now in a GPL module */
    memset( &granum2_adf, 0, sizeof(struct pt_adf) );
    err = adfsensor_kernel_onetime_init(&granum2_adf, pip_isr, cio_isr);
    XASSERT(err==0,err);

    /* Our PIP callback is disabled by default */
    pip_callback_enable = false;

    /* Our PP callback is enabled by default */
    pp_callback_enable = true;

    /* Our CIO callback is disabled by default */
    cio_callback_enable = false;

    /* Send out initial PP state */
    smech_adf_paper_event_callback(SCAN_ADF_PAPER_PRESENT, adf_sensor_paper_present(), 0);

    intAttach( INTNUM_CDMA_IRQ, 0, cdmaISR, 0 );
    intEnable(INTNUM_CDMA_IRQ);
    
    //レジスタにメモリマッピングする
    if (!request_mem_region(CDMA_ADDR_CH5, sizeof(CDMA_REGS_t), "cdma5")) {
        printk("request_mem_region failed ch5.\n");
    }
    else {
        cdma_addr_ADF_TOP = (CDMA_REGS_t*)ioremap(CDMA_ADDR_CH5, sizeof(CDMA_REGS_t));
    }
    if (!request_mem_region(CDMA_ADDR_CH6, sizeof(CDMA_REGS_t), "cdma6")) {
        printk("request_mem_region failed ch6.\n");
    }
    else {
        cdma_addr_ADF_END = (CDMA_REGS_t*)ioremap(CDMA_ADDR_CH6, sizeof(CDMA_REGS_t));
    }

}

