#!/usr/bin/perl

# **********************************************************************************
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Copyright (c) 2009-2015, Marvell International Ltd.
#
# Alternatively, this software may be distributed under the terms of the GNU
# General Public License Version 2, and any use shall comply with the terms and
# conditions of the GPL.  A copy of the GPL is available at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
#
# THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
# IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
# ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
# this warranty disclaimer.
# ************************************************************************************


# $Id$

package asm_sccplite;
use strict;

use Exporter;
our @ISA = ("Exporter");
our @EXPORT = qw( $WIDTH
                  $MEM_SIZE
                  $REGS
                  $REGS_EXT
                  $FIELDS
                  $INSTRS );

our ( $WIDTH,
      $MEM_SIZE,
      $REGS,
      $REGS_EXT,
      $MODES,
      $FIELDS,
      $INSTRS );

##########################################################################

$WIDTH         = 2;      # internal width (width of sram and size of instructions)

$MEM_SIZE      = 1024;   # how big is the memory....used to check whether binary fits

##########################################################################


# Edit this structure to add new registers
# Format is NAME => REGCODE

$REGS = {
         bptr0   => 0x0,
         bptr1   => 0x1,
         g0      => 0x2,
         g1      => 0x3,
         g2      => 0x4,
         g3      => 0x5,
         g4      => 0x6,
         g5      => 0x7
        };

$REGS_EXT = {
             bptr0      => 0x00,
             bptr1      => 0x01,
             g0         => 0x02,
             g1         => 0x03,
             g2         => 0x04,
             g3         => 0x05,
             g4         => 0x06,
             g5         => 0x07,
             
             sp         => 0x08,
             mode       => 0x09,
             gpio_oe    => 0x0a,
             gpio_o     => 0x0b,
             gpio_i     => 0x0c,
             trig       => 0x0d,
             int        => 0x0e,
             par        => 0x0f,
             trig_rd    => 0x10,
             ext_addr_l => 0x11,
             ext_addr_u => 0x12,
             ext_data_l => 0x13,
             ext_data_u => 0x14,
             ext_err    => 0x15
            };

$MODES = {
             mode_mem2sccp    => 0,
             mode_sccp2mem    => 1,
             mode_d0msb       => 2,
             mode_d1msb       => 3,
             mode_gpiobank    => 4,
             mode_trigbank    => 6,
             mode_oe_override => 8,
             mode_oe_sense    => 9,
             mode_clk_term    => 10,
             mode_ext_err_int => 11,
             mode_stackerr    => 15,
            };

foreach (keys %{$REGS}) {
  $REGS->{uc($_)} = $REGS->{$_};
}
foreach (keys %{$REGS_EXT}) {
  $REGS_EXT->{uc($_)} = $REGS_EXT->{$_};
}
foreach (keys %{$MODES}) {
  $MODES->{uc($_)} = $MODES->{$_};
}


##########################################################################


# define the fields that appear in the instructions
# 'width' defines the field width
# 'encode' restricts the field values that are allowed and is used to 
#    translate the field value to a number
$FIELDS = {
           a_en      => { width    => 1 },
           a_opcode  => { width    => 4 },
           b_en      => { width    => 1 },
           b_opcode  => { width    => 1 },
           c_en      => { width    => 1 },
           c_opcode  => { width    => 1 },
           d_en      => { width    => 1 },
           d_opcode  => { width    => 4 },
           e_en      => { width    => 1 },
           e_opcode  => { width    => 4 },
           f_en      => { width    => 1 },
           f_opcode  => { width    => 4 },
           g_en      => { width    => 1 },
           g_opcode  => { width    => 5 },
           h_en      => { width    => 1 },
           h_opcode  => { width    => 3 },
           i_en      => { width    => 1 },
           i_opcode  => { width    => 4 },
           
           reg1      => { width    => 3,
                          encoding => $REGS
                        },
           reg0      => { width    => 3,
                          encoding => $REGS
                        },
		     
           reg_ext1 =>  { width    => 5,
                          encoding => $REGS_EXT
                        },
           reg_ext0 =>  { width    => 5,
                          encoding => $REGS_EXT
                        },
		     
           gpio      => { width    => 4 },
           trig      => { width    => 4 },
           int       => { width    => 4 },
           msb       => { width    => 4 },
           lsb       => { width    => 4 },

           oe        => { width    => 1 },
           val       => { width    => 1 },
           pin       => { width    => 1 },
           clk       => { width    => 2 },
           rotate    => { width    => 1 },
           
           sram      => { width    => 10 },
           immed     => { width    => 8 },
           delay     => { width    => -1 },
           flags     => { width    => 2 },
           shft      => { width    => 4 },
           mode      => { width    => 4,
                          encoding => $MODES
                        },
           mode_val  => { width    => 1 },           # mode value

           read      => { width    => 1 },           # for ext operations
           inc       => { width    => 1 },
           size      => { width    => 2 },
           
          };


##########################################################################

# layout the instructions by defining the field and field positions
# 'lsb' defines where each fields least significant bit lies
# 'value' defines what a fields value is, so that field isn't given a value
#    in the assembly code

$INSTRS = {
           # Tier A
           jmp           => { a_en      => { lsb   => 15,
                                             value => 0x1  },
                              a_opcode  => { lsb   => 11,
                                             value => 0x0  },
                              sram      => { lsb   => 0    }
                            },
           jsr           => { a_en      => { lsb   => 15,
                                             value => 0x1  },
                              a_opcode  => { lsb   => 11,
                                             value => 0x1  },
                              sram      => { lsb   => 0    }
                            },
           mov           => { a_en      => { lsb   => 15,
                                             value => 0x1  },
                              a_opcode  => { lsb   => 11,
                                             value => 0x2  },
                              reg_ext1  => { lsb   => 5    },
                              reg_ext0  => { lsb   => 0    } 
                            },
           ldi           => { a_en      => { lsb   => 15,
                                             value => 0x1  },
                              a_opcode  => { lsb   => 11,
                                             value => 0x3  },
                              reg1      => { lsb   => 8    },
                              immed     => { lsb   => 0    } 
                            },
           addi          => { a_en      => { lsb   => 15,
                                             value => 0x1  },
                              a_opcode  => { lsb   => 11,
                                             value => 0x4  },
                              reg1      => { lsb   => 8    },
                              immed     => { lsb   => 0    } 
                            },
           subi          => { a_en      => { lsb   => 15,
                                             value => 0x1  },
                              a_opcode  => { lsb   => 11,
                                             value => 0x5  },
                              reg1      => { lsb   => 8    },
                              immed     => { lsb   => 0    } 
                            },
           andi          => { a_en      => { lsb   => 15,
                                             value => 0x1  },
                              a_opcode  => { lsb   => 11,
                                             value => 0x6  },
                              reg1      => { lsb   => 8    },
                              immed     => { lsb   => 0    } 
                            },
           ori           => { a_en      => { lsb   => 15,
                                             value => 0x1  },
                              a_opcode  => { lsb   => 11,
                                             value => 0x7  },
                              reg1      => { lsb   => 8    },
                              immed     => { lsb   => 0    } 
                            },
           cmpi          => { a_en      => { lsb   => 15,
                                             value => 0x1  },
                              a_opcode  => { lsb   => 11,
                                             value => 0x8  },
                              reg1      => { lsb   => 8    },
                              immed     => { lsb   => 0    } 
                            },
           sdat          => { a_en      => { lsb   => 15,
                                             value => 0x1  },
                              a_opcode  => { lsb   => 11,
                                             value => 0x9  },
                              oe        => { lsb   => 8    },
                              pin       => { lsb   => 7    },
                              reg0      => { lsb   => 4    },
                              gpio      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           psgpio        => { a_en      => { lsb   => 15,
                                             value => 0x1  },
                              a_opcode  => { lsb   => 11,
                                             value => 0xa  },
                              reg1      => { lsb   => 8    },
                              msb       => { lsb   => 4    },
                              lsb       => { lsb   => 0    }
                            },
           prgpio        => { a_en      => { lsb   => 15,
                                             value => 0x1  },
                              a_opcode  => { lsb   => 11,
                                             value => 0xb  },
                              reg1      => { lsb   => 8    },
                              msb       => { lsb   => 4    },
                              lsb       => { lsb   => 0    }
                            },


           # Tier B
           ldr           => { b_en      => { lsb   => 14,
                                             value => 0x1  },
                              b_opcode  => { lsb   => 13,  
                                             value => 0x0  },
                              reg1      => { lsb   => 10   },
                              sram      => { lsb   => 0    }
                            },
           str           => { b_en      => { lsb   => 14,
                                             value => 0x1  },
                              b_opcode  => { lsb   => 13,  
                                             value => 0x1  },
                              reg1      => { lsb   => 10   },
                              sram      => { lsb   => 0    }
                            },

           
           # Tier C
           je            => { c_en      => { lsb   => 13,
                                             value => 0x1  },
                              c_opcode  => { lsb   => 12,  
                                             value => 0x0  },
                              flags     => { lsb   => 10,  
                                             value => 0x0  },
                              sram      => { lsb   => 0    }
                            },
           jne           => { c_en      => { lsb   => 13,
                                             value => 0x1  },
                              c_opcode  => { lsb   => 12,  
                                             value => 0x0  },
                              flags     => { lsb   => 10,  
                                             value => 0x1  },
                              sram      => { lsb   => 0    }
                            },
           jgt           => { c_en      => { lsb   => 13,
                                             value => 0x1  },
                              c_opcode  => { lsb   => 12,
                                             value => 0x0  },
                              flags     => { lsb   => 10,
                                             value => 0x2  },
                              sram      => { lsb   => 0    }
                            },
           jlt           => { c_en      => { lsb   => 13,
                                             value => 0x1  },
                              c_opcode  => { lsb   => 12,
                                             value => 0x0  },
                              flags     => { lsb   => 10,
                                             value => 0x3  },
                              sram      => { lsb   => 0    }
                            },
           tgpioc        => { c_en      => { lsb   => 13,
                                             value => 0x1  },
                              c_opcode  => { lsb   => 12,
                                             value => 0x1  },
                              clk       => { lsb   => 10    },
                              reg1      => { lsb   => 7    },
                              reg0      => { lsb   => 4    },
                              gpio      => { lsb   => 0    }
                            },

           
           # Tier D
           sdatr         => { d_en      => { lsb   => 12,
                                             value => 0x1  },
                              d_opcode  => { lsb   => 8,
                                             value => 0x0  },
                              oe        => { lsb   => 3,
                                             order => 4    },
                              pin       => { lsb   => 7,
                                             order => 3    },
                              reg1      => { lsb   => 4,
                                             order => 2    },
                              reg0      => { lsb   => 0,
                                             order => 1    },
                              delay     => { lsb   => -1,
                                             order => 0    },
                            },
           rdat          => { d_en      => { lsb   => 12,
                                             value => 0x1  },
                              d_opcode  => { lsb   => 8,
                                             value => 0x1  },
                              pin       => { lsb   => 7    },
                              reg1      => { lsb   => 4    },
                              gpio      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           nopi          => { d_en      => { lsb   => 12,
                                             value => 0x1  },
                              d_opcode  => { lsb   => 8,
                                             value => 0x2  },
                              immed     => { lsb   => 0    } 
                            },
           shli          => { d_en      => { lsb   => 12,
                                             value => 0x1  },
                              d_opcode  => { lsb   => 8,
                                             value => 0x3  },
                              rotate    => { lsb   => 7,
                                             value => 0x0  },
                              reg1      => { lsb   => 4    },
                              shft      => { lsb   => 0    }
                            },
           roli          => { d_en      => { lsb   => 12,
                                             value => 0x1  },
                              d_opcode  => { lsb   => 8,
                                             value => 0x3  },
                              rotate    => { lsb   => 7,
                                             value => 0x1  },
                              reg1      => { lsb   => 4    },
                              shft      => { lsb   => 0    }
                            },
           shri          => { d_en      => { lsb   => 12,
                                             value => 0x1  },
                              d_opcode  => { lsb   => 8,
                                             value => 0x4  },
                              rotate    => { lsb   => 7,
                                             value => 0x0  },
                              reg1      => { lsb   => 4    },
                              shft      => { lsb   => 0    }
                            },
           rori          => { d_en      => { lsb   => 12,
                                             value => 0x1  },
                              d_opcode  => { lsb   => 8,
                                             value => 0x4  },
                              rotate    => { lsb   => 7,
                                             value => 0x1  },
                              reg1      => { lsb   => 4    },
                              shft      => { lsb   => 0    }
                            },
           smod          => { d_en      => { lsb   => 12,
                                             value => 0x1  },
                              d_opcode  => { lsb   => 8,
                                             value => 0x5  },
                              mode_val  => { lsb   => 4    },
                              mode      => { lsb   => 0    }
                            },

           
           # Tier E
           rdatr         => { e_en      => { lsb   => 11,
                                             value => 0x1  },
                              e_opcode  => { lsb   => 7,
                                             value => 0x0  },
                              pin       => { lsb   => 3,
                                             order => 3    },
                              reg1      => { lsb   => 4,
                                             order => 2    },
                              reg0      => { lsb   => 0,
                                             order => 1    },
                              delay     => { lsb   => -1,
                                             order => 0    }
                            },
           rgpio         => { e_en      => { lsb   => 11,
                                             value => 0x1  },
                              e_opcode  => { lsb   => 7,
                                             value => 0x1  },
                              reg1      => { lsb   => 4    },
                              gpio      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           shl           => { e_en      => { lsb   => 11,
                                             value => 0x1  },
                              e_opcode  => { lsb   => 7,
                                             value => 0x2  },
                              reg1      => { lsb   => 4    },
                              rotate    => { lsb   => 3,
                                             value => 0x0  },
                              reg0      => { lsb   => 0    }
                            },
           rol           => { e_en      => { lsb   => 11,
                                             value => 0x1  },
                              e_opcode  => { lsb   => 7,
                                             value => 0x2  },
                              reg1      => { lsb   => 4    },
                              rotate    => { lsb   => 3,
                                             value => 0x1  },
                              reg0      => { lsb   => 0    }
                            },
           shr           => { e_en      => { lsb   => 11,
                                             value => 0x1  },
                              e_opcode  => { lsb   => 7,
                                             value => 0x3  },
                              reg1      => { lsb   => 4    },
                              rotate    => { lsb   => 3,
                                             value => 0x0  },
                              reg0      => { lsb   => 0    }
                            },
           ror           => { e_en      => { lsb   => 11,
                                             value => 0x1  },
                              e_opcode  => { lsb   => 7,
                                             value => 0x3  },
                              reg1      => { lsb   => 4    },
                              rotate    => { lsb   => 3,
                                             value => 0x1  },
                              reg0      => { lsb   => 0    }
                            },
           smodr         => { e_en      => { lsb   => 11,
                                             value => 0x1  },
                              e_opcode  => { lsb   => 7,
                                             value => 0x4  },
                              reg1      => { lsb   => 4    },
                              mode      => { lsb   => 0    }
                            },
           rmod          => { e_en      => { lsb   => 11,
                                             value => 0x1  },
                              e_opcode  => { lsb   => 7,
                                             value => 0x5  },
                              reg1      => { lsb   => 4    },
                              mode      => { lsb   => 0    }
                            },

           
           # Tier F
           sgpio         => { f_en      => { lsb   => 10,
                                             value => 0x1  },
                              f_opcode  => { lsb   => 6,
                                             value => 0x0  },
                              oe        => { lsb   => 5    },
                              val       => { lsb   => 4    },
                              gpio      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           # alias to sgpio
           sck           => { f_en      => { lsb   => 10,
                                             value => 0x1  },
                              f_opcode  => { lsb   => 6,
                                             value => 0x0  },
                              oe        => { lsb   => 5    },
                              val       => { lsb   => 4    },
                              gpio      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           rgpior        => { f_en      => { lsb   => 10,
                                             value => 0x1  },
                              f_opcode  => { lsb   => 6,
                                             value => 0x1  },
                              reg1      => { lsb   => 3    },
                              reg0      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           add           => { f_en      => { lsb   => 10,
                                             value => 0x1  },
                              f_opcode  => { lsb   => 6,
                                             value => 0x2  },
                              reg1      => { lsb   => 3    },
                              reg0      => { lsb   => 0    }
                            },
           sub           => { f_en      => { lsb   => 10,
                                             value => 0x1  },
                              f_opcode  => { lsb   => 6,
                                             value => 0x3  },
                              reg1      => { lsb   => 3    },
                              reg0      => { lsb   => 0    }
                            },
           and           => { f_en      => { lsb   => 10,
                                             value => 0x1  },
                              f_opcode  => { lsb   => 6,
                                             value => 0x4  },
                              reg1      => { lsb   => 3    },
                              reg0      => { lsb   => 0    }
                            },
           or            => { f_en      => { lsb   => 10,
                                             value => 0x1 },
                              f_opcode  => { lsb   => 6,
                                             value => 0x5  },
                              reg1      => { lsb   => 3    },
                              reg0      => { lsb   => 0    }
                            },
           xor           => { f_en      => { lsb   => 10,
                                             value => 0x1  },
                              f_opcode  => { lsb   => 6,
                                             value => 0x6  },
                              reg1      => { lsb   => 3    },
                              reg0      => { lsb   => 0    }
                            },
           cmp           => { f_en      => { lsb   => 10,
                                             value => 0x1  },
                              f_opcode  => { lsb   => 6,
                                             value => 0x7  },
                              reg1      => { lsb   => 3    },
                              reg0      => { lsb   => 0    }
                            },
           ldri          => { f_en      => { lsb   => 10,
                                             value => 0x1  },
                              f_opcode  => { lsb   => 6,
                                             value => 0x8  },
                              reg1      => { lsb   => 3    },
                              reg0      => { lsb   => 0    }
                            },
           stri          => { f_en      => { lsb   => 10,
                                             value => 0x1  },
                              f_opcode  => { lsb   => 6,
                                             value => 0x9  },
                              reg1      => { lsb   => 3    },
                              reg0      => { lsb   => 0    }
                            },
           awmg          => { f_en      => { lsb   => 10,
                                             value => 0x1  },
                              f_opcode  => { lsb   => 6,
                                             value => 0xa  },
                              reg1      => { lsb   => 3    },
                              reg0      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },

           
           # Tier G
           tgpior        => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x0  },
                              oe        => { lsb   => 3    },
                              reg0      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           # alias to tgpior
           tckr          => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x0  },
                              oe        => { lsb   => 3    },
                              reg0      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           awt           => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x1  },
                              trig      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           clt           => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x2  },
                              trig      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           sint          => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x3  },
                              int       => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           wint          => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x4  },
                              int       => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           tmod          => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x5  },
                              mode      => { lsb   => 0    }
                            },
           rdxl          => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x6  },
                              read      => { lsb   => 3,
                                             value => 1    },
                              inc       => { lsb   => 2    },
                              size      => { lsb   => 0,
                                             value => 2    }
                            },
           # alias to rdxl
           rdx           => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x6  },
                              read      => { lsb   => 3,
                                             value => 1    },
                              inc       => { lsb   => 2    },
                              size      => { lsb   => 0,
                                             value => 2    }
                            },
           rdxw          => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x6  },
                              read      => { lsb   => 3,
                                             value => 1    },
                              inc       => { lsb   => 2    },
                              size      => { lsb   => 0,
                                             value => 1    }
                            },
           rdxb          => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x6  },
                              read      => { lsb   => 3,
                                             value => 1    },
                              inc       => { lsb   => 2    },
                              size      => { lsb   => 0,
                                             value => 0    }
                            },
           wrxl          => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x6  },
                              read      => { lsb   => 3,
                                             value => 0    },
                              inc       => { lsb   => 2    },
                              size      => { lsb   => 0,
                                             value => 2    }
                            },
           # alias to wrxl
           wrx           => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x6  },
                              read      => { lsb   => 3,
                                             value => 0    },
                              inc       => { lsb   => 2    },
                              size      => { lsb   => 0,
                                             value => 2    }
                            },
           wrxw          => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x6  },
                              read      => { lsb   => 3,
                                             value => 0    },
                              inc       => { lsb   => 2    },
                              size      => { lsb   => 0,
                                             value => 1    }
                            },
           wrxb          => { g_en      => { lsb   => 9,
                                             value => 0x1  },
                              g_opcode  => { lsb   => 4,
                                             value => 0x6  },
                              read      => { lsb   => 3,
                                             value => 0    },
                              inc       => { lsb   => 2    },
                              size      => { lsb   => 0,
                                             value => 0    }
                            },

           
           # Tier H
           sgpior        => { h_en      => { lsb   => 8,
                                             value => 0x1  },
                              h_opcode  => { lsb   => 5,
                                             value => 0x0  },
                              oe        => { lsb   => 4   },
                              val       => { lsb   => 3    },
                              reg0      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           # alias to sgpior
           sckr          => { h_en      => { lsb   => 8,
                                             value => 0x1  },
                              h_opcode  => { lsb   => 5,
                                             value => 0x0  },
                              oe        => { lsb   => 4    },
                              val       => { lsb   => 3    },
                              reg0      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           tgpio         => { h_en      => { lsb   => 8,
                                             value => 0x1  },
                              h_opcode  => { lsb   => 5,
                                             value => 0x1  },
                              oe        => { lsb   => 4    },
                              gpio      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           # alias to tgpio
           tck           => { h_en      => { lsb   => 8,
                                             value => 0x1  },
                              h_opcode  => { lsb   => 5,
                                             value => 0x1  },
                              oe        => { lsb   => 4    },
                              gpio      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           awg           => { h_en      => { lsb   => 8,
                                             value => 0x1  },
                              h_opcode  => { lsb   => 5,
                                             value => 0x2  },
                              val       => { lsb   => 4    },
                              gpio      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },

           
           # Tier I
           tgpios        => { i_opcode  => { lsb   => 4,
                                             value => 0x0  },
                              clk       => { lsb   => 0    }
                            },
           nop           => { i_opcode  => { lsb   => 4,
                                             value => 0x1  },
                              reg0      => { lsb   => 0    }
                            },
           inv           => { i_opcode  => { lsb   => 4,
                                             value => 0x2  },
                              reg0      => { lsb   => 0    }
                            },
           push          => { i_opcode  => { lsb   => 4,
                                             value => 0x3  },
                              reg0      => { lsb   => 0    }
                            },
           pop           => { i_opcode  => { lsb   => 4,
                                             value => 0x4  },
                              reg1      => { lsb   => 0    }
                            },
           rts           => { i_opcode  => { lsb   => 4,
                                             value => 0x5  }
                            },
           awmt          => { i_opcode  => { lsb   => 4,
                                             value => 0x6  },
                              reg0      => { lsb   => 0    },
                              delay     => { lsb   => -1   }
                            },
           halt          => { i_opcode  => { lsb   => 4,
                                             value => 0x7  }
                            },
          };



foreach (keys %{$INSTRS}) {
  $INSTRS->{uc($_)} = $INSTRS->{$_};
}
			
1;
