#!/usr/bin/perl

# **********************************************************************************
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Copyright (c) 2009-2015, Marvell International Ltd.
#
# Alternatively, this software may be distributed under the terms of the GNU
# General Public License Version 2, and any use shall comply with the terms and
# conditions of the GPL.  A copy of the GPL is available at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
#
# THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
# IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
# ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
# this warranty disclaimer.
# ************************************************************************************


# $Id$

package asm;
use strict;
use asm_sccplite;
use Storable;
use POSIX qw(ceil floor);

use Exporter;
our @ISA = ("Exporter");
our @EXPORT = qw( 
                  $DIRECTIVES
                  $SYM_HASH
                  $DEFINE_HASH
                  
                  fill_hex
                  get_word
                  eqi
                  nei
                  check_data
                  sym_subs
                  define_subs
                  encode_instr
                  extract_bits
                  add_sym
                  add_define
                  get_reqd_fields
                  get_constant_fields
                  get_cmd
                  get_num_fields
                  get_opcodes
                  xlat_val
                  rev_op
               );

#######################################################

# valid directives...
# can be equ, data or any instruction
our $DIRECTIVES = {equ    => { name  => "",
                               value => "" },
                   data   => { value => "" }, 
                   org    => { value => "" }, 
                   
                   %{$INSTRS}
                  };


foreach (keys %{$DIRECTIVES}) {
  $DIRECTIVES->{uc($_)} = $DIRECTIVES->{$_};
}


# will be populated with symbols
our $SYM_HASH;

# will be populated with preprocessor defines
our $DEFINE_HASH;

#######################################################

sub fill_hex {
  my $HEX = shift;
  my $addr = shift;
  my $width = shift;
  my $hex_data = shift;
  my $infile = shift;
  my $linenum = shift;

  for (my $j=$width-1; $j>=0; $j--) {
    my $hex_addr = $addr+($width-1-$j);
    if (defined $HEX->[$hex_addr]) {
      $hex_addr = sprintf("0x%x", $hex_addr);
      die "Error:  $infile($linenum) - Code overwrite at byte address '$hex_addr'!\n";
    }
    $HEX->[$hex_addr] = substr($hex_data, 2*$j, 2);
  }
}

#######################################################

sub get_word {
  my $HEX_PTR = shift;
  my $i = shift;

  my $lword;
  for (my $j=$WIDTH-1; $j>=0; $j--) {
    if (!defined $HEX_PTR->[$i+$j]) {
      $lword .= "00";
    }
    else {
      $lword .= $HEX_PTR->[$i+$j];
    }
  }
  return $lword;
}

#######################################################

sub eqi {
  my $a = shift;
  my $b = shift;

  if ($a eq $b || $a eq uc($b)) {
    return 1;
  }
  else {
    return 0;
  }
}

sub nei {
  my $a = shift;
  my $b = shift;

  if ($a ne $b && $a ne uc($b)) {
    return 1;
  }
  else {
    return 0;
  }
}

#######################################################

# make full copies of hashes/arrays
sub deepcopy {
  if (ref $_[0] eq 'HASH') {
    return { map(deepcopy($_), %{$_[0]}) };
  }
  elsif (ref $_[0] eq 'ARRAY') {
    return [ map(deepcopy($_), @{$_[0]}) ];
  }
  $_[0]
}

#######################################################

sub check_data {
  my $value = shift;
  my $width = shift;
  my $file = shift;
  my $linenum = shift;
  
  my $unsigned_upper = 2<<$width - 1;
  my $signed_upper   = (2<<($width-1)) - 1;
  my $signed_lower   = -(2<<($width-1));
  
  # check the data value is numerical
  if ($value !~ /^\d+$/) {
    die "Error:  $file($linenum) - Invalid value '$value'\n";
  }
  
  # check that the numerical data can fit within the width
  if ($value < 0) {
    if ($value < $signed_lower) {
      die "Error:  $file($linenum) - Signed data value '$value' is out of range ($signed_lower to $signed_upper)\n";
    }
  }
  else {
    if ($value > $unsigned_upper) {
      die "Error:  $file($linenum) - Unsigned data value '$value' is out of range (0 to $unsigned_upper)\n";
    }
  }
}

#######################################################

sub sym_subs {
  my $tokens = shift;
  my $file = shift;
  my $linenum = shift;
  my $cmd = get_cmd($tokens);
  
  my $i=0;
  foreach my $token (@{$tokens}) {
    # special case, do not substitute the NAME field of equ
    unless (eqi($cmd,"equ") && $i==0) {
      # symbol subs
      if (exists ($SYM_HASH->{$token})) {
        $token = $SYM_HASH->{$token};
      }
    }
    # hex conversion
    if ($token =~ /^0x/) {
      unless ($token =~ /^0x[0-9a-fA-F]+$/) {
        die "Error:  $file($linenum) - Invalid hex value '$token'!\n";
      }
      $token = hex($token);
    }
    $i++;
  }
}

#######################################################

sub define_subs {
  my $tokens = shift;
  my $file = shift;
  my $linenum = shift;

  local $" = "";
  for(my $i=0; $i<@{$tokens}; $i++) {
    # split by any math symbols
    my @sub_tokens = split /(\s+|\+|\-|\/|\*|\%|\&|\||\^|\(|\)|\>\>|\<\<|ceil|floor)/, $tokens->[$i];
    foreach my $sub_token (@sub_tokens) {
      # define subs
      if (exists ($DEFINE_HASH->{$sub_token})) {
        $sub_token = $DEFINE_HASH->{$sub_token};
      }
    }
    $tokens->[$i] = "@sub_tokens";
  }
}

#######################################################

sub encode_instr {
  my $tokens = shift;
  my $file = shift;
  my $linenum = shift;
  
  my $cmd = get_cmd($tokens);
  
  # valid command?
  if (!exists $INSTRS->{$cmd}) {
    die "Error:  $file($linenum) - Invalid command '$cmd'\n";
  }
  
  # create a copy of the instruction hash to build the instruction
  my $INSTR_MAP = deepcopy($INSTRS->{$cmd});
    
  # get the fields required by the instruction
  my @fields = @{get_reqd_fields($cmd)};
  my $num_req = @fields;
  
  # enough operands provided?
  if (@{$tokens} != $num_req+1) {
    my $usage = get_instr_usage($cmd);
    die "Error:  $file($linenum) - Command '$cmd' requires $num_req arguments\n$usage\n";
  }
  
  # check the fields
  for (my $i=1; $i<@{$tokens}; $i++) {
    my $field = shift @fields;
    my $token = $tokens->[$i];
    
    # any constraints for the field (ex register name)
    if (exists $FIELDS->{$field}->{encoding}) {
      my $encoding = $FIELDS->{$field}->{encoding};
      if (!exists $encoding->{$token}) {
        die "Error:  $file($linenum) - Token '$token' is invalid\n";
      }
      else {
        # translate the token to a coding
        $token = $encoding->{$token};
      }
    }
    else {
      # should be numerical
      if ($token !~ /^\d+$/) {
        die "Error:  $file($linenum) - Invalid value '$token'\n";
      }

      # is an op defined?  If so, perform the op on the token
      my $op = $INSTRS->{$cmd}->{$field}->{op};
      if ($op ne "" ) {
        $token = eval("$token $op");
      }
      
      # check the width
      my $width = $FIELDS->{$field}->{width};
      if ($width > 0) {
        check_data($token, $width, $file, $linenum);
      }
    }
    
    # save the value in the instruction map
    $INSTR_MAP->{$field}->{value} = $token;
  }
  
  # create the instruction code
  my $coded_instr = 0x0;
  foreach my $field (keys %{$INSTR_MAP}) {
    #    print $field . " | " . $INSTR_MAP->{$field}->{lsb} . " | " . $INSTR_MAP->{$field}->{value} . "\n";
    
    # special case (don't actually encode negative fields)...ex = delay
    unless ($INSTR_MAP->{$field}->{lsb} < 0) {
      $coded_instr = $coded_instr | ($INSTR_MAP->{$field}->{value} << $INSTR_MAP->{$field}->{lsb});
    }
  }
  
  return $coded_instr;
}

#######################################################

sub extract_bits {
  my $hex = shift;
  my $lower = shift;
  my $width = shift;
  
  my $shift = (hex($hex) >> $lower);
  my $mask  = (2 ** $width) - 1;
  
  return $shift & $mask;
}

#######################################################

sub add_sym {
  my $name = shift;
  my $value = shift;
  my $file = shift;
  my $linenum = shift;
  
  # check if the symbol has already been defined
  if (exists $SYM_HASH->{$name}) {
    die "Error:  $file($linenum) - Symbol '$name' has been defined previously\n";
  }
  
  if ($value !~ /^".+"$/) {
    # tokenize by arithmetic operations (+,-,/,*,%,>>,<<,&,|,^,(,)
    my @tokens = split /\s+|(\+|\-|\/|\*|\%|\&|\||\^|\(|\)|\>\>|\<\<|ceil|floor)/, $value;

    # do sym substitution on any tokens
    foreach my $token (@tokens) {
      if (exists $SYM_HASH->{$token}) {
        $token = $SYM_HASH->{$token};    
        
        # check if we have an operator or a decimal/hex number
        if ($token !~ /\+|\-|\/|\*|\%|\&|\||\^|\(|\)|\>\>|\<\<|ceil|floor|[\d.]+|0x[\da-f]+/i) {
          die "Error:  $file($linenum) - Invalid token '$token'\n";
        }
      }
    }
    
    $value = "@tokens";
    $value = eval($value);
  }
  
  # add the symbol entry
  $SYM_HASH->{$name} = $value;
}

#######################################################

sub add_define {
  my $name = shift;
  my $value = shift;
  my $file = shift;
  my $linenum = shift;
  
  if ($value !~ /^".+"$/) {
    # tokenize by arithmetic operations (+,-,/,*,%,>>,<<,&,|,^,(,)
    my @tokens = split /\s+|(\+|\-|\/|\*|\%|\&|\||\^|\(|\)|\>\>|\<\<|ceil|floor)/, $value;

    # do define substitution on any tokens
    foreach my $token (@tokens) {
      if (exists $DEFINE_HASH->{$token}) {
        $token = $DEFINE_HASH->{$token};
      }
    }
    
    $value = "@tokens";
    if ($value =~ /\+|\-|\/|\*|\%|\&|\||\^|\(|\)|\>\>|\<\<|ceil|floor/) {
      $value = eval($value);
    }
  }

  # add the symbol entry
  $DEFINE_HASH->{$name} = $value;
}

#######################################################

sub get_reqd_fields {
  my $cmd = shift;
  my @fields;
  
  my @tmp = keys(%{$INSTRS->{$cmd}});
  foreach (@tmp) {
    unless (exists $INSTRS->{$cmd}->{$_}->{value}) {
      push @fields, $_;
    }
  }
  
  @fields = sort {
    if (defined $INSTRS->{$cmd}->{$b}->{order}) {
      $INSTRS->{$cmd}->{$b}->{order} <=> $INSTRS->{$cmd}->{$a}->{order};
    }
    else {
      $INSTRS->{$cmd}->{$b}->{lsb} <=> $INSTRS->{$cmd}->{$a}->{lsb}
    }
  } @fields;

  return \@fields;
}

#######################################################

sub get_instr_usage {
  my $cmd = shift;
  my @fields;
  
  my @tmp = keys(%{$INSTRS->{$cmd}});
  foreach (@tmp) {
    unless (exists $INSTRS->{$cmd}->{$_}->{value}) {
      push @fields, $_;
    }
  }

  @fields = sort {
    if (defined $INSTRS->{$cmd}->{$b}->{order}) {
      $INSTRS->{$cmd}->{$b}->{order} <=> $INSTRS->{$cmd}->{$a}->{order};
    }
    else {
      $INSTRS->{$cmd}->{$b}->{lsb} <=> $INSTRS->{$cmd}->{$a}->{lsb}
    }
  } @fields;

  my $usage = "USAGE:  '$cmd @fields'";
  return $usage;
}

#######################################################

sub get_constant_fields {
  my $cmd = shift;
  my @fields;
  
  my @tmp = keys(%{$INSTRS->{$cmd}});
  foreach (@tmp) {
    if (exists $INSTRS->{$cmd}->{$_}->{value}) {
      push @fields, $_;
    }
  }

  @fields = sort {
    if (defined $INSTRS->{$cmd}->{$b}->{order}) {
      $INSTRS->{$cmd}->{$b}->{order} <=> $INSTRS->{$cmd}->{$a}->{order};
    }
    else {
      $INSTRS->{$cmd}->{$b}->{lsb} <=> $INSTRS->{$cmd}->{$a}->{lsb}
    }
  } @fields;
  
  return \@fields;
}


#######################################################

sub get_cmd {
  my $tokens = shift;
  my $cmd;
  
  if ($tokens->[1] =~ /^(equ|data|EQU|DATA)$/) {
    $cmd = $1;
  }
  else {
    $cmd = $tokens->[0];
  }
  
  return $cmd;
}

#######################################################

sub get_num_fields {
  my $cmd = shift;
  
  my @fields = keys %{$INSTRS->{$cmd}};
  my $num_req = 0;
  foreach my $field (@fields) {
    unless (exists $INSTRS->{$cmd}->{$field}->{value}) {
      $num_req++;
    }
  }
  
  return $num_req;
}

#######################################################

sub get_opcodes {
  my $OPCODES;
  
  foreach my $instr (keys %{$INSTRS}) {
    my $value = $INSTRS->{$instr}->{opcode}->{value};
    $OPCODES->{$value} = $instr;
  }
  
  return $OPCODES;
}


#######################################################

sub xlat_val {
  my $field = shift;
  my $val = shift;
  
  if (exists $FIELDS->{$field}->{encoding}) {
    my $encoding = $FIELDS->{$field}->{encoding};
    my %rev_enc  = reverse (%{$encoding});
    $val = $rev_enc{$val};
  }
  
  return $val;
}

#######################################################

sub rev_op {
  my $val = shift;
  my $op = shift;
  
  if ($op =~ /([%&|^])/) {
    die "Error:  Unable to reverse operation $1\n";
  }
  else {
    $op =~ s/\+/\#sub#/g;
    $op =~ s/\-/\#add#/g;
    $op =~ s/\//\#mult#/g;
    $op =~ s/\*/\#div#/g;
    $op =~ s/>>/\#shl#/g;
    $op =~ s/<</\#shr#/g;
    
    $op =~ s/#sub#/\-/g;
    $op =~ s/#add#/\+/g;
    $op =~ s/#mult#/\*/g;
    $op =~ s/#div#/\//g;
    $op =~ s/#shl#/<</g;
    $op =~ s/#shr#/>>/g;
  }
  
  $val = eval("$val $op");
  
  return $val;
}


1;
