/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/

/** 
* \file afe_wm8213.c
* 
* \brief AFE driver for wolfson micro 8213 AFE.
* 
* This driver uses a SCCP-lite interface to interact with the 
* AFE's registers.
* 
**/
#ifndef MODULE_TEST
#include <linux/init.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/delay.h>
#else
#include "cpp_stub.h"
#endif

#include "afe_driver.h"
#include "sccp/wolfson_sccplt.h"

#include "sccplite.h"

#define SCCP_DEV_INDEX 0

/* SCCPLite microcode creates these register locations, offset at SRAM address
 * zero. These are 16-bit values mapped onto 32-bit addresses! (SCCPLite is 16-bit). 
 */
//#define OFFSET_CMD  0x40
//#define OFFSET_REG  0x44
//#define OFFSET_VAL  0x00
//#define OFFSET_DBG  3

#define START_ADDR  0x24

#define CMD_WRITE_FB   0x3
#define CMD_READ_FB    0x1
#define CMD_WRITE_DF   0x7
#define CMD_READ_DF    0x5

#define AFE_REG_DEVICE_ID				0x00
#define AFE_REG_SETUP_REG1				0x01
#define AFE_REG_SETUP_REG2				0x02
#define AFE_REG_SETUP_REG3				0x03
#define AFE_REG_SOFTWARE_RESET			0x04
#define AFE_REG_DEVICE_ID_REVISION		0x05
#define AFE_REG_SETUP_REG4				0x06
#define AFE_REG_SETUP_REG5				0x07
#define AFE_REG_SETUP_REG6				0x08
#define AFE_REG_BLC_RED_TARGET			0x09
#define AFE_REG_BLC_GREEN_TARGET		0x0A
#define AFE_REG_BLC_BLUE_TARGET			0x0B
#define AFE_REG_BLC_CONTROL1			0x0C
#define AFE_REG_BLC_CONTROL2			0x0D
#define AFE_REG_BLC_CONTROL3			0x0E
#define AFE_REG_BLC_CONTROL4			0x0F
#define AFE_REG_DAC_VALUE_RED			0x20
#define AFE_REG_DAC_VALUE_GREEN			0x21
#define AFE_REG_DAC_VALUE_BLUE			0x22
#define AFE_REG_DAC_VALUE_RGB			0x23
#define AFE_REG_PGA_GAIN_LSB_RED		0x24
#define AFE_REG_PGA_GAIN_LSB_GREEN		0x25
#define AFE_REG_PGA_GAIN_LSB_BLUE		0x26
#define AFE_REG_PGA_GAIN_LSB_RGB		0x27
#define AFE_REG_PGA_GAIN_MSBS_RED		0x28
#define AFE_REG_PGA_GAIN_MSBS_GREEN		0x29
#define AFE_REG_PGA_GAIN_MSBS_BLUE		0x2A
#define AFE_REG_PGA_GAIN_MSBS_RGB		0x2B
#define AFE_REG_BLC_CONTROL5			0x2C
#define AFE_REG_BLC_CONTROL6			0x2D

#define AFE_REG_DEVICE_ID_DF			0x80
#define AFE_REG_SETUP_REG1_DF			0x81
#define AFE_REG_SETUP_REG2_DF			0x82
#define AFE_REG_SETUP_REG3_DF			0x83
#define AFE_REG_SOFTWARE_RESET_DF		0x84
#define AFE_REG_DEVICE_ID_REVISION_DF	0x85
#define AFE_REG_SETUP_REG4_DF			0x86
#define AFE_REG_SETUP_REG5_DF			0x87
#define AFE_REG_SETUP_REG6_DF			0x88
#define AFE_REG_BLC_RED_TARGET_DF		0x89
#define AFE_REG_BLC_GREEN_TARGET_DF		0x8A
#define AFE_REG_BLC_BLUE_TARGET_DF		0x8B
#define AFE_REG_BLC_CONTROL1_DF			0x8C
#define AFE_REG_BLC_CONTROL2_DF			0x8D
#define AFE_REG_BLC_CONTROL3_DF			0x8E
#define AFE_REG_BLC_CONTROL4_DF			0x8F
#define AFE_REG_DAC_VALUE_RED_DF		0xA0
#define AFE_REG_DAC_VALUE_GREEN_DF		0xA1
#define AFE_REG_DAC_VALUE_BLUE_DF		0xA2
#define AFE_REG_DAC_VALUE_RGB_DF		0xA3
#define AFE_REG_PGA_GAIN_LSB_RED_DF		0xA4
#define AFE_REG_PGA_GAIN_LSB_GREEN_DF	0xA5
#define AFE_REG_PGA_GAIN_LSB_BLUE_DF	0xA6
#define AFE_REG_PGA_GAIN_LSB_RGB_DF		0xA7
#define AFE_REG_PGA_GAIN_MSBS_RED_DF	0xA8
#define AFE_REG_PGA_GAIN_MSBS_GREEN_DF	0xA9
#define AFE_REG_PGA_GAIN_MSBS_BLUE_DF	0xAA
#define AFE_REG_PGA_GAIN_MSBS_RGB_DF	0xAB
#define AFE_REG_BLC_CONTROL5_DF			0xAC
#define AFE_REG_BLC_CONTROL6_DF			0xAD



#define NAME_VAL_PAIR( regname ) { #regname, regname }

typedef struct 
{
    const char *reg_name;
    uint32_t    reg_offset;
} reg_name_val_t;

const static reg_name_val_t all_dump_regs[] = 
{
	NAME_VAL_PAIR(AFE_REG_DEVICE_ID),
	NAME_VAL_PAIR(AFE_REG_SETUP_REG1),
	NAME_VAL_PAIR(AFE_REG_SETUP_REG2),
	NAME_VAL_PAIR(AFE_REG_SETUP_REG3),
	NAME_VAL_PAIR(AFE_REG_SOFTWARE_RESET),
	NAME_VAL_PAIR(AFE_REG_DEVICE_ID_REVISION),
	NAME_VAL_PAIR(AFE_REG_SETUP_REG4),
	NAME_VAL_PAIR(AFE_REG_SETUP_REG5),
	NAME_VAL_PAIR(AFE_REG_SETUP_REG6),
	NAME_VAL_PAIR(AFE_REG_BLC_RED_TARGET),
	NAME_VAL_PAIR(AFE_REG_BLC_GREEN_TARGET),
	NAME_VAL_PAIR(AFE_REG_BLC_BLUE_TARGET),
	NAME_VAL_PAIR(AFE_REG_BLC_CONTROL1),
	NAME_VAL_PAIR(AFE_REG_BLC_CONTROL2),
	NAME_VAL_PAIR(AFE_REG_BLC_CONTROL3),
	NAME_VAL_PAIR(AFE_REG_BLC_CONTROL4),
	NAME_VAL_PAIR(AFE_REG_DAC_VALUE_RED),
	NAME_VAL_PAIR(AFE_REG_DAC_VALUE_GREEN),
	NAME_VAL_PAIR(AFE_REG_DAC_VALUE_BLUE),
	NAME_VAL_PAIR(AFE_REG_DAC_VALUE_RGB),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_LSB_RED),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_LSB_GREEN),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_LSB_BLUE),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_LSB_RGB),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_MSBS_RED),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_MSBS_GREEN),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_MSBS_BLUE),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_MSBS_RGB),
	NAME_VAL_PAIR(AFE_REG_BLC_CONTROL5),
	NAME_VAL_PAIR(AFE_REG_BLC_CONTROL6),

	NAME_VAL_PAIR(AFE_REG_DEVICE_ID_DF),
	NAME_VAL_PAIR(AFE_REG_SETUP_REG1_DF),
	NAME_VAL_PAIR(AFE_REG_SETUP_REG2_DF),
	NAME_VAL_PAIR(AFE_REG_SETUP_REG3_DF),
	NAME_VAL_PAIR(AFE_REG_SOFTWARE_RESET_DF),
	NAME_VAL_PAIR(AFE_REG_DEVICE_ID_REVISION_DF),
	NAME_VAL_PAIR(AFE_REG_SETUP_REG4_DF),
	NAME_VAL_PAIR(AFE_REG_SETUP_REG5_DF),
	NAME_VAL_PAIR(AFE_REG_SETUP_REG6_DF),
	NAME_VAL_PAIR(AFE_REG_BLC_RED_TARGET_DF),
	NAME_VAL_PAIR(AFE_REG_BLC_GREEN_TARGET_DF),
	NAME_VAL_PAIR(AFE_REG_BLC_BLUE_TARGET_DF),
	NAME_VAL_PAIR(AFE_REG_BLC_CONTROL1_DF),
	NAME_VAL_PAIR(AFE_REG_BLC_CONTROL2_DF),
	NAME_VAL_PAIR(AFE_REG_BLC_CONTROL3_DF),
	NAME_VAL_PAIR(AFE_REG_BLC_CONTROL4_DF),
	NAME_VAL_PAIR(AFE_REG_DAC_VALUE_RED_DF),
	NAME_VAL_PAIR(AFE_REG_DAC_VALUE_GREEN_DF),
	NAME_VAL_PAIR(AFE_REG_DAC_VALUE_BLUE_DF),
	NAME_VAL_PAIR(AFE_REG_DAC_VALUE_RGB_DF),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_LSB_RED_DF),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_LSB_GREEN_DF),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_LSB_BLUE_DF),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_LSB_RGB_DF),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_MSBS_RED_DF),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_MSBS_GREEN_DF),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_MSBS_BLUE_DF),
	NAME_VAL_PAIR(AFE_REG_PGA_GAIN_MSBS_RGB_DF),
	NAME_VAL_PAIR(AFE_REG_BLC_CONTROL5_DF),
	NAME_VAL_PAIR(AFE_REG_BLC_CONTROL6_DF)
};

uint32_t reg_read(void *dev, uint32_t reg_index)
{
    uint16_t address;
    uint16_t cmd;
	uint16_t data = 0;

    address = reg_index & 0x3F;
	address |= 0x10; //read request
	if( (reg_index & 0x80) > 0){
		// DF
	    cmd = CMD_READ_DF;
		address |= 0x40;
	} else {
		// FB
	    cmd = CMD_READ_FB;
	}

    sccp_write_sram_words( SCCP_DEV_INDEX, 0x22, &address, sizeof(address) );
    sccp_write_sram_words( SCCP_DEV_INDEX, 0x20, &cmd, sizeof(cmd) );

    do
    {
        // Wait for command to be handled
        usleep_range( 10, 50 );
        sccp_read_sram_words( SCCP_DEV_INDEX, 0x20, &data, sizeof(data) );
    } while( data != 0 );

    sccp_read_sram_words( SCCP_DEV_INDEX, 2, &data, sizeof(data) );
    sccp_clr_interrupt( SCCP_DEV_INDEX, 0);
    //pr_alert("wm8213 AFE RD:%d,%d\n",reg_index,data);
    return data;
}

int reg_write(void *dev, uint32_t reg_index, uint32_t reg_value)
{
    uint16_t address;
    uint16_t cmd;
    uint16_t value = reg_value;
    uint16_t data = 0;
	
    //pr_alert("wm8213 AFE WR:%d,%d\n",reg_index,reg_value);
    address = reg_index & 0x3F;
	if( (reg_index & 0x80) > 0){
		// DF
	    cmd = CMD_WRITE_DF;
		address |= 0x40;
	} else {
		// FB
	    cmd = CMD_WRITE_FB;
	}

    //pr_alert("wm8213 ADDR:%d\n",address);
	sccp_write_sram_words( SCCP_DEV_INDEX, 0x22, &address, sizeof(address) );
    sccp_write_sram_words( SCCP_DEV_INDEX, 0x0, &value, sizeof(value) );
    sccp_write_sram_words( SCCP_DEV_INDEX, 0x20, &cmd, sizeof(cmd) );

    do
    {
        // Wait for command to be handled
        usleep_range( 10, 50 );
        sccp_read_sram_words( SCCP_DEV_INDEX, 0x20, &data, sizeof(data) );
    } while( data != 0 );
    sccp_clr_interrupt( SCCP_DEV_INDEX, 0);
    return 0;
}


static int reset(void *dev)
{
    int err = 0;

//    reg_write( dev, AFE_REG_SW_RESET, 0x00 );

    return err;
}

static int soft_setup(void *dev, void* config)
{
    int err = 0;

    return err;
}

static int set_gains(void *dev, uint32_t red, uint32_t green, uint32_t blue)
{
    int err = 0;
    pr_alert("Can't set gains!!\n");
    return err;
}

static int set_offsets(void *dev, int32_t red, int32_t green, int32_t blue)
{
    int err = 0;
    pr_alert("Can't set offsets!!\n");
    return err;
}

static int get_bits_per_pixel(void *dev)
{
    return 16;
}

static int get_name(void *dev, char *buffer, int buf_len)
{
    int err = 0;

    strncpy( buffer, "wm8213", buf_len );
    buffer[buf_len - 1] = '\0';

    return err;
}

static int reg_dump(void *dev)
{
    int err = 0;
    int i;

    pr_alert("wm8213 AFE registers:\n");

    for ( i = 0; i < sizeof(all_dump_regs) / sizeof(all_dump_regs[0]); i++ )
    {
        uint32_t val;

        val = reg_read( dev, all_dump_regs[i].reg_offset );

        pr_alert("    %-21s: 0x%02x\n", all_dump_regs[i].reg_name, val );
    }

    return err;
}
#ifndef MODULE_TEST
static afe_driver_functions_t wm8213_functions =
{
    .reset              = reset,
    .soft_setup         = soft_setup,

    .set_gains          = set_gains,
    .set_offsets        = set_offsets,

    .get_bits_per_pixel = get_bits_per_pixel,

    .get_name           = get_name,
    .reg_read           = reg_read,
    .reg_write          = reg_write,
    .reg_dump           = reg_dump,
};
#else
static afe_driver_functions_t wm8213_functions;
#endif

static int wm8213_init(void)
{
    sccp_write_microcode( SCCP_DEV_INDEX, 0, wolfson_sccplt, sizeof(wolfson_sccplt) );
    sccp_processor_start( SCCP_DEV_INDEX, START_ADDR );
//    reg_write( SCCP_DEV_INDEX, AFE_REG_SW_RESET, 0x00 );

    afe_register_device( &wm8213_functions, NULL );

    return  0;
}

static void wm8213_exit(void)
{
    afe_remove_device( &wm8213_functions, NULL );

    sccp_processor_stop( SCCP_DEV_INDEX );

    return;
}
#ifndef MODULE_TEST
module_init(wm8213_init);
module_exit(wm8213_exit);

MODULE_AUTHOR("Copyright (c) 2014 Marvell International Ltd. All Rights Reserved");
MODULE_DESCRIPTION("Device Driver for wm8213 AFE");

MODULE_LICENSE("Dual MPL/GPL");
MODULE_VERSION("2014_JUL_14");

#endif
