/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <stdio.h>
#include <sys/types.h>
#include <dirent.h>
#include <mqueue.h>

#include "jpeghw_api.h" // JPEGHW library
#include "jpeg_strip_adaptor.h"
#include "jpeg_page_adaptor.h"
#include "jpeg_message.h"

#include "logger.h"
#include "dma_buffer.h"
#include "uio_lib_api.h"

#define DBG_PRFX "jpeg_strip_compress: "
#define LOGGER_MODULE_MASK DEBUG_LOGGER_MODULE_DEVICES | LOGGER_SUBMODULE_BIT(20)

//static int filenum = 0;
#define JPEG_FILE_DEFAULT_IMAGE_HEIGHT_MAX 8400 // 14 inch default, needs backpatch based on adf height
                                                 // using jpeg_jpeghw_fixup_page_length() callback

struct strip_info_s
{
    struct BigBuffer_s *big_buffer;
    uint32_t lines_in_buffer;
    bool is_last_strip;
};

struct jpeg_page_info_s
{
    uint32_t image_height;
    uint32_t image_width;
    uint8_t quality;
    uint8_t bytes_per_pixel;
    uint8_t mcu_width;
    uint8_t mcu_height;
    bool auto_header;
};

struct jpeg_jpeghw_page_adaptor 
{
    jpeg_page_adaptor_BaseClassMembers; //< put this macro in your subclass

    struct jpeghw_compress_struct compress_info;
    struct jpeghw_compression_mgr jcmgr;
    struct jpeghw_error_mgr jerr;
    bool error;

    FILE* dst_fp; /**< downstream dst stream handle */
};

struct jpeg_jpeghw_strip_adaptor 
{
    jpeg_strip_adaptor_BaseClassMembers;  // put this macro in your subclass 
    struct BigBuffer_s* bb; //< input data as dma_buffer.
};

// externs
extern void logger_init();

#define JPEG_MEM_DST_MGR_BUFFER_SIZE (64 * 1024)   ///< might want to have this be configurable.
void cmgr_init(struct jpeghw_compress_struct * cinfo) {
    struct jpeg_jpeghw_page_adaptor *jpa =
            (struct jpeg_jpeghw_page_adaptor*) cinfo->common.client_context;

    DBG_ASSERT(jpa);
}

void cmgr_term(struct jpeghw_compress_struct * cinfo) {
    struct jpeg_jpeghw_page_adaptor *jpa =
            (struct jpeg_jpeghw_page_adaptor*) cinfo->common.client_context;

    DBG_ASSERT(jpa);
}

struct BigBuffer_s* cmgr_get_output_buffer(struct jpeghw_compress_struct * cinfo) 
{
    struct jpeg_jpeghw_page_adaptor *jpa =
            (struct jpeg_jpeghw_page_adaptor*) cinfo->common.client_context;

    DBG_ASSERT(jpa);

    struct BigBuffer_s* output_buffer = dma_buffer_malloc(0, JPEG_MEM_DST_MGR_BUFFER_SIZE);

    return output_buffer;
}

// This is the function that receives jpeg data from the compressor.
// In this example the data is written to a file.
// Obviously the data could be sent elsewhere if so desired.
bool cmgr_send_output_buffer(struct jpeghw_compress_struct * cinfo, 
                             struct BigBuffer_s* output_buffer, uint32_t bytes_in_buffer) 
{
    struct jpeg_jpeghw_page_adaptor *jpa =
            (struct jpeg_jpeghw_page_adaptor*) cinfo->common.client_context;

    DBG_ASSERT(jpa);
    DBG_ASSERT(output_buffer);

    char *data = dma_buffer_mmap_forcpu(output_buffer);
    if (data)
    {
        //printf("write %d bytes of datadata to %p\n", bytes_in_buffer, jpa->dst_fp);
        fwrite(data, 1, bytes_in_buffer, jpa->dst_fp);
        dma_buffer_unmmap_forcpu(output_buffer);
    }

    BigBuffer_Free(output_buffer);

    return true;
}

void my_error_exit(struct jpeghw_common_struct * cinfo) {
    printf(">>>>>jpeg internal exit");
    // do a longjmp here
    exit(-1);
}

void jpeg_attach_page(struct jpeg_page_adaptor *base_page, void *junk) 
{
   printf("%s\n",__func__);
    struct jpeg_jpeghw_page_adaptor* page =
            (struct jpeg_jpeghw_page_adaptor *) base_page;

    if (page == NULL)
    {
        DBG_PRINTF_ERR("%s() Invalid page parameter!\n", __func__);
        return;
    }

    jpeghw_start_compress(&page->compress_info);
}

void jpeg_send_page(struct jpeg_page_adaptor *base_page, void *junk) 
{
    DBG_PRINTF_INFO("%s(%p)\n", __func__, base_page);

    // send the page information downstream
}

void jpeg_close_page(struct jpeg_page_adaptor *base_page) 
{
    struct jpeg_jpeghw_page_adaptor *jpa =
            (struct jpeg_jpeghw_page_adaptor *) base_page;

    if (jpa == NULL)
    {
        return;
    }

    if (jpeghw_finish_compress(&jpa->compress_info) != e_JPEGHW_SUCCESS)
    {
        printf(">>>>>%s() ERROR - compress failed!\n", __func__);
        jpa->error = true;
    }
}

#if 0
// update img height in results file with given src lines
int jpeg_fixup_page_length(struct jpeg_page_adaptor* base_page) 
{
    if (!base_page)
    {
        return FAIL;
    }

    uint32_t lines_encoded = base_page->strip_out_height;
    struct jpeg_jpeghw_page_adaptor* page =
            (struct jpeg_jpeghw_page_adaptor *) base_page;

    if (lines_encoded != page->compress_info.common.image_height)
    {
        FILE *dst_file = fopen(page->dst_filename, "r+"); // read/write dst
        if (dst_file) { // good file (stream)
            lines_encoded = page_jpeg_fixup_page_length(dst_file, lines_encoded);
        }

        if (dst_file)
            fclose(dst_file);
    }

    return lines_encoded;
}
#endif

void jpeg_attach_from_strip(struct jpeg_page_adaptor *base_page,
        struct jpeg_strip_adaptor *base_strip, void *sinfo)
{
    if (!base_page || !base_strip || !sinfo) return;

    struct jpeg_jpeghw_strip_adaptor *jfsa =
            (struct jpeg_jpeghw_strip_adaptor *) base_strip;

    struct strip_info_s *strip_info = (struct strip_info_s *)sinfo;
    jfsa->bb = strip_info->big_buffer;
    jfsa->lines_in = strip_info->lines_in_buffer;
    jfsa->is_end_of_page = strip_info->is_last_strip;
}

void jpeg_free_in_strip(struct jpeg_strip_adaptor* base_strip) 
{
    struct jpeg_jpeghw_strip_adaptor *strip =
            (struct jpeg_jpeghw_strip_adaptor *) base_strip;
    DBG_PRINTF_INFO("%s() strip_adaptor %p\n", __func__, strip);

    if (strip->bb) {
        DBG_PRINTF_INFO("%s() relinquish strip jsa %p bb %p\n", __func__, strip,
                strip->bb);

        strip->bb = BigBuffer_Free(strip->bb); // reclaim payload
    }
}

void jpeg_recv_strip_jpeghw(struct jpeg_page_adaptor *base_page,
        struct jpeg_strip_adaptor *base_strip, void *sinfo)
{
    //printf("==%s\n",__func__);
    if (!base_page || !base_strip || !sinfo) return;

    struct jpeg_jpeghw_strip_adaptor *strip =
            (struct jpeg_jpeghw_strip_adaptor *) base_strip;
    struct jpeg_jpeghw_page_adaptor* page =
            (struct jpeg_jpeghw_page_adaptor *) base_page;

    strip->adopt_bb_from_blob(base_page, base_strip, sinfo); 

    uint32_t lines=0;

    if (strip->bb)
    {
       //#define aTEST
       #ifdef aTEST
       lines = strip->lines_in; 
       posix_sleep_us(5000);
       #else
       //printf("  state:%d\n", page->compress_info.common.global_jpeg_state);
       lines = jpeghw_write_scanlines(&page->compress_info, strip->bb,
                                 strip->lines_in, strip->is_end_of_page);
       #endif

        //printf("  last error:%d\n", page->compress_info.common.last_error);
        if (lines < strip->lines_in) 
        {
            printf( ">>>>>%s() WARNING - not all Scanlines were processed (%d/%d)!\n",
                    __func__, lines, strip->lines_in);
            page->error = true;

            if (lines == 0) 
            {
                strip->bb = BigBuffer_Free(strip->bb);
            }
        } 
        else 
        {
            jpa_from_page_at_output_eoi(base_page, strip->lines_in);
        }

        #ifdef TEST
        strip->bb = BigBuffer_Free(strip->bb);
        #endif
        strip->bb = NULL; // passed ownership downstream
    }

    // after output with lines computation?   need progress object ?
    strip->free_in_strip(base_strip);
}

static struct jpeg_jpeghw_page_adaptor *jpeg_page_adaptor_constructor(struct jpeg_page_info_s *pinfo) 
{
    struct jpeg_jpeghw_page_adaptor *jpa = 0;

    jpa = (struct jpeg_jpeghw_page_adaptor *) MEM_CALLOC(
            sizeof(struct jpeg_jpeghw_page_adaptor), 1);
    if (!jpa)
        return 0;

    jpa = (struct jpeg_jpeghw_page_adaptor*) construct_jpeg_page_adaptor(
            (struct jpeg_page_adaptor*) jpa);

    bool warning = false;
    jpeghw_error_type_t ret = e_JPEGHW_ERROR;
    do
    {
        if (ret == e_JPEGHW_ERR_NOT_AVAILABLE)
        {
            if (!warning)
            {
                printf("WARNING: Unable to acquire compressor, retrying...\n");
                warning = true;
            }

            posix_sleep_us(100);
        }

        ret = jpeghw_create_compress(&jpa->compress_info, JPEGHW_CONFIG_FLAG_BLOCK_JPEG_CORE_AVAIL);

    } while (ret == e_JPEGHW_ERR_NOT_AVAILABLE);

    if (ret == e_JPEGHW_SUCCESS)
    {
        jpa->adopt_from_page = jpeg_attach_page;
        jpa->jpeg_send_page = jpeg_send_page;
        jpa->page_close = jpeg_close_page;

        // set cmgr callbacks
        jpa->compress_info.cmgr = &jpa->jcmgr;
        jpa->compress_info.cmgr->init = cmgr_init;
        jpa->compress_info.cmgr->get_output_buffer = cmgr_get_output_buffer;
        jpa->compress_info.cmgr->send_output_buffer = cmgr_send_output_buffer;
        jpa->compress_info.cmgr->term = cmgr_term;

        jpa->compress_info.cmgr->get_quant_table = 0;//cmgr_get_quant_table;
        jpa->compress_info.cmgr->get_huff_table = 0; //cmgr_get_huff_table;
        jpa->compress_info.common.scanline_timeout = 0;
        jpa->compress_info.common.progress_monitor = 0;

        jpa->compress_info.common.image_height = pinfo->image_height;
        jpa->compress_info.common.image_width = pinfo->image_width;
        jpa->compress_info.common.bytes_per_pixel = pinfo->bytes_per_pixel;

        /// todo: add error manage to prevent libjpeg from calling exit().
        jpa->jerr.error_exit = my_error_exit;
        jpa->compress_info.common.err = jpeghw_std_error(&jpa->jerr);

        jpa->compress_info.quality = pinfo->quality;
        jpa->compress_info.auto_generate_jpeg_header = pinfo->auto_header;

        // back pointer to allow callbacks to have context connection.
        jpa->compress_info.common.client_context = (void *)jpa;
    }
    else
    {
        memFree(jpa);
        jpa = NULL;
    }

    return (struct jpeg_jpeghw_page_adaptor *) jpa;
}

static void jpeg_page_adaptor_destructor(struct jpeg_jpeghw_page_adaptor *jpa) 
{
    DBG_ASSERT(jpa);

    if ( jpa  )
    {
        jpeghw_destroy_compress(&jpa->compress_info);

        memFree(jpa);
    }
}

static struct jpeg_jpeghw_strip_adaptor *jpeg_strip_adaptor_constructor()
{
    struct jpeg_jpeghw_strip_adaptor *jsa = 0;

    jsa = (struct jpeg_jpeghw_strip_adaptor *) MEM_MALLOC(
            sizeof(struct jpeg_jpeghw_strip_adaptor));
    if (!jsa)
        return NULL;

    jsa = (struct jpeg_jpeghw_strip_adaptor*) construct_jpeg_strip_adaptor(
            (struct jpeg_strip_adaptor*) jsa);

    jsa->adopt_bb_from_blob = jpeg_attach_from_strip;
    jsa->recv_strip = jpeg_recv_strip_jpeghw;

    jsa->free_in_strip = jpeg_free_in_strip; // default

    return (struct jpeg_jpeghw_strip_adaptor *) jsa;
}

static void jpeg_strip_adaptor_destructor(struct jpeg_jpeghw_strip_adaptor *jfsa)
{
    DBG_ASSERT(jfsa);
    if (jfsa)
    {
        memFree(jfsa);
    }
}

#define NUM_COMPRESSORS 2
struct jpeg_adaptors_s
{
   struct jpeg_jpeghw_page_adaptor *page_adaptor;
   struct jpeg_jpeghw_strip_adaptor *strip_adaptor;
};


int jpeg_adaptor_start_page(struct jpeg_adaptors_s *jpeg_adaptors, struct jpeg_page_info_s *pinfo) 
{
    jpeg_adaptors->page_adaptor = jpeg_page_adaptor_constructor(pinfo);

    jpeg_adaptors->strip_adaptor = jpeg_strip_adaptor_constructor();

    jpeg_recv_page((struct jpeg_page_adaptor *)jpeg_adaptors->page_adaptor, (void *)pinfo);

    return 0;
}

void jpeg_adaptor_end_page(struct jpeg_adaptors_s *jpeg_adaptors) 
{
    jpa_from_page_close((struct jpeg_page_adaptor *)jpeg_adaptors->page_adaptor);

    jpeg_strip_adaptor_destructor(jpeg_adaptors->strip_adaptor);
    jpeg_page_adaptor_destructor(jpeg_adaptors->page_adaptor);
}

int jpeg_adaptor_process_strip(struct jpeg_adaptors_s *jpeg_adaptors, struct strip_info_s *sinfo) 
{
   jpeg_recv_strip((struct jpeg_page_adaptor *)jpeg_adaptors->page_adaptor,
                   (struct jpeg_strip_adaptor *)jpeg_adaptors->strip_adaptor, sinfo);

   return 0;
}

void initialize_jpeg_compressor()
{
   memInitMemory(0, 1024 * 1024 * 128); // init memory pool uses dmaalloc.ko, implies 1 process for now.
   logger_init();  // controlled by /home/root/logger_config_file if present
   uio_lib_init(); // init uio uses /dev/uio*
   jpeghw_init();
}

void * open_jpeg_compressor(uint32_t image_width, uint32_t image_height, uint32_t bits_per_pixel)
{
    struct jpeg_adaptors_s *jpeg_adaptors=0;

    printf("%s\n", __func__);

    struct jpeg_page_info_s pinfo;
    pinfo.image_width = image_width;
    pinfo.image_height = image_height;
    pinfo.bytes_per_pixel = bits_per_pixel/8;
    pinfo.mcu_width = pinfo.bytes_per_pixel == 1 ? 8 : 16;
    pinfo.mcu_height = pinfo.mcu_width;
    pinfo.quality = 99;
    pinfo.auto_header = true;

    jpeg_adaptors = malloc(sizeof(struct jpeg_adaptors_s));
    jpeg_adaptor_start_page(jpeg_adaptors, &pinfo);

    return jpeg_adaptors;
}

void close_jpeg_compressor(struct jpeg_adaptors_s *jpeg_adaptors)
{
   printf("%s\n", __func__);
   jpeg_adaptor_end_page(jpeg_adaptors);
}

void destroy_jpeg_compressor(struct jpeg_adaptors_s *jpeg_adaptors)
{
   printf("%s\n", __func__);
   free(jpeg_adaptors);
   jpeghw_terminate();
}

int jpeg_compressor_process_strip(struct jpeg_adaptors_s  *jpeg_adaptors, 
         struct BigBuffer_s *ibb, uint32_t num_bytes, uint32_t num_rows, bool last_buffer)
{
    struct strip_info_s sinfo; 
    sinfo.big_buffer = ibb;
    sinfo.lines_in_buffer = num_rows;
    sinfo.is_last_strip = last_buffer;

    jpeg_adaptor_process_strip(jpeg_adaptors, &sinfo);

    return 0;
}

void * jpeg_compress_thread(void * arg)
{
    int32_t bytes_read=0;
    struct scan_data_s scan_data;
    struct scan_message_s *scan_msg = (struct scan_message_s *)arg;
    mqd_t mqd = 0;
    struct jpeg_adaptors_s *jpeg_adaptors=0;
    struct BigBuffer_s *big_buffer;
    char outputfile[256];
    char mq_name[256];
    bool ofile_is_open=false;

    strcpy(outputfile, scan_msg->output_filename);
    strcpy(mq_name, scan_msg->mq_name);

    mqd = mq_open(mq_name, O_RDONLY);
    if(mqd == -1 && errno == ENOENT)
    {
       printf("ERROR:  %s there is no message queue\n", __func__);
       return NULL;
    }

    while(1) // loop forever
    {
        struct mq_attr attr;
        unsigned int prio=0;

        mq_getattr(mqd, &attr);

        if (attr.mq_curmsgs > 0)
        {
           uint32_t num_bytes=0;

           bytes_read = mq_receive(mqd, (char *)&scan_data, attr.mq_msgsize, &prio);

           if(bytes_read == sizeof(scan_data))
           {
              if (scan_data.data_type == SCAN_STRIP_DATA)
              {
                 num_bytes = scan_data.meta_data.num_rows * 
                             scan_data.meta_data.pixel_width * 
                             scan_data.meta_data.bits_per_pixel / 8;

                 big_buffer = dma_buffer_from_kernel(&scan_data.dma_alloc);

                 //printf(".");
                 //if (scan_data.meta_data.last_buffer) printf("-L-");
                 //fflush(stdout);

                 jpeg_compressor_process_strip(jpeg_adaptors, big_buffer, num_bytes,
                            scan_data.meta_data.num_rows, scan_data.meta_data.last_buffer);
              }
              else if (scan_data.data_type == SCAN_PAGE_START)
              {
                 printf("=-=Start of page received, %s\n", mq_name);
                 printf("         width:%d\n", scan_data.meta_data.pixel_width);
                 printf("         height:%d\n", scan_data.meta_data.num_rows);
                 printf("         bpp:%d\n", scan_data.meta_data.bits_per_pixel);
                 initialize_jpeg_compressor();
                 jpeg_adaptors = open_jpeg_compressor( scan_data.meta_data.pixel_width,
                       scan_data.meta_data.num_rows, scan_data.meta_data.bits_per_pixel);
                 jpeg_adaptors->page_adaptor->dst_fp = fopen(outputfile, "w");
                 ofile_is_open = true;
              }
              else if (scan_data.data_type == SCAN_PAGE_END)
              {
                 printf("=-=End of page received %s\n", mq_name);
                 close_jpeg_compressor(jpeg_adaptors);
                 fclose(jpeg_adaptors->page_adaptor->dst_fp);
                 ofile_is_open = false;
                 destroy_jpeg_compressor(jpeg_adaptors);
              }
              else if (scan_data.data_type == SCAN_THRD_STOP)
              {
                 printf("=-=scan thread stop received %s\n", mq_name);
                 if(ofile_is_open)
                 {
                    fclose(jpeg_adaptors->page_adaptor->dst_fp);
                 }
                 break;
              }
              else
              {
                 printf("=-=scan_data_type:%d\n", scan_data.data_type);
              }
           }
           else printf("message len %d != read len %d\n", sizeof(scan_data), bytes_read);

           if(bytes_read == -1)
           {
              printf("--- read error %s\n", strerror(errno));
              usleep(10);
           }
        }
        else usleep(10);
    }
    
    mq_close(mqd);

    return NULL;
}

