#!/usr/bin/python
#
# ===========================================================================
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Copyright (c) 2011-2015, Marvell International Ltd.
#
# Alternatively, this software may be distributed under the terms of the GNU
# General Public License Version 2, and any use shall comply with the terms and
# conditions of the GPL.  A copy of the GPL is available at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
#
# THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
# IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
# ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
# this warranty disclaimer.
# ================================================================================

# 


# Pretend to be the scan block. Decode and (ascii) graphically display the
# clocks. For now, just want to decode a complete register dump and remind
# myself of which registers are interrelated.
#
# davep 27-July-06
#
# davep 01-Dec-2007
#   add pixel period validation
#

import sys
import re
import StringIO
import codecs

import scif
import bits
import regdump
from register import register, four_field_register

g_debug = 0

#asic="mvss01"
#asic="mv6120"
asic="mv6170"

class FieldOutOfPixelPeriodRange( Exception ) :
    """A register field exceeds the currently configured pixel period."""
    def __init__( self, register_name, register_field, field_value, pixel_period ) :
        self.register_name = register_name
        self.register_field = register_field
        self.field_value = field_value
        self.pixel_period = pixel_period

    def __str__( self ) :
        return "register=%s field=%s value=%d PIXPER=%d" % \
                         ( self.register_name, self.register_field,
                         self.field_value, self.pixel_period )

class pixper_validator_mixin :
    """Validate all applicable register fields against the pixel period."""

    # davep 01-dec-2007; add some sanity checks so I don't make an ass of
    # myself too many more times.
    #
    # "Valid values for each Clk's EDGEi value are from 0 -> SCFG2.PIXPER"
    # (from the C Scan HLD, in the section on the "Sensor Clocks1 #
    # Register")  so let's make sure our clocks are in and under the pixel
    # period
    
    def pixper_validate( self, pixper ) :
        for field in self.pixper_field_names :
            if self.field_values.has_key( field ) :
#                print "validating %s.%s" % (self.name,field)
                # As of 01-Dec-2007, all disabled clocks that compare against
                # the pixel period are set to 0xff (8-bits). Could change in
                # the future but for now, filter against 0xff.
                if self.field_values[field] != 0xff and self.field_values[field] > pixper :
                    raise FieldOutOfPixelPeriodRange( self.name, field,\
                            self.field_values[field], pixper )

        
class register_scfg1_2005( register )  :
    # davep 05-Jul-2012 ; rename to register_scfg1_2005 to indicate from circa
    # 2005 asic scan block. Will need to choose asic to pick the right
    # registers.

    p1clken=0 #< sensor phase 1 clocks enable */
    p2clken=0 #< sensor phase 2 clocks enable */
    rsclken=0 #< sensor reset clocks enable */
    cpclken=0 #< sensor CP clocks enable */
    mclken=0  #< AFE master clock enable */
    vsclken=0 #< AFE VS (video sample?) clock enable */
    clclken=0 #< AFE clamp clock enable */
    lclpen=0  #< AFE line clamp clock enable */
    leden=0   #< enable LEDR, LEDG, LEDB (3 bits) */
    bulben=0  #< enable bulb PBM generator on the LEDB pin */
    ssmen=0  #< sensor clock start signal mask enable (4 bits)
             #          1000=P1Clk 0100=P2Clk 0010=RSClk 0001=SPClk */
    smpol=0  #< sensor clock start signal mask polarities (4 bits)
             #          1000=P1Clk 0100=P2Clk 0010=RSClk 0001=SPClk */
    p1apol=0 #< P1Clk polarity; 0=initial low, asserts high; 1=initial high, asserts low */
    sppol=0  #< start pulse polarity */
    scpol=0  #< sensor clock polarities (4 bits)
             #          1000=P1Clk 0100=P2Clk 0010=RSClk 0001=SPClk */
    acpol=0  #< AFE clock polarities (4 bits)
             #          1000=MClk 0100=VSClk 0010=CD1Clk 0001=LCClk */

    field_names = ( "p1clken", "p2clken", "rsclken", "cpclken", "mclken", "vsclken", "clclken",
        "lclpen", "leden_b", "leden_g", "leden_r", "bulben", "ssmen_p1",
        "ssmen_p2", "ssmen_rs", "ssmen_cp", 
        "smpol_p1", "smpol_p2", "smpol_rs", "smpol_cp", 
        "p1apol", 
        "reserve_1", "reserve_2",
        "sppol",
        "scpol_p1", "scpol_p2", "scpol_rs", "scpol_cp", 
        "acpol_m", "acpol_vs", "acpol_cd1", "acpol_lc", )
              
    def __init__( self, register_value ) :
        # make sure I didn't screw something up 
        assert( len(self.field_names)==32)

        register.__init__( self, register_value )

    def decode( self ) :
        r = self.register_value

        for i in range(31,-1,-1) :
            if g_debug > 0 :
                print "%s %#x %#x" % ( self.field_names[31-i], (1<<i), (1<<i)&r )
            num = (r & (1L<<i)) >> i
            self.field_values[ self.field_names[31-i] ] = num

class register_scfg1_mv6170( register ) : 
    # davep 05-Jul-2012 ; adding Mv6170 SCFG1
    field_names = ( "p1clken", "p2clken", "rsclken", "cpclken", "mclken", "vsclken", "clclken",
        "lclpen", "bulben", "ssmen", "smpol",  "p1apol", "sptog", "sppol", "scpol", "acpol" 
    )
    shift_masks = (
       (31,1), 
       (30,1),
       (29,1),
       (28,1),
       (27,1),
       (26,1),
       (25,1),
       (24,1),
       (20,1),
       (16,0xf),
       (12,0xf),
       (11,1),
       (9,1),
       (8,1),
       (4,0xf),
       (0,0xf)
    )


class register_scfg2_2005( register ) :
    # davep 05-Jul-2012 ; rename to register_scfg2_2005 to indicate from circa
    # 2005 asic scan block. Will need to choose asic to pick the right
    # registers.

    field_names = ( "cmode", "stype", "ssel", "afewidth", "pixper" )

    def decode( self ) :
        r = self.register_value

        self.field_values["cmode"] = (r & scif.SCIF_SCFG2_CMODE(-1)) >> 31
        self.field_values["stype"] = (r & scif.SCIF_SCFG2_STYPE(-1)) >> 30
        self.field_values["ssel"] = (r & scif.SCIF_SCFG2_SSEL(-1)) >> 28
        self.field_values["afewidth"] = (r & scif.SCIF_SCFG2_AFEWIDTH(-1)) >> 27
        self.field_values["pixper"] = (r & scif.SCIF_SCFG2_PIXPER(-1)) 
    
class register_scfg2_mv6170( register ) : 
    # davep 05-Jul-2012 ; adding mv6170
    field_names = ( "cmode", "stype", "ssel", "afewidth", 
    "expmask", "leden1", "leden0", "expdiv",
    "pixper" )

    shift_masks = ( 
        (31,1),
        (30,1),
        (28,0x3),
        (26,0x3),
        (25,1),
        (19,0x7),
        (16,0x7),
        (8,0xff),
        (0,0xff),
    )
    
class register_stcfg1( four_field_register ) :
    field_names = ( "spgen", "mskextd", "ssme1", "ssme2" )

class register_ccfg( register ) : 
    field_names = ( "cpmode", "clmode", "cle1", "cle2", "acle2" )

    def decode( self ) :
        r = self.register_value 

        self.field_values["cpmode"] = ( r & scif.SCIF_CCFG_CPMODE(~0) ) >> 31
        self.field_values["clmode"] = ( r & scif.SCIF_CCFG_CLMODE(~0) ) >> 30
        self.field_values["cle1"] = ( r & scif.SCIF_CCFG_CLE1(~0) ) >> 16
        self.field_values["cle2"] = ( r & scif.SCIF_CCFG_CCLE2(~0) ) >> 8
        self.field_values["acle2"] = ( r & scif.SCIF_CCFG_ACLE2(~0) ) >> 0

class register_sclk1( four_field_register, pixper_validator_mixin ) :
    field_names = ( "p1e1", "p1e2", "p1e3", "p1e4" )
    pixper_field_names = field_names

class register_sclk2( four_field_register, pixper_validator_mixin ) :
    field_names = ( "p1e5", "p1e6", "p1e7", "p1e8" )
    pixper_field_names = field_names

class register_sclk3( four_field_register, pixper_validator_mixin ) :
    field_names = ( "p2e1", "p2e2", "p2e3", "p2e4" )
    pixper_field_names = field_names

class register_sclk4( four_field_register, pixper_validator_mixin ) :
    field_names = ( "p2e5", "p2e6", "p2e7", "p2e8" )
    pixper_field_names = field_names

class register_sclk5( four_field_register, pixper_validator_mixin ) :
    field_names = ( "rse1", "rse2", "rse3", "rse4" )
    pixper_field_names = field_names

class register_sclk6( four_field_register, pixper_validator_mixin ) :
    field_names = ( "cpe1", "cpe2", "cpe3", "cpe4" )
    pixper_field_names = field_names

class register_aclk1( four_field_register, pixper_validator_mixin ) :
    field_names = ( "mce1", "mce2", "mce3", "mce4" )
    pixper_field_names = field_names

class register_aclk2( four_field_register, pixper_validator_mixin ) :
    field_names = ( "mce5", "mce6", "mce7", "mce8" )
    pixper_field_names = field_names

class register_aclk3( four_field_register, pixper_validator_mixin ) :
    field_names = ( "mce9", "mce10", "mce11", "mce12" )
    pixper_field_names = field_names

class register_aclk4( four_field_register, pixper_validator_mixin ) :
    field_names = ( "vse1", "vse2", "vse3", "vse4" )
    pixper_field_names = field_names

class register_aclk5( four_field_register, pixper_validator_mixin ) :
    field_names = ( "cd1e1", "cd1e2", "cd1e3", "cd1e4" )
    pixper_field_names = field_names

class register_aclk6( four_field_register, pixper_validator_mixin ) :
    field_names = ( "mclk13", "mclk14", "mclk15", "mclk16" )
    pixper_field_names = field_names

class register_sstat( register ) :
    field_names = ( "scanyr", "cmdstat", "qempty", "qfull", "qovr", "iovr",
        "afepcomp", "scmdcomp" )

    def decode( self ) :
        r = self.register_value

        self.field_values[ "scanyr" ] = (r & scif.SCIF_SSTAT_SCANYR(-1)) >> 16

        # decode the single bit fields
        for i in range(0,7) :
            if g_debug > 0 :
                print "%s %#x %#x" % ( self.field_names[7-i], (1<<i), (1<<i)&r )
            num = (r & (1<<i)) >> i
            self.field_values[ self.field_names[7-i] ] = num

class register_scanx( register ) :
    field_names = ( "scanx",  )

    def decode( self ) :
        r = self.register_value

        self.field_values[ "scanx" ] = int(r)

class register_scycle_2005( register ) :
    # davep 11-Jan-2011 ; deprecated the C/H ASIC SCYCLE register
    field_names = ( "mstepp", "scycle" )

    def decode( self ) :
        r = self.register_value

        self.field_values["mstepp"] = (r & scif.SCIF_SCYCLE_MSTEPP(-1)) >> 8
        self.field_values["scycle"] = (r & scif.SCIF_SCYCLE_SCYCLE(-1))

class register_scycle( register ) :
    # davep 11-Jan-2011 ; added new (as of Mv6120 ASIC) ExpPer "Exposure
    # Period Data" field
    field_names = ( "expper", "scycle" )

    def decode( self ) :
        r = self.register_value

        self.field_values["expper"] = (r & scif.SCIF_SCYCLE_EXPPER(-1)) >> 16
        self.field_values["scycle"] = (r & scif.SCIF_SCYCLE_SCYCLE(-1))

class register_sctrl( register ) :
    # davep 13-Nov-2012 ; update to SCTRL with the UMSEN, "Unexpected Motor
    # Sync" interrupt 
    field_names = ( "umsen", "cmdqen", "termcmd", "sreset", "afepie", "scmdie",
                    "run", "scanen" )
    shift_masks = ((7,1),
                   (6,1),
                   (5,1),
                   (4,1),
                   (3,1),
                   (2,1),
                   (1,1),
                   (0,1) )

class register_scmd_2005( register ) :
    field_names = ( "ramp", "sdata", "smrun", "sclkper", "scany" )

    def decode( self ) :
        r = self.register_value

        self.field_values["ramp"] = (r & scif.SCIF_SCMD_RAMP(-1)) >> 31 
        self.field_values["sdata"] = (r & scif.SCIF_SCMD_SDATA(-1)) >> 30 
        self.field_values["smrun"] = (r & scif.SCIF_SCMD_SMRUN(-1)) >> 29 
        self.field_values["sclkper"] = (r & scif.SCIF_SCMD_SCLKPER(-1)) >> 16
        self.field_values["scany"] = (r & scif.SCIF_SCMD_SCANY(-1)) 

class mvss01_register_scmd( register_scmd_2005 ) :
    # davep 27-Apr-2011 ; adding Mvss01 SCMD
    name = "SCMD"
    field_names = ( "eoi", "sdata", "sptogval", "sclkper", "scany" )

    shift_masks = ( 
        ( 31, 1 ),
        ( 30, 1 ),
        ( 29, 1 ),
        ( 16, 0x1fff ), # 13 bits 
        (  0, 0xffff ), # 16 bits 
    )

class register_stcfg2( register ) :
    field_names = ( "spe1", "spe2" )

    def decode( self ) :
        r = self.register_value

        self.field_values["spe1"] = (r & scif.SCIF_STCFG2_SPE1(-1)) >> 16
        self.field_values["spe2"] = (r & scif.SCIF_STCFG2_SPE2(-1)) 

class register_stcfg3( register ) :
    field_names = ( "aph1e1", "aph1e2", "aph1e3", "aph1e4" )

    def decode( self ) :
        r = self.register_value

        self.field_values["aph1e1"] = (r & scif.SCIF_STCFG3_APH1E1(-1)) >> 24
        self.field_values["aph1e2"] = (r & scif.SCIF_STCFG3_APH1E2(-1)) >> 16
        self.field_values["aph1e3"] = (r & scif.SCIF_STCFG3_APH1E3(-1)) >> 8
        self.field_values["aph1e4"] = (r & scif.SCIF_STCFG3_APH1E4(-1)) 

class register_stcfg4( register ) :
    field_names = ( "aph1e5", "aph1e6", "aph1e7", "aph1e8" )

    def decode( self ) :
        r = self.register_value

        self.field_values["aph1e5"] = (r & scif.SCIF_STCFG4_APH1E5(-1)) >> 24
        self.field_values["aph1e6"] = (r & scif.SCIF_STCFG4_APH1E6(-1)) >> 16
        self.field_values["aph1e7"] = (r & scif.SCIF_STCFG4_APH1E7(-1)) >> 8
        self.field_values["aph1e8"] = (r & scif.SCIF_STCFG4_APH1E8(-1)) 

class register_mcfg( register ) :
    field_names = ( "dataen", "lsmode", "m1type", "sm1type", "m2type",\
                    "sm2type"  )

    def decode( self ) :
        r = self.register_value

        self.field_values["dataen"] = (r & scif.SCIF_MCFG_DATAEN(-1)) >> 15
        self.field_values["lsmode"] = (r & scif.SCIF_MCFG_LSMODE(-1)) >> 13
        self.field_values["m1type"] = (r & scif.SCIF_MCFG_M1TYPE(-1)) >> 11
        self.field_values["sm1type"] = (r & scif.SCIF_MCFG_SM1TYPE(-1)) >> 8
        self.field_values["m2type"] = (r & scif.SCIF_MCFG_M2TYPE(-1)) >> 3
        self.field_values["sm2type"] = (r & scif.SCIF_MCFG_SM2TYPE(-1)) 

class register_ldata( register, pixper_validator_mixin ) :
    field_names = ( "data_sample_A", "data_code_A", "data_sample_B", "data_code_B" )
    pixper_field_names = ( field_names[0], field_names[2] )

    def decode( self ) :
        r = self.register_value

        a_data = (scif.SCIF_LDATA_DS_A(-1) & r) >> 24
        a_code = (scif.SCIF_LDATA_DC_A(-1) & r) >> 16
        b_data = (scif.SCIF_LDATA_DS_B(-1) & r) >> 8 
        b_code = scif.SCIF_LDATA_DC_B(-1) & r

        self.field_values["data_sample_A"] = a_data
        self.field_values["data_code_A"] = a_code
        self.field_values["data_sample_B"] = b_data
        self.field_values["data_code_B"] = b_code

# davep 07-Jul-2012 ; adding PSEQ1, PSEQ2 (amazed they were missing so long)
class PSEQ_field( object ) : 
    seq_complete = 0
    datatype_complete = 0
    output_valid = 0
    color = 0

    def __init__( self, value ) : 
        self.value = value
        self.decode()

    def decode( self ) : 
        # seq_complete is bit 6
        self.seq_complete = (self.value&0x20) >> 5

        # datatype complete is bit 5
        self.datatype_complete = (self.value&0x10) >> 4

        # output_valid is bit 4
        self.output_valid = (self.value&0x08) >> 3

        # final three bits are color
        self.color = self.value & 0x7

    def __repr__( self ) : 
        s = "S={0} P={1} V={2} C={3}".format( 
                self.seq_complete,
                self.datatype_complete,
                self.output_valid,
                self.color )
        return s

class PSEQ_register( register ) : 
    pseq_list = []
    def decode( self ) : 
        register.decode(self)
        self.pseq_decode_all()

    def verbose_print( self ) : 
        s = register.verbose_print(self)
        s += "\n" + ",".join( str(pseq) for pseq in self.pseq_list )
        return s

    def pseq_decode_all( self) : 
        self.pseq_list = [ 
                self.pseq_decode(self.field_values[name]) \
                    for name in self.field_names if name.startswith("pseq") ]

    def pseq_decode( self, pseq_field ) : 
        return PSEQ_field( pseq_field )

class register_pseq1( PSEQ_register ) : 
    field_names = ( "pseq1", "pseq2", "pseq3", "pseq4" )
    shift_masks = ( 
        ( 24, 0x3f ),
        ( 16, 0x3f ),
        (  8, 0x3f ),
        (  0, 0x3f )
    )

class register_pseq2( PSEQ_register ) : 
    field_names = ( "pseq5", "pseq6")
    shift_masks = ( 
        (  8, 0x3f ),
        (  0, 0x3f )
    )

# davep 15-Mar-2013 ; adding CBI Arbitration Register. Not in all ASICs.
class register_cfgarb( register ) : 
    field_names = ("extenable","parenable",)
    shift_masks = ( 
        ( 16, 0x07 ),
        (  0, 0x03 ),
    )

def split_8bit( value ) :
    """Split a 32-bit value into 4 8-bit values. Returns an array of those
    8-bit values.
    """

    shifts = ( 24, 16, 8, 0 )
    return [ (value & (0xffL<<s)) >> s for s in shifts ] 

def get_afe_ldata( reg_dump, scan_regs ) :
    # davep 19-Dec-2012 ; guess at the ASIC configuration based on number of
    # LDATA registers. Some ASICs have 4, some have 6, some have two sets of
    # six (dual sensor support).
    #
    # We want to build an ordered list of the LDATA values.

    ldata_reg_list = [ (k,hex(reg_dump[k])) for k in reg_dump if "ldata" in k ] 
    if g_debug > 0 :
        print ldata_reg_list

    num_ldata = len(ldata_reg_list)

    # build an ordered list of LDATA registers
    if num_ldata==6:
        # there are six LDATA registers numbered 1 to 6
        # two possibilities for register names:
        #   e.g., "ldata10" vs "ldata1"  (note trailing zero)
        if "ldata10" in reg_dump :
            fmt = "ldata{0}0"
        else :
            fmt = "ldata{0}"
        ldata_names = [ fmt.format(num) for num in range(1,6+1) ]
    elif num_ldata==12 : 
        # There are two sets of six numbered 10,20,30,...,60 (first sensor)
        # and 11,21,31,...61 (second sensor)
        #  LDATA10=0x0001090a  LDATA20=0x13011c0a  LDATA30=0x26012f1a
        #  LDATA40=0xff00ff00  LDATA50=0xff00ff00  LDATA60=0xff00ff00
        #  
        #  LDATA11=0xff00ff00  LDATA21=0xff00ff00  LDATA31=0xff00ff00
        #  LDATA41=0xff00ff00  LDATA51=0xff00ff00  LDATA61=0xff00ff00
        #
        ldata_names = [ "ldata{0}{1}".format(first,second) 
                        for second in range(0,1+1) 
                        for first in range(1,6+1) ]
    else:
        # davep 11-Feb-2013 ; as of this writing, we only have 6 or 12 LDATA 
        # (4 has dropped off the supported chip list)
        assert 0, num_ldata

    if g_debug > 0 :
        print ldata_names

    ldata = []
    for name in ldata_names :
#        print name, "%#010x"%reg_dump[name]
        reg_value = reg_dump[name]
        
        a_data = (scif.SCIF_LDATA_DS_A(-1) & reg_value) >> 24
        a_code = (scif.SCIF_LDATA_DC_A(-1) & reg_value) >> 16
        b_data = (scif.SCIF_LDATA_DS_B(-1) & reg_value) >> 8 
        b_code = scif.SCIF_LDATA_DC_B(-1) & reg_value
        # 0xff indicates a disabled edge
        if a_data != 0xff :
            ldata.append( (a_data,a_code) )
        if b_data != 0xff : 
            ldata.append( (b_data,b_code) )

    return ldata

def parse_8bit_clocks( reg_dump, names ) :

    clocks = []
    for name in names :
#        print name, "%#010x" % reg_dump[name]
        # davep 12-Jul-2011 ; adding support for ACLK6 ; handle dumps without
        # the new ACLK6 register
        if not name in reg_dump : 
            print "parse_8bit_clocks() : ignoring reg \"{0}\" not in register dump".format( name )
            continue

        reg_value = reg_dump[name]

        # note: blanket assuming 0xff indicates a disabled clock edge 
        clocks.extend(  [ reg for reg in split_8bit(reg_value) if reg != 0xff ] )

    return clocks 

def get_afe_mclk( reg_dump, scan_regs ) :
    # davep 12-Jul-2011 ; adding support for ACLK6
    return parse_8bit_clocks( reg_dump, ("aclk1", "aclk2", "aclk3", "aclk6" ) )

def get_afe_vsmp( reg_dump, scan_regs ) :
    return parse_8bit_clocks( reg_dump, ( "aclk4",  ) )

def get_afe_cd1( reg_dump, scan_regs ) :
    return parse_8bit_clocks( reg_dump, ( "aclk5",  ) )

def get_sensor_clocks( reg_dump, scan_regs ) :
    return parse_8bit_clocks( reg_dump, ( "sclk1", "sclk2" ) )

def get_sensor_clocks2( reg_dump, scan_regs ) :
    return parse_8bit_clocks( reg_dump, ( "sclk3", "sclk4" ) )

def get_sensor_rs( reg_dump, scan_regs ) :
    return parse_8bit_clocks( reg_dump, ( "sclk5", ) )
    
def get_sensor_ap1( reg_dump, scan_regs ) :
    return parse_8bit_clocks( reg_dump, ( "stcfg3", ) )

def led2005_pwm( reg_dump ) : 
    # davep 02-Jul-2012 ; decode scanblock circa 2005 
    led = {}
    fields = split_8bit( reg_dump[ "lpwm" ] ) 
    led["pwmdiv"] = fields[0] 

    led["red_duty"] = fields[1]
    led["green_duty"] = fields[2]
    led["blue_duty"] = fields[3]

#    led["red_duty%"] = (float(led["red_duty"]) / 128.0) * 100.0

    led["gates"] = []
    for i in (0,1,2) :
        gate = reg_dump[ "led%d" % i ]
        led["gates"].append( ( int((gate & 0xffff0000L)>>16), gate&0xffffL ) )

    return led
    
def led_pwm0_pwm1_pwm( reg_list ) : 
    # davep 02-Jul-2012 ; decode scan blocks with LPWM0, LPWM1 registers
    pass

def get_led_pwm( reg_dump ) :
    # davep 11-Jan-2011 ; handle scan blocks with no LPWM register
    if reg_dump.has_key("lpwm") : 
        return led2005_pwm( reg_dump ) 

    # davep 02-Jul-2012 ; add support for scan blocks with LPWM0, LPWM1
    if reg_dump.has_key("lpwm0") :
        return led_pwm0_pwm1_pwm( reg_dump )

    return None


def draw_clock( clock, period ) :
    edges = clock["edges"]
    polarity = clock["polarity"]

    clock_str = ""

    up=1
    down=0

    # davep 02-Aug-2012 ; let's try unicode to get some better lines
    # http://en.wikipedia.org/wiki/Box-drawing_character
    # http://www.unicode.org/charts/PDF/U2500.pdf
#    state_chars = ( u'\u2581', u'\u2594' )
#    transition_chars = (u'\u2572',u'\u2571')

    state_chars = ( "_", "-" )
    transition_chars = ('v','^')

    flip = lambda bit : (1,0)[bit]

    state = polarity
    last_state = state 
    for i in range(period) :
        last_state = state
        if i in edges :
            state = flip(state)
#        clock_str += state_chars[state] 
        if last_state != state :         
            if last_state :
                # high -> low
#                clock_str += u'\u2572'
#                clock_str += "v"
                clock_str += transition_chars[down]
            else :
                # low -> high
#                clock_str += u'\u2571'
#                clock_str += '^'
                clock_str += transition_chars[up]
        else : 
            clock_str += state_chars[state] 

    return clock_str

def draw_clocks( list_of_clocks, max_length ) :

    return [ draw_clock( clock, max_length ) for clock in list_of_clocks ]

def old_draw_clocks( list_of_clocks, max_length ) :
    
    if g_debug > 0 :
        print "drawing clocks..."

    clocks = []
    for clk in list_of_clocks : 
        if g_debug > 0 :
            print clk

        iter = clk["edges"].__iter__()

        if len(clk["edges"]) == 0 :
            clocks.append( { "next": 0xff, 
                             "iter": iter, 
                             "str": "", 
                             "state": clk["polarity"] } )
        else :
            
            clocks.append( { "next": iter.next(), 
                             "iter": iter, 
                             "str": "", 
                             "state": clk["polarity"] } )

    state = ( "_", "-" )

    flip = lambda bit : (1,0)[bit]

    for i in range( 0, max_length ) :
        
        for clk in clocks : 
            if i >= clk["next"] : 
                clk["state"] = flip( clk["state"] )
                try :
                    clk["next"] = clk["iter"].next()
                except StopIteration: 
                    clk["next"] = 0xff

            clk["str"] += state[ clk["state"] ] 

    return [ clk["str"] for clk in clocks ]

#def read_reglist( infile ) :
#
#    reg_re = re.compile( "^(\s*\w+=0x[\da-fA-F]+\s*)+$" ) 
#
#    lines = ""
#    line_counter = 0
#    while 1 :
#        line = infile.readline()
#        if len(line) <= 0 :
#            break
#
#        line_counter += 1
#
#        # kill leading/trailing crap
#        line = line.strip()
#
#        # if it doesn't look like a scan register dump, silently drop it (quick
#        # and dirty check)
#        if not reg_re.match( line ) :
#            if g_debug > 0 :
#                print "ignoring invalid register data on line %d" % line_counter
#            continue
#
#        # return a big long string of all the clocks separated by whitespace
#        lines += " " + line 
#
#    return lines
#
#def read_reglist_from_file( infilename ) :
#    infile = open( infilename, "r" )
#    lines = read_reglist( infile )
#    infile.close()
#    return lines

def calculate_sclkper( pixels_per_line, busclk, PIXPER ) :
    # Given a desired line time in number of pixels, the bus clk speed,
    # calculate what SCLKPER must be.
    #
    # I'm sick and tired of trying to understand PIXPER/SCYCLE/SCLKPER over and
    # over and over again.  Went through it for the umpteenth time and I'm now
    # going to write it all up here so I don't have to go through it again.
    #
    # davep 09-Jan-2008
    #
    # NOTE: pixels_per_line may or may not be the number of pixels in the
    # sensor. (WTF!?) Let me explain. In this function, the pixels_per_line is
    # the number of pixels between Start Pulses. When scanning with a motor,
    # the sensor can capture data faster than we can move the motor. So our
    # Start Pulses need to be far enough apart to take into account the time it
    # takes to move to the next line. For example, suppose the sensor has a
    # 500ns pixel.  The sensor could deliver 3200 pixels in 1.6ms.  But suppose
    # the motor takes us 5ms to get to the next line. Thus we need our Start
    # Pulses 5ms apart.  Using d=rt, we have:
    #
    #   num_pixels = (1pixel/500ns) * 5ms
    #   num_pixels = 5ms / 500ns
    #   num_pixels = 5e-3 / 500e-9
    #   num_pixels = 10000
    #
    # Thus pixels_per_line passed to this function would be 10000.
    #
    # r (rate) = one pixel every 500 ns
    # t (time) = 5ms
    #
    # 
    # Another note: any variable IN ALL CAPS is a register or register field.
    # For example, PIXPER is the SCFG2 register's pixel period field. 
    #
    
    # Assumes SCYCLE is hardwired to 31 in the firmware as it has been in the
    # firmware since forever. I don't see us ever changing this.
    SCYCLE = 31

    #
    # Equations from pages 28,29 in the scan block document:
    #
    #  Pix Period = (PIXPER+1) * ClkPeriod   
    #
    #  Scan Clock = 1/((SCLKPER+1)*(Pix Period)) Hz
    #
    #  ScanLine Clock = (Scan Clock) * (1/(SCYCLE+1)) HZ
    #
    #
    # PIXPER, SCYCLE, ClkPeriod are constants.
    # Take those equations, solve for SCLKPER (and get really confused in the
    # process).
    # 

    # how long in seconds for a real pixel
    pixel_period = float(PIXPER+1) * (1 / float(busclk))

    #  BIG FAT NOTE!  THE SCAN CLOCK DOESN'T REALLY EXIST. Its only use in life
    #  is to calculate ScanLine Clock. It's not a signal that can be seen
    #  somewhere. It's confusing and I always get asked if we can see the Scan 
    #  Clock. NO. It's not real.  So this function does its best to hide it.

    # how long in seconds between Start Pulses
    scan_line_period = pixels_per_line * pixel_period

    print "pixel period=%s seconds" % str(pixel_period)
    print "scan line period=%s seconds" % str(scan_line_period)

    # Time to start dusting off my algebra brain. Using a lot of intermediate
    # steps to make the units (frequency, period, seconds, etc) COMPLETELY
    # clear. Anything without a unit is a variable that doesn't reflect
    # anything in the real world (scan_clock, I'm looking at you!).
    scan_line_freq = 1/scan_line_period
    scan_clock = (SCYCLE+1) * scan_line_freq # via eq 2 above

    SCLKPER = 1 / (scan_clock * pixel_period)

    # SCMD.SCLKPER counts from zero so subtract one to make the new value
    # reflect the proper register value.
    SCLKPER = int(round(SCLKPER)) - 1

    return SCLKPER

def tos( *foo ) : 
    return [ str(q) for q in foo ]  

def decode_scan_registers( reg_list, system_clock ) : 
    scan_regs = scif.make_all_registers()

    return_list = []
#    p = lambda *s : return_list.append( "".join( [ "{0}".format(s) for q in s ] ) )
    p = lambda *s : return_list.append( "".join( [ str(q) for q in s ] ) )

    afe_ldata_fields = get_afe_ldata( reg_list, scan_regs )
#    afe_ldata = { "edges" : get_afe_mclk( reg_list, scan_regs ) }
    afe_ldata = {}
    afe_ldata["edges"] = [ n[0] for n in afe_ldata_fields ]
    afe_ldata["codes"] = [ n[1] for n in afe_ldata_fields ]
    afe_ldata["polarity"] = 0  # initial value low
    p( "afe_ldata=", afe_ldata )
    print "afe_ldata=", afe_ldata 

    # AFE master clock edges
    afe_mclks = { "edges" : get_afe_mclk( reg_list, scan_regs ) }
    afe_mclks["polarity"] = (reg_list["scfg1"] & scif.SCIF_SCFG1_ACPOL_MCLK(-1)) > 0 
    p( "afe_mclks=",afe_mclks )

    # AFE video sample edges
    afe_vsmp = { "edges" : get_afe_vsmp( reg_list, scan_regs ) }
    afe_vsmp["polarity"] = (reg_list["scfg1"] & scif.SCIF_SCFG1_ACPOL_VSCLK(-1)) > 0 
    p( "afe_vsmp=", afe_vsmp )

    # AFE CD1 
    afe_cd1 = { "edges" : get_afe_cd1( reg_list, scan_regs ) }
    afe_cd1["polarity"] = (reg_list["scfg1"] & scif.SCIF_SCFG1_ACPOL_CD1CLK(-1)) > 0 
    p( "afe_cd1=", afe_cd1 )
    
    # phase 1 clock edges
    sensor_p1_edges = { "edges" : get_sensor_clocks( reg_list, scan_regs ) }
    sensor_p1_edges["polarity"] = (reg_list["scfg1"] & scif.SCIF_SCFG1_SCPOL_P1CLK(-1)) > 0 
    p( "sensor_p1=", sensor_p1_edges )

    # davep 22-Mar-2013 ; add Phase2 clock edges
    sensor_p2_edges = { "edges" : get_sensor_clocks2( reg_list, scan_regs ) }
    sensor_p2_edges["polarity"] = (reg_list["scfg1"] & scif.SCIF_SCFG1_SCPOL_P2CLK(-1)) > 0 
    p( "sensor_p2=", sensor_p2_edges )

    # sensor reset pulse edges
    sensor_rs = { "edges" :  get_sensor_rs( reg_list, scan_regs ) }
    sensor_rs["polarity"] = (reg_list["scfg1"] & scif.SCIF_SCFG1_SCPOL_RSCLK(-1)) > 0 
    p( "sensor_rs=", sensor_rs )

    # additional phase 1 edges
    sensor_ap1 = { "edges" : get_sensor_ap1( reg_list, scan_regs ) }
    sensor_ap1["polarity"] = 0
    p( "sensor_ap1=", sensor_ap1 )

    led_pwm = get_led_pwm( reg_list )
    if led_pwm : 
        p( "led_pwm=",led_pwm )
        p( "" )
    
    # davep 05-Jul-2012 ; adding new mv6170 scfg[12] and pushing older
    # scfg[12] into new class name
    if asic=="mv6170" : 
        register_scfg1 = register_scfg1_mv6170
        register_scfg2 = register_scfg2_mv6170
    else :
        register_scfg1 = register_scfg1_2005
        register_scfg2 = register_scfg2_2005

    if asic=="mvss01" :
        register_scmd = mvss01_register_scmd
    else :
        register_scmd = register_scmd_2005
    
    # calculate pixel line time, etc. 
    # davep 01-Dec-2007 ; do this before we print because I need PIXPER in the
    # register dump code below
    scfg2 = register_scfg2( reg_list["scfg2"] )
    PIXPER = scfg2.field_values["pixper"] 

    # XXX hardwire PIXPER for experiments
#    PIXPER = 30

    # big dump of bunch of registers; using a tuple because I want to preserve
    # the order
    dump_it = ( ( "scfg1", register_scfg1 ),
                ( "scfg2", register_scfg2 ),
                ( "sctrl", register_sctrl ),
                ( "scmd", register_scmd ),
                ( "sstat", register_sstat ),
                ( "scanx", register_scanx ),
                ( "scycle", register_scycle ),
                ( "stcfg1", register_stcfg1 ),
                ( "stcfg2", register_stcfg2 ),
                ( "stcfg3", register_stcfg3 ),
                ( "stcfg4", register_stcfg4 ),
                ( "ccfg", register_ccfg ),
                ( "sclk1", register_sclk1 ),
                ( "sclk2", register_sclk2 ),
                ( "sclk3", register_sclk3 ),
                ( "sclk4", register_sclk4 ),
                ( "sclk5", register_sclk5 ),
                ( "aclk1", register_aclk1 ), 
                ( "aclk2", register_aclk2 ), 
                ( "aclk3", register_aclk3 ), 
                ( "aclk4", register_aclk4 ), 
                ( "aclk5", register_aclk5 ),
                ( "aclk6", register_aclk6 ),
#                ( "ldata1", register_ldata ),
#                ( "ldata2", register_ldata ),
#                ( "ldata3", register_ldata ),
#                ( "ldata4", register_ldata ),
                ( "pseq1", register_pseq1 ),
                ( "pseq2", register_pseq2 ),
                ( "mcfg", register_mcfg ),

                # davep 15-Mar-2013 ; adding CBI Arbitration Register.
                # Not in all ASICs.
                ( "cfgarb", register_cfgarb ),   
              ) 
    # davep 25-Sep-2007 ; XXX silence this code 
#    dump_it = ()

    pixel_period_error_count = 0
    for reg in dump_it :
        reg_name = reg[0]
        reg_class = reg[1]

        reg = reg_class( reg_list[reg_name] )
        p( reg.verbose_print() )

        # davep 01-Dec-2007 ; adding PIXPER validation
        if reg_class.__dict__.has_key( "pixper_field_names" ) :
            try :
                reg.pixper_validate( PIXPER )
            except FieldOutOfPixelPeriodRange, pp_error :
                print >>sys.stderr, "WARNING! PixelPeriod exceeded! "+str(pp_error)
                pixel_period_error_count += 1

        # blank line between registers
        p( "" )

    scmd = register_scmd( reg_list["scmd"] )
    scycle = register_scycle( reg_list["scycle"] )
    SCLKPER = scmd.field_values["sclkper"]
    SCYCLE = scycle.field_values["scycle"]
    p( "PIXPER=%d SCYCLE=%d SCLKPER=%d" % (PIXPER, SCYCLE, SCLKPER) )

    p( "system clock=",system_clock )
    pixel_period = (PIXPER+1) / system_clock
    p( "real time pixel period=",(pixel_period*1e9)," nanosec" )

    if SCLKPER == 0 : 
        p( "SCLKPER is 0 in the register dump. Usually caused by SCMD not running yet." )
        p( "Without SCLKPER, this script cannot calculate line times." )
    else:
        scan_clock = float((SCLKPER+1)*(PIXPER+1))
        scan_line_clock = scan_clock * (SCYCLE+1)

        p( "scan_clock=%f scan_line_clock=%f" % (scan_clock,scan_line_clock))
        p( "real time scan line=", (scan_line_clock/system_clock)*1e3, "millisec" )

        p( "line time in pixels=",(SCLKPER+1)*(SCYCLE+1) )

    # get strings of some clocks (drawn as asci)
    (mclk_str, vsmp_str, sensor_p1_str, sensor_p2_str, sensor_rs_str ) = \
        draw_clocks( (afe_mclks, afe_vsmp, sensor_p1_edges, sensor_p2_edges, sensor_rs), PIXPER+1 )
    rsmp_str, = draw_clocks( (afe_cd1,), PIXPER+1 )
    afe_ldata_str, = draw_clocks( (afe_ldata,), PIXPER+1 )

    # draw a simple number list
    s = ""
    for i in range(PIXPER+1) :
        s += "%x" % (i%16)
    p( "     ", s*2 )

    p( "pixl=", "*"*(PIXPER+1))
    p( "sclk=",sensor_p1_str*2 )
    p( "p2ck=",sensor_p2_str*2 )
#    p( "  rs=",sensor_rs_str*2 ) )
    p( "mclk=",mclk_str*2 )
    p( "vsmp=",vsmp_str*2 )
    p( "rsmp=",rsmp_str*2 )
    p( "ldat=",afe_ldata_str*2 )

    p( "" )

    # davep 10-Jul-2012 ; adding comma separated clocks suitable for copy/paste
    # into scanclocks.html
    scanclocks_html = (
        ( "P1Clk", sensor_p1_edges ),
        ( "AFE MCLK", afe_mclks ),
        ( "AFE VSMP", afe_vsmp ),
        ( "AFE LDATA", afe_ldata),
    )
    for s in scanclocks_html :
        p( s[0] + ": " + ",".join( "{0}".format(n) for n in s[1]["edges"] ) )

    # if we had any fields outside the PIXPER, print it big bold letters
    # (Make it big and obnoxious for now; will probably tone it down later)
    if pixel_period_error_count > 0 :
        p( "" ) 
        errmsg = "** WARNING! WARNING! WARNING! %d pixel period errors! **" % pixel_period_error_count
        p( "*" * len(errmsg) )
        p( errmsg )
        p( "*" * len(errmsg) )
        p( "" )

    return return_list

def parse( lines, scan_block_clock ) : 
    infile = StringIO.StringIO( "\n".join( lines ) )
    fields = regdump.decode_register_dump( infile )
    infile.close()

    strlist = decode_scan_registers( fields, scan_block_clock ) 
    return strlist 

def main() :

    if len(sys.argv) != 3 :
        print >>sys.stderr, ("usage: %s busclock logfile" % sys.argv[0])
        print >>sys.stderr, ("for example, \"%s 115 log.txt\"" % sys.argv[0])
        sys.exit(1)

    system_clock_e6 = int(sys.argv[1])
    system_clock = float( system_clock_e6 * 1e6)

    infilename = sys.argv[2]
    infile = open(infilename,"r")

    reg_list = regdump.decode_register_dump( infile )
#    print reg_list

    rlist = decode_scan_registers( reg_list, system_clock ) 
    print "\n".join( rlist )

    infile.close()

if __name__ == '__main__' :
    main()

