#!/usr/bin/python
#
# ===========================================================================
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Copyright (c) 2011-2015, Marvell International Ltd.
#
# Alternatively, this software may be distributed under the terms of the GNU
# General Public License Version 2, and any use shall comply with the terms and
# conditions of the GPL.  A copy of the GPL is available at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
#
# THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
# IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
# ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
# this warranty disclaimer.
# ================================================================================

# 


# Functions to parse a register dump like:
# 
#    SCFG1=0xACEA28A8  SCFG2=0x7000001D   SCFG3=0x00000000   SCTRL=0x00000047
#    SCMD=0x40A6001D   SSTAT=0x00130050   SIACK=0x000FFFFF   SCANX=0x00000A4F 
#    SCYCLE=0x00001F1F STCFG1=0x00000062  STCFG2=0x00010061 STCFG3=0x02080E14
#    ...
#
# Moved/consolidated this code from my original scan.py file.
#
# davep 03-Jan-2008
# 
# davep 04-Oct-2010 ; added a 'main'
#
# davep 16-Oct-2013 ; return an OrderedDict from decode_register_dump() so can
#                       preserve the order found in the firmware's dump

import re
from collections import OrderedDict

import hexdump

g_debug = 0

def parse_register_dump( infile ) :
    # Search the file for data that matches the NAME=VALUE paradigm. Gather
    # those lines into a single string and return that string.
     
    # davep 18-Jan-2013 ; add (?:0x)? to the regex to catch register dumps
    # without the leading 0x (some C libraries don't put 0x on %p) 
    reg_re = re.compile( "^(\s*\w+=(?:0x)?[\da-fA-F]+\s*)+$" ) 

    lines = ""
    line_counter = 0
    
    # Note I'm reading to end-of-file.  My current usage is to copy/paste the
    # dump from a terminal window into a small file.  If run against a file
    # with multiple dumps, we'll cheerfully capture any and all register dumps
    # in that file. Probably not what's wanted so be careful with the input.
    while 1 :
        line = infile.readline()

        if len(line) <= 0 :
            break

        line_counter += 1

        # kill leading/trailing crap
        line = line.strip()

        if g_debug > 0 : 
            print "line={0}".format( line )

        # davep 22-Feb-2011 (T minus 4) ; check for the crap I'm getting from
        # screen's logfile (cut/paste changes 0x0d0a into "^M")
        if len(line)>2 and line[-2]=='^' and line[-1]=='M' :
            line = line[:-2]
        
        if len(line) <= 0 :
            continue

        # if it doesn't look like my stadard register dump, silently drop it (quick
        # and dirty check)
        if not reg_re.match( line ) :
            if g_debug > 0 :
                print "ignoring invalid register data on line %d" % line_counter
            continue

        # return a big long string of all the clocks separated by whitespace
        lines += " " + line 

    return lines

def decode_register_dump( infile ) :
    # Parse a register dump like:
    #
    #    SCFG1=0xACEA28A8   SCFG2=0x7000001D   SCFG3=0x00000000   SCTRL=0x00000047
    #
    # and return a hash like:
    #
    # {'scfg1': 2901026984L, 'scfg2': 1879048221, 'stcfg3': 34082324,  
    #   'sctrl': 71 }
    #

    # davep 17-Oct-2013 ; upgrade to an ordered dictionary (requires Python 2.7
    # or greater)
    reg_list = OrderedDict()
#    reg_list = {}

    dump_str = parse_register_dump( infile )

    for reg in dump_str.split() :
        fields = reg.split("=")
        name,value = fields[0].lower(), int(fields[1],16)
        reg_list[name] = value

    return reg_list

def parse_file( infilename ) :
    with open( infilename, "r" ) as infile :
        return decode_register_dump( infile )

def hash_to_class( fields_hash, class_list ) :
    instance_list = []

    for reg_name,reg_class in class_list :
        if reg_name in fields_hash :
            obj = reg_class( fields_hash[reg_name] )
            instance_list.append( obj )
            obj = None

    return instance_list

if __name__=='__main__' :
    import sys

    print decode_register_dump( sys.stdin )


