#!/usr/bin/python
#
# ===========================================================================
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Copyright (c) 2011-2015, Marvell International Ltd.
#
# Alternatively, this software may be distributed under the terms of the GNU
# General Public License Version 2, and any use shall comply with the terms and
# conditions of the GPL.  A copy of the GPL is available at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
#
# THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
# IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
# ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
# this warranty disclaimer.
# ================================================================================

# 


# Send a file to icefile via Python. Rewrite from a C app.
# davep 12-Nov-2010
#
# Rewrote icefile to write to ICETest HW block instead of PIE. Much simpler.
# Add more brains to host application.
# davep 06-Jun-2011 
#
# Use PIL instead of raw pixel files. 
# davep 08-Jun-2011

import sys
import struct
import os
import stat
import array
import socket
from PIL import Image
import StringIO

import hexdump
import connection
import icetest

ICEFILE_PACKET_COOKIE = 0x1b5b356d

ICEFILE_READ_BUFFER_SIZE = 1024*50

ICEFILE_OPEN_MAX_CHANNELS = 6

# TODO move this into icetest.py? pic.py? somewhere?
PIC_ADCN_ANT_COLOR = lambda x : (((x) & 0x7) << 18)
PIC_ADCN_ANT_DATATYPE = lambda x : (((x) & 0x3) << 16)
PIC_ADCN_ANT_DATA = lambda x :(((x) & 0xffff) << 0)

PIC_ADCN_ANT_GET_COLOR = lambda x : (((x) >> 18) & 0x7)
PIC_ADCN_ANT_GET_DATATYPE = lambda x : (((x) >> 16) & 0x3) 
PIC_ADCN_ANT_GET_DATA = lambda x :((x) & 0xffff)

def icefile_open( icefile_list, dim_x, dim_y ) :

    print "icefile_open()" 

    # stupid human checks
    assert len(icefile_list) in (1,3), len(icefile_list)

    for icefile in icefile_list : 
        assert icefile["bits_per_pixel"] in (8,16), icefile["bits_per_pixel"] 

        # make sure all entries have the same size
        assert icefile["total_rows"]==icefile_list[0]["total_rows"]
        assert icefile["pixels_per_row"]==icefile_list[0]["pixels_per_row"]

    channel_list = []
    for icefile in icefile_list : 
        channel_list.append( icefile["icetest_color"] )

    image_info = {
        "total_rows" : dim_y,
        "pixels_per_row" : dim_x,
        "rows_per_buffer" : 60
    }

    print "icefile_open image_info={0}".format( image_info )

    # sanity check what the firmware sent us, hard fail if we find something we
    # don't like 
    #assert image_info["rows_per_buffer"] > 0, image_info["rows_per_buffer"]
    assert image_info["pixels_per_row"] > 0, image_info["pixels_per_row"] 
    assert image_info["total_rows"] > 0, image_info["total_rows"] 
    # these checks mostly for corrupted packets
    #assert image_info["rows_per_buffer"] < 1024, image_info["rows_per_buffer"]
    assert image_info["pixels_per_row"] < 65536, image_info["pixels_per_row"] 
    
    return image_info

def encode_pixels_for_icetest( pixelrow, icetest_color=icetest.PIC_CBI_MEVEN ) : 

#    print "encode pixels len={0} color={1}".format( len(pixelrow), icetest_color )

    # input an 8-bpp pixel
    pixelarray = array.array( "B" )
    pixelarray.fromstring( pixelrow )

    # output a 32-bit icetest pixel+metadata
    icetestarray = array.array( "I" )
    assert icetestarray.itemsize==4, icetestarray.itemsize
    icetestarray.fromlist( [ n<<8 for n in pixelarray ] )

    # first pixel
    icetestarray[0] = PIC_ADCN_ANT_COLOR(icetest_color) \
                  | PIC_ADCN_ANT_DATATYPE( icetest.PIC_CBI_FIRP )\
                  | PIC_ADCN_ANT_DATA( icetestarray[0] )
    # normal pixel
    for i in range(1,len(pixelarray)) :
        icetestarray[i] = PIC_ADCN_ANT_COLOR(icetest_color) \
                      | PIC_ADCN_ANT_DATATYPE( icetest.PIC_CBI_NORMP )\
                      | PIC_ADCN_ANT_DATA( icetestarray[i] )
    # last pixel
    icetestarray[i] = PIC_ADCN_ANT_COLOR(icetest_color) \
                  | PIC_ADCN_ANT_DATATYPE( icetest.PIC_CBI_LASTP )\
                  | PIC_ADCN_ANT_DATA( icetestarray[i] )

    buf = icetestarray.tostring()

    assert len(buf)==len(pixelrow)*4, len(buf)

    return buf

def encode_file( icefile, image_info ) :
    # icefile is the host file
    # image_info is from the firmware

    print "encode_file() {0} image_info={1}".format( icefile["filename"], image_info )

    # The image_info contains the icebuf info from the firmware. We want to
    # send packets that are exactly multiples of the icebuf data size. When the
    # icebuf->data lines up correctly, USB can DMA directly into the buffer
    # saving us many memcpy()'s.

#    icebuf_size = image_info["rows_per_buffer"] * image_info["bytes_per_row"] * 5


#    infile = open( icefile["name"] )
    infile = icefile["infile"]

    row_counter = 0

    while row_counter < image_info["total_rows"] :

        # for now, just send one row at a time; eventually read/encode a bigger
        # chunk
        bytes_to_read = icefile["bytes_per_row"] 

        pixelbuf = infile.read( bytes_to_read )
        if len(pixelbuf)<=0 :
            # end of file
            break

        if len(pixelbuf) != bytes_to_read :
            raise Exception( 
                    "short read from {0}: is your pixels per row correct?".format( 
                        icefile["name"] ) )

        # XXX assume 8-bpp
        pixelbuf_num_pixels = len(pixelbuf)

        row_counter += pixelbuf_num_pixels / icefile["bytes_per_row"]

        assert row_counter <= image_info["total_rows"]
        
        if pixelbuf_num_pixels < image_info["pixels_per_row"] :
            # pad image out to width the firmware expects
            num_pad_pixels = image_info["pixels_per_row"] - pixelbuf_num_pixels
            pad = array.array("B")
            pad.fromlist( [ 0xff ] * num_pad_pixels )
            pixelbuf += pad.tostring()

        assert len(pixelbuf)>=image_info["pixels_per_row"], len(pixelbuf)

        buf = encode_pixels_for_icetest( pixelbuf[0:image_info["pixels_per_row"]], 
                                            icefile["icetest_color"] ) 

#        print "color={0} sent rows={1} remaining={2} of {3}".format( 
#             icefile["icetest_color"], row_counter,
#             image_info["total_rows"]-row_counter, icefile["filename"] ) 

        # more to send
        yield buf

    infile.close()

    # davep 06-Jun-2011 ; pad the bottom of the image to match the necessary
    # rows the firmware needs 
    rows_padding = image_info["total_rows"] - row_counter
    assert rows_padding >=0 , rows_padding 

    if rows_padding > 0 : 
        # the 4 is because we're sending 32-bit values for each pixel
        bytes_padding = image_info["pixels_per_row"] * 4

        print "need rows={0} bytes={1} padding".format( rows_padding, bytes_padding )
        
        # make a row of white pixels, send it repeatedly
        a = array.array( "B" )
        a.fromlist( [0xff]*image_info["pixels_per_row"] )

        buf = encode_pixels_for_icetest( a.tostring(), icefile["icetest_color"] ) 

#        dpkt = DataPacket( icefile["icetest_color"], payload=buf,
#                                num_rows=1, bytes_per_row=len(buf) )
#        dpkt.pack()

        while rows_padding > 0 : 
#            fh.write( dpkt.netbuf )
#            fh.flush()
            rows_padding -= 1
            print "write color={2} rows={0} padding remaining={1}".format( 
                1, rows_padding, icefile["icetest_color"] )

            # more to send
            yield buf

    # done!
    yield None

def icefile_save_files( icefile_list, image_info, outfilename ) :

    print "icefile_save_files() image_info={0}".format( image_info )

    of = open(outfilename, "wb")

    # make a list of our encode_file generators
    genr_list = [ encode_file( icefile, image_info ) for icefile in icefile_list ]
    genr_done_list = [ False ] * len(genr_list)

    buf_list = []

    # how much data is the firmware demanding?
    pixels_per_row = image_info["pixels_per_row"]
    bytes_per_row = pixels_per_row * 4

    print genr_done_list, not reduce(lambda x,y : x and y, genr_done_list) 
    a = not reduce(lambda x,y : x and y, genr_done_list) 

    assert not reduce(lambda x,y : x and y, genr_done_list) 

    total_bytes = 0
    bytes_written = 0

    # while there are any channels not yet done
    while not reduce(lambda x,y : x and y, genr_done_list) : 

        # round robin through each channel, sending data
        for idx,genr in enumerate(genr_list) : 

            # if this channel is done, skip it 
            if genr_done_list[idx] : 
                continue

            buf = genr.next() 

            if buf is None :
                # end of this file; don't talk to this channel anymore
                genr_done_list[idx] = True
                continue

            assert len(buf)==bytes_per_row, (len(buf),bytes_per_row)
            buf_list.append( buf )
            buf = None

            # if we hit our rows-per-buffer count, send what we have
            if len(buf_list)==image_info["rows_per_buffer"] : 
                bytes_written = of.write( "".join(buf_list) )
                total_bytes += len("".join(buf_list))
                buf_list = []

                print "total_bytes_written={0}".format(total_bytes)

    # anything left over, send it now
    if len(buf_list) > 0 :
        of.write( "".join(buf_list) )
        buf_list = []

    of.close();
    print "done saving file(s)"
    
def icefile_close( ) : 
    print "icefile_close()"

def isfile( devfilepath ) :  
    statinfo = os.stat( devfilepath )
    mode = statinfo[ stat.ST_MODE ]
    return stat.S_ISCHR(mode), stat.S_ISSOCK(mode)

def run_icefile( icefile_list, dim_x, dim_y, outfilename ) : 

    image_info = icefile_open( icefile_list, dim_x, dim_y )

    if image_info is None : 
        # open failed
        icefile_close( )
        return

    icefile_save_files( icefile_list, image_info, outfilename )

    icefile_close( )

def usage() : 
    msg = """icefile_prep.py - reformat an image file for use with icefile_linux
Only 8-bpp currently supported. Icefile must be enabled in firmware.

usage:
    icefile_prep.py infile pix_per_row total_rows outfile
Example:
    icefile_prep.py q60.tif 2580 3320 q60.ice
"""
    print msg

def parse_args() : 

    if len(sys.argv) != 5 :
        usage()
        sys.exit(1)

    infilename  = sys.argv[1]
    dim_x       = int(sys.argv[2])
    dim_y       = int(sys.argv[3])
    outfilename = sys.argv[4]

    return infilename, dim_x, dim_y, outfilename

def load_icefile_list( filename ) : 

    icefile_list = []

    im = Image.open( filename ) 
    im.load()
    print im.mode

    # davep 14-Jan-2013 ; attempt to convert RGBA to pure RGB image
    if im.mode=="RGBA":
        im2 = im.convert("RGB")
        print im2.mode
        im = im2
        del im2

    band_names = im.getbands()
    print im.size

    print filename

    # convert PIL band name to Icetest CBI color name
    band_to_icetest_color = {
        'R' : icetest.PIC_CBI_CEVEN_0,
        'G' : icetest.PIC_CBI_CEVEN_1,
        'B' : icetest.PIC_CBI_CEVEN_2,

        'L' : icetest.PIC_CBI_MEVEN,
        'P' : icetest.PIC_CBI_MEVEN,
    }

    for band_name, band in zip(band_names, im.split()) : 
        icefile = {}
        icefile["filename"] = filename
        icefile["icetest_color"] = band_to_icetest_color[band_name]
        icefile["pixels_per_row"] = im.size[0]
        icefile["total_rows"] = im.size[1]
        # assume an 8-bpp pixel
        icefile["bits_per_pixel"] = 8
        icefile["bytes_per_row"] = icefile["pixels_per_row"] / (icefile["bits_per_pixel"]/8)
        icefile["infile"] = StringIO.StringIO( band.tostring() )

        icefile_list.append( icefile )

    return icefile_list

def main() :
#    test()
#    sys.exit(0)

#    test_icefile_encode( filename_list[0], pixels_per_row )
#    sys.exit(0)

    (infilename, dim_x, dim_y, outfilename) = parse_args()

    icefile_list = load_icefile_list( infilename )
    print icefile_list

    run_icefile( icefile_list, dim_x, dim_y, outfilename )
    
if __name__ == '__main__' :
    main()

