#!/usr/bin/python
#
# ===========================================================================
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Copyright (c) 2011-2015, Marvell International Ltd.
#
# Alternatively, this software may be distributed under the terms of the GNU
# General Public License Version 2, and any use shall comply with the terms and
# conditions of the GPL.  A copy of the GPL is available at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
#
# THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
# IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
# ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
# this warranty disclaimer.
# ================================================================================

# 


# decode icetest register dump
#
# davep 25-Mar-2011

import sys

import regdump
from register import register as Register

class TestConfig( Register ) : 
    name = "TestConfig"
    field_names = ( 'pic_rate', 'soft_reset', 'enable', 'mode' )

    shift_masks = ( 
        (8,0xff),
        (2,1),
        (1,1),
        (0,1) )

class TestIDMA_Config( Register ) : 
    name = "IDMA_Config"
    field_names = ('replicate', 'enable', 'out_width', 'burst_len' )

    shift_masks = (
        (24,1),
        (16,1),
        (8,0x07),
        (0,0x03) )

class TestIDMA_Status( Register ) : 
    name = "IDMA_Status"
    field_names = ('soi', 'eoi', 'reset', 'full_cbuf', 'empty_cbuf',
                   'full_dbuf', 'empty_dbuf', 'packer_empty', 'dma_busy' )

    shift_masks = (
        (8,1),
        (7,1),
        (6,1),
        (5,1),
        (4,1),
        (3,1),
        (2,1),
        (1,1),
        (0,1), )

class TestIDMA_line_width( Register ) :
    name = "TestIDMA_line_width"
    field_names = ( "line_width", )
    shift_masks = ( 
        ( 0, 0xffff ),
    )

class TestIDMA_Interrupt( Register ) : 
    field_names = ( "bad_rresp", "bad_bresp", "reset", "eoi", "fin", "who" )
    shift_masks = ( 
        ( 5, 1 ),
        ( 4, 1 ),
        ( 3, 1 ),
        ( 2, 1 ),
        ( 1, 1 ),
        ( 0, 1 )
    )

class TestIDMA_InterruptEnable( TestIDMA_Interrupt ):
    name = "IntEn"

class TestIDMA_InterruptPend( TestIDMA_Interrupt ):
    name = "IntPend"

#/* Color (3 bits) */
PIC_CBI_CEVEN_0 =  0   # /* Color 0 (even or non-staggered) */
PIC_CBI_CEVEN_1 =  1   # /* Color 1 (even or non-staggered) */
PIC_CBI_CEVEN_2 =  2   # /* Color 2 (even or non-staggered) */
PIC_CBI_MEVEN   =  3   # /* Monochrome (even or non-staggereed) */
PIC_CBI_CODD_0  =  4   # /* Color 0 (odd) */
PIC_CBI_CODD_1  =  5   # /* Color 1 (odd) */
PIC_CBI_CODD_2  =  6   # /* Color 2 (odd) */
PIC_CBI_MODD    =  7   # /* Monochrome (odd) */


#/* Datatype (2 bits) */
PIC_CBI_NORMP   =  0   # /* normal data pixel */
PIC_CBI_FIRP    =  1   # /* first data pixel in line */
PIC_CBI_LASTP   =  2   # /* last data pixel in line */
PIC_CBI_EOI     =  3   # /* first data pixel in last line */
#PIC_CBI_FIRPLL  =  3   # /* first data pixel in last line */

color_names = ( 
    # don't bother with the 'even' nonsense because staggered sensors aren't
    # supported anyway so we'll never see the 'odd' kruft
    "red",
    "green",
    "blue",
    "mono",
)

datatype_names = (
    "normp",
    "firp",
    "lastp",
    # meaning of b'11 changed from FIRPLL (First-Pixel-Last-Line) to EOI
    # (End-of-Image) in Gemstone or so
    "eoi"
)

class ICETest_Data( Register ) : 
    """The 32-bit value jammed into PIC ADCN ANT or ICETest IDMA. A 16-bit
    pixel combined with metadata."""

    name = "ICETestData"
    field_names = ( "exposure", "colortype", "datatype", "data" )
    shift_masks = (
        ( 21, 1 ),
        ( 18, 0x07 ),
        ( 16, 0x03 ),
        (  0, 0xffff )
    )

    def __repr__( self ):
        dt_name = datatype_names[ self.datatype ]
        color_name = color_names[ self.colortype ]

        s = ""
        s += "{0}={1} ".format( "exposure", self.exposure )
        s += "{0}={1} ({2}) ".format( "datatype", self.datatype, dt_name )
        s += "{0}={1} ({2}) ".format( "colortype", self.colortype, color_name )
        s += "{0}={1:#06x} ".format( "data", self.data )

        return s

def print_fields( fields_hash ) : 
    print("config={0:#010x}".format( fields_hash["config"] ))
    print("dma_cfg={0:#010x}".format( fields_hash["dmacfg"] ))
    print("dma_status={0:#010x}".format( fields_hash["dmastatus"] ))
    print("dma_line_width={0:#010x}".format( fields_hash["line_width"] ))
    print("dma_int_en={0:#010x}".format( fields_hash["dmaint_en"] ))
    print("dma_int_pend={0:#010x}".format( fields_hash["dmaint_st"] ))

    test_config = TestConfig( fields_hash["config"] )
    print("Config={0:#010x} {1}".format( test_config.register_value, test_config ))

    idma_cfg = TestIDMA_Config( fields_hash["dmacfg"] )
    print("IDMA_Config={0:#010x} {1}".format( idma_cfg.register_value, idma_cfg ))

    idma_status = TestIDMA_Status( fields_hash["dmastatus"] )
    print("IDMA_Status={0:#010x} {1}".format( idma_status.register_value, idma_status ))

    idma_line_width = TestIDMA_line_width( fields_hash["line_width"] )
    print("IDMA_line_width={0:#010x} {1}".format( idma_line_width.register_value, idma_line_width ))

    idma_en = TestIDMA_InterruptEnable( fields_hash["dmaint_en"] )
    print("IDMA_IntEn={0:#010x} {1}".format( idma_en.register_value, idma_en ))
    idma_pend = TestIDMA_InterruptPend( fields_hash["dmaint_st"] )
    print("IDMA_IntPend={0:#010x} {1}".format( idma_pend.register_value, idma_pend ))
    
def main() :
    if len(sys.argv) > 1 : 
        infilename = sys.argv[1]
        fields = regdump.parse_file( infilename )
        print_fields( fields )
    
#    test_config = TestConfig( 0x00006400 )
#    print "Config={0:#010x} {1}".format( test_config.register_value, test_config )
#
#    idma_cfg = TestIDMA_Config( 0x01010502 )
#    print "IDMA_Config={0:#010x} {1}".format( idma_cfg.register_value, idma_cfg )
#
#    idma_status = TestIDMA_Status( 0x00000056 )
#    print "IDMA_Status={0:#010x} {1}".format( idma_status.register_value, idma_status )

if __name__=='__main__':
    main()
