#!/usr/bin/python
#
# ===========================================================================
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Copyright (c) 2014-2015, Marvell International Ltd.
#
# Alternatively, this software may be distributed under the terms of the GNU
# General Public License Version 2, and any use shall comply with the terms and
# conditions of the GPL.  A copy of the GPL is available at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
#
# THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
# IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
# ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
# this warranty disclaimer.
# ================================================================================

# 


# Combines two icefile images together to test dual scan
#
import sys
import struct
import os
import stat
import array
import socket
from PIL import Image
import io

import hexdump
import connection
import icetest

def usage() : 
    msg = """dual_icefile_prep.py - reformat two image files for use with icefile_linux
Only 8-bpp currently supported. Icefile must be enabled in firmware.

usage:
    dual_icefile_prep.py infile_front infile_back pix_per_row total_rows strip_lines outfile
Example:
    icefile_prep.py front.tif back.tif 2580 3300 128 dual.ice
"""
    print(msg)

def parse_args() : 

    if len(sys.argv) != 7 :
        usage()
        sys.exit(1)

    filename_f  = sys.argv[1]
    filename_b  = sys.argv[2]
    dim_x       = int(sys.argv[3])
    dim_y       = int(sys.argv[4])
    strip_lines = int(sys.argv[5])
    outfilename = sys.argv[6]

    return filename_f, filename_b, dim_x, dim_y, strip_lines, outfilename

# values read from binary files are strings in python < 3
# but bytes/int in python 3+ (bytes for the whole buffer, ints for
# the individual values in the buffer
def getintval(val):
    if sys.version_info.major < 3:
        return (ord(val))
    else:
        return (val)

# python2 and 3 write values to binary files very differently
# python3 wants to write a bytes object as bins, and
# python2 writes chars/strs as bins
def writeintvalto_binfile(outfile, val):
    if sys.version_info.major < 3:
        outfile.write(chr(val))
    else:
        outfile.write(bytes([val]))

def main() :
    (filename_f, filename_b, dim_x, dim_y, strip_lines, outfilename) = parse_args()

    dim_y = int((dim_y + strip_lines - 1) / strip_lines) * strip_lines

    outfilename_front = outfilename + ".front"
    outfilename_back = outfilename + ".back"

    os.system("python icefile_prep.py " + filename_f + " " + str(dim_x) + " " + str(dim_y) + " " + outfilename_front)
    os.system("python icefile_prep.py " + filename_b + " " + str(dim_x) + " " + str(dim_y) + " " + outfilename_back)

    front_file = open(outfilename_front, 'rb')
    back_file  = open(outfilename_back, 'rb')

    out_file = open(outfilename, 'wb')

    words_per_strip = dim_x * strip_lines * 4
    total_lines = 0
    which_byte = 0

    print("combining files...")
    while 1:
        buffer = front_file.read(words_per_strip)
        if len(buffer) < words_per_strip:
            break
        out_file.write(buffer)

        buffer = back_file.read(words_per_strip)
        for next_value in buffer:
            next_byte = getintval(next_value)
            if which_byte == 2:
                next_byte = next_byte | 0x80
            writeintvalto_binfile(out_file, next_byte)
            which_byte += 1
            which_byte &= 0x3

        total_lines += strip_lines
        print("lines: " + str(total_lines))

    out_file.close()
    front_file.close()
    back_file.close()

    os.remove(outfilename_front)
    os.remove(outfilename_back)
    
    if ((total_lines != dim_y) and (total_lines != dim_y * 3)):
        print("WARNING - number of lines isn't what we would expect.")

if __name__ == '__main__' :
    main()

