#!/usr/bin/python
#
# ===========================================================================
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Copyright (c) 2011-2015, Marvell International Ltd.
#
# Alternatively, this software may be distributed under the terms of the GNU
# General Public License Version 2, and any use shall comply with the terms and
# conditions of the GPL.  A copy of the GPL is available at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
#
# THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
# IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
# ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
# this warranty disclaimer.
# ================================================================================

# 


import signal 
import os
import errno
import socket
import logging

# Implements some file handle methods so can hide win32file objects behind the
# usual Python read/write methods.
#
# davep Nov/Dec 2009.
#
# $Header: //unity/firmware/modules/common/scan/acl/connection.py#8 $

# the default read timeout in milliseconds
READ_TIMEOUT_MS = 3*1000

class ConnectionError( Exception ) :
    def __init__( self, data_so_far="" ) :
        Exception.__init__(self)

        if 0 :
            print("data_so_far=", hexdump.dump( data_so_far, 16 ))

        # data_so_far is the response string that has accumulated up to the
        # point the connection failed
        self.data_so_far = data_so_far

class ConnectionTimeout( ConnectionError )  :
    def __init__( self, data_so_far="" ) :
        ConnectionError.__init__(self, data_so_far )

class Connection( object ) :
    """Like a file handle.  Created to create non-blocking reads around the
       POSIX and Win32 file handles.  Could probably be extended to network
       sockets, too."""
    
    def __init__( self, connection_handle, timeout_ms=READ_TIMEOUT_MS ) :
        self.connection_handle = connection_handle
        self.timeout_ms = timeout_ms

    def __del__( self ) :
        if self.connection_handle : 
            self.connection_handle.close()

    def read( self, buflen ) :
        # default is to assume a Python file handle
        assert self.connection_handle
        return self.connection_handle.read( buflen )

    def write( self, buf ) :
        assert self.connection_handle
        return self.connection_handle.write( buf )

    def flush( self ) :
        assert self.connection_handle
        return self.connection_handle.flush()


def alarm_handler( signum, stack ) :
    if 0 :
        print("alarm!")

class PosixFileConnection( Connection ) :

    def __init__( self, connection_handle, timeout_ms=READ_TIMEOUT_MS ):
        Connection.__init__( self, connection_handle, timeout_ms )

        # we're going to need the low level file descriptor so we can do
        # interruptible IO
        self.file_descriptor = self.connection_handle.fileno()

    def _read_( self, buflen ) :
        try : 
            buf = os.read( self.file_descriptor, buflen )
        except OSError as e :
            print(e)
            # errno.EINTR is "interrupted system call"
            if e.errno == errno.EINTR :
                raise ConnectionTimeout()
            else :
                # wtf happened!? 
                raise

        return buf

    def read( self, buflen ) :
        # FIXME I shouldn't be capturing/releasing this signal for each and
        # every read. Should read a big buffer of data into a local cache.
        signal.signal( signal.SIGALRM, alarm_handler )

        # alarm() wants seconds so round up our milliseconds 
        signal.alarm( int(round(self.timeout_ms/1000.0)) )

        # call the (potentially child) method to do the read
        buf = self._read_( buflen )
#        try :
#            buf = os.read( self.file_descriptor, buflen )
#        except OSError,e :
#            # errno.EINTR is "interrupted system call"
#            if e.errno == errno.EINTR :
#                raise ConnectionTimeout()
#            else :
#                # wtf happened!? 
#                raise

        # cancel the timeout alarm
        signal.alarm( 0 )

        signal.signal( signal.SIGALRM, signal.SIG_DFL )

        return buf

    def write( self, buf ) :

#        print "write {0}".format( len(buf) )

        signal.signal( signal.SIGALRM, alarm_handler )

        # alarm() wants seconds so round up our milliseconds 
        signal.alarm( int(round(self.timeout_ms/1000.0)) )

        self.connection_handle.write( buf )

        # cancel the timeout alarm
        signal.alarm( 0 )

        signal.signal( signal.SIGALRM, signal.SIG_DFL )

#        print "write finished"

#    def flush( self ) : 
#        print "file flush"  

    def close( self ) :
        self.connection_handle.close()

class SocketConnection( PosixFileConnection ) :
    def __init__( self, connection_handle ):
        PosixFileConnection.__init__( self, connection_handle )

        # self.connection_handle is a socket.socket() object

    def _read_( self, buflen ) :
        try : 
            buf = self.connection_handle.recv( buflen )
        except socket.error as e :
            # errno.EINTR is "interrupted system call"
            if e.errno == errno.EINTR :
                raise ConnectionTimeout()
            else :
                # wtf happened!? 
                raise

        return buf

    def write( self, buf ) :
        # TODO add timeout
        return self.connection_handle.send( buf )

    def flush( self ) :
        pass


def test_unix_socket() :
    # davep 01-Mar-2010 ; adding unix domain socket suppor
    logging.basicConfig( level=logging.DEBUG )

    s = socket.socket( socket.AF_UNIX, socket.SOCK_STREAM )

    # assume my usbproxy
    s.connect( "usbproxysock" )

    c = SocketConnection( s )

    import acl
    import acldump
    acl.get_fwversion( c )
    acl.get_build_info( c )
    acldump.dump( c )

    s.close()

def test_linux() :
    # run some Linux tests
    fh = open( "/dev/usb/lp0", "r+b" )
    conn_handle = PosixFileConnection( fh )

    # should fail with a ConnectionTimeout
    print("trying read with timeout... (should fail)")
    try : 
        buf = conn_handle.read( 1024 )
    except ConnectionTimeout as e :
        pass

    import acl
    import pjl

    acl.get_fwversion( conn_handle )

    # acl.py's PJL reads character by character so it's a good torture test
    try :
        pjl.pjl_get_info( conn_handle )
    except ConnectionTimeout as e :
        print(e)

    conn_handle.close()
    
def test_win32() : 
    # Test using a regular file
    import win32connection

    win32connection.run_tests()

if __name__=='__main__' :
    import sys

    if sys.platform[:5] == 'linux':
#        test_unix_socket()
        test_linux()
    elif sys.platform[:3] == 'win':
        test_win32()

