#!/usr/bin/python

#
# ===========================================================================
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Copyright (c) 2011-2015, Marvell International Ltd.
#
# Alternatively, this software may be distributed under the terms of the GNU
# General Public License Version 2, and any use shall comply with the terms and
# conditions of the GPL.  A copy of the GPL is available at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
#
# THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
# IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
# ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
# this warranty disclaimer.
# ================================================================================

# 
# Simple script to parse a scan of the Kodak Q-60 Color Image Target
# David Poole 13-Mar-2006

# Read a file of 8-bit pixels (e.g., the Red plane). Using the measured offsets
# of the image and the DPI, find the center of each color square. 
#
# Calculate the average of an NxN matrix around the center pixel.
#
# Measuring an original, each large color square (e.g., A1) is 1/4" square. 
# Upper left corner of A1 is at (44/60" x 37/60").
#
# Measuring a scan (our margins will clip some of the top and left), the upper
# left corner of A1 is at pixel [214,174] at 300 DPI. 
#
# Version History:
# 1.0.0  davep 14-Mar-2006  
#   First public appearance
#
# 1.0.1  davep 14-Mar-2006
#   Added parsing of Grayscale strip 
#
# 1.5.0  davep 15-Aug-2006
#   Added calculation of mean and standard deviation
#
# 2.0.0 davep 06-Apr-2015
#   Update to Python3, PIL, NumPy. Merge other Python scripts into this script
#   so only need to run one script. 

import sys
import os
import getopt
import math
import numpy as np
from PIL import Image
import logging
import csv

import color
import plot
import gretag as it8

dlog = logging.getLogger("q60")

VERSION = "2.0.0"

default_it8_file = "R2200703.Q60"

g_valid_dpi_list = (150,300,600,1200)

# origin is top,left corner of A1
q60_image_info = {
   300  : {  "dpi" : 300,
#             "origin_x" : 204,
             "origin_x" : 195,
             "origin_y" : 192,
#             "origin_y" : 174,
             "pixels_per_square_x" : 77,
             "pixels_per_square_y" : 76,
             "sample_size" : 20,
             "pixels_per_row" : 2544
           },
}

# constants
q60_rows = ( 'A', 'B', 'C', 'D', 'E' , 'F', 'G', 'H', 'I', 'J', 'K', 'L' )
q60_columns = list(range( 0, 24))
#q60_columns = range( 0, 6 )

# user can tell us to make little thumbnail images of all the sample squares
g_make_thumbnails = False

g_make_csv = False

# don't send these fields to the CSV file; (I still capture the data to make
# thumbnails)
g_squares_to_ignore = [ "%s%d" % (x,y) for x in q60_rows for y in (0,23) ]
face_r = ( 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H' )
face_c = ( 20, 21, 22 )
face = [ "%s%d" % (x,y) for x in face_r for y in face_c ]
g_squares_to_ignore.extend( face )


def write_csv(sample_data,color_name) : 
    outfilename = "{0}.csv".format(color_name)

    names = ( "name", "mean", "median", "stddev" )
    with open(outfilename,"w") as outfile :
        writer = csv.DictWriter( outfile, names )
        writer.writeheader()
        for sample in sample_data : 
            if sample["name"] not in g_squares_to_ignore :
                writer.writerow( sample )

    dlog.info("wrote {0}".format(outfilename))

def parse_row( image_info, row_ndata, row_name, color_name ) :
    squares_data = []
    for column_of_square in q60_columns : 
        outfile_basename = "{0}{1:02}_{2}".format(row_name,column_of_square,color_name[0])
        outfile_ext = "dat"
        outfile_name = outfile_basename + "." + outfile_ext

        # skip the empty left edge of the image
        col = image_info["origin_x"]

        # move in to our target square
        col += (column_of_square-1) * image_info["pixels_per_square_x"]

        # sample around the midpoint of each square
        col += image_info["pixels_per_square_x"]//2 - image_info["sample_size"]//2

        # sample halfway down into each square
        row = row_ndata.shape[0]//2 - image_info["sample_size"]//2

        pixels_ndata = row_ndata[ row:row+image_info["sample_size"], col:col+image_info["sample_size"] ]

        # make a name that closely matches the Kodak Q60 reference data file field
        if row_name=="GS" :
            name = "{0}{1}".format(row_name,column_of_square)
        else:
            name = "{0}{1:02}".format(row_name,column_of_square)

        squares_data.append( { "name": name, 
                                 "mean": pixels_ndata.mean(), 
                                 "median": np.median(pixels_ndata),
                                 "stddev" : pixels_ndata.std() } )

        if g_make_thumbnails :
            outfilename = "{0}.tif".format(outfile_basename)
            save_image( pixels_ndata, outfilename )

    return squares_data

def load_image(infilename):
    img = Image.open(infilename)
    img.load()
    dlog.debug("\"{0}\" rows={1} cols={2} mode={3} bands={4}".format(
        infilename,img.size[1],img.size[0],img.mode,img.getbands()))

    ndata = np.asarray(img,dtype="uint8")
    dlog.debug("ndata shape={0} min={1} max={2} mean={3} std={4}".format(
        ndata.shape,ndata.min(),ndata.max(),ndata.mean(),ndata.std()))

    return ndata

def save_image(ndata,outfilename):
    img = Image.fromarray( ndata, "L" )
    img.save( outfilename )
    dlog.info("wrote {0}".format(outfilename))

def parse_plane( image_info, color_plane, color_name ) :
    sample_data = []
    row = image_info["origin_y"]
    for row_name in q60_rows : 
        row_ndata = color_plane[row:row+image_info["pixels_per_square_y"],:]

        if g_make_thumbnails :
            outfilename = "row{0}_{1}.tif".format(row_name,color_name[0])
            save_image( row_ndata, outfilename )

        row_data = parse_row( image_info, row_ndata, row_name, color_name ) 
        sample_data.extend( row_data )
        
        row += image_info["pixels_per_square_y"] 

    # size of an entire row of squares

    # Parse grayscale strip; it's two rows down from our position when we leave
    # the previous loop. The grayscale strip is twice as tall as the previous
    # squares so we'll just drop down to the bottom of it and treat it as
    # another row of squares.
    
    row += image_info["pixels_per_square_y"] * 2
    row_ndata = color_plane[row:row+image_info["pixels_per_square_y"],:]
    row_data = parse_row( image_info, row_ndata, "GS", color_name )
    sample_data.extend( row_data )

    write_csv( sample_data, color_name )
    return sample_data

def parse_image( image_info, infilename ) :
    ndata = load_image(infilename)

    rgb_data = [ parse_plane(image_info,ndata[:,:,idx],color_name) for idx,color_name in enumerate(("red","green","blue"))]

    def rgb_data_iter(rgb_data):
        for i in range(len(rgb_data[0])):
            yield rgb_data[0][i]["name"],rgb_data[0][i]["mean"],rgb_data[1][i]["mean"],rgb_data[2][i]["mean"]

    rgb = rgb_data_iter(rgb_data);

    data_dict = {}
    for patchname,r,g,b in rgb: 
        data_dict[patchname] = { "rgb": (r,g,b) }

    return data_dict

def load_it8_file(infilename):
    # The Q60 reference data has a few records we have under a different name.
    # IT8 Dmin == our GS00
    # IT8 Dmax == our GS23
    # So create duplicate records with our names to make our life easier down the road.
    q60_data = it8.parse(infilename)

    for i in range(len(q60_data)) : 
        record = q60_data[i]
        if record["SAMPLE_ID"] == "Dmin" :
            record["SAMPLE_ID"] = "GS0"
        elif record["SAMPLE_ID"] == "Dmax" :
            record["SAMPLE_ID"] = "GS23"

    return q60_data

def plot_image( image_info, infilename, it8_filename ) :
    data_dict = parse_image( image_info, infilename )

    # convert the RGB to Lab
    plot_dict = {}
    for patch in data_dict :
        print(patch)
        XYZ = color.RGB_to_XYZ( data_dict[patch]["rgb"] )
        plot_dict[patch] = { "test": color.XYZ_to_Lab( XYZ ) }

    # make a 
    it8_data = load_it8_file(it8_filename)
    for target in it8_data:
        patch_name = target["SAMPLE_ID"]
        target_Lab = ( target["LAB_L"], target["LAB_A"], target["LAB_B"] )
        assert patch_name in plot_dict, patch_name
        plot_dict[patch_name]["target"] = target_Lab

    plot.plot( plot_dict )

def usage() : 
    print("q60.py v{0}".format(VERSION),file=sys.stderr)
    s="""Parse 8-bit pixel planar data of Kodak Q-60 test target.
Create graphs comparing scanned image to target values. Optionally create thumbnails 
of each Q60 patch. Optionally create CSV file of each channels' RGB statistics.

usage:
q60.py [options] image_filename IT8_filename
  -i filename    Input filename of 8-bit pixels (required).
  -o filename    Output filename for CSV values of pixel means.
  -t             Thumbnails. Create thumbnails of each sample.
  -c             CSV. Write CSV of red,green,blue channels' statistics.
  -d dpi         Integer, DPI of scan. Currently 150,300,600,1200 are supported.

image_filename  Image of scanned Q60
It8_filename    .Q60 file with XYZ,Lab of the Q60 target (default={0})
""".format(default_it8_file)

    print(s,file=sys.stderr)
    
def parse_args() : 
    if len(sys.argv)==1 :
        usage()
        sys.exit(0)

    cmdline_args = {}

    try :
        opts,args = getopt.getopt( sys.argv[1:], "htcd:" )
    except getopt.GetoptError as e:
        print(e)
        print("Use -h for help.")
        sys.exit(1)

    cmdline_args["make_thumbnails"] = False
    cmdline_args["make_csv"] = False

    for opt,arg in opts :
        if opt=="-h":
            usage()
            sys.exit(0)

        elif opt in ( "-t" ) :
            cmdline_args["make_thumbnails"] = True

        elif opt in ( "-c" ) :
            cmdline_args["make_csv"] = True

        elif opt in ("-d" ) :
            try : 
                dpi = int(arg)
            except ValueError :
                print("Invalid integer \"%s\" for dpi." % arg, file=sys.stderr)
                sys.exit(1)
            if dpi not in g_valid_dpi_list :
                dpistr = ",".join( [ str(x) for x in g_valid_dpi_list ] ) 
                print("Unsupported DPI \"%d\". Supported DPIs are %s." % (dpi, dpistr), file=sys.stderr)
                sys.exit(1)

            cmdline_args["dpi"] = dpi

    if len(args) == 1 :
        cmdline_args["infilename"] = args[0]
        cmdline_args["it8_filename"] = default_it8_file
    elif len(args) == 2 :
        cmdline_args["infilename"] = args[0]
        cmdline_args["it8_filename"] = args[1]
    else:
        print("invalid command line: need two input filenames.", file=sys.stderr)
        sys.exit(1)

    return cmdline_args

def main() :

    args = parse_args()

    dpi = args.get( "dpi", 300 )

    global g_make_thumbnails
    global g_make_csv
    g_make_thumbnails = args["make_thumbnails"]
    g_make_csv = args["make_csv"]

    plot_image( q60_image_info[dpi], args["infilename"], args["it8_filename"] )

if __name__ == '__main__':
    fmt = "%(filename)s %(name)s %(message)s"
#    fmt = "%(filename)s %(lineno)d %(name)s %(message)s"
#    logging.basicConfig( level=logging.INFO, format=fmt )
    logging.basicConfig( level=logging.DEBUG, format=fmt )

    main()

