#!/usr/bin/python
#
# ===========================================================================
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Copyright (c) 2011-2015, Marvell International Ltd.
#
# Alternatively, this software may be distributed under the terms of the GNU
# General Public License Version 2, and any use shall comply with the terms and
# conditions of the GPL.  A copy of the GPL is available at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
#
# THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
# IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
# ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
# this warranty disclaimer.
# ================================================================================

# 


# simple script to parse a GretagMacbeth (tm) SpectroScan CSV file
# davep 5-Apr-06
#
# Format is tab separated data and is pretty simple to read. Fields are between
# a "BEGIN_DATA_FORMAT" and "END_DATA_FORMAT" pair. Data is between
# "BEGIN_DATA" and "END_DATA" pair. 
#
# Find the data format. Use as keys for a hash of each line of data. Store the
# data in an array.
#
# Extended to calculate XYZ to ??? values.
# davep 25-May-06
#
# Added ability to parse files using space as record separator and not just
# tab.
# davep 15-Aug-06
#
# $Id: gretag.py 406 2008-05-13 13:40:07Z davep $

import sys

import color

verbose = 0
debug = 0

def _look_for( fileh, string ) :
    if debug :
        print("looking for \"%s\"..." % string)

    while 1 :
        line = fileh.readline()

        if len(line) == 0 :
            break

        line = line.strip()

#        if verbose > 1 :
#            print line

        if line==string :
            break

    if line != string :
        return None
    return string

def parse( filename ) :

    if debug :
        print("opening file \"%s\"" % filename) 

    f = open( filename, "r" )

    # look for "BEGIN_DATA_FORMAT" field
    line = _look_for( f, "BEGIN_DATA_FORMAT" )
    if line is None :
        raise Exception("No BEGIN_DATA_FORMAT field")

    # data format can be on multiple lines; read until we get a line with
    # "END_DATA_FORMAT". Split each line, append fields to the data_fields[]
    # array
    data_fields = []
    while 1 :
        # read the data format line
        line = f.readline().strip()
        if len(line) <= 0 :
            # end of file or error
            raise Exception("Missing END_DATA_FORMAT field")
            
        if line=="END_DATA_FORMAT" :
            break

        # split either on spaces or tabs
        if line.find( "\t" ) > -1 :
            separator = "\t"
        else :
            separator = " "

        data_fields.extend(  line.split( separator ) )

    if verbose > 0 or debug > 0 :
        print(data_fields) 
    assert len(data_fields)>1 

    # we'll make an array of hashes with the data fields as keys
    data_array = []

#    line = _look_for( f, "END_DATA_FORMAT" )
#    if line is None :
#        raise "Missing END_DATA_FORMAT field"

    # look for the start of data
    line = _look_for( f, "BEGIN_DATA" )
    if line is None :
        raise Exception("Missing BEGIN_DATA field")

    line_number = 0
    # read data
    while 1 :
        line = f.readline()
        line_number += 1

        if len(line) <= 0 :
            break

        line = line.strip()

        # skip comments
        if line[0] == "#" : 
            continue

        # look for the end of data marker
        if line=="END_DATA" :
            break

        # doing the weird stuff with the list comprehension to kill those extra
        # empty fields that seem to pop up if there are two separators in a row
        data = [ field for field in line.split( separator ) if len(field)>0 ]

        if len(data) != len(data_fields) :
            print("it8 parse failure at line %d of %s" % (line_number, filename), file=sys.stderr)
            print("length of data (%d) doesn't match data fields (%d)" % (len(data), len(data_fields)), file=sys.stderr)
        hash = {}

        for i in range(0,len(data_fields)) :
            hash[ data_fields[i] ] = data[i].strip()

        data_array.append( hash )
    
    f.close()

    return data_array


def test_gretag_parse() :
    if len(sys.argv) < 2 :
        print("usage: %s gretag_csv_file_to_parse.csv [field(s) to print]" % sys.argv[0], file=sys.stderr)
        print("example:", file=sys.stderr)
        print("%s q60.csv A10 B10 C10" % sys.argv[0], file=sys.stderr)
        print("will parse 'q60.csv' and print the fields A10,B10,C10", file=sys.stderr)
        sys.exit(1)

    filename = sys.argv[1] 

    fields = parse( filename )

    for f in fields :
        print(f) 

    print()
    print(len(list(fields[0].keys())), "fields:")
    print(list(fields[0].keys()))
    print(len(fields),"records found.")

    printme = sys.argv[2:]
    if len(printme) > 0 :
        for f in fields :
            for p in printme :
                print(f[p], end=' ')
            print()
        
if __name__ == "__main__" :
    test_gretag_parse()

