#!/usr/bin/python
#
# ===========================================================================
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this file,
# You can obtain one at http://mozilla.org/MPL/2.0/.
#
# Copyright (c) 2011-2015, Marvell International Ltd.
#
# Alternatively, this software may be distributed under the terms of the GNU
# General Public License Version 2, and any use shall comply with the terms and
# conditions of the GPL.  A copy of the GPL is available at
# http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
#
# THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
# IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
# ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
# this warranty disclaimer.
# ================================================================================

# 


# Various color space conversion functions.  Moved into own script today.
#
# davep 28-July-06
# $Id: color.py 406 2008-05-13 13:40:07Z davep $
  
import sys

# Observer = 2 degrees, Illuminant = D65  from Table 3-3
# _Understanding Color Management_, Sharma. ISBN 1401814476
#
# Also http://www.easyrgb.com/math.php?MATH=M15#text15
# Also _The Measurement of Colour_ 3e, Hunt. ISBN 0863433871 page 264
D65_X = 95.047
D65_Y = 100.0
D65_Z = 108.883 

# Illuminant=D50 
# From _The Measurement of Colour_ 3e, Hunt. ISBN 0863433871 page 264
D50_X = 96.42
D50_Y = 100.0
D50_Z = 82.49

# Note that I'm usually hardwiring to D65 illumination here.
illuminant = ( D65_X, D65_Y, D65_Z )
#illuminant = ( D50_X, D50_Y, D50_Z )

# textbook RGB -> YCC as we're using in the firmware
rgb_to_ycc = [
     0.2990,  0.5870,  0.1140,
    -0.1687, -0.3313,  0.5000,
     0.5000, -0.4187, -0.0813,
]
 
# http://www.couleur.org/index.php?page=transformations#YCbCr
x_srgb_to_ycc = [
    # usually normalized by 2**12 in PIE
     0.2989,  0.5866,  0.1145,
    -0.1688, -0.3312,  0.5000,
     0.5000, -0.4184, -0.0816,
]

# textbook YCC -> RGB 
ycc_to_rgb = [
    1.0,  0,       1.4022,
    1.0, -0.3456, -0.7145,
    1.0,  1.7710,  0,
]

def xassert( flag, value ) :
    if not flag :
        raise AssertionError( value )

def make_triplet( *args ) :
#    if type(args[0])==type( 0 ) or type(args[0])==type(0.0) :
    if len(args) == 3 :
        # function( 10, 10, 10 ) or function(10.0,10.0,10.0)
        a = args[0]
        b = args[1]
        c = args[2]
    elif type(args[0]) == type( () ) or type(args[0]) == type( [] ):
        # function( (10, 10, 10) )
        a = args[0][0]
        b = args[0][1]
        c = args[0][2]
    else :
        xassert( 0, len(args) )

    return (a,b,c)

def XYZ_to_Lab( *args ) :
    # Can't remember where I originally found this formula but have confirmed
    # it in R.W.G.Hunt's _The Reproduction of Color_, 5e, p777.
    # 
    # davep 16-Aug-06 
    
    (X,Y,Z) = make_triplet( *args ) 

    # cube root of ratio
    X_term = pow( (X/illuminant[0]), (1.0/3.0))
    Y_term = pow( (Y/illuminant[1]), (1.0/3.0))
    Z_term = pow( (Z/illuminant[2]), (1.0/3.0))

    L_star = 116.0 * Y_term - 16
    a_star = 500.0 * ( X_term - Y_term )
    b_star = 200.0 * ( Y_term - Z_term )

    # davep 16-Aug-06 ; enforce clipping
    # L = Luminance [0,100]
    # a = [-128 (green), 127 (red)]
    # b = [-128 (blue), 127 (yellow)]
    L_star = min( 100, max( 0, L_star ))
    a_star = min( 127, max( -128, a_star ))
    b_star = min( 127, max( -128, b_star ))

    return ( L_star, a_star, b_star )

def Lab_to_XYZ( *args ) :
    # http://www.easyrgb.com/math.php?MATH=M8#text8
    
    (L_star, a_star, b_star ) = make_triplet( *args ) 

    var_Y = ( L_star + 16 ) / 116.0
    var_X = a_star / 500.0 + var_Y
    var_Z = var_Y - b_star / 200.0

    var_Y_cubed = pow( var_Y, 3.0 )
    var_X_cubed = pow( var_X, 3.0 )
    var_Z_cubed = pow( var_Z, 3.0 )

    if var_Y_cubed > 0.008856 :
        var_Y = var_Y_cubed
    else : 
        var_Y = ( var_Y - 16 / 116 ) / 7.787

    if var_X_cubed > 0.008856 :
        var_X = var_X_cubed
    else :
        var_X = ( var_X - 16 / 116 ) / 7.787

    if var_Z_cubed > 0.008856 :
        var_Z = var_Z_cubed
    else :
        var_Z = ( var_Z - 16 / 116 ) / 7.787

    X = illuminant[0] * var_X 
    Y = illuminant[1] * var_Y
    Z = illuminant[2] * var_Z

    return (X,Y,Z)

def XYZ_to_RGB( *args ) :
    # http://www.easyrgb.com/math.php?MATH=M1#text1

    (X,Y,Z) = make_triplet( *args )

    # Observer = 2 degrees, Illuminant = D65
    var_X = X / 100.0        # Where X = 0 /  95.047
    var_Y = Y / 100.0        # Where Y = 0 / 100.000
    var_Z = Z / 100.0        # Where Z = 0 / 108.883

    var_R = var_X *  3.2406 + var_Y * -1.5372 + var_Z * -0.4986
    var_G = var_X * -0.9689 + var_Y *  1.8758 + var_Z *  0.0415
    var_B = var_X *  0.0557 + var_Y * -0.2040 + var_Z *  1.0570

    if var_R > 0.0031308 : 
        var_R = 1.055 * pow( var_R , ( 1 / 2.4 ) ) - 0.055
    else :
        var_R = 12.92 * var_R

    if var_G > 0.0031308 :
        var_G = 1.055 * pow( var_G , ( 1 / 2.4 ) ) - 0.055
    else :
        var_G = 12.92 * var_G

    if var_B > 0.0031308 :
        var_B = 1.055 * pow( var_B , ( 1 / 2.4 ) ) - 0.055
    else :
        var_B = 12.92 * var_B

    R = var_R * 255
    G = var_G * 255
    B = var_B * 255

    return ( int(R+0.5), int(G+0.5), int(B+0.5) )

def RGB_to_XYZ( *args ) :
    # http://www.easyrgb.com/math.php?MATH=M2#text2

    (R,G,B) = make_triplet( *args )

    # Observer = 2 degrees, Illuminant = D65
     
    var_R = ( R / 255.0 )        # Where R = 0 / 255
    var_G = ( G / 255.0 )        # Where G = 0 / 255
    var_B = ( B / 255.0 )        # Where B = 0 / 255

    if var_R > 0.04045 :
        var_R = pow( ( ( var_R + 0.055 ) / 1.055 ), 2.4) 
    else :
        var_R = var_R / 12.92

    if var_G > 0.04045 :
        var_G = pow( ( ( var_G + 0.055 ) / 1.055 ), 2.4 )
    else :  
        var_G = var_G / 12.92

    if var_B > 0.04045 :
        var_B = pow( ( ( var_B + 0.055 ) / 1.055 ), 2.4 )
    else :
        var_B = var_B / 12.92

    var_R = var_R * 100
    var_G = var_G * 100
    var_B = var_B * 100

    # //Observer = 2 degrees, Illuminant = D65
    X = var_R * 0.4124 + var_G * 0.3576 + var_B * 0.1805
    Y = var_R * 0.2126 + var_G * 0.7152 + var_B * 0.0722
    Z = var_R * 0.0193 + var_G * 0.1192 + var_B * 0.9505

    return (X,Y,Z)

def XYZ_to_xyY( *args ) :
    # CIE 1931 xyY
    # http://en.wikipedia.org/wiki/CIE_1931_color_space
    # and
    # http://www.easyrgb.com/math.php?MATH=M3#text3
     
    # thankfully Python is case sensitive...
    
    (X,Y,Z) = make_triplet( *args )

    x = X / float( X + Y + Z )
    y = Y / float( X + Y + Z )

    return (x, y, Y)
    
def xyY_to_XYZ( *args ) :
    # CIE 1931 xyY to tristimulus
    # http://www.easyrgb.com/math.php?MATH=M4#text4
    # and
    # http://www.brucelindbloom.com/index.html?Equations.html
     
    # thankfully Python is case sensitive...
    
    (x,y,Y) = make_triplet( *args )

    X = float( x*Y ) / float(y)
    Z = float((1-x-y)*Y) / float(y)

    return (X, Y, Z)


def RGB_to_YCC( *args ) :
    
    (R,G,B) = make_triplet( *args )

#    clip_y = lambda x : max(0,min(100,x))
    clip_y = lambda x : max(0,min(255,x))
    clip_cbcr = lambda x : max(-128,min(127,x))

    Y  = clip_y( (rgb_to_ycc[0] * R + rgb_to_ycc[1] * G + rgb_to_ycc[2] * B) )
    Cb = clip_cbcr( (rgb_to_ycc[3] * R + rgb_to_ycc[4] * G + rgb_to_ycc[5] * B) )
    Cr = clip_cbcr( (rgb_to_ycc[6] * R + rgb_to_ycc[7] * G + rgb_to_ycc[8] * B) )

    return (Y,Cb,Cr)

def YCC_to_RGB( *args ) :

    (Y,Cb,Cr) = make_triplet( *args )

    clip_rgb = lambda x : max(0,min(255,int(round(x))))

#    R = ( (ycc_to_rgb[0] * Y + ycc_to_rgb[1] * Cb + ycc_to_rgb[2] * Cr) )
#    G = ( (ycc_to_rgb[3] * Y + ycc_to_rgb[4] * Cb + ycc_to_rgb[5] * Cr) )
#    B = ( (ycc_to_rgb[6] * Y + ycc_to_rgb[7] * Cb + ycc_to_rgb[8] * Cr) )
    R = clip_rgb( (ycc_to_rgb[0] * Y + ycc_to_rgb[1] * Cb + ycc_to_rgb[2] * Cr) )
    G = clip_rgb( (ycc_to_rgb[3] * Y + ycc_to_rgb[4] * Cb + ycc_to_rgb[5] * Cr) )
    B = clip_rgb( (ycc_to_rgb[6] * Y + ycc_to_rgb[7] * Cb + ycc_to_rgb[8] * Cr) )

    return (R,G,B)

def main() :
    color_from = sys.argv[1]
    color_to = sys.argv[2]

    (a,b,c) = make_triplet( [ float(num) for num in sys.argv[3:6] ] )

    convert = "%s_to_%s" % (color_from,color_to)

    cmd = "%s( %f, %f, %f )" % (convert, a, b, c )
    print(cmd)
    print(eval( cmd ))

if __name__ == '__main__' :
    main()

