/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2004-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <stdlib.h>

#include "lassert.h"

#include "scantypes.h"
#include "siqdata.h"


/* This stuff doesn't live in siqdata.c so I can build the parser as a
 * standalone host-based executable. siqdata.c has too many dependencies on
 * scanvars.c
 */
 
/** convert human names to internal numbers 
 * \todo TODO switch to hash table because this list is going to get HUGE
 */
const struct siqdata_name g_siq_names[] = {
    { "name",      4, SIQ_NAME },

    /* davep 01-Nov-2012 ; adding "user" settings so can use siqfile as a
     * pickle/unpickle for all scan settings. These settings are ignored in a
     * siqfile and are only used in Linux userspace/kernelspace communication.
     * 
     */
    { "dpi", 3, SIQ_USER_DPI },
    { "scale", 5, SIQ_USER_SCALE },
    { "area", 4, SIQ_USER_AREA },
    { "capture_cal", 11, SIQ_CAPTURE_CAL },
    { "pieout", 6, SIQ_PIEOUT },
    { "bpp", 3, SIQ_BPP },
    { "docsrc", 6, SIQ_DOC_SRC },

    /* calibration tuning */
    { "cal_white_digital", 17, SIQ_CAL_WHITE_DIGITAL },
    { "cal_black_digital", 17, SIQ_CAL_BLACK_DIGITAL },
    { "cal_white_analog",  16, SIQ_CAL_WHITE_ANALOG },
    { "cal_black_analog",  16, SIQ_CAL_BLACK_ANALOG },
    { "cal_analog_offset", 17, SIQ_CAL_ANALOG_OFFSET },
    { "cal_analog_gain",   15, SIQ_CAL_ANALOG_GAIN },
    { "cal_led_pwm",       11, SIQ_CAL_LED_PWM },
    { "cal_led_exp",       11, SIQ_CAL_LED_EXP },
    { "cal_mode",           8, SIQ_CAL_MODE },

    /* IQ tuning */
    { "white_clip",      10, SIQ_WHITE_CLIP_POINT },
    { "black_clip",      10, SIQ_BLACK_CLIP_POINT },
    { "gamma",            5, SIQ_GAMMA },
    { "contrast",         8, SIQ_CONTRAST },
    { "brightness",      10, SIQ_BRIGHTNESS },
    { "copy_brightness", 15, SIQ_COPY_BRIGHTNESS },
    { "sharpness",        9, SIQ_SHARPNESS },
    { "rgb_tint",         8, SIQ_RGB_TINT },

    /* 5x5 filter implemented in firmware (only for emergencies) */
    { "sw_filter",        9, SIQ_SW_FILTER },

    /* firmware margins; support strange sensors PIC can't handle. Also
     * overfloaded to support firmware chipgap correction.
     */
    { "fw_margins",     10, SIQ_FW_MARGINS },

    /* PIC Bit Depth Reduction (BDR) */
    { "pic_bdr_enable", 14, SIQ_PIC_BDR_ENABLE },
    { "pic_bdr_lut", 11, SIQ_PIC_BDR_LUT },

    /* PIE Color Space Converter (CSC) */
    { "csc_enable_sensor0", 18, SIQ_CSC_ENABLE_SENSOR0 },
    { "csc_enable_sensor1", 18, SIQ_CSC_ENABLE_SENSOR1 },
    { "csc_rgbycc_matrix",  17, SIQ_CSC_RGBYCC_MATRIX },
    { "csc_yccrgb_matrix",  17, SIQ_CSC_YCCRGB_MATRIX },
    { "csc_rgbsrgb_bypass", 18, SIQ_CSC_RGBSRGB_BYPASS },
    { "csc_rgbycc_bypass",  17, SIQ_CSC_RGBYCC_BYPASS },
    { "csc_yccrgb_bypass",  17, SIQ_CSC_YCCRGB_BYPASS },
    { "csc_r_lut_sensor0",  17, SIQ_CSC_R_LUT_SENSOR0 },
    { "csc_g_lut_sensor0",  17, SIQ_CSC_G_LUT_SENSOR0 },
    { "csc_b_lut_sensor0",  17, SIQ_CSC_B_LUT_SENSOR0 },
    { "csc_r_lut_sensor1",  17, SIQ_CSC_R_LUT_SENSOR1 },
    { "csc_g_lut_sensor1",  17, SIQ_CSC_G_LUT_SENSOR1 },
    { "csc_b_lut_sensor1",  17, SIQ_CSC_B_LUT_SENSOR1 },

    /* PIE SubCube Color Space Converter (SCCSC) */
    { "sccsc_enable_sensor0", 20, SIQ_SCCSC_ENABLE_SENSOR0 },
    { "sccsc_enable_sensor1", 20, SIQ_SCCSC_ENABLE_SENSOR1 },
    { "sccsc_lut_sensor0",    17, SIQ_SCCSC_LUT_SENSOR0 },
    { "sccsc_lut_sensor1",    17, SIQ_SCCSC_LUT_SENSOR1 },
    { "sccsc_whitethresh0_sensor0", 26, SIQ_SCCSC_WHITETHRESH_CH0_SEN0 },
    { "sccsc_whitethresh1_sensor0", 26, SIQ_SCCSC_WHITETHRESH_CH1_SEN0 },
    { "sccsc_whitethresh2_sensor0", 26, SIQ_SCCSC_WHITETHRESH_CH2_SEN0 },
    { "sccsc_whitethresh0_sensor1", 26, SIQ_SCCSC_WHITETHRESH_CH0_SEN1 },
    { "sccsc_whitethresh1_sensor1", 26, SIQ_SCCSC_WHITETHRESH_CH1_SEN1 },
    { "sccsc_whitethresh2_sensor1", 26, SIQ_SCCSC_WHITETHRESH_CH2_SEN1 },

    /* Generic PIE settings */
    { "pie_pad_to_pixel_width", 22, SIQ_PIE_PAD_TO_PIXEL_WIDTH },
    { "use_pie",                 7, SIQ_PIE_ENABLE },

    /* PIE Tone Contract and Neutral Sensitivity (TCNS) */
    { "tcns_enable",  11, SIQ_TCNS_ENABLE },
    { "tcns_lut_y",   10, SIQ_TCNS_LUT_Y },
    { "tcns_lut_cb",  11, SIQ_TCNS_LUT_CB },
    { "tcns_lut_cr",  11, SIQ_TCNS_LUT_CR },

    /* PIE color shift, CIS color fringe removal (CS) - Sensor 0 */
    { "cis_colorshift_enable_sensor0",     29, SIQ_CIS_COLORSHIFT_ENABLE_SENSOR0 },
    { "cis_colorshift_round_mode_sensor0", 33, SIQ_CIS_COLORSHIFT_ROUND_MODE_SENSOR0 },
    { "cis_colorshift_mmode_sensor0",      28, SIQ_CIS_COLORSHIFT_MMODE_SENSOR0 },
    { "cis_colorshift_mult_sensor0",       27, SIQ_CIS_COLORSHIFT_MULT_SENSOR0 },

    /* PIE color shift, CIS color fringe removal (CS) - Sensor 1 */
    { "cis_colorshift_enable_sensor1",     29, SIQ_CIS_COLORSHIFT_ENABLE_SENSOR1 },
    { "cis_colorshift_round_mode_sensor1", 33, SIQ_CIS_COLORSHIFT_ROUND_MODE_SENSOR1 },
    { "cis_colorshift_mmode_sensor1",      28, SIQ_CIS_COLORSHIFT_MMODE_SENSOR1 },
    { "cis_colorshift_mult_sensor1",       27, SIQ_CIS_COLORSHIFT_MULT_SENSOR1 },

    /* PIE MultiFilter (DSMF) */
    { "dsmf_enable",                       11, SIQ_DSMF_ENABLE  },
    { "dsmf_test_mixed",                   15, SIQ_DSMF_TEST_MIXED },
    { "dsmf_test_halftone",                18, SIQ_DSMF_TEST_HALFTONE },
    { "dsmf_test_text",                    14, SIQ_DSMF_TEST_TEXT },
    { "dsmf_test_isolated",                18, SIQ_DSMF_TEST_ISOLATED },
    { "dsmf_alpha_filter_en",              20, SIQ_DSMF_ALPHA_FILTER_EN },
    { "dsmf_filt_idx_sel",                 17, SIQ_DSMF_FILT_IDX_SEL },
    { "dsmf_ot_out_sel",                   15, SIQ_DSMF_OT_OUT_SEL },
    { "dsmf_proj_horiz_thresh_x55",        26, SIQ_DSMF_PROJ_HORIZ_THRESH_X55 },
    { "dsmf_proj_vert_thresh_x55",         25, SIQ_DSMF_PROJ_VERT_THRESH_X55 },
    { "dsmf_proj_diag_thresh_x25",         25, SIQ_DSMF_PROJ_DIAG_THRESH_X25 },
    { "dsmf_isolated_horiz_thresh",        26, SIQ_DSMF_ISOLATED_HORIZ_THRESH },
    { "dsmf_isolated_vert_thresh",         25, SIQ_DSMF_ISOLATED_VERT_THRESH },
    { "dsmf_isolated_diag_thresh",         25, SIQ_DSMF_ISOLATED_DIAG_THRESH },
    { "dsmf_3x3_text_horiz_thresh",        26, SIQ_DSMF_3X3_TEXT_HORIZ_THRESH },
    { "dsmf_3x3_text_vert_thresh",         25, SIQ_DSMF_3X3_TEXT_VERT_THRESH },
    { "dsmf_3x3_text_diag_thresh",         25, SIQ_DSMF_3X3_TEXT_DIAG_THRESH },
    { "dsmf_3x3_mixed_horiz_thresh",       27, SIQ_DSMF_3X3_MIXED_HORIZ_THRESH },
    { "dsmf_3x3_mixed_vert_thresh",        26, SIQ_DSMF_3X3_MIXED_VERT_THRESH },
    { "dsmf_3x3_mixed_diag_thresh",        26, SIQ_DSMF_3X3_MIXED_DIAG_THRESH },
    { "dsmf_5x5_text_horiz_thresh",        26, SIQ_DSMF_5X5_TEXT_HORIZ_THRESH },
    { "dsmf_5x5_text_vert_thresh",         25, SIQ_DSMF_5X5_TEXT_VERT_THRESH },
    { "dsmf_5x5_text_diag_thresh",         25, SIQ_DSMF_5X5_TEXT_DIAG_THRESH },
    { "dsmf_5x5_mixed_horiz_thresh",       27, SIQ_DSMF_5X5_MIXED_HORIZ_THRESH },
    { "dsmf_5x5_mixed_vert_thresh",        26, SIQ_DSMF_5X5_MIXED_VERT_THRESH },
    { "dsmf_5x5_mixed_diag_thresh",        26, SIQ_DSMF_5X5_MIXED_DIAG_THRESH },
    { "dsmf_line_horiz_thresh",            22, SIQ_DSMF_LINE_HORIZ_THRESH },
    { "dsmf_line_vert_thresh",             21, SIQ_DSMF_LINE_VERT_THRESH },
    { "dsmf_line_diag_thresh",             21, SIQ_DSMF_LINE_DIAG_THRESH },
    { "dsmf_line_diag1_thresh",            22, SIQ_DSMF_LINE_DIAG1_THRESH },
    { "dsmf_line_diag2_thresh",            22, SIQ_DSMF_LINE_DIAG2_THRESH },
    { "dsmf_scale_10",                     13, SIQ_DSMF_SCALE_10 },
    { "dsmf_scale_15",                     13, SIQ_DSMF_SCALE_15 },
    { "dsmf_scale_20",                     13, SIQ_DSMF_SCALE_20 },
    { "dsmf_output_scale",                 17, SIQ_DSMF_OUTPUT_SCALE },
    { "dsmf_offset",                       11, SIQ_DSMF_OFFSET },
    { "dsmf_enable_erosion",               19, SIQ_DSMF_ENABLE_EROSION },
    { "dsmf_enable_dilation",              20, SIQ_DSMF_ENABLE_DILATION },
    // use_mf_ai_lut implied by any of the following 4:
    { "dsmf_ai_lut0_y",                    14, SIQ_DSMF_AI_LUT0_Y  },
    { "dsmf_ai_lut0_cbcr",                 17, SIQ_DSMF_AI_LUT0_CBCR  },
    { "dsmf_ai_lut1_y",                    14, SIQ_DSMF_AI_LUT1_Y  },
    { "dsmf_ai_lut1_cbcr",                 17, SIQ_DSMF_AI_LUT1_CBCR  },
    { "dsmf_var_coeff_y_f0",               19, SIQ_DSMF_VAR_COEFF_Y_F0 },
    { "dsmf_var_coeff_y_f1",               19, SIQ_DSMF_VAR_COEFF_Y_F1 },
    { "dsmf_var_coeff_y_f2",               19, SIQ_DSMF_VAR_COEFF_Y_F2 },
    { "dsmf_var_coeff_y_f3",               19, SIQ_DSMF_VAR_COEFF_Y_F3 },
    { "dsmf_var_coeff_y_mult",             21, SIQ_DSMF_VAR_COEFF_Y_MULT },
    { "dsmf_var_coeff_y_shift",            22, SIQ_DSMF_VAR_COEFF_Y_SHIFT },
    { "dsmf_var_coeff_chroma_f0",          24, SIQ_DSMF_VAR_COEFF_CHROMA_F0 },
    { "dsmf_var_coeff_chroma_f1",          24, SIQ_DSMF_VAR_COEFF_CHROMA_F1 },
    { "dsmf_var_coeff_chroma_f2",          24, SIQ_DSMF_VAR_COEFF_CHROMA_F2 },
    { "dsmf_var_coeff_chroma_f3",          24, SIQ_DSMF_VAR_COEFF_CHROMA_F3 },
    { "dsmf_var_coeff_chroma_mult",        26, SIQ_DSMF_VAR_COEFF_CHROMA_MULT },
    { "dsmf_var_coeff_chroma_shift",       27, SIQ_DSMF_VAR_COEFF_CHROMA_SHIFT },
    { "dsmf_filtctl_conv_filter_coef_sel", 33, SIQ_DSMF_FILTCTL_CONV_FILTER_COEF_SEL },
    { "dsmf_filtctl_ai_lut_y_sel",         25, SIQ_DSMF_FILTCTL_AI_LUT_Y_SEL },
    { "dsmf_filtctl_ai_lut_c_sel",         25, SIQ_DSMF_FILTCTL_AI_LUT_C_SEL },
    { "dsmf_filtctl_unsharp_mask_enable",  32, SIQ_DSMF_FILTCTL_UNSHARP_MASK_ENABLE },
    { "dsmf_filtctl_textenhance_enable",   31, SIQ_DSMF_FILTCTL_TEXTENHANCE_ENABLE },

    { "te_count_black_min",                18, SIQ_TE_COUNT_BLACK_MIN },
    { "te_count_white_min",                18, SIQ_TE_COUNT_WHITE_MIN },
    { "te_count_white_plus_black_min",     29, SIQ_TE_COUNT_WHITE_PLUS_BLACK_MIN },
    { "te_center_color_max",               19, SIQ_TE_CENTER_COLOR_MAX },
    { "te_intense_color_max",              20, SIQ_TE_INTENSE_COLOR_MAX },
    { "te_black_y_max",                    14, SIQ_TE_BLACK_Y_MAX },
    { "te_black_CbCr_max",                 17, SIQ_TE_BLACK_CbCr_MAX },
    { "te_white_y_min",                    14, SIQ_TE_WHITE_Y_MIN },
    { "te_white_CbCr_max",                 17, SIQ_TE_WHITE_CbCr_MAX },
    { "te_use_5x5_matrix",                 17, SIQ_TE_USE_5X5_MATRIX },

    /* HalfPack, the HalfToner/BitPacker. Deprecated--moved to another
     * hardware block. Only the firmware based ordered dither emulator remains.
     */
    { "hp_bpp",          6,  SIQ_HP_BPP },
    { "hp_horiz_rep",   12,  SIQ_HP_HORIZ_REP },
    { "hp_size_lut",    11,  SIQ_HP_SIZE_LUT },
    { "hp_sbias_lut",   12,  SIQ_HP_SBIAS_LUT },
    { "hp_density_lut", 14,  SIQ_HP_DENSITY_LUT },
    { "hp_diffus_lut",  13,  SIQ_HP_DIFFUS_LUT },
    { "hp_alg_invert_data", 18, SIQ_HP_ALG_INVERT_DATA },
    { "hp_alg_accum_err",16,  SIQ_HP_ALG_ACCERR  },
    { "hp_alg_sbias_idx",16,  SIQ_HP_ALG_SBIAS_I },
    { "hp_alg_sbias_dot",16,  SIQ_HP_ALG_SBIAS_D },
    { "hp_alg_doadot",   13,  SIQ_HP_ALG_DOADOT },
    { "hp_alg_shift",    12,  SIQ_HP_ALG_SHIFT },
    { "hp_alg_errweight1",17,  SIQ_HP_ALG_ERRW1 },
    { "hp_alg_errweight2",17,  SIQ_HP_ALG_ERRW2 },
    { "hp_alg_errweight3",17,  SIQ_HP_ALG_ERRW3 },
    { "hp_bias_sram_cfg", 16,  SIQ_HP_BIAS_SRAM_CFG },
    { "hp_bias_reg0",    12,   SIQ_HP_BIAS_REG0 },
    { "hp_bias_reg1",    12,   SIQ_HP_BIAS_REG1 },

    /* must have an empty entry to indicate end of list */
    { 0, 0, 0 },
};

/**
 * \brief return human name for internal siqfile key 
 *
 * Used to convert a siqfile data key into a nice readable string. 
 *
 * \author David Poole
 * \date 29-Dec-2005
 *
 */

const char *
siqdata_find_name_for_key( int key )
{
    int i;

    /* linear search, blah blah blah */
    for( i=0 ; ; i++ ) {
        if( g_siq_names[i].key == key ) {
            return g_siq_names[i].str;
        }
    }
    return "(unknown key!)";
}

/**
 * \brief get first name in list of siq data names; initialize the iterator
 *
 * See also siqdata_get_next_name() 
 *
 *
 * \author David Poole
 * \date 25-Jun-2006
 *
 */

const struct siqdata_name * 
siqdata_get_first_name( void )
{
    return &g_siq_names[0];
}

/**
 * \brief  A rough iterator on g_siq_names[].
 *
 * See also siqdata_get_first_name()
 *
 *
 * \author David Poole
 * \date 25-Jun-2006
 *
 */

const struct siqdata_name * 
siqdata_get_next_name( const struct siqdata_name *name )
{
    /* stupid human checks */
    if( name==NULL ) {
        return NULL;
    }

    XASSERT( name >= g_siq_names, (uint32_t)name );    
    XASSERT( name < g_siq_names+sizeof(g_siq_names), (uint32_t)name );

    /* somehow already on last element!? */
    if( name->len == 0 ) {
        return NULL;
    }

    /* move to next element in our array */
    name++;

    /* if on end-of-list marker, return NULL */
    if( name->len==0 ) {
        return NULL;
    }
            
    return name;
}

