/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2007-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * \file scantest.c
 *
 * \brief Code for running simple tests on the sensor.
 *
 * Do NOT NOT call these functions unless you know what you're doing! Don't
 * make me hurt you.
 *
 */
 
#include <string.h>
#include <limits.h>

#include "scos.h"

#include "lassert.h"
#include "memAPI.h"
#include "regAddrs.h"
#include "cpu_api.h"
#include "list.h"
#include "regAddrs.h" 

#include "scancore.h"
#include "scantypes.h"
#include "scandbg.h"
#include "scan.h"
#include "scanimg.h"
#include "scantest.h"
#include "scanblk_if.h"
#include "scanif.h"
#include "scancmdq.h"

#include "pic_handle_if.h"
#include "pic_if.h"
#include "pic_convenience_if.h"
#include "pic.h"
#include "scansen.h"
#include "scanvars.h"
#include "safetylock.h"
#include "icedma.h"
#include "scantask.h"
#include "ostools.h"
#ifdef HAVE_PIC_DESCRIPTOR_DMA 
#include "picdma_descrip.h"
#include "picdma_descrip_test.h"
#endif
#include "scands.h"
#include "afe_if.h"

// set this constant to the actual sensor you want to debug.
// If you only have 1 sensor, the value 1 works nicely.
// If you have 2 sensors, set to 2 for the 2nd sensor, set to 1 for the 1st sensor
// If you have 3 sensors, set to 4 for the 3rd sensor, etc, std bitmask
#define SCANTEST_SENSOR 0x1

/* define SCANTEST_DEBUG to turn on extra verbose messages; help for debugging
 * the tests themselves
 */
//#define SCANTEST_DEBUG

static bool scmd_done;

static int scantest_scancmdq_isr( void )
{
    /*
     * BIG FAT NOTE!  THIS IS AN INTERRUPT HANDLER! 
     */

    scmd_done = true;

    /* fake out scan_interrupt() */
    return SCAN_CMDQ_SUCCESS;
}

static scan_err_t check_for_messages( struct scos_mq * msgq )
{
    scan_msg_t msg;
    scan_err_t scerr;

    /* wait for N usec for message; also serves as a handy delay */
    scerr = scos_mq_receive( msgq, (void *)&msg, 1000 );
    XASSERT( scerr==SCANERR_NONE||scerr==SCANERR_TIMEDOUT, scerr );

    if( scerr==0 ) {
        switch( msg.msgType ) {
            case SMSG_SCAN_CANCEL :
                scerr = SCANERR_SCAN_CANCELLED;
                break;

            case SCAN_DBGMSG_DUMP_AFE :
                afe_reg_dump( 0 );  // ToDo - handle multiple afe ifaces
                break;

            case SCAN_DBGMSG_DUMP_SCAN :
                scif_dump();
                break;

            case SMSG_AFE_HW_FAILURE :
                /* davep 19-Jul-2012 ; catch AFE hardware failure; can happen in
                 * interrupt context so ISR will send the thread this message
                 */
                dbg1( "%s afe failure param1=0x%x\n", __FUNCTION__, msg.param1 );
                scerr = SCANERR_SCAN_CANCELLED;
                break;

            default :
                /* at this stage of the game, we're only expecting something from
                 * the scan subsystem
                 */
                if(msg.msgType == SMSG_SCAN_MECH_STATUS) {
                    dbg1("%s: ignoring SMSG_SCAN_MECH_STATUS\n", __FUNCTION__);
                    break;
                }
                XASSERT( 0, msg.msgType );
        }
    }

    return scerr;
}

static void blocking_cancel_scmd( void )
{
    /* davep 24-Jul-2012 ; simpler, brute force cancel. Shut everything down. */
    scan_interrupt_disable();
    scif_control(false, false, false); // stop scan, no interrupt
    scif_clock(false, false);          // disable scan clocks
}

scan_err_t scantest_sensor( uint32_t sensor_num, struct pic_handle_t *pic_handle, scan_cmode_t cmode, uint32_t dpi, uint32_t num_lines )
{
    scan_err_t scerr, test_scerr;
    scan_cmdq_isr_t old_cmdq_isr;
    uint32_t scanx, sclkper;
    uint32_t pixper;
    struct scos_mq * msgq;
    uint16_t scany;
    uint32_t sensor_bitmask;

    dbg2( "%s cmode=%d dpi=%d num_lines=%d\n", __FUNCTION__, 
                cmode, dpi, num_lines );

    /* we're going to run SCMD so grab the cmdq interrupt from the official
     * owner (we'll restore it before we leave this function)
     */
    old_cmdq_isr  = scan_set_cmdq_interrupt( scantest_scancmdq_isr );
    scan_interrupt_enable();

    scerr = scands_get_integer_with_default( "scantest_sensor", &sensor_bitmask, SCANTEST_SENSOR);
    XASSERT( scerr==0, scerr );
    
    /* start the scan block and sensor/afe from a clean slate */
    scerr = scanimg_open( cmode, dpi, sensor_bitmask );
    XASSERT( scerr==0, scerr );

    /* tell sensor/afe how we're scanning */
    scerr = scanimg_setup();
    XASSERT( scerr==0, scerr );

    scanimg_get_scanx( &scanx );
    scanimg_get_pixper( &pixper );

    /* we'll need internal scanline mode so we don't need a motor */
    scif_set_internal_scanline_mode( cmode, dpi, scanx, &sclkper );
    dbg2( "%s scanx=%d sclkper=%d pixper=%d\n", 
                __FUNCTION__, scanx, sclkper, pixper );

    /* davep 25-Feb-2013 ; explicitly put PIC into bypass */
#ifdef HAVE_PIC_DESCRIPTOR_DMA 
    pic_soft_setup(sensor_num);
    pic_output_dma_set_disable_mode(pic_handle, PIC_OUT_DMA_DISCARD_DATA);
#endif

    pic_do_configure(pic_handle, sensor_num);

    /* away we go! */
    scanimg_run();

    /* Turn on scan block. After this call, we're ready to run. All we need is
     * some stuff pushed into SCMD
     */
    scif_control(true,true, false);
    scif_clock(true, false);

    /* is this why I'm not getting my PIC overflow? */
    pic_interrupt_enable(sensor_num);
//    pic_interrupt_disable();

    dbg2( "%s waiting for num_lines=%d to complete\n", __FUNCTION__, num_lines );

    /* get the scantask message queue so we can check for cancel and debug
     * messages 
     */
    msgq = scantask_get_msgq();

    test_scerr = SCANERR_NONE;

    /* davep 18-Feb-2008 ; Interesting note: I found doing a scif_dump()
     * here caused the command to never start. Hypothesis is reading the
     * registers at the moment I pushed in an SCMD caused some sort of
     * confusion in the Scan block state machines.
     *
     * After some further tinkering, decided even peeking at SSTAT.SCMDSTAT
     * might be causing problems. I'd plug in an SCMD and it just wouldn't
     * start. 
     *
     * So I think I'm going to avoid hassling the scan block while it's
     * working (i.e, don't poll the scan block). Switched to using the
     * global scmd_done flag.
     */
    scmd_done = false;

    /* davep 09-Nov-2010 ; I've changed how we send in scan commands. Previous
     * code would send one row at a time, waiting for scmd_done. We would
     * brutally spin on scmd_done. Something changed so the serial terminal
     * locks up while we spin loop. I added a pthread_yield() but I'm worried
     * that would change the timing of the SCMDs. 
     *
     * I changed the algorithm. Now we push in as many lines as we can then
     * politely spin waiting for the command to complete.
     */

    while( num_lines ) {
        scany = MIN( SCIF_SCMD_SCANY(~0), num_lines );

        num_lines -= scany;

        /* scan command to get some lines with no data */
        scif_send_simple_scmd( false, sclkper, scany );

        while( !scmd_done ) {
            /* davep 30-Sep-2010 ;  peak into msgq looking for a cancel */
            scerr = check_for_messages( msgq );
            if( scerr==SCANERR_SCAN_CANCELLED ) {
                blocking_cancel_scmd();

                /* leave our loop */
                dbg2( "%s test cancelled\n", __FUNCTION__ );
                num_lines = 0;
                test_scerr = scerr;
                break;
            }
            else { 
                XASSERT( scerr==SCANERR_NONE||scerr==SCANERR_TIMEDOUT, scerr );
                pthread_yield();
            }
        }

        scmd_done = false;
    }

//    XASSERT( !scif_command_executing(), *SCIF_SSTAT );

    scanimg_stop();
    scanimg_close();

    /* davep 24-Jul-2012 ; a certain digital afe control block seems to throw
     * an error during cancel so need to clean out any pending messages
     */
    while( 1 ) {
        scerr = check_for_messages( msgq );
        if( scerr==SCANERR_NONE||scerr==SCANERR_TIMEDOUT ) {
            break;
        }
    }

    /* put back the previous handler (clean up after ourselves) */
    scan_set_cmdq_interrupt( old_cmdq_isr );

    dbg2( "%s done\n", __FUNCTION__ );

    return test_scerr;
}

scan_err_t scantest_simple_sensor( uint32_t scan_sensor, struct pic_handle_t *pic_handle )
{
    scan_err_t scerr;

    dbg2( "%s start\n", __FUNCTION__ );

    scerr = scantest_sensor( scan_sensor, pic_handle, SCAN_CMODE_MONO, 300, 100 );
    if( scerr != 0 ) {
        return scerr;
    }

    scerr = scantest_sensor( scan_sensor, pic_handle, SCAN_CMODE_COLOR, 300, 100 );
    if( scerr != 0 ) {
        return scerr;
    }

    scerr = scantest_sensor( scan_sensor, pic_handle, SCAN_CMODE_MONO, 600, 100 );
    if( scerr != 0 ) {
        return scerr;
    }

    scerr = scantest_sensor( scan_sensor, pic_handle, SCAN_CMODE_COLOR, 600, 100 );
    if( scerr != 0 ) {
        return scerr;
    }

    /* MORE HERE */

    dbg2( "%s done\n", __FUNCTION__ );

    /* success! */
    return 0;
}

uint32_t chan_bitmap;
DECLARE_COMPLETION(scantest_sensor_with_data_complete);
void scantest_sensor_with_data_callback(struct pic_output_dma_interrupt_info *int_info, void *callback_data)
{
    printk("Received interrupt from odma channel %d\n", int_info->chan_num);
    if (int_info->bad_rresp)
    {
        printk("ERROR!!! BAD RRESP!!! TEST FAILED!!!!\n");
        return;
    }
    if (int_info->bad_bresp)
    {
        printk("ERROR!!! BAD BRESP!!! TEST FAILED!!!!\n");
        return;
    }
    if (int_info->dir_err)
    {
        printk("ERROR!!!  DIR_ERR!!! TEST FAILED!!!!\n");
        return;
    }
    if (int_info->chg_line_align_err)
    {
        printk("ERROR!!!  CHG_LINE_ALIGN_ERR!!! TEST FAILED!!!!\n");
        return;
    }
    if (int_info->chg_line_align_err)
    {
        printk("ERROR!!!  CHG_LINE_ALIGN_ERR!!! TEST FAILED!!!!\n");
        return;
    }
    if (int_info->eol_align_err)
    {
        printk("ERROR!!!  EOL_ALIGN_ERR!!! TEST FAILED!!!!\n");
        return;
    }
    if (int_info->eoi_align_err)
    {
        printk("ERROR!!!  EOI_ALIGN_ERR!!! TEST FAILED!!!!\n");
        return;
    }
    if (int_info->eoi_err)
    {
        printk("ERROR!!!  EOI_ERR!!! TEST FAILED!!!!\n");
        return;
    }
    dbg2("EOImage was %s, XFER_END was %s\n",int_info->eoi?"set":"not set", int_info->xfer_end?"set":"not set");
    
    dbg2("chan_bitmap 0x%X & with chan_num=%d, (value 0x%X)\n", chan_bitmap, int_info->chan_num, (~(1<<int_info->chan_num)));

    // only clear bit for channel if we got a xfer end interrupt
    if (int_info->xfer_end)
    {
        chan_bitmap = chan_bitmap & (~(1<<int_info->chan_num));
        
        if (chan_bitmap == 0)
        {
            complete(&scantest_sensor_with_data_complete);
        }
    }
}
scan_err_t scantest_sensor_with_data(uint32_t sensor_num, struct pic_handle_t *pic_handle,
                                      scan_cmode_t cmode, uint32_t dpi, uint32_t num_lines)
{
    scan_err_t scerr;
    int i;
    scan_cmdq_isr_t old_cmdq_isr;
    uint32_t scanx, sclkper, pic_bytes_per_row;
    uint8_t *picbuf, channel, pic_num_channels;
    dma_addr_t picbuf_phys_addr;
    uint32_t num_rows;
    uint8_t *pic_buffers[ PIC_WDMA_NUM_CHANNELS ];
    uint8_t pic_channels[ PIC_WDMA_NUM_CHANNELS ];
    bool test_success;
    pic_bitpack_mode bitpack_mode;
    uint32_t sensor_bitmask;
    uint8_t pic_instance, channum;

    dbg2( "%s cmode=%d dpi=%d num_lines=%d\n", __FUNCTION__, 
                cmode, dpi, num_lines );

    /* we're going to run SCMD so grab the cmdq interrupt from the official
     * owner (we'll restore it before we leave this function)
     */
    scan_interrupt_disable();
    old_cmdq_isr  = scan_set_cmdq_interrupt( scantest_scancmdq_isr );
    scan_interrupt_enable();

    pic_interrupt_disable(sensor_num);
    /*
     *
     *   Static/Const fields steps
     *
     */

    scerr = scands_get_integer_with_default( "scantest_sensor", &sensor_bitmask, SCANTEST_SENSOR);
    XASSERT( scerr==0, scerr );
    
    /* start the scan block and sensor/afe from a clean slate */
    scerr = scanimg_open( cmode, dpi, sensor_bitmask );
    XASSERT( scerr==0, scerr );
    /* set pic to a clean slate (all blocks will be in bypass after this call) */
    pic_soft_setup(sensor_num);
    /* 
     *
     *   Per Scan Setup steps
     *
     */

    /* tell sensor/afe how we're scanning */
    scerr = scanimg_setup();
    XASSERT( scerr==0, scerr );
    /* we'll need internal scanline mode so we don't need a motor */
    scanimg_get_scanx( &scanx );
    scif_set_internal_scanline_mode( cmode, dpi, scanx, &sclkper );

    dbg2( "%s scanx=%d sclkper=%d\n", __FUNCTION__, scanx, sclkper );


    /* 
     *
     *  Set up PIC
     *
     */
    pic_output_dma_set_disable_mode(pic_handle, PIC_OUT_DMA_PROCESS_DATA);

    bitpack_mode = pic_wdma_get_bitpack_mode(pic_handle);
    /* how many bytes will we need? we'll leave PIC margins in bypass for this
     * test so PIC WDMA will get all the pixels from the scan block
     */
    if( bitpack_mode == PIC_BITPACK_16BIT ) {
        pic_bytes_per_row = scanx * 2;
    }
    else {
        pic_bytes_per_row = scanx;
    }

    /* we're not using PIC margins (all PIC blocks will be in bypass for this
     * test) so we need to make sure Scan has set up SCANX to be modulo our DMA
     * width
     */
    if( pic_bytes_per_row % ICE_DMA_BYTE_ALIGN != 0 ) {
        /* we'll have to ask the scan block to pad up so we'll be able to
         * survive DMA 
         */

        if( bitpack_mode == PIC_BITPACK_16BIT ) {
            /* we'll get 16-bit pixels so we'll need to make sure scanx number
             * of pixels is aligned to 1/2 our burst size
             *
             * e.g.,  16-bit pixels and 16-byte DMA burst size means we'll need
             * 8 pixels per burst.
             */
            scanimg_align_scanx( &scanx, ICE_DMA_BYTE_ALIGN/2, SCANX_ALIGN_PAD );
            scanimg_set_scanx( scanx );
            pic_bytes_per_row = scanx * 2;
        }
        else {
            /* we'll get 8-bit pixels so align PIXELs to our byte burst size */
            scanimg_align_scanx( &scanx, ICE_DMA_BYTE_ALIGN, SCANX_ALIGN_PAD );
            scanimg_set_scanx( scanx );
            pic_bytes_per_row = scanx;
        }
    }

    XASSERT( (pic_bytes_per_row % ICE_DMA_BYTE_ALIGN)==0, pic_bytes_per_row );

    if( cmode==SCAN_CMODE_MONO ) {
        num_rows = num_lines;
        pic_num_channels = 1;
        pic_channels[0] = PIC_DMA_CHANNEL_MEVEN;
    }
    else {
        num_rows = num_lines;
        pic_num_channels = 3;
        pic_channels[0] = PIC_DMA_CHANNEL_CEVEN_0;
        pic_channels[1] = PIC_DMA_CHANNEL_CEVEN_1;
        pic_channels[2] = PIC_DMA_CHANNEL_CEVEN_2;
    }

    for( i=0 ; i<pic_num_channels ; i++ ) {
        channel = pic_channels[i];

        picbuf = ice_dma_alloc_buffer( pic_bytes_per_row * num_rows );
        if( picbuf == NULL ) {
            printk( "%s failed to get %d bytes of memory for PIC buffer for channel %d\n",
                        __FUNCTION__, pic_bytes_per_row * num_rows, channel );

            /* clean up after ourselves before we leave */
            do {
                i--;
                pic_buffers[i] = CACHE_BYPASS_CLEAR( pic_buffers[i] );
                PTR_FREE( pic_buffers[i] );
            }
            while( i>0 );

            scanimg_close();
            scan_set_cmdq_interrupt( old_cmdq_isr );
            return SCANERR_OUT_OF_MEMORY;
        }

        dbg2( "%s channel=%d ptr=%p datalen=%d\n", __FUNCTION__, channel,
                    picbuf, pic_bytes_per_row*num_rows );
        /* put into the DMA-able memory area */
        picbuf_phys_addr = dma_map_single(NULL, picbuf, pic_bytes_per_row*num_rows, DMA_BIDIRECTIONAL);
        XASSERT(picbuf_phys_addr, (uint32_t) picbuf);

        /* initialize to a value so we can debug the buffer if something goes
         * wrong 
         */
        memset( picbuf, 0xee, pic_bytes_per_row * num_rows );
        cpu_dcache_writeback_region( picbuf, pic_bytes_per_row * num_rows );

#ifdef HAVE_PIC_DESCRIPTOR_DMA
        scerr = pic_wdma_open(pic_handle, pic_channels, pic_num_channels, 1, num_rows, pic_bytes_per_row);
        XASSERT(scerr>0, scerr);
#endif

        convert_channel_idx_to_pic_instance_channum(channel, &pic_instance, &channum);
        pic_output_dma_channel_set_enable_dma(pic_handle, PIC_WDMA_CFG_ENABLE, channum);
        pic_output_dma_channel_set_linesize(pic_handle, pic_bytes_per_row, channum);
        pic_odma_register_callback_dma_channel(pic_handle, scantest_sensor_with_data_callback, NULL, channum);
        pic_wdma_channel_load( channel, picbuf, (dma_addr_t)picbuf_phys_addr, num_rows, pic_bytes_per_row );

        // enable interrupts on each channel
        pic_enable_output_dma_channel_irqs(pic_handle, NULL, true, channum);
        
        /* save the pointer so we can free it later */
        PTR_ASSIGN( pic_buffers[i], picbuf );
    }

    /* davep 04-Oct-2010 ; have to re-write the bitpack mode because recent
     * changes put the bitpack mode into the individual DMA channels.  The
     * above pic_wdma_channel_reset() wipes out the previous setting. 
     *
     * In earlier ASICs, the bitpack mode was stored in an overall config
     * register as it was in earlier ASICs. This code will cause no harm in
     * older ASICs.
     */
    pic_wdma_set_bitpack_mode(pic_handle, bitpack_mode);

    XASSERT(pic_wdma_get_bitpack_mode(pic_handle)==bitpack_mode, bitpack_mode);

    pic_enable_common_irqs(pic_handle, NULL, true);  // enable all pic ints
    /*
     *
     *  Fire in the hole!
     *
     */
    pic_do_configure(pic_handle, sensor_num);
    // pic_dump(sensor_num);

    // inform the ISR how many XFER_END interrupts it should receive
    chan_bitmap = 0;
    // now wait for the interrupt(s) from the PIC output dma
    for (i=0;i<pic_num_channels;i++)
    {
        chan_bitmap = chan_bitmap | (1<<i);
    }
    // end of setting up the ISR
    
    // ok, now start the pic odmas
    pic_cwdma_late_launch();
    
    scerr = scanimg_run();
    XASSERT( scerr==0, scerr );

    /* Turn on scan block. After this call, we're ready to run. All we need is
     * some stuff pushed into SCMD.
     */
    scif_control(true,true, false);
    scif_clock(true, false);

    scif_dump();

    /* scan command to get our lines */
    scif_send_simple_scmd( true, sclkper, num_lines );

    wait_for_completion_interruptible(&scantest_sensor_with_data_complete);

    // we're done now, so as part of the cleanup, remove the callback so we aren't called again
    for (i=0;i<pic_num_channels;i++)
    {
        channel = pic_channels[i];
        convert_channel_idx_to_pic_instance_channum(channel, &pic_instance, &channum);
        pic_odma_register_callback_dma_channel(pic_handle, NULL, NULL, channum);
    }
    pic_do_configure(pic_handle, sensor_num);
    
    if (chan_bitmap == 0)
    {
        printk( "%s PIC finished successfully\n", __FUNCTION__ );
        test_success = true;
    }
    else
    {
        printk( "%s Interrupt Handler ERROR\n", __FUNCTION__ );
        test_success = false;
    }

    
#ifdef SCANTEST_DEBUG
    pic_dump();
#endif

    /* turn things off */
    scerr = scanimg_stop();
    XASSERT( scerr==0, scerr );

    for( i=0 ; i<pic_num_channels ; i++ ) {
        pic_wdma_channel_disable( pic_channels[i] );
    }

    // cancel requires we have the pic_wdma_interrupt set up
    for (i=0;i<pic_num_channels;i++)
    {
        channel = pic_channels[i];
        convert_channel_idx_to_pic_instance_channum(channel, &pic_instance, &channum);
        pic_odma_register_callback_dma_channel(pic_handle, pic_wdma_interrupt, NULL, channum);
    }
    pic_do_configure(pic_handle, sensor_num);

    pic_wdma_cancel(); // clean stuff up before cleaning up
    pic_wdma_close(sensor_num);
    pic_reset(sensor_num);
    /* (clean up after ourselves) */
    for( i=0 ; i<pic_num_channels ; i++ ) {
#ifdef SCANTEST_DEBUG
        scanlog_hex_dump( pic_buffers[i], 256 );
#endif
        pic_buffers[i] = CACHE_BYPASS_CLEAR( pic_buffers[i] );
        PTR_FREE( pic_buffers[i] );
    }

    scanimg_close();
    scif_control(false,false, false);
    scif_clock(false, false);

    /* put back the previous handlers */
    scan_set_cmdq_interrupt( old_cmdq_isr );

    dbg2( "%s done\n", __FUNCTION__ );

    /* success? */
    if (test_success)
        printk("TEST COMPLETED SUCESSFULLY!!!\n");
    else
        printk("TEST FAILED!!!\n");
    return test_success ? SCANERR_NONE : SCANERR_GENERIC_FAIL;
}

scan_err_t scantest_with_data(uint32_t scan_sensor, struct pic_handle_t *pic_handle)
{
    scan_err_t scerr;

    dbg2("%s start\n", __FUNCTION__);

    printk("1st scantest with data with starting - Mono 300 DPI\n");
    scerr = scantest_sensor_with_data(scan_sensor, pic_handle, SCAN_CMODE_MONO, 300, 100);
    if (scerr != 0)
    {
        printk("scerr was %d\n", scerr);
        return scerr;
    }

    printk("2nd scantest with data with starting - Mono 600 DPI\n");
    scerr = scantest_sensor_with_data(scan_sensor, pic_handle, SCAN_CMODE_MONO, 600, 25);
    if (scerr != 0)
    {
        printk("scerr was %d\n", scerr);        
        return scerr;
    }

    // FIXME - get the color test working
//    printk("3rd scantest with data with starting\n");
//    scerr = scantest_sensor_with_data(scan_sensor, pic_handle, SCAN_CMODE_COLOR, 300, 100);
//    if (scerr != 0)
//    {
//        printk("scerr was %d\n", scerr);
//        return scerr;
//    }

    /* success! */
    return 0;
}

scan_err_t scantest_run_finger_test( void )
{
    scan_err_t scerr;
    uint32_t num32;
    int dpi;
    scan_cmode_t cmode;
    uint32_t sensor_to_test = 0;
    struct pic_handle_t *pic_handle = NULL;

    dpi = 300;
    cmode = SCAN_CMODE_MONO;

    /* davep 09-Feb-2011 ; adding scands */
    scerr = scands_get_integer_with_default( "fingertest_dpi", &num32, 300 );
    XASSERT( scerr==0, scerr );

    /* Normally I'd return SCANERR_INVALID_PARAM on the following sanity
     * checks.  However, by design, the scandbg caller will assert fail on a
     * return value != SCANERR_NONE. So, I return SCANERR_NONE.  Smile and
     * wave, boys. Smile and wave.
     */

    if( num32 != 300 && num32 != 600 && num32 != 1200 ) {
        /* finger test is configured with invalid dpi */
        dbg2( "%s bad scands dpi value %d; finger test not started\n", __FUNCTION__, num32 );
        return SCANERR_NONE;
    }
    dpi = num32;

    scerr = scands_get_integer_with_default( "fingertest_cmode", &num32, (uint32_t)SCAN_CMODE_MONO );
    XASSERT( scerr==0, scerr );

    if( (scan_cmode_t)num32 != SCAN_CMODE_MONO && (scan_cmode_t)num32 != SCAN_CMODE_COLOR ) {
        /* finger test is configured with invalid colormode */
        dbg2( "%s ignoring bad scands cmode value %d; finger test not started\n", __FUNCTION__, num32 );
        return SCANERR_NONE;
    }
    cmode = (scan_cmode_t)num32;

    /* davep 30-Mar-2008 ; nifty idea from SandraC's; bump to insane numlines to
     * do a finger test 
     */
    /* davep 30-Sep-2010 ; making cancelable */
    while(1)
    { 
        pic_handle = pic_create_new_default_handle();
        scerr = scantest_sensor( sensor_to_test, pic_handle, cmode, dpi, INT_MAX );
        if (scerr==SCANERR_SCAN_CANCELLED)
        {
            dbg2( "%s finger test canceled\n", __FUNCTION__ );
            break;
        }

        XASSERT( scerr==0, scerr );
    }

    pic_do_free_handle(pic_handle);

    return SCANERR_NONE;
}

scan_err_t scantest_run_sensor_to_pic_test( void )
{
    scan_err_t scerr;
    uint32_t scan_sensor = 0; 
    struct pic_handle_t *pic_handle;

    dbg1( "%s start\n", __FUNCTION__ );

    pic_handle = pic_create_new_default_handle();

#ifdef HAVE_PIC_DESCRIPTOR_DMA 
    pic_soft_setup(scan_sensor);
#endif

    // set pic to 16 bits per pixel for this scantest
    pic_wdma_set_bpp(pic_handle, 16);
    
    scerr = scantest_with_data(scan_sensor, pic_handle);
    XASSERT( scerr==0, scerr );

    dbg1( "%s:scantest_with_data done\n", __FUNCTION__ );

    pic_do_free_handle(pic_handle);

    return SCANERR_NONE;
}
