/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2007-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/** 
 * Description:
 *
 * Generic SDK sensor driver.  Implements the ScanSensor class interface required
 * by the ScanImaging (scanimg.c) class.  Note that this is a stub file.  The functions
 * here do not talk to any specific hardware - that is up to the user of the SDK to
 * implement.
 *
 * For details as to what to implement in each function, please refer to scansen.h
 *
 *
 **/
 
#include <string.h>
#include <stdbool.h>
#include <stdint.h>

#include "lassert.h"

#include "scantypes.h"
#include "scandbg.h"
#include "scanhw.h"
#include "scanblk_if.h"
#include "scanif.h"
#include "scan.h"
#include "scanplat.h"
#include "scansen.h"
#include "cisx_if.h"
#include "cisx.h"


/**
 * \brief Product specific sensor margins
 *
 * Customize for your specific sensor
 */
#if 0
static const scan_sensor_margin_t sensor_margin_table[] = {
    {300,  24, 2580,  0},  
    {600,  24, 5160,  0},
    {1200, 24, 10320, 0},
    /* end of list marker */
    { -1, 0, 0, 0 }
};
#endif
static const scan_sensor_margin_t sensor_margin_table[] = {
    {300,  0, 2592,  4+1},  
    {600,  0, 5184,  4+1},
    {1200, 0, 10368, 4+1},
    /* end of list marker */
    { -1, 0, 0, 0 }
};


/**
 * \brief Power up initialization of the scan sensor 
 *
 * Initializes the scan sensor and any associated internal structures, variables, etc.
 * This function will be called one time at system startup.  
 *
 * \retval scan_err_t any scan error that occured (if all is well, SCANERR_NONE)
 *
 * \remarks If there's nothing that needs to be done, just return SCANERR_NONE.
 **/
scan_err_t scansen_onetime_init(void)
{
    dbg1("STUB FUNCTION - please implement SDK function %s\n",__FUNCTION__);
    return SCANERR_NONE;
}

void scansen_get_sensor_conf( uint32_t sensor_num, scan_cmode_t cmode, uint32_t sensor_dpi, SENSOR_CONF *sensor )
{
    dbg1( "%s warning--using stub function\n", __FUNCTION__ );

    memset( sensor, 0, sizeof(SENSOR_CONF) );

    /* cal.c checks if this is non-zero; not sure if it matters since trying to
     * run with a stub sensor is, um, bad
     */
    sensor->sensor_freq_khz = 1;
}

/**
 * \brief Return the margin table
 *
 * Uses a platform-specific call to get the margin table, then searches
 * through it to find the appropriate entry for the requested dpi, then
 * returns just that entry.
 *
 * \param[in] dpi  the dpi of the margin entry to be returned
 *
 * \param[out] margins  the margin entry that matches the requested dpi
 *
 * \retval scan_err_t any scan error that occured
 *  SCANERR_NONE - entry was found and returned
 *  SCANERR_INVALID_PARAM - margin table has invalid entry
 *  SCANERR_NO_ENTRY - the requested dpi was not found in the margin table
 *
 **/
scan_err_t scansen_get_margins(uint32_t dpi, scan_sensor_margin_t *margins)
{
    int i;

    dbg1("STUB FUNCTION - please implement/verify SDK function %s\n",__FUNCTION__);

    /* One possible implementation of scansen_get_margins is as follows: */

    memset( margins, 0, sizeof(margins) );

    for( i=0 ; ; i++ ) {
        if( sensor_margin_table[i].dpi==-1 ) {
            dbg1( "%s invalid/unknown DPI %ld\n", __FUNCTION__, dpi );
            return SCANERR_INVALID_PARAM;
        }
        else if( sensor_margin_table[i].dpi == dpi ) {
            memcpy( margins, &sensor_margin_table[i], sizeof(scan_sensor_margin_t) );
            return SCANERR_NONE;
        }
    }

    /* should never get here - the requested dpi was not found in the margin table */
    return SCANERR_NO_ENTRY;
}

/**
 * \brief Setup the scan sensor for the requested dpi and color mode
 *
 * A scan sensor can support a variety of resolutions and modes (color, mono).
 * Based on these values, the sensor may require different settings
 * (e.g. the sensor needs a signal asserted to indicate 600 dpi).
 *
 * \param[in] cmode  the requested color mode
 *
 * \param[in] dpi  the requested dpi
 *
 * \param[out] scanx  the scan line width 
 *
 * \retval scan_err_t any scan error that occured
 *  SCANERR_NONE - dpi was found, and scan sensor set up
 *  SCANERR_INVALID_PARAM - requested dpi is not supported
 *
 * \remark - an even number of edges is required by the ASIC
 **/
scan_err_t scansen_setup(scan_cmode_t cmode, uint32_t dpi, uint32_t scanx)
{
    // This is very specific to the exact sensor
    switch( dpi )
    {
    case 300 : 
        dbg1("STUB FUNCTION - please implement 300 dpi for SDK function %s\n",__FUNCTION__);
        break;
        
    case 600 :
        dbg1("STUB FUNCTION - please implement 600 dpi for SDK function %s\n",__FUNCTION__);
        break;

    case 1200 :
        dbg1("STUB FUNCTION - please implement 1200 dpi for SDK function %s\n",__FUNCTION__);
        break;
        
    default :
        /* if your resolution isn't supported, optionally assert, and/or return
           SCANERR_INVALID_PARAM */
        // XASSERT(0,dpi);
        dbg1("STUB FUNCTION - please implement default dpi for SDK function %s\n",__FUNCTION__);
        return SCANERR_NONE;
    }
    return SCANERR_NONE;    
}

/**
 * \brief Setup the illumination time for each LED 
 *  
 *  Normally, the 3 LEDs (RGB) are on in sequence for a given amount
 *  of time.  The gate values for each color specify how much time
 *  each of the LEDs should be on.  If the gate values are all set to
 *  the same value, then each LED will be on for the same amount of time
 *  (when the time comes in the sequence for the LED to be on). 
 *
 * \param[in] cmode  the requested color mode
 * \param[in] pixels_per_line  the number of pixels on a scan line
 * \param[in] gateR - the percentage of time the red LED should be on
 * \param[in] gateG - the percentage of time the green LED should be on
 * \param[in] gateB - the percentage of time the blue LED should be on 
 *
 * davep 04-Jun-2012 ; Removing floats. The gate value is an integer in [0,100]
 *                     for 0,100%.  Note there is no decimal precision (like
 *                     with the analog gain).
 **/
void scansen_set_leds( uint32_t sensor_num, scan_cmode_t cmode, uint32_t pixels_per_line, 
                      uint32_t gateR, uint32_t gateG, uint32_t gateB)
{
    dbg1("STUB FUNCTION - please implement/verify/verify SDK function %s\n",__FUNCTION__);

    dbg2( "%s cmode=%d ppl=%ld r=%d g=%d b=%d\n", __FUNCTION__, cmode,
                pixels_per_line, gateR, gateG, gateB );
}

/**
 * \brief Enable the LEDs individually
 *
 * Each LED (RGB) can be individually enabled or disabled.
 *
 * \param[in] enable_red  enable or disable the red LED:
 *            true=enable, false=disable
 * \param[in] enable_green  enable or disable the green LED:
 *            true=enable, false=disable
 * \param[in] enable_blue  enable or disable the blue LED:
 *            true=enable, false=disable
 *
 **/
void scansen_led_enable( uint32_t sensor_num, bool enable_red, bool enable_green, bool enable_blue )
{
    scif_led_enable( sensor_num, enable_red, enable_green, enable_blue );
}

/**
 * \brief Send the LED percentile to the hardware
 *
 * Each LED (RGB) can be individually set for strength.  Range is 0-100%
 * This can be used to set power, pwm duty cycle, voltage, current, etc.
 *
 * \param[in] dutyR  strength for the red LED
 * \param[in] dutyG  strength for the green LED
 * \param[in] dutyB  strength for the blue LED
 *
 * davep 04-Jun-2012 ; Removing floats. The gate value is an integer in [0,100]
 *                     for 0,100%.  Note there is no decimal precision (like
 *                     with the analog gain).
 **/
void scansen_set_led_percentile( uint32_t sensor_num,uint32_t dutyR, uint32_t dutyG, uint32_t dutyB)
{
    dbg1("STUB FUNCTION - please implement/verify SDK function %s\n",__FUNCTION__);
    // some boards are set up to require a reduction in LED power to prevent overdriving
    // the LEDs.  If your board requires a maximum of 80% power, the following code is
    // the sample code to support that.
    // Specifically, since the led duty cycle can be set from 0-100, and if 100 is too high
    // a value for your board, make sure to set the led pwm duty cycle
    // from 0-80.  If a value of > 80 would overdrive the LEDs, it is prevented here.
    // This is a simple scale down 0->0 100->80.  
    
    scif_led_duty(sensor_num,1*128, dutyR, dutyG, dutyB);
}

/**
 * \brief Send the analog offsets for the 3 colors to the AFE
 *
 * Take platform independent analog offset values, convert them to the
 * values needed for our AFE, and write them to the AFE.  This function
 * is normally used by calibration which uses this function to set the offset,
 * take measurements, and compute the final offset.  This function assures
 * that the values written to the afe stay within the values the AFE supports.
 *
 * \param[in] red_offset  platform-independent, red analog offset value
 * \param[in] green_offset  platform-independent, green analog offset value
 * \param[in] blue_offset  platform-independent, blue analog offset value 
 *
 * The range of the input parms is 0-65535
 */
void scansen_set_analog_offset(uint32_t sensor_num, int red_offset, int green_offset, int blue_offset)
{
    dbg1("STUB FUNCTION - please implement/verify SDK function %s\n",__FUNCTION__);
}

/**
 * \brief Send the analog gain for the 3 colors to the AFE
 *
 * Take platform independent analog gain values, convert them to the
 * values needed for our AFE, and write them to the AFE.  This function
 * is normally used by calibration which uses this function to set the gain
 * take measurements, and compute the final gain.  This function assures
 * that the values written to the afe are appropriate for the AFEs gain function
 *
 * \param[in] red_gain  platform-independent, red analog gain value
 * \param[in] green_gain  platform-independent, green analog gain value
 * \param[in] blue_gain  platform-independent, blue analog gain value 
 *
 * There is no absolute range, as the gain parameter is the ratio of the
 * desired output pixel to the un-gained output pixel.
 *
 * davep 04-Jun-2012 ; Removing floats. The gate value is an integer in
 *                      [0,10000] for 0,100.00%.  Using float*100 to have two
 *                      decimal digits of precision. For example, gain=3.14
 *                      would be passed as integer value 314
 */

void scansen_set_analog_gain(uint32_t sensor_num, uint32_t red_gain, uint32_t green_gain, uint32_t blue_gain)
{
    dbg1("STUB FUNCTION - please implement/verify SDK function %s\n",__FUNCTION__);
}

scan_err_t scansen_run( void )
{
    return SCANERR_NONE;
}

scan_err_t scansen_stop( void )
{
    return SCANERR_NONE;
}

scan_err_t scansen_pause( void )
{
    return SCANERR_NONE;
}

scan_err_t scansen_resume( void )
{
    return SCANERR_NONE;
}


// HACK: remove this table from the STUB file.  The stub scansen_get_cisx_conf
//       should just return SCANERR_NOT_IMPLEMENTED!
struct scan_cisx_conf {
    uint32_t                dpi;           
    struct cisx_sensor_conf cisx_conf;
};

static const struct scan_cisx_conf scansen_cisx_conf_table[] = {
    {300,   {3, {{91, 864, 0},
                 {91, 864, 0},
                 {91, 864, 0},
                },
             CISX_INT_CFG_CCD_ORDER_012,
             0,
            },
    },
    {600,   {3, {{91, 1728, 0},
                 {91, 1728, 0},
                 {91, 1728, 0},
                },
             CISX_INT_CFG_CCD_ORDER_012,
             0,
            },
    },
    {1200,  {3, {{91, 3456, 0},
                 {91, 3456, 0},
                 {91, 3456, 0},
                },
             CISX_INT_CFG_CCD_ORDER_012,
             0,
            },
    },
    /* end of list marker */
    {-1,   {0, {{0, 0, 0},
                {0, 0, 0},
                {0, 0, 0},
               },
            0,
            0,
           },
    }    
};    

scan_err_t scansen_get_cisx_conf( uint32_t dpi, struct cisx_sensor_conf *cisxsensor )
{
    int i;

    for (i=0; ; i++) 
    {
        if (scansen_cisx_conf_table[i].dpi == -1) 
        {
            dbg1("%s: invalid/unknown DPI %ld\n", __FUNCTION__, dpi);
            XASSERT(0, dpi);
        }
        else if (scansen_cisx_conf_table[i].dpi == dpi) 
        {
            memcpy(cisxsensor, &scansen_cisx_conf_table[i].cisx_conf, sizeof(struct cisx_sensor_conf));

            cisxsensor->cbiout_pace = 0;

            break;
        }
    }
    return SCANERR_NONE;
    //return SCANERR_NOT_IMPLEMENTED;
}

// hack, SANDRA REMOVE
typedef struct {

    scan_cmode_t   cmode;

    uint32_t         dpi;

    uint32_t         scanx;

} scan_scanx_conf_t;

static const scan_scanx_conf_t scansen_scanx_table[] = {
    {SCAN_CMODE_MONO,  300,  (91*3)+2592  }, 
    {SCAN_CMODE_MONO,  600,  (91*3)+5184  }, 
    {SCAN_CMODE_MONO,  1200, (91*3)+10368 }, 
    {SCAN_CMODE_COLOR, 300,  (91*3)+2592  }, 
    {SCAN_CMODE_COLOR, 600,  (91*3)+5184  }, 
    {SCAN_CMODE_COLOR, 1200, (91*3)+10368 },
    /* end of list marker */
    {0, -1, 0}
};

scan_err_t scansen_calc_scanx( scan_cmode_t cmode, uint32_t dpi, uint32_t *scanx )
{
    scan_scanx_conf_t scanx_conf;
    int i;

    for (i=0; ; i++) 
    {
        if (scansen_scanx_table[i].dpi == -1) 
        {
            dbg1("%s: invalid/unknown DPI %ld\n", __FUNCTION__, dpi);
            XASSERT(0, dpi);
        }
        else if ((scansen_scanx_table[i].cmode == cmode) && (scansen_scanx_table[i].dpi == dpi)) 
        {
            memcpy(&scanx_conf, &scansen_scanx_table[i], sizeof(scan_scanx_conf_t));
            dbg2("%s: cmode=%d dpi=%d scanx=%d\n", __FUNCTION__,
                                                   scanx_conf.cmode,
                                                   scanx_conf.dpi,
                                                   scanx_conf.scanx);
            *scanx = scanx_conf.scanx;
            break;
        }
    }

    return SCANERR_NONE;
}


