/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



/**
 * \file scansen-common.c
 *
 * \brief Scan Sensor common functions
 *
 * Implements scan sensor common functions (not OEM specific)
 *
 */

#include "lassert.h"
#include "scantypes.h"
#include "scandbg.h"
#include "scansen.h"
#include "scanvars.h"


/**
 * \brief (Debug) Dump a sensor gap structure
 *
 **/
void scansen_chipgap_dump(const struct scan_sensor_chipgap *gaps)
{
    int i;

    if (gaps != NULL)
    {
        dbg2("%s first_gap=%d width=%d num_gaps=%d\n", __FUNCTION__,
               gaps->first_gap_offset, 
               gaps->gap_to_gap_width,
               gaps->num_chipgap_list);

        for (i = 0; i < SCANSEN_CHIPGAP_MAX_GAPS; i++)
        {
            dbg2("%d ", gaps->chipgap_list[i]);
        }
        dbg2("\n");
    }
}


/**
 * \brief Map sensor chipgap from sensor pixels to margin relative pixels
 *
 * Sensor chipgaps are specified by OEM code in terms of raw sensor pixels.
 * Once the raw pixels go through the PIC top margin block, the pixel numbering
 * will not be the same.  We must re-map the sensor gaps based on the PIC top
 * margin settings (pixel area in scanvars).
 *
 * \param[in]  chipgap settings, sensor pixels
 * \param[out] chipgap settings, margin relative pixels
 * 
 * \return scan_err_t any scan error that occured
 * \retval SCANERR_NONE - chipgap was remapped
 * \retval SCANERR_INVALID_PARAM - requested table pointers invalid
 * \retval SCANERR_GENERIC_FAIL - no overlap between gaps and margins
 **/
scan_err_t scansen_chipgap_map_to_margins(const struct scan_sensor_chipgap *sensor_gaps, 
                                                struct scan_sensor_chipgap *margin_gaps)
{
    int      i;
    int      num_valid_segments = 0;
    uint32_t gap_pixel_loc; 
    const struct scanvars  *sv;
    long int margin_left_pixel, margin_right_pixel;
 
    // Quick sanity check on input pointers
    if (sensor_gaps == NULL || margin_gaps == NULL)
    {
        dbg2("%s: invalid input parameters\n", __FUNCTION__);
        return SCANERR_INVALID_PARAM;
    }

    // We have pointers, put the output data into a known state before doing
    // anything else
    memset( margin_gaps, 0, sizeof(struct scan_sensor_chipgap) );

    // Make sure the sensor gap list has some entries
    if ( (sensor_gaps->num_chipgap_list == 0) || 
         (sensor_gaps->num_chipgap_list > 1 && sensor_gaps->gap_to_gap_width == 0) )
    {
        dbg2("%s: invalid input parameters num_chipgap_list=%d gap_to_gap_width=%d\n", 
              __FUNCTION__, sensor_gaps->num_chipgap_list, sensor_gaps->gap_to_gap_width);
        return SCANERR_INVALID_PARAM;
    }
   
    // Peek at the scanvars and grab the margins (part of the pixel area)
    sv = scanvar_peek();
    margin_left_pixel  = sv->scan_area_pixels.x;
    margin_right_pixel = sv->scan_area_pixels.x + sv->scan_area_pixels.width - 1;

    dbg2("%s: left_mar=%d right_mar=%d\n", __FUNCTION__, margin_left_pixel, margin_right_pixel);
    dbg2("%s: Sensor gap list:\n", __FUNCTION__);
    scansen_chipgap_dump(sensor_gaps);

    // March through the gap list looking for something inside the margins. Start off
    // at the first gap location and then increment by the gap width.
    // NOTE:  we don't have to worry about what the inserted gap pixels do to the
    //        pixel numbering. That is only an issue downstream of chipgap ... SEP.
    gap_pixel_loc = sensor_gaps->first_gap_offset;

    for (i = 0; i < sensor_gaps->num_chipgap_list; i++)
    {
        // Looking for gaps within the margins
        if ( gap_pixel_loc >= margin_left_pixel && gap_pixel_loc <= margin_right_pixel ) 
        {
            // If this is the first valid segment, set first_gap_offset and gap width
            if (num_valid_segments == 0)
            {
                // Must convert first sensor gap location to be margin relative!
                margin_gaps->first_gap_offset = gap_pixel_loc - margin_left_pixel;
                margin_gaps->gap_to_gap_width = sensor_gaps->gap_to_gap_width;
            }

            // Copy the gap value
            margin_gaps->chipgap_list[num_valid_segments] = sensor_gaps->chipgap_list[i]; 
            margin_gaps->num_chipgap_list++;

            num_valid_segments++;
        }

        // Move to the next gap location
        gap_pixel_loc += sensor_gaps->gap_to_gap_width;
    }

    dbg2("%s: Margin gap list:\n", __FUNCTION__);
    scansen_chipgap_dump(margin_gaps);

    // Let the caller know if we came up empty
    if (num_valid_segments == 0)
    {
        return SCANERR_GENERIC_FAIL;
    }

    // We mapped some entries
    return SCANERR_NONE;
}

