/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2006-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * \file scanman.c
 *
 * \brief Scan Manager 
 *
 *  This file's entire reason for existence is to sit between Scantask and
 *  whatever architecture the rest of The System might create.  
 *
 *  ScanMan exists to keep Scantask software architecture independent.
 */
    
/* davep 23-Jul-2008 ; added -D__AGMESSAGE_H_ to the makefile to prevent
 * agMessage.h from being included in the scan code.  But we need agMessage.h in
 * this file.
 */
#ifdef __AGMESSAGE_H__
#undef __AGMESSAGE_H__
#endif

#include <string.h> 
#include <pthread.h>

#include "posix_ostools.h"
#include "scos.h"

#include "ATypes.h"
#include "list.h"           
#include "lassert.h"
#include "agResourceMgr.h"
#include "agMessage.h"
#include "agRouter.h"
#include "memAPI.h"
#include "utils.h"
#include "agConnectMgr.h"
#include "sys_init_api.h"
#include "data_access_api.h"
#include "error_types.h"
#include "platform.h"
#include "event_observer.h"
#include "response_subject.h"

#include "scancore.h"
#include "scantypes.h"
#include "scanvars.h"
#include "scanlib.h"
#include "scanman.h"
#include "scantools.h"
#include "scandbg.h"
#include "scanhw.h"
#include "scan.h"
#include "scandbg.h"
#include "cal.h"
#include "calstats.h"
#include "pie.h"
#include "scanACL.h"
#include "smirb.h"
#include "scantask.h"
#include "cmdline.h"
#include "scanhwerr.h"
#include "scands.h"
#include "demo_scanapp.h"
#include "icefileapp.h"
#include "scanmech.h"
#include "scanmsg.h"
#include "safetylock.h"
#include "icedma.h"

/* enable to add more debug messages */
#define SCANMAN_DEBUG

#ifdef SCANMAN_DEBUG
    #define SCANMAN_ASSERT(x)  ASSERT((x))
#else
    #define SCANMAN_ASSERT(x)
#endif

#define SCANMAN_QUEUE_SIZE 20 
#define SCANMAN_TASK_PRI     THR_PRI_NORMAL
#define SCANMAN_STACK_SIZE   (POSIX_MIN_STACK_SIZE  * 4)// has to be big enough for pthreads
#define NUMSCANMODULES 2

/** handy macro to track the ScanMan state machine transitions */
#define STATE_TRANSITION(new_state)  (state_transition( (new_state), __LINE__ ))

/* davep 16-Apr-2013 ; test/debug feature; pretend we're scanning on a dual
 * scan system (sensor on both sides of paper)
 */
//#define FAKE_DUAL_SCAN 1

// align the stack
ALIGN(8)
static UINT8 scanman_stack[SCANMAN_STACK_SIZE];
static pthread_t scanman_task_id; 
static mqd_t scanman_msgq;
//static MESSAGE scanman_msgq_buffer[SCANMAN_QUEUE_SIZE];

typedef enum {
    SMSTATE_IDLE=1,
    SMSTATE_STARTING=2,
    SMSTATE_SCANNING=3,
    SMSTATE_CANCELING=4,
    SMSTATE_FAILING=5,
    SMSTATE_WAITING=6,
    SMSTATE_WAITING_FOR_SCANTASK_1=7,
    SMSTATE_WAITING_FOR_SJM=8,
    SMSTATE_WAITING_FOR_APP=9,
    SMSTATE_WAITING_FOR_SCANTASK_2=10,
} scanman_state;

static scanman_state g_curr_state;

/* davep 06-Dec-2005 ; state transition table
 * davep 13-Nov-2008 ; new state transition table
 * davep 24-Jun-2009 ; scanman v2 with no stopy copy support.
 * davep 27-Jun-2009 ; http://10.71.116.60/view/Main/ScanManv2 
 *
 * Scanman needs to maintain state in order to know when to start/stop the
 * scantask. Cancel brought out a lot of problems with scanman/scantask's
 * interaction so I had to make this state machine. 
 *
 */

/* ScanMan vs Scantask.
 *
 * ScanMan exists to protect Scantask from the rest of the system.  ScanTask
 * controls all the scan hardware and, as such, is extremely complex, containing
 * several message queues and state machines.  ScanMan prevents other threads
 * from interfering with Scantask.
 *
 * ScanMan 'blackout' period. With the previous paragraph in mind, there are
 * times when Scantask cannot talk to the rest of the system. A cancel in a bad
 * spot could be misinterpreted and cause certain operations (e.g., physical paper
 * present flag scans) to fail. This is the "Scantask blackout period".  The
 * 'waiting4scantask' state is the Scantask blackout. We wait for the
 * SMSG_SCAN_READY before we can talk to Scantask again.
 *
 *
 * Some Random Notes.
 *
 * 'scanning' receives SMSG_SCAN_FAILED when Scantask fails (paper jam, mispick,
 * etc.)  ScanMan will call the SJM to cancel the current job. 
 *
 * When 'scanning' receives MSG_CANCEL, ScanMan will ask Scantask to cancel.
 * ScanMan waits for the SMSG_SCAN_FAILED indicating Scantask has successfully
 * canceled. Then ScanMan waits for 
 *                                                     
 * In the 'scanning' state, we could receive multiple SMSG_PAGE_START,
 * SMSG_PAGE_END when we're scanning from ADF.
 *
 * waiting4scantask vs waiting4sjm. On leaving the 'scanning' state, ScanMan has
 * to rendezvous between two different threads: the SJM (System Job Manager) and
 * Scantask. On SCAN_FAILED, ScanMan asks SJM to cancel the job then goes to the
 * 'failing' state. There is no need to cancel Scantask. At this point, ScanMan
 * has to wait for both Scantask and SJM. ScanMan waits for the MSG_CANCEL from
 * SJM and the SMSG_SCAN_READY from Scantask. Those messages can arrive in
 * either order but both have to be received in order to go back to IDLE.
 *
 *
 */

/**
 * \brief  track our state transitions 
 *
 * \author David Poole
 * \date 21-Sep-2007
 *
 */

static void state_transition( scanman_state new_state, int line_number )
{
    dbg2( "scanman state from %d to %d at %d\n", 
                g_curr_state, new_state, line_number );
    g_curr_state = new_state;

    /* davep 01-Jun-2012 ; adding observer events to make control panel happy */
    switch( new_state ) {
        case SMSTATE_IDLE :
            scanman_observer_notify_event( SCANMAN_EVENT_IDLE, 1 );
            break;

        case SMSTATE_STARTING :
            scanman_observer_notify_event( SCANMAN_EVENT_SCAN_START, 1 );
            break;

        case SMSTATE_SCANNING :
            scanman_observer_notify_event( SCANMAN_EVENT_SCAN_RUNNING, 1 );
            break;

        case SMSTATE_CANCELING : 
        case SMSTATE_FAILING : 
            scanman_observer_notify_event( SCANMAN_EVENT_SCAN_CANCEL, 1 );
            break;

        case SMSTATE_WAITING : 
            /* we get into the waiting state on a successful completion of a
             * scan (i.e., a scan that wasn't canceled or failed due to some
             * error)
             */
            scanman_observer_notify_event( SCANMAN_EVENT_SCAN_COMPLETE, 1 );
            break;

        default : 
            /* huh? */
            break;
    }

}

/**
 * \brief return pointer to my pxthread id 
 *
 *  Originally created to verify function caller thread context.
 *
 * \author David Poole
 * \date 15-Sep-2011
 *
 */

pthread_t scanman_thread_identify( void )
{
    return scanman_task_id;
}

 /**
  * \brief  Blocking send a message to ScanMan
  *
  * Wrapped the Message Router in a function so we can separate the Router from
  * Scantask.  (Working on the Scan/Copy SDK.)
  *
  * This could be called from interrupt context!
  *
  * \author David Poole
  * \date 23-Jun-2008
  *
  * davep 23-Jul-2008 ; moving the scan subsystem (scantask) to using its
  * own message (scan_msg_t) as part of the BSDK. 
  *
  * Scantask uses scanman_msg_send() to communicate messages outside his
  * own scan_task() thread.  scanman_msg_send() translates the
  * scan_msg_t structure to a MESSAGE structure.
  *
  * The fields of scan_msg_t and MESSAGE are exactly the same. 
  * 
  * ScanMan will receive MESSAGE but the values in msgType will be the SMSG_xxx
  * values from scancore.h and scantypes.h which leaves us with the possibility
  * of collision with agMessage.h AGMSG values.
  *
  * So cheat and set the high bit on our outgoing SMSG_xxx msgType so ScanMan
  * can tell where the message came from.
  *
  * THIS IS A KLUDGE!  Yeah, yeah, yeah.  
  *
  */

scan_err_t scanman_msg_send( scan_msg_t *scan_msg )
{
    error_type_t err;
    MESSAGE sys_msg;

#define MESSAGE_FROM_SCANTASK_FLAG (1<<31)

    /* NOTE! This function can be called from interrupt context! */

    /* Another note. This is the ONE and ONLY ONE place where we move a
     * scan_msg_t to a MESSAGE.
     */
    sys_msg.msgType = MESSAGE_FROM_SCANTASK_FLAG | scan_msg->msgType;
    sys_msg.param1 = scan_msg->param1;
    sys_msg.param2 = scan_msg->param2;
    sys_msg.param3 = scan_msg->param3;

    err = SYMsgSend( SCANMANID, &sys_msg );
    if( err != OK ) {
        XASSERT( err==OK, err );
        return SCANERR_MSG_FAIL;
    }

    return SCANERR_NONE;
}

/**
 * \brief Updates a int data store varaiable
 *
 * \author Alan Smithee 
 *
 * We cannot update the scands from interrupt context (e.g., the paper present flag). 
 *
 * Using ScanMan instead of ScanTask because ST has many message loops and ST
 * has the taskqueue.
 *
 **/

static void handle_paper_event_msg( scan_adf_paper_event_t adfevt, UINT32 value  )
{
    scan_err_t scerr;

    dbg2( "scanman received adf paper event evt=%d value=%d\n", adfevt, value );

    switch( adfevt ) {
        case SCAN_ADF_PAPER_PRESENT :
            /* value contains the current status 
             * value==0 paper not present
             * value==1 paper present
             */

            scerr = scands_set_integer( "adf_paper_present", value && 1 );
            XASSERT( scerr==SCANERR_NONE, scerr );

            scanman_observer_notify_event( SCANMAN_EVENT_ADF_PAPER_PRESENT, value );
            break;

        case SCAN_ADF_PAPER_NOPICK:
            scerr = scands_set_integer( "adf_paper_nopick", value && 1 );
            XASSERT( scerr==SCANERR_NONE, scerr );

            scanman_observer_notify_event( SCANMAN_EVENT_ADF_PAPER_NOPICK, value );
            break;

        case SCAN_ADF_PAPER_JAM : 
            /* value==1 paper jammed
             * value==0 paper jam cleared
             */
            scerr = scands_set_integer( "adf_paper_jam", value && 1 );
            XASSERT( scerr==SCANERR_NONE, scerr );

            scanman_observer_notify_event( SCANMAN_EVENT_ADF_PAPER_JAM, value );
            break;

        default : 
            /* ignore */
            break;
    }

}

/**
 * \brief Blocking wait for Scantask to be ready
 *
 * This function handles the "scantask blackout period".
 *
 * Blocks waiting for Scantask's SMSG_SCAN_READY message.
 *
 * \author David Poole
 * \date 05-Dec-2005
 * \date 27-Jun-2009 (removed the cancel, turned into a general blocking wait)
 *
 */

static void blocking_wait_for_scantask_ready( void )
{
    UINT pxretcode;
    MESSAGE sys_msg;
    BOOL scantask_done;

    dbg2( "%s\n", __FUNCTION__ );

    scantask_done = FALSE;

    /* Wait for scantask to finish canceling; Scantask will reply with an
     * SMSG_SCAN_FAILED message
     */
    while( !scantask_done) {
        pxretcode = posix_wait_for_message( scanman_msgq, (char *)&sys_msg, sizeof(MESSAGE),
                                            5*USEC_PER_SECOND );
        XASSERT( pxretcode == 0 || pxretcode == ETIMEDOUT, pxretcode );

        if( pxretcode != 0 ) {
            /* scanman must wait for a message from scantask telling us he's
             * done before we can leave
             */
            dbg1("%s no messages; wait some more\n", __FUNCTION__ );

            /* stay in message loop */
            continue;
        }
        
        /* We should only be getting messages from scantask! Once we enter a
         * cancel state, nothing else should be talking to us.
         */
        /* davep 05-Aug-2009 ; (Bug 13698) we can get into this state with
         * cancel. Cancel is anything but clean. Have to handle (and ignore)
         * messages from the app.
         */
//        XASSERT( sys_msg.msgType & MESSAGE_FROM_SCANTASK_FLAG, 
//                    sys_msg.msgType );

        switch( sys_msg.msgType ) {
            case MSG_SCAN_JOBSTART:
                /* ignore */
                dbg1("scanman received jobstart in state=%d\n", g_curr_state );
                break;

            case MSG_SCAN_JOBEND :
                /* ignore */
                dbg1( "scanman received jobend in state=%d\n", g_curr_state );
                break;

            case MSG_CANCELJOB: 
                /* we're already canceling or finishing so ignore */
                dbg1( "scanman received canceljob in state=%d\n", g_curr_state );
                break;

            case (SMSG_PAGE_DATA | MESSAGE_FROM_SCANTASK_FLAG) :
                dbg1( "scanman cancel received pagedata\n" );
                /* free the data; ignore the rest of the message */
                PTR_FREE( sys_msg.param3 );
                break;

            case (SMSG_PAGE_START | MESSAGE_FROM_SCANTASK_FLAG) :
                /* page_start is scantask telling us he's starting data (sort of
                 * like a "file open")
                 */
                dbg1( "scanman cancel received pagestart\n");
                break;

            case (SMSG_PAGE_END | MESSAGE_FROM_SCANTASK_FLAG):
                /* page_end is scantask telling us he's done sending data (sort
                 * of like a "file close")
                 */
                dbg1( "scanman cancel received pageend\n");
                break;
            
            case (SMSG_SCAN_END | MESSAGE_FROM_SCANTASK_FLAG):
                /* scan_end is scantask telling us he's done with the scan but
                 * he's not yet ready to accept another scan
                 */
                dbg1( "scanman cancel received scanend\n");
                break;
            
            case (SMSG_SCAN_READY | MESSAGE_FROM_SCANTASK_FLAG):
                /* scan_ready is the "ack" from scantask telling us he's done */
                dbg1( "scanman cancel received scan_ready\n");
                scantask_done = TRUE;
                STATE_TRANSITION( SMSTATE_IDLE );
                break;

            case (SMSG_SCAN_FAILED | MESSAGE_FROM_SCANTASK_FLAG):
                dbg1( "scanman cancel received scan_failed\n");
                STATE_TRANSITION( SMSTATE_WAITING_FOR_SCANTASK_1 );
                break;

            /* the following are messages that can occur normally during a
             * cancel; the default case will assert so we can catch anything we
             * don't expect 
             *
             * XXX do we want to go back to simply ignoring the unexpected
             * messages?
             */
            case (SMSG_SCAN_SIZE | MESSAGE_FROM_SCANTASK_FLAG) :
                dbg1( "scanman cancel received scan_size\n");
                break;

            case (SMSG_DATA_BLOB | MESSAGE_FROM_SCANTASK_FLAG):
                /* davep 27-Jun-2009 ; oops. Amazing this bug slipped through
                 * for so long.  The data blob contains a malloc'd buffer which
                 * needs free.
                 */
                PTR_FREE( sys_msg.param3 );
                break;

            case (SMSG_SCAN_ADF_PAPER_EVENT | MESSAGE_FROM_SCANTASK_FLAG) :
                handle_paper_event_msg( sys_msg.param1, sys_msg.param2 );
                break;

            case (SMSG_CAL_CALIBRATION_IN_PROGRESS | MESSAGE_FROM_SCANTASK_FLAG) :
                scanman_observer_notify_event( SCANMAN_EVENT_SCAN_CALIBRATION_RUNNING, 1 );
                break;

            case (SMSG_CAL_CALIBRATION_DONE | MESSAGE_FROM_SCANTASK_FLAG) :
                scanman_observer_notify_event( SCANMAN_EVENT_SCAN_CALIBRATION_COMPLETE, 1 );
                /* send another 'scanning' status to make the status transition
                 * a little more intuitive
                 * scan running -> calibration running -> cal complete -> scan running 
                 */
                scanman_observer_notify_event( SCANMAN_EVENT_SCAN_RUNNING, 1 );
                break;

            default:
                /* assert here so we can get a quick notice if we get any weird
                 * messages we should handle in the cancel state
                 */
                XASSERT( 0, sys_msg.msgType );
                break;
        }
    } /* end while */

    dbg2( "%s done\n", __FUNCTION__ );
}

/**
 * \brief send message to job manager acking our cancel message 
 *
 * I have no clue what weird_unknown_parameter is but it comes from the cancel
 * message and we have to send it along in the ack.
 *
 * \author David Poole
 * \date 12-Jun-2007
 *
 */

static void
ack_cancel_msg( void *weird_unknown_parameter )
{
    MESSAGE send_msg;
    error_type_t err;

    /* tell job manager we've successfully cancelled */
    send_msg.msgType = MSG_CANCELJOB;
    send_msg.param1 = SYS_ACK;
    send_msg.param2 = SCANMANID;
    send_msg.param3 = weird_unknown_parameter;
    err = SYMsgSend( SJMID, &send_msg);
    XASSERT( err==OK, err );
}


static void
free_recipe_msg( CURRENT_RESOURCE *job_resources )
{
    MESSAGE send_msg;
    error_type_t err;

    send_msg.msgType = MSG_FREERECIPE;
    send_msg.param1 = 0;
    send_msg.param2 = 0;
    send_msg.param3 = job_resources;
    err = SYMsgSend( SJMID, &send_msg );
    XASSERT( err==OK, err );
}

/**
 * \brief Finish the scanman cancel
 *
 * Finish the cancel process.  ACK the System Job Manager, release my resources,
 * sanity check Scantask.
 *
 * \author David Poole
 * \date 10-Nov-2005
 *
 */

static void cancel_scanman( void *weird_unknown_parameter, 
                            CURRENT_RESOURCE *job_resources )
{
//    ULONG msgcnt;

    dbg2( "%s\n", __FUNCTION__ );

    /* tell job manager we've successfully cancelled */
    ack_cancel_msg( weird_unknown_parameter );

    /* release myself as a resource */
    free_recipe_msg( job_resources );

    /* our message queue should be EMPTY (this is not foolproof; we could be
     * racing something)
     */
    /* davep 07-Apr-2006 ; tweak for bug 928  */
//    msgcnt = get_queue_pending( &scanman_msgq );
//    XASSERT( msgcnt==0, msgcnt );

    /* davep 27-Jun-2009 ; XXX temp remove this assert; hitting it with a cancel
     * during N-UP. Not sure this assert is valid anymore (SMECH_POLL?)
     */
    /* it's a bit rude to be peeking into scantask's message queue but we need
     * to make sure scantask is idle as well (again, this is not foolproof)
     */
//    msgcnt = scantask_get_queue_pending();
//    XASSERT( msgcnt==0, msgcnt );

    dbg2( "%s done\n", __FUNCTION__ );
}

/**
 * \brief  Cancel the current job
 *
 * Sends the "cancel job" message to the system job manager.
 *
 * \author David Poole
 * \date 26-Jan-2007
 *
 */

static void scanman_cancel_job( void )
{
    MESSAGE sys_msg;
    error_type_t err;

    dbg2( "%s\n", __FUNCTION__ );

    sys_msg.msgType = MSG_CANCELJOB;
    sys_msg.param1 = SYS_REQUEST;
    sys_msg.param2 = 0;
    sys_msg.param3 = (void*)e_Scanner;

    err = SYMsgSend( SJMID, &sys_msg );
    XASSERT( err==OK, err );
}

/**
 * \brief  register scanner with The System
 *
 * Copy/Paste code from ScanManInit() after a race condition was discovered. I
 * Have no idea what this code means or what it does.
 *
 * The Scantask thread is now dynamically allocated and isn't known until the
 * thread is fully up and running.  We now have to wait until Scantask is fully
 * running before calling the ResourceRegister().
 *
 * \author David Poole
 * \date 08-Jul-2008
 *
 */

static void register_scanman_with_system( void )
{
    dbg2( "%s\n", __FUNCTION__ );

    /* everyone who allocates memory must be in the list of threads registered
     * with the resource manager
     */

    /* davep 29-Dec-2008 ; (Bug 11386) Don't register Scantask as a resource
     * anymore. No longer necessary since we went to the new memory manager. Not
     * registering Scantask means we can register ScanMan before the entire
     * system is running. 
     * */
    rm_register(e_Scanner, SCANMANID);
}

/**************************************************
 * Function name   :  posix_initialize
 *    returns      :  none
 * Created by      :  David Poole
 * Date created    :  26-Aug-2008
 * Description     : 
 *
 *  Tell tx_posix to initialize the ThreadX fake POSIX subsystem.
 *
 * Notes           : 
 *
 **************************************************/

//#ifdef HAVE_TX_POSIX_2008
//static void posix_initialize( void )
//{
//    UINT txretcode;

//    txretcode = tx_posix_initialize();
//    XASSERT( txretcode==TX_SUCCESS, txretcode );
//}
//#endif

/**
 * \brief  Do the steps necessary for scanman's clean job completion.
 *
 * By clean, I mean "not a cancel". 
 *
 *
 * \author David Poole
 * \date 16-Nov-2008
 *
 */

static void finish_job( CURRENT_RESOURCE **p_job_resources )
{
    CURRENT_RESOURCE *job_resources;

    /* use a temporary so code is a little less pointer confusing */
    PTR_ASSIGN( job_resources, *p_job_resources );

    ASSERT( job_resources != NULL );

    /* davep 17-Oct-2007 ; Bug 7498 ; disable possible calibration
     * between pages of a single job ; turn it back on now that the
     * job is done
     */
    scan_set_config_no_calcache_timeout( FALSE );

    /* release myself as a resource */
    free_recipe_msg( job_resources );

    /* we're now released */
    job_resources = NULL;
}

#ifdef FAKE_DUAL_SCAN
/**
 * \brief  Duplicate a data message payload into a newly malloc'd buffer.
 *
 * Created to mimic dual scan (sensor on both side of paper) by sending the
 * same data twice. Because the downstream code will always free() the data,
 * cannot simply send the same pointer twice.
 *
 * \author David Poole
 * \date 16-Apr-2013
 */

static scan_err_t make_dup_data_msg( MESSAGE *msg, uint32_t bytes_per_row ) 
{
    int datalen;
    uint16_t num_rows;
    scan_data_type dtype;
    uint8_t *src_ptr, *dst_ptr;
    int malloc_fail_counter;
    bool last;

    msg_scan_planedata_decode( msg, &num_rows, &dtype, &src_ptr, &last );

    msg->param2 |= SMSG_DATA_SET_PAGE_SIDE( 1 );

    datalen = num_rows * bytes_per_row;

    /* dup the data */
    malloc_fail_counter= 0;
    while( 1 ) {
        dst_ptr = MEM_MALLOC_ALIGN( datalen, cpu_get_dcache_line_size() );
        if( dst_ptr ) {
            break;
        }

        malloc_fail_counter += 1;
        if( malloc_fail_counter > 10 ) {
            dbg2( "%s unable to allocate bytes=%d; cancelling scan\n", 
                    __FUNCTION__, datalen);
            scantask_cancel_msg();
            return SCANERR_OUT_OF_MEMORY;
        }
        posix_sleep_seconds(1);
    }

    memcpy( dst_ptr, src_ptr, datalen );
    PTR_ASSIGN(msg->param3,dst_ptr);
    return SCANERR_NONE;
}
#endif

/**
 * \brief ScanMan task
 *
 *
 * \author Brad Smith
 * \date 
 *
 *
 **/

static void* scanman_task(void* unused)
{
    UINT pxretcode;
    MESSAGE sys_msg;
    error_type_t err;
    CURRENT_RESOURCE *job_resources;
    scan_err_t scerr;
    ScannerHWError scantask_hw_err;
    struct scanvars *sv;
    struct scanvars *cached_sv;
    uint32_t running_scan_task_id;
    struct scanman_ioctl_request_block *smirb;
    uint32_t tmp_task_id;
    uint32_t pixels_per_row_padded, pixels_per_row, total_rows,
             bits_per_pixel, bytes_per_row;

    /* wait for rest of system to become fully initialized */
    SysWaitForInit();
    
    scerr = scanman_obs_navel_gazer_init();
    /* if my test/debug navel gazer fails to init, we don't care */

    scanman_observer_notify_event( SCANMAN_EVENT_INITIALIZING, 1 );

    scan_cmdline_init();

    // Start scan worker task.
    scerr = scantask_init( scanman_msg_send );
    XASSERT( scerr==SCANERR_NONE, scerr );

    scantask_hw_err = ScannerHWError_None;

    cached_sv = NULL;

    memset( &sys_msg, 0, sizeof(sys_msg) );
    
    // Wait for scantask to initialize.
    // Don't do anything else until that happens.
    while( sys_msg.msgType != (SMSG_SCAN_INIT_DONE | MESSAGE_FROM_SCANTASK_FLAG) )
    {
        pxretcode = posix_wait_for_message( scanman_msgq, (char *)&sys_msg, sizeof(MESSAGE),
                                            POSIX_WAIT_FOREVER );
        XASSERT( pxretcode == 0, pxretcode );

        switch( sys_msg.msgType ) {
            case (SMSG_SCAN_INIT_DONE | MESSAGE_FROM_SCANTASK_FLAG) :
                /* check for an error */
                scantask_hw_err = (ScannerHWError)sys_msg.param1;

                if( scantask_hw_err != ScannerHWError_None ) {
                    dbg1( "%s scantask failed to init! err=%d\n", 
                                __FUNCTION__, scantask_hw_err );
                    scanman_observer_notify_event( SCANMAN_EVENT_SCAN_OFFLINE, 1 );
                }
                else {
                    /* scantask is up and running successfully */
                    dbg1( "%s scantask initialized successfully\n", __FUNCTION__ );
                    /* the state transition to idle below will send the observer notify
                     * event 
                     */
                    //send a different event to let UI knows that boot initialization has finished
                    scanman_observer_notify_event( SCANMAN_EVENT_INITIALIZED, 1 );
                }

                /* Note we continue on from here whether or not Scantask
                 * successfully initialized. Why? Well, why not? What else should we
                 * do? 
                 */
                break;

            /* davep 04-Feb-2011 ; add handling of paper event; need this early
             * in the boot process (event before INIT_DONE) so the hardware can
             * send us an initial state
             */
            case (SMSG_SCAN_ADF_PAPER_EVENT | MESSAGE_FROM_SCANTASK_FLAG) :
                handle_paper_event_msg( sys_msg.param1, sys_msg.param2 );
                break;

            default : 
                dbg1( "%s unwanted message msgType=%d param1=%d param2=%d param3=%p\n",
                            __FUNCTION__, sys_msg.msgType, sys_msg.param1,
                            sys_msg.param2, sys_msg.param3 );
                break;
        }
    }

#ifdef HAVE_ACL
    registerScanACLCmds();
#endif

    job_resources = NULL;

    STATE_TRANSITION(SMSTATE_IDLE);

    icefileapp_onetime_init();

    while( 1 )
    {
        pxretcode = posix_wait_for_message( scanman_msgq, (char *)&sys_msg, sizeof(MESSAGE),
                                            POSIX_WAIT_FOREVER );
        XASSERT( pxretcode == 0, pxretcode );

        if( pxretcode != 0 ) {
            continue;
        }

        switch (sys_msg.msgType)
        {
            case MSG_RESOURCES:
                dbg1("scanman received resources in state=%d\n", g_curr_state );

                dbg2("%s resource msg: param1=%#lx param2=%#lx param3=%p\n", 
                            __FUNCTION__, sys_msg.param1, sys_msg.param2,
                            sys_msg.param3 );

                /* make sure we correctly ended the last job */
                XASSERT( g_curr_state==SMSTATE_IDLE, (UINT32)g_curr_state );
                XASSERT( job_resources==NULL, (UINT32)job_resources );

                job_resources = (CURRENT_RESOURCE *) sys_msg.param3;
                dbg2("%s src=%d dst=%d\n", __FUNCTION__, 
                            job_resources->Source,
                            job_resources->Destination );

                STATE_TRANSITION( SMSTATE_STARTING );
                break; 

            case MSG_ACKRECIPE :
                dbg1("scanman received ackrecipe in state=%d\n", g_curr_state );

                /* ignore for now; this message is from the System Job Manager
                 * telling us all Resource messages have been sent (avoids race
                 * conditions)
                 */

                /* davep 17-Jan-2012 ; don't start job if we're in an
                 * error state 
                 */
                if( !scanner_is_alive() ) {
                    dbg2( "%s scanner is offline hwerr=%d\n", __FUNCTION__, scan_get_hwerr() );
                    scanman_cancel_job();
                    break;
                }

                /* davep 07-Apr-2006 ; bug 928; gently ignore unwanted messages */
                if( g_curr_state!=SMSTATE_STARTING ) {
                    dbg1( "%s unexpected message type=%d in state=%d\n", 
                                __FUNCTION__, sys_msg.msgType, g_curr_state );
                    SCANMAN_ASSERT(0);
                    break;
                }
                /* no need to change state */
                break;

            case MSG_SCAN_JOBSTART:
                dbg1("scanman received jobstart in state=%d\n", g_curr_state );

                /* davep 17-Jan-2012 ; don't start job if we're in an
                 * error state (duplicate of same check in MSG_ACKRECIPE to
                 * handle race conditions)
                 */
                if( !scanner_is_alive() ) {
                    dbg2( "%s scanner is offline hwerr=%d\n", __FUNCTION__, scan_get_hwerr() );
                    scanman_cancel_job();
                    break;
                }

                switch( g_curr_state ) {
                    case SMSTATE_STARTING :
                    case SMSTATE_WAITING_FOR_APP :
                        /* app wants another scan */
                        ASSERT( job_resources != NULL );

                        /* davep 11-Nov-2008 ; Bug 11161 ; require scanvar in a message */
                        ASSERT( sys_msg.param3 != NULL );
                        PTR_ASSIGN( sv, sys_msg.param3 );
                        XASSERT( sv->cookie==SCANVAR_COOKIE, sv->cookie );

                        /* davep 02-Sep-2009 ;  claim the scantask data path for myself */
                        scanvar_set_scan_callback( sv, scanman_msg_send );

                        /* tell Scantask to get off his butt and send us some data */
                        scerr = scantask_scan_start( sv, &running_scan_task_id );
                        XASSERT( scerr==0, scerr );

                        STATE_TRANSITION( SMSTATE_SCANNING );
                        break;

                    case SMSTATE_WAITING :
                        /* davep 11-Nov-2008 ; Bug 11161 ; require scanvar in a message */
                        ASSERT( sys_msg.param3 != NULL );

                        /* davep 01-Nov-2011 ; make sure we don't wind up with
                         * /two/ pending cached scanvars (bad bad thing
                         * happened)
                         */
                        XASSERT( cached_sv==NULL, cached_sv->id );

                        /* We get a scanvar in this message but the scanner is
                         * still busy (we have to wait for the SMSG_SCAN_READY).
                         * So save the scanvar for when we get the
                         * SMSG_SCAN_READY.
                         */
                        PTR_ASSIGN( cached_sv, sys_msg.param3 );
                        XASSERT( cached_sv->cookie==SCANVAR_COOKIE, cached_sv->cookie );

                        /* davep 01-Nov-2011 ;  claim the scantask data path for myself */
                        scanvar_set_scan_callback( cached_sv, scanman_msg_send );

                        STATE_TRANSITION( SMSTATE_WAITING_FOR_SCANTASK_2 );
                        break;

                    case SMSTATE_IDLE :
                        /* davep 21-Sep-2009 ; Bug 12224 ; race condition
                         * between scanman and copyapp on a cancel. Scanman
                         * handles his cancel, ack's the cancel, then goes back
                         * to idle. Meanwhile, copyapp has already sent the
                         * jobstart.
                         *
                         * Simply ignore the jobstart. 
                         *
                         * Note the jobstart contains alloc'd memory (the
                         * scanvar).  But the downstream thread (e.g., copyapp)
                         * is required to free that on a cancel.
                         */
                        dbg1( "scanman ignore jobstart in state=%d\n", g_curr_state );
                        break;

                    case SMSTATE_WAITING_FOR_SCANTASK_2 :
                        /* davep 16-Oct-2009 ; (Bug 14574) kludge to ignore
                         * double-tap button presses causing extra PAGESTART
                         * messages.
                         *
                         * Ignore the message but verify the same scanvar is
                         * sent.
                         */
                        dbg1( "%s ignore extra jobstart while in state=%d\n", 
                                    __FUNCTION__, g_curr_state );
                        ASSERT( sys_msg.param3 != NULL );
                        PTR_ASSIGN( sv, sys_msg.param3 );
                        XASSERT( sv->cookie==SCANVAR_COOKIE, sv->cookie );
                        XASSERT( cached_sv==sv, (UINT32)sv );
                        break;

                    default :
                        /* wtf!? */
                        XASSERT( 0, g_curr_state );
                        break;
                }
                break;

            case MSG_SCAN_JOBEND :
                dbg1( "scanman received jobend in state=%d\n", g_curr_state );

                switch( g_curr_state ) {
                    case SMSTATE_WAITING :
                        STATE_TRANSITION( SMSTATE_WAITING_FOR_SCANTASK_1 );
                        break;

                    case SMSTATE_WAITING_FOR_APP :
                        /* we are now done done done; clean up ourselves with
                         * the SysJobMgr 
                         */
                        finish_job( &job_resources );
                        STATE_TRANSITION( SMSTATE_IDLE );
                        break;

                    case SMSTATE_IDLE :
                        /* davep 21-Sep-2009 ; Bug 12224 ; see commments in
                         * MSG_SCAN_JOBSTART for more detail
                         */
                        dbg1( "scanman ignore jobend in state=%d\n", g_curr_state );
                        break;

                    default :
                        XASSERT( 0, g_curr_state );
                        break;
                }
                break;

            case MSG_CANCELJOB: 
                dbg1( "scanman received canceljob in state=%d\n", g_curr_state );

                /* lotsa places we can receive a cancel so easier to specify
                 * where we shouldn't get a cancel
                 */
                if( g_curr_state==SMSTATE_IDLE ) {
                    dbg1( "%s unexpected message type=%d in state=%d\n", 
                                __FUNCTION__, sys_msg.msgType, g_curr_state );
                    SCANMAN_ASSERT(0);
                    break;
                }

                /* davep 17-Oct-2007 ; Bug 7498 ; disable possible calibration
                 * between pages of a single job ; turn it back on now that the
                 * job is done
                 */
                scan_set_config_no_calcache_timeout( FALSE );

                /* Only cancel scantask if scantask is actually running.
                 * Scantask should only be running when we told it to.
                 */
                switch( g_curr_state ) {
                    case SMSTATE_SCANNING :
                        STATE_TRANSITION( SMSTATE_CANCELING );

                        /* tell scantask to cancel then wait for it to complete */
                        scan_task_status_t task_status;
                        scerr = scantask_scan_cancel( running_scan_task_id,
                                                        &task_status );

                        /* the cancel will tell us the status of the task. If
                         * the task was pending (i.e., scantask had never even
                         * seen it yet), we don't have to wait for scantask.
                         */
                        if( task_status==SCAN_TASK_STATUS_RUNNING ) {
                            XASSERT( scerr==SCANERR_NONE, scerr );
                            blocking_wait_for_scantask_ready();
                        }
                        else {
                            /* jump straight to idle since we don't have to wait
                             * for scantask (we canceled before we even started
                             * or scantask has already completed the task we
                             * tried to cancel)
                             */
                            STATE_TRANSITION( SMSTATE_IDLE );
                        }

                        /* If we get here, we should have passed from the
                         * 'canceling' state to 'waiting' state then to 'idle'
                         * state.  The blocking_cancel_scantask() function blocks
                         * waiting for the SMSG_SCAN_READY.
                         */
                        XASSERT( g_curr_state==SMSTATE_IDLE, g_curr_state );
                        break;

                    case SMSTATE_FAILING :
                    case SMSTATE_WAITING :
                    case SMSTATE_WAITING_FOR_SCANTASK_1 :
                    case SMSTATE_WAITING_FOR_SCANTASK_2 :
                        /* everything is almost clean but we have to wait for
                         * scantask to go back to his ready state
                         */
                        STATE_TRANSITION( SMSTATE_WAITING_FOR_SCANTASK_1 );

                        /* Blocking wait for scantask's SMSG_SCAN_READY. We don't
                         * send the cancel to scantask because scantask is already
                         * in his blackout period. 
                         */
                        blocking_wait_for_scantask_ready();
                        XASSERT( g_curr_state==SMSTATE_IDLE, g_curr_state );
                        break;

                    default :
                        /* keep going, clean up scanman */
                        break;
                }

                /** \bug  WTF is param3? See Cory V. Atkin for all the exciting
                 * details!
                 */
                cancel_scanman( sys_msg.param3, job_resources );

                /* we're now released */
                job_resources = NULL;

                if( g_curr_state != SMSTATE_IDLE ) {
                    STATE_TRANSITION( SMSTATE_IDLE );
                }
                break;


            case MSG_SCAN_DEV_REQUEST :
                dbg2( "scanman received dev request in state=%d\n", g_curr_state );

                smirb = sys_msg.param3;

                /* verify the smirb */
                scanman_smirb_sanity( smirb );

//                /* test/debug--refuse it */
//                scanman_observer_notify_event( SCANMAN_EVENT_DEV_REQUEST_REFUSE, request_id );
//                break;

                scerr = scantask_ioctl_request( smirb, &tmp_task_id );
                if( scerr != SCANERR_NONE ) {
                    /* scantask doesn't want it so we must refuse it */
                    dbg2( "%s scantask refused request id=%d scerr=%d\n", __FUNCTION__, smirb->id, scerr );
                    scanman_observer_notify_event( SCANMAN_EVENT_DEV_REQUEST_REFUSE, smirb->id );

                    scanman_smirb_free( &smirb );
                }
                else { 
                    dbg2( "%s queued ioctl request id=%d scantaskid=%d\n", __FUNCTION__, 
                                smirb->id, tmp_task_id );
                }
                break;

            /*
             *
             *  The following are all messages from Scantask to me.  Need to
             *  rewrite the message then resend it to The Cloud Beyond.
             * 
             */
            case (SMSG_SCAN_SIZE | MESSAGE_FROM_SCANTASK_FLAG) :
                dbg1( "scanman received scansize in state=%d\n", g_curr_state );

                ASSERT( job_resources != NULL );
                XASSERT( g_curr_state==SMSTATE_SCANNING, (UINT32)g_curr_state );

                /* rewrite and pass on the msg to my downstream */
                sys_msg.msgType = MSG_SCAN_SIZE;
                err = SYMsgSend( job_resources->Destination, &sys_msg);
                XASSERT( err==OK, err );

                /* davep 16-Apr-2013 ; testing dual sided scan using single
                 * sided ADF; need to duplicate every buffer coming through so
                 * must know the scan size
                 */
                msg_scan_size_decode( &sys_msg, &pixels_per_row_padded,
                        &pixels_per_row, &total_rows, &bits_per_pixel );
                bytes_per_row = (bits_per_pixel/8) * pixels_per_row;
                break;

            case (SMSG_PAGE_START | MESSAGE_FROM_SCANTASK_FLAG) :
                dbg2( "scanman received pagestart in state=%d\n", g_curr_state );

                /* davep 07-Jun-2009 ; adding SMSG_PAGE_START for better ADF
                 * support.
                 */
                XASSERT( g_curr_state==SMSTATE_SCANNING, (UINT32)g_curr_state );
                ASSERT( job_resources != NULL );

                sys_msg.msgType = MSG_SCAN_PAGESTART;
                err = SYMsgSend( job_resources->Destination, &sys_msg);
                XASSERT( err==OK, err );

#ifdef FAKE_DUAL_SCAN
                /* davep 16-Apr-2013 ; test/debug ; duplicate every
                 * start/end/pagedata message for a side2 experiment
                 */
                sys_msg.param1 |= SMSG_PAGE_START_SET_PAGE_SIDE(1);
                err = SYMsgSend( job_resources->Destination, &sys_msg);
                XASSERT( err==OK, err );
#endif
                break;

            case (SMSG_PAGE_DATA | MESSAGE_FROM_SCANTASK_FLAG) :
                /* davep 07-Apr-2006 ; bug 928; we'll still assert here
                 * because we don't want to leak memory. Also, we shouldn't get
                 * any abberant data messages from scantask because we're
                 * responsible for stopping scantask
                 */
                XASSERT( g_curr_state==SMSTATE_SCANNING, (UINT32)g_curr_state );
                ASSERT( job_resources != NULL );

                sys_msg.msgType = MSG_SCAN_PLANEDATA;
                err = SYMsgSend( job_resources->Destination, &sys_msg);
                XASSERT( err==OK, err );

#ifdef FAKE_DUAL_SCAN
                /* davep 16-Apr-2013 ; test/debug; duplicate every
                 * start/end/pagedata message for a side2 experiment
                 */
                scerr = make_dup_data_msg(&sys_msg,bytes_per_row);
                if( scerr==SCANERR_NONE ) {
                    err = SYMsgSend( job_resources->Destination, &sys_msg);
                    XASSERT( err==OK, err );
                }
#endif

                /* davep 18-May-2012 ; XXX temp debug (commenting out but
                 * leaving in code because quite useful)
                 */
#if 0
                if( 1 ) { 
                    uint16_t num_rows;
                    scan_data_type dtype;
                    uint8_t *data_ptr;
                    bool last_buffer;
                    msg_scan_planedata_decode( &sys_msg, &num_rows, &dtype, &data_ptr, &last_buffer );

                    dbg2( "%s nr=%d dtype=%d data_ptr=%p last=%d\n", __FUNCTION__, num_rows, dtype, data_ptr, last_buffer );
                } 
#endif
                /* don't change state */
                break;
            
            case (SMSG_PAGE_END | MESSAGE_FROM_SCANTASK_FLAG):
                dbg2( "scanman received pageend in state=%d\n", g_curr_state );
                XASSERT( g_curr_state==SMSTATE_SCANNING, (UINT32)g_curr_state );
                ASSERT( job_resources != NULL );

                sys_msg.msgType = MSG_SCAN_PAGEEND;
                err = SYMsgSend(job_resources->Destination, &sys_msg);
                XASSERT( err==OK, err );

#ifdef FAKE_DUAL_SCAN
                /* davep 16-Apr-2013 ; test/debug ; duplicate every
                 * start/end/pagedata message for a side2 experiment
                 */
                sys_msg.param1 |= SMSG_PAGE_START_SET_PAGE_SIDE(1);
                err = SYMsgSend( job_resources->Destination, &sys_msg);
                XASSERT( err==OK, err );
#endif
                break;

            case (SMSG_SCAN_END | MESSAGE_FROM_SCANTASK_FLAG):
                /* davep 07-Jun-2009 ; better ADF support ; use SCAN_END instead
                 * of PAGE_END to indicate end of scantask
                 */
                dbg1( "scanman received scanend in state=%d\n", g_curr_state );

                /* davep 07-Apr-2006 ; bug 928; gently ignore unwanted messages */
                if( g_curr_state!=SMSTATE_SCANNING ) {
                    dbg1( "%s unexpected message type=%d in state=%d\n", 
                                __FUNCTION__, sys_msg.msgType, g_curr_state );
                    SCANMAN_ASSERT(0);
                    break;
                }

                /* davep 11-Nov-2008 ; Bug 11161 ; now must have a pointer to
                 * scanvar in our response
                 */
                ASSERT( sys_msg.param3 != NULL );
                sv = sys_msg.param3;
                XASSERT( sv->cookie==SCANVAR_COOKIE, sv->cookie );

                ASSERT( job_resources != NULL );

                /* rewrite and pass on the pageend msg to my downstream */
                sys_msg.msgType = MSG_SCAN_JOBEND;
                err = SYMsgSend( job_resources->Destination, &sys_msg);
                XASSERT( err==OK, err );

                /* davep 17-Oct-2007 ; Bug 7498 ; disable calibration between
                 * pages of a single job; after the first page finishes (so the
                 * first page can trigger a cal if necessary), stop cal. When
                 * the job finshes, restart ability to cal if the cache entry
                 * has timed out.
                 */
                scan_set_config_no_calcache_timeout( TRUE );

                /* davep 07-Jun-2009 ; now we enter the blackout state -- don't
                 * send anything to Scantask!  TODO add a bunch of asserts to
                 * make sure ScanMan doesn't talk to Scantask when in the
                 * blackout
                 */
                STATE_TRANSITION( SMSTATE_WAITING );
                break;

            case (SMSG_SCAN_READY | MESSAGE_FROM_SCANTASK_FLAG):
                dbg1( "scanman received scanready in state=%d\n", g_curr_state );

                switch( g_curr_state ) {
                    case SMSTATE_FAILING :
                        /* During a failed scan, we have to sync the SCAN_READY
                         * and the SystemJobManager sending us the CANCEL. They 
                         * can arrive in either order so have to handle both.
                         */
                        STATE_TRANSITION( SMSTATE_WAITING_FOR_SJM );
                        break;

                    case SMSTATE_WAITING :
                        /* Scantask tells us he's ready. Now we wait for the App
                         * to tell us want he wants us to do.  We could get a
                         * MSG_SCAN_JOBSTART and we'll do another scan. Or we
                         * could get a MSG_SCAN_JOBEND and we're done.
                         */
                        STATE_TRANSITION( SMSTATE_WAITING_FOR_APP );
                        break;

                    case SMSTATE_WAITING_FOR_SCANTASK_1 :
                        /* We're done. Clean up ourselves with the SysJobMgr */
                        finish_job( &job_resources );
                        STATE_TRANSITION( SMSTATE_IDLE );
                        break;

                    case SMSTATE_WAITING_FOR_SCANTASK_2 :
                        /* we should have a cached scanvar from a previously
                         * received MSG_SCAN_JOBSTART 
                         */
                        ASSERT( cached_sv != NULL );

                        PTR_ASSIGN( sv, cached_sv );

                        /* make sure the callback is still us (better safe than
                         * sorry)
                         */
                        XASSERT( sv->scan_callback_fn==scanman_msg_send,
                                (UINT32)sv->scan_callback_fn );

                        /* tell Scantask to get off his butt and send us some data */
                        scerr = scantask_scan_start( sv, &running_scan_task_id );
                        XASSERT( scerr==0, scerr );

                        STATE_TRANSITION( SMSTATE_SCANNING );
                        break;

                    default :
                        /* wtf!? */
                        XASSERT( 0, g_curr_state );
                        break;
                }
                break;

            case (SMSG_SCAN_FAILED | MESSAGE_FROM_SCANTASK_FLAG):
                dbg1( "scanman received scanfailed in state=%d\n", g_curr_state );

                switch( g_curr_state ) {
                    case SMSTATE_SCANNING :
                        scanman_cancel_job();
                        /* wait for SMSG_SCAN_READY and MSG_CANCEL */
                        STATE_TRANSITION( SMSTATE_FAILING );
                        break;

                    case SMSTATE_CANCELING :
                        /* wait for SMSG_SCAN_READY */
                        STATE_TRANSITION( SMSTATE_WAITING_FOR_SCANTASK_1 );
                        break;

                    /* davep 30-Apr-2009 ; handle fail message while waiting for
                     * scantask. Is happening if the PP fails after a
                     * successful scan. 
                     */
                    case SMSTATE_WAITING_FOR_SCANTASK_1 :
                        /* stay calm, don't worry. wait for the ready */
                        break;

                    default :
                        /* wtf!? */
                        XASSERT( 0, g_curr_state );
                        break;
                }
                break;

            case (SMSG_DATA_BLOB | MESSAGE_FROM_SCANTASK_FLAG):
                dbg1( "scanman received data blob in state=%d\n", g_curr_state );
                /* rewrite to an agMessage/scanmsg.h message and resend */
                sys_msg.msgType = MSG_SCAN_DATA_BLOB;

                err = SYMsgSend( job_resources->Destination, &sys_msg);
                XASSERT( err==OK, err );

                /* don't change state */
                break;

            case (SMSG_SCAN_ADF_PAPER_EVENT | MESSAGE_FROM_SCANTASK_FLAG) :
                handle_paper_event_msg( sys_msg.param1, sys_msg.param2 );
                break;

            case (SMSG_SCAN_MECH_IOCTL | MESSAGE_FROM_SCANTASK_FLAG) :
                dbg2( "scanman received mech ioctl response in state=%d\n", g_curr_state );

                smirb = sys_msg.param3;

                /* is this one of my smirbs? */
                scanman_smirb_sanity( smirb );

                if( smirb->scerr==SCANERR_NONE ) {
                    scanman_observer_notify_event( SCANMAN_EVENT_DEV_REQUEST_SUCCESS, smirb->id );
                }
                else {
                    scanman_observer_notify_event( SCANMAN_EVENT_DEV_REQUEST_FAIL, smirb->id );
                }
                scanman_smirb_free( &smirb );
                break;

            case (SMSG_CAL_CALIBRATION_IN_PROGRESS | MESSAGE_FROM_SCANTASK_FLAG) :
                scanman_observer_notify_event( SCANMAN_EVENT_SCAN_CALIBRATION_RUNNING, 1 );
                break;

            case (SMSG_CAL_CALIBRATION_DONE | MESSAGE_FROM_SCANTASK_FLAG) :
                scanman_observer_notify_event( SCANMAN_EVENT_SCAN_CALIBRATION_COMPLETE, 1 );
                /* send another 'scanning' status to make the status transition
                 * a little more intuitive
                 * scan running -> calibration running -> cal complete -> scan running 
                 */
                scanman_observer_notify_event( SCANMAN_EVENT_SCAN_RUNNING, 1 );
                break;

            default:
                dbg1("scanman unknown message type=%#x param1=%#lx\n", 
                            sys_msg.msgType, sys_msg.param1);

                /* davep 24-Jul-2008 ; add the assert back while I move
                 * Scantask from MESSAGE to scan_msg_t
                 */
                XASSERT( 0, sys_msg.msgType );
                break;
        }

    } /* end while(1) message loop */

    /* should never get here! */
    ASSERT(0);
    return 0;
}

void ScanManInit(void)
{
    UINT pxretcode;
    scan_err_t scerr;

    /* davep 23-Jul-2008 ; I'm moving internal use of MESSAGE to scan_msg_t so
     * we don't have to ship agMessage.h to the BSDK customers.
     *
     * But I'm still using the mq_send(), etc, functions which are hardwired to
     * send/recieve only MESSAGE sized chunks.
     */
    XASSERT( sizeof(scan_msg_t)==sizeof(MESSAGE), sizeof(scan_msg_t) );

    /* create the message queue we'll use to interact with the rest of the
     * system 
     */
    posix_create_message_queue( &scanman_msgq, "/scanman", SCANMAN_QUEUE_SIZE,
                                sizeof(MESSAGE));   

    router_register_queue(SCANMANID, scanman_msgq);  // Register a message queue.

    scerr = scanman_obs_onetime_init();
    XASSERT( scerr==SCANERR_NONE, scerr );

    scerr = scanman_smirb_onetime_init();
    XASSERT( scerr==SCANERR_NONE, scerr );

    pxretcode = posix_create_thread( &scanman_task_id, &scanman_task, 0, "scanman",
                                    scanman_stack, SCANMAN_STACK_SIZE, POSIX_THR_PRI_NORMAL );
    XASSERT(pxretcode==0, pxretcode);

    register_scanman_with_system();

    /* unless we're building with very specific debug code, all of demo_scanapp
     * is #def'd to a no-op
     */
    demo_scanapp_init();
}
 
