/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2011-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



/*
 * Scan block LED control.
 *
 * davep 03-Jul-2012 ; adding 6170 multiple sensor support.
 */

#include <stdint.h>
#include <stdbool.h>

#include "lassert.h"
#include "regAddrs.h"
//#include "SCAN_TOP_regstructs.h"
//#include "SCAN_TOP_regmasks.h"

#include "scantypes.h"
#include "scancore.h"
#include "scanblk_if.h"
#include "scanif.h"
#include "scandbg.h"
#include "scansen.h"

// scif_led_duty - compute time on in pixel for each LED
// based on the requested duty cycle for each LED, compute the amount of time
// that the LEDs should be on during the lpwmperiod.  Turn on at 0, off at the calculated time.
// Note that we don't set the LEDPOL - 0 is the reset value we write in scif_led_reset, and
// 0 is pwm normally low - so we are already set to the proper value.
void scif_led_duty(uint32_t sensor_num, unsigned int lpwmperiod, uint32_t dutyR, uint32_t dutyG, uint32_t dutyB)
{
    struct scanblk_LPWMCONFIG0_cfg lpwmconfig0;
    struct scanblk_LPWMCONFIG1_cfg lpwmconfig1;
    struct scanblk_LPWM00_cfg      lpwm00;
    struct scanblk_LPWM10_cfg      lpwm10;
    struct scanblk_LPWM20_cfg      lpwm20;
    struct scanblk_LPWM01_cfg      lpwm01;
    struct scanblk_LPWM11_cfg      lpwm11;
    struct scanblk_LPWM21_cfg      lpwm21;



    uint32_t lpwmmin, lpwmmax, ledRoff, ledGoff, ledBoff;

    dbg2( "%s s#=%d dutyR=%d dutyG=%d dutyB=%d (these are duty cycle percentages!) pwmperiod=%d\n", __FUNCTION__, sensor_num, dutyR, dutyG, dutyB,lpwmperiod );


    lpwmmax = lpwmperiod - 1;

    // lowest value 0 - in case caller wants 100%, starting at 0 allows that
    lpwmmin = 0;

    // The duty cycle is the amount of time an LED is on during a pixel time.  If the pixel period
    // is 100, and the duty cycle is 20%, you can turn on the LED at time 79, and off at time 99.
    // On at 1, off at 21 would also be equally mathematically valid
    ledRoff = (lpwmperiod * dutyR) / 100 + lpwmmin;
    if (ledRoff > lpwmmax)
        ledRoff = lpwmmax;
    ledGoff = (lpwmperiod * dutyG) / 100 + lpwmmin;
    if (ledGoff > lpwmmax)
        ledGoff = lpwmmax;
    ledBoff = (lpwmperiod * dutyB) / 100 + lpwmmin;
    if (ledBoff > lpwmmax)
        ledBoff = lpwmmax;

    // special case - 100% means don't allow the LEDs to turn off at all for noise reduction.  so
    // for this case, we set the led*off to be *higher* than the lpwmperiod, and the ASIC will never
    // turn off the LED

    if (dutyR == 100)
        ledRoff = lpwmperiod + 1;
    if (dutyG == 100)
        ledGoff = lpwmperiod + 1;
    if (dutyB == 100)
        ledBoff = lpwmperiod + 1;

    if (sensor_num == 0)
    {
        /* Clear structures */
        memset(&lpwmconfig0, 0x0, sizeof(lpwmconfig0));
        memset(&lpwm00,      0x0, sizeof(lpwm00));
        memset(&lpwm10,      0x0, sizeof(lpwm10));
        memset(&lpwm20,      0x0, sizeof(lpwm20));

        /* Set bits */
        lpwmconfig0.lpwmper = lpwmperiod;
        lpwm00.led0on       = lpwmmin;
        lpwm00.led0off      = ledRoff;
        lpwm10.led1on       = lpwmmin;
        lpwm10.led1off      = ledGoff;
        lpwm20.led2on       = lpwmmin;
        lpwm20.led2off      = ledBoff;

        /* Set valid bits */
        lpwmconfig0.lpwmper_valid = true;
        lpwm00.led0on_valid       = true;
        lpwm00.led0off_valid      = true;
        lpwm10.led1on_valid       = true;
        lpwm10.led1off_valid      = true;
        lpwm20.led2on_valid       = true;
        lpwm20.led2off_valid      = true;

        /* Write registers */
        scanblk_set_LPWMCONFIG0(&lpwmconfig0);
        scanblk_set_LPWM00(&lpwm00);
        scanblk_set_LPWM10(&lpwm10);
        scanblk_set_LPWM20(&lpwm20);

    }
    else
    {
        /* Clear structures */
        memset(&lpwmconfig1, 0x0, sizeof(lpwmconfig1));
        memset(&lpwm01,      0x0, sizeof(lpwm01));
        memset(&lpwm11,      0x0, sizeof(lpwm11));
        memset(&lpwm21,      0x0, sizeof(lpwm21));

        /* Set bits */
        lpwmconfig1.lpwmper = lpwmperiod;
        lpwm01.led0on       = lpwmmin;
        lpwm01.led0off      = ledRoff;
        lpwm11.led1on       = lpwmmin;
        lpwm11.led1off      = ledGoff;
        lpwm21.led2on       = lpwmmin;
        lpwm21.led2off      = ledBoff;

        /* Set valid bits */
        lpwmconfig1.lpwmper_valid = true;
        lpwm01.led0on_valid       = true;
        lpwm01.led0off_valid      = true;
        lpwm11.led1on_valid       = true;
        lpwm11.led1off_valid      = true;
        lpwm21.led2on_valid       = true;
        lpwm21.led2off_valid      = true;

        /* Write registers */
        scanblk_set_LPWMCONFIG1(&lpwmconfig1);
        scanblk_set_LPWM01(&lpwm01);
        scanblk_set_LPWM11(&lpwm11);
        scanblk_set_LPWM21(&lpwm21);
    }
}

void scif_ledR_gate(uint32_t sensor_num, unsigned int pos, unsigned int neg)
{
    struct scanblk_LED00_cfg led00;
    struct scanblk_LED01_cfg led01;

    if (sensor_num == 0)
    {
        memset(&led00, 0x0, sizeof(led00));
        led00.led0pos       = pos;
        led00.led0neg       = neg;
        led00.led0pos_valid = true;
        led00.led0neg_valid = true;
        scanblk_set_LED00(&led00);
    }
    else
    {
        memset(&led01, 0x0, sizeof(led01));
        led01.led0pos       = pos;
        led01.led0neg       = neg;
        led01.led0pos_valid = true;
        led01.led0neg_valid = true;
        scanblk_set_LED01(&led01);
    }
}
EXPORT_SYMBOL(scif_ledR_gate);

void scif_ledG_gate(uint32_t sensor_num, unsigned int pos, unsigned int neg)
{
    struct scanblk_LED10_cfg led10;
    struct scanblk_LED11_cfg led11;

    if (sensor_num == 0)
    {
        memset(&led10, 0x0, sizeof(led10));
        led10.led1pos       = pos;
        led10.led1neg       = neg;
        led10.led1pos_valid = true;
        led10.led1neg_valid = true;
        scanblk_set_LED10(&led10);
    }
    else
    {
        memset(&led11, 0x0, sizeof(led11));
        led11.led1pos       = pos;
        led11.led1neg       = neg;
        led11.led1pos_valid = true;
        led11.led1neg_valid = true;
        scanblk_set_LED11(&led11);
    }
}
EXPORT_SYMBOL(scif_ledG_gate);

void scif_ledB_gate(uint32_t sensor_num, unsigned int pos, unsigned int neg)
{
    struct scanblk_LED20_cfg led20;
    struct scanblk_LED21_cfg led21;

    if (sensor_num == 0)
    {
        memset(&led20, 0x0, sizeof(led20));
        led20.led2pos       = pos;
        led20.led2neg       = neg;
        led20.led2pos_valid = true;
        led20.led2neg_valid = true;
        scanblk_set_LED20(&led20);
    }
    else
    {
        memset(&led21, 0x0, sizeof(led21));
        led21.led2pos       = pos;
        led21.led2neg       = neg;
        led21.led2pos_valid = true;
        led21.led2neg_valid = true;
        scanblk_set_LED21(&led21);
    }
}
EXPORT_SYMBOL(scif_ledB_gate);

void scif_led_enable(uint32_t sensor_num, int onR, int onG, int onB)
{
    uint32_t num32;

    struct scanblk_SCFG2_cfg scfg2;
    memset(&scfg2, 0x0, sizeof(scfg2));
//    dbg2( "%s onR=%d onG=%d onB=%d, s#=%d\n", __FUNCTION__, onR, onG, onB,sensor_num );

    //num32 = (onR<<2) | (onG<<1) | onB;
    num32 = (onB<<2) | (onG<<1) | onR;
    if (sensor_num == 0)
    {
        scfg2.leden0 = num32;
        scfg2.leden0_valid = true;
    }
    else
    {
        scfg2.leden1 = num32;
        scfg2.leden1_valid = true;
    }

    scanblk_set_SCFG2(&scfg2);

    /* XXX temp debug */
//    dbg2( "%s %d\n", __FUNCTION__, __LINE__ );
//    scif_dump();
//    afe_dump();
}

void scif_led_reset( void )
{
    struct scanblk_LPWMCONFIG0_cfg lpwmconfig0;
    struct scanblk_LPWMCONFIG1_cfg lpwmconfig1;
    struct scanblk_LPWM00_cfg      lpwm00;
    struct scanblk_LPWM10_cfg      lpwm10;
    struct scanblk_LPWM20_cfg      lpwm20;
    struct scanblk_LPWM01_cfg      lpwm01;
    struct scanblk_LPWM11_cfg      lpwm11;
    struct scanblk_LPWM21_cfg      lpwm21;

    uint32_t sensor_num;


    /* looks like zero is a safe reset value for all these */
    memset(&lpwmconfig0, 0x0, sizeof(lpwmconfig0));
    memset(&lpwm00,      0x0, sizeof(lpwm00));
    memset(&lpwm10,      0x0, sizeof(lpwm10));
    memset(&lpwm20,      0x0, sizeof(lpwm20));
    memset(&lpwmconfig1, 0x0, sizeof(lpwmconfig1));
    memset(&lpwm01,      0x0, sizeof(lpwm01));
    memset(&lpwm11,      0x0, sizeof(lpwm11));
    memset(&lpwm21,      0x0, sizeof(lpwm21));

    /* Set valid bits */
    lpwmconfig0.lpwmper_valid = true;
    lpwmconfig0.ledpol_valid  = true;
    lpwm00.led0on_valid       = true;
    lpwm00.led0off_valid      = true;
    lpwm10.led1on_valid       = true;
    lpwm10.led1off_valid      = true;
    lpwm20.led2on_valid       = true;
    lpwm20.led2off_valid      = true;
    lpwmconfig1.lpwmper_valid = true;
    lpwmconfig1.ledpol_valid  = true;
    lpwm01.led0on_valid       = true;
    lpwm01.led0off_valid      = true;
    lpwm11.led1on_valid       = true;
    lpwm11.led1off_valid      = true;
    lpwm21.led2on_valid       = true;
    lpwm21.led2off_valid      = true;

    /* Write registers */
    scanblk_set_LPWMCONFIG0(&lpwmconfig0);
    scanblk_set_LPWM00(&lpwm00);
    scanblk_set_LPWM10(&lpwm10);
    scanblk_set_LPWM20(&lpwm20);
    scanblk_set_LPWMCONFIG1(&lpwmconfig1);
    scanblk_set_LPWM01(&lpwm01);
    scanblk_set_LPWM11(&lpwm11);
    scanblk_set_LPWM21(&lpwm21);

    for (sensor_num=0; sensor_num < scansen_get_num_sensors(); sensor_num++)
    {
        scif_ledR_gate(sensor_num,0,0);
        scif_ledG_gate(sensor_num,0,0);
        scif_ledB_gate(sensor_num,0,0);
    }
}

void scif_led_dump( void )
{
    printk("STUBBED OUT : %s\n", __FUNCTION__);
#if 0
    dbg1( "%8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LED00", scif_regs->LED00,
            "LED10", scif_regs->LED10,
            "LED20", scif_regs->LED20
        );
    dbg1( "%8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LPWMCfg0", scif_regs->LPWMConfig0,
            "LPWM00", scif_regs->LPWM00,
            "LPWM10", scif_regs->LPWM10,
            "LPWM20", scif_regs->LPWM20
        );
    dbg1( "%8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LED01", scif_regs->LED01,
            "LED11", scif_regs->LED11,
            "LED21", scif_regs->LED21
        );
    dbg1( "%8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LPWMCfg1", scif_regs->LPWMConfig1,
            "LPWM01", scif_regs->LPWM01,
            "LPWM11", scif_regs->LPWM11,
            "LPWM21", scif_regs->LPWM21
        );
#endif
}

