/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2006-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



// Things to remember (some lessons painfully learned - e.g. by smoking a motor):
//  PIC left/right margins need to be the same for each RGB channel or set to bypass.
//  SCANX needs to be div 2 for staggered color CCD data; data goes to 6 PIC channels.
//  SCANX must be a multiple of 16 for PIC DMA, or margins need to trim to multiple.
//  Don't set DataType bit in PSEQ after first RGB triplet on staggered CCD data.
//  Set CCD VBulb to 11V (Aruba JP20) and leave on, don't cycle on off.
//  Red is the first color in the CCD sensor to see the page.
//  One flatbed stepper motor full step is 1/300 of inch.
//  8 microsteps = 1 full step on flatbed.
//  MnHOLD in L6219 Control LATCHES the last VRef Motor PWM (may not be low current).
//  MnSCALE in L6219 Control scales 0-93.75% in 6.25% increments.
//  NDATA in AFE Control should not account for dummy bits.
//  SENINIT in AFE Control is the initial state of SLOAD before it is asserted.
//  SENINIT in AFE Control matches signal pin polarity 1=high, 0=low.
//  SSTAT_QEMPTY might glitch high for 1 clock at end of a scan command.
//  SSTAT_QEMPTY will be stable when SSTAT_SCMDCOMP is set.
//  afe_auto_* must be called after any afe_read/write functions.
//  The LED Gate edges appear to be one pixel clock behind in numbering
//  The LED Gate neg edge must be <= pixclks-2 or the next LED cycle is missed

#include <stdint.h>

#include "scos.h"

#include "regAddrs.h"
#include "lassert.h"
#include "hwconfig_api.h"
#include "cpu_api.h"
#include "hwconfig_api.h"
#include "regAddrs.h"
//#include "SCAN_TOP_regstructs.h"
//#include "SCAN_TOP_regmasks.h"

#include "scantypes.h"
#include "scancore.h"
#include "scandbg.h"
#include "scanblk_if.h"
#include "scanif.h"
#include "scanreset.h"
#include "scan.h"
#include "scands.h"
#include "scanplat.h"
#include "safeint.h"

static struct scanblk_irqs enabled_interrupts = {
    .unexp_mtr_sync    = 1,
    .queue_overrun     = 1,
    .cmd_int_overrun   = 1,
    .scan_cmd_complete = 1,
};

/* * * * * * * * * * Sensor * * * * * * * * * */

unsigned int scif_sys_clocks_pixel(void)
{
    struct scanblk_SCFG2_cfg cfg;
    scanblk_get_SCFG2(&cfg);
    return cfg.pixper + 1;
}

/* * * * * * * * * * Control & Status * * * * * * * * * */

void scif_control(int run, int interrupt, bool in_irq_context)
{
    struct scanblk_SCTRL_cfg sctrl;
    
    memset(&sctrl, 0x0, sizeof(sctrl));

    sctrl.run_valid = true;
    sctrl.scmdie_valid = true;
    if (run)
    {
        sctrl.run = 1;
    }
    else
    {
        sctrl.run = 0; // Unnecessary (memset) but here for clarity
    }
    if (interrupt)
    {
        sctrl.scmdie = 1;
    }
    else
    {
        sctrl.scmdie = 0; // Unnecessary (memset) but here for clarity
    }

    // We need to know if we are in interrupt context here: if the low level driver
    // uses the wrong protection method (spinlock) ... very bad things will happen. 
    if (in_irq_context) {
        scanblk_set_SCTRL_isr(&sctrl);
    }
    else {
        scanblk_set_SCTRL(&sctrl);
    }
}

uint32_t scif_interrupt_disable( void )
{
    uint32_t was_enabled;
    struct scanblk_irqs interrupts;

    scanblk_get_enabled_ints(&interrupts);
    was_enabled = interrupts.unexp_mtr_sync
                | interrupts.queue_overrun | interrupts.cmd_int_overrun
                | interrupts.scan_cmd_complete;
    scanblk_disable_int(&interrupts);
    return was_enabled;
}

void scif_interrupt_enable( void )
{
    /* turn them all on */
    scanblk_enable_int(&enabled_interrupts);
}

void scif_clock(int enable, bool in_irq_context)
{
    struct scanblk_SCTRL_cfg sctrl;
    memset(&sctrl, 0x0, sizeof(sctrl));

    sctrl.scanen = enable;
    sctrl.scanen_valid = true;

    // We need to know if we are in interrupt context here: if the low level driver
    // uses the wrong protection method (spinlock) ... very bad things will happen. 
    if (in_irq_context) {
        scanblk_set_SCTRL_isr(&sctrl);
    }
    else {
        scanblk_set_SCTRL(&sctrl);
    }
}

// FIXME - we really need to be queueing the scanblk_SCMD_cfg structures, and not the uint32_t
// that would eliminate this #include
#include "SCAN_regmasks.h"
uint32_t scif_command( bool sdata, unsigned int sclkper, uint16_t scany)
{
    uint32_t reg=0;
    /* ballen 06/2014
     * I'm trying to move everything to the new driver architecture,
     * but this command is a little bit too entwined for the moment.
     * Since it doesn't actually write to hw, I'm leaving it with it's
     * own local definitions of these commands until I can finish the
     * port later.
     */
    /* davep 17-Mar-2008 ; XXX temp debug */
    ASSERT(sclkper>0);

    /* burtp 03-Aug-2009; Catch evil doers that would seek to roll us over */
    ASSERT(scany>0);

    // build up command from the system regmasks.
    // FIXME - we really need to be queueing the scanblk_SCMD_cfg structures, and not the uint32_t
    // reference all of the callers of this fcn, and the cmdq that stores the uint32_t
    reg = SCAN_SCMD_SDATA_REPLACE_VAL(reg, sdata);
    reg = SCAN_SCMD_SCLKPER_REPLACE_VAL(reg, sclkper-1);
    reg = SCAN_SCMD_SCANY_REPLACE_VAL(reg, scany-1);
    return(reg);
}

//uint32_t scif_status(void)
//{
//    return *SCIF_SSTAT;
//}

bool scif_command_executing(void)
{
    struct scanblk_SSTAT_cfg cfg;
    scanblk_get_SSTAT(&cfg);
    return (cfg.cmdstat) && true;
}

#if 0
void scif_interrupt_wait(struct scanblk_SSTAT_cfg * ints)
{
    struct scanblk_SSTAT_cfg cfg;
    scanblk_get_SSTAT(&cfg);

    if(ints->unexpsync)
    {
        /* wait for condition */
        while(cfg.unexpsync == 0) { scanblk_get_SSTAT(&cfg); }
    }
    if(ints->qovr)
    {
        /* wait for condition */
        while(cfg.qovr == 0) { scanblk_get_SSTAT(&cfg); }
    }
    if(ints->iovr)
    {
        /* wait for condition */
        while(cfg.iovr == 0) { scanblk_get_SSTAT(&cfg); }
    }
    if(ints->scmdack)
    {
        /* wait for condition */
        while(cfg.scmdack == 0) { scanblk_get_SSTAT(&cfg); }
    }
}
#endif

/**
 * \brief Setup the scan block CMODE (Color Mode) bit
 *
 * SCFG2.CMODE==0 => monochrome scan
 * SCFG2.CMODE==1 => color scan
 *
 * \author David Poole
 * \date 26-Feb-2008
 *
 */

void
scif_set_cmode( int cmode )
{
    struct scanblk_SCFG2_cfg cfg;
    memset(&cfg, 0x0, sizeof(cfg));

    cfg.cmode_valid = true;
    if( cmode==SCAN_CMODE_MONO ) {
        /* clear the cmode bit indicating mono mode */
        cfg.cmode = 0;
    }
    else {
        /* set the cmode bit indicating color mode */
        cfg.cmode = 1;
    }

    scanblk_set_SCFG2(&cfg);
}


/**
 * \brief Setup the scan block clamp config register (CCFG) register
 * and turn on the clamp clock enable in scan config 1 register
 *
 * Note, when you use the clamp clock to generate edges for scanner resolution,
 * set ccd_clamp_mode = 1 (line mode), and set sensor_clamp_edges[0] and [1] to
 * any edges you need for changing the clamp line.
 *
 * If youi don't want any edges for scanner resolution (you only want to have the
 * line asserted or deasserted), then set the ccd_camp_mode = 0 (pixel mode) which
 * seems to make the ASIC ignore the sensor_clamp_edges.  Then be sure to set
 * the clamp_mode_polarity = 0 for deassert, or = for assert.
 *
 * It looks odd, but we use CPCLK as the ccd clamp clock as per Doug Keithly
 *
 * \author Sandra Capri
 * \date 21-Apr-2008
 *
 */

void scif_set_clamp_clock(uint8_t ccd_clamp_mode, uint8_t afe_clamp_mode,
                          uint8_t clamp_mode_polarity, uint8_t *sensor_clamp_edges)
{
    struct scanblk_CCFG_cfg ccfg_cfg;
    struct scanblk_SCFG1_cfg scfg1_cfg;

    memset(&ccfg_cfg, 0x0, sizeof(ccfg_cfg));
    ccfg_cfg.cpmode_valid = true;
    ccfg_cfg.clmode_valid = true;
    ccfg_cfg.cle1_valid   = true;
    ccfg_cfg.ccle2_valid  = true;
    ccfg_cfg.acle2_valid  = true;
    ccfg_cfg.cpmode       = ccd_clamp_mode;
    ccfg_cfg.clmode       = afe_clamp_mode;
    ccfg_cfg.cle1         = sensor_clamp_edges[0];
    ccfg_cfg.ccle2        = sensor_clamp_edges[1];
    ccfg_cfg.acle2        = sensor_clamp_edges[2];
    scanblk_set_CCFG(&ccfg_cfg);

    memset(&scfg1_cfg, 0x0, sizeof(scfg1_cfg));
    scfg1_cfg.cpclken_valid = true;
    scfg1_cfg.scpol         = true;
    scfg1_cfg.cpclken       = 1;
    scfg1_cfg.scpol         = clamp_mode_polarity;
    scanblk_set_SCFG1(&scfg1_cfg);
}

#if 0
static struct scan_register {
    const char *name;
    volatile uint32_t *addr;
} regs[] = {
    { "SCFG1",    SCIF_SCFG1, },
    { "SCFG2",    SCIF_SCFG2, },
    { "SCFG3",    SCIF_SCFG3, },
    { "SCTRL",    SCIF_SCTRL, },
    { "SCMD",    SCIF_SCMD, },
    { "SSTAT",    SCIF_SSTAT, },
    { "SIACK",    SCIF_SIACK, },
    { "SCANX",    SCIF_SCANX, },
    { "SCYCLE",    SCIF_SCYCLE, },
    { "STCFG1",    SCIF_STCFG1, },
    { "STCFG2",    SCIF_STCFG2, },
    { "STCFG3",    SCIF_STCFG3, },
    { "STCFG4",    SCIF_STCFG4, },
    { "CCFG",    SCIF_CCFG, },
    { "SCLK1",    SCIF_SCLK1, },
    { "SCLK2",    SCIF_SCLK2, },
    { "SCLK3",    SCIF_SCLK3, },
    { "SCLK4",    SCIF_SCLK4, },
    { "SCLK5",    SCIF_SCLK5, },
    { "SCLK6",    SCIF_SCLK6, },
    { "ACLK1",    SCIF_ACLK1, },
    { "ACLK2",    SCIF_ACLK2, },
    { "ACLK3",    SCIF_ACLK3, },
    { "ACLK4",    SCIF_ACLK4, },
    { "ACLK5",    SCIF_ACLK5, },
#ifdef HAVE_SCIF_ACLK6
    { "ACLK6",    SCIF_ACLK6, },
#endif
    { "BPWM",    SCIF_BPWM, },
#ifdef HAVE_SCAN_BLOCK_2005
    { "LPWM",    SCIF_LPWM, },
#endif
#ifdef SCIF_CFGARB
    { "CFGARB",    SCIF_CFGARB, },
#endif
    { "LED0",    SCIF_LED0, },
    { "LED1",    SCIF_LED1, },
    { "LED2",    SCIF_LED2, },
    { "MCFG",    SCIF_MCFG, },
#ifdef HAVE_SCAN_BLOCK_2005
    { "L6219",    SCIF_6219MCTRL, },
//    { "A3967",    SCIF_3967MCTRL, },
//    { "T62209",    SCIF_62209MCTRL, },
    { "MPWMCFG",SCIF_MPWMCFG, },
    { "PCTRLA",    SCIF_PCTRLA, },
    { "PCTRLB",    SCIF_PCTRLB, },
    { "PCTRLC",    SCIF_PCTRLC, },
    { "PCTRLD",    SCIF_PCTRLD, },
#endif
    { "AFEPC",    SCIF_AFEPC, },
//    { "AFEPD",    SCIF_AFEPD, },

    /* davep 12-Oct-2010 ; if we have LPWM0, assume we have both */
#ifdef SCIF_LPWM0
    { "LPWM0",    SCIF_LPWM0, },
    { "LPWM1",    SCIF_LPWM1, },
#endif
};
#endif

void scif_dump(void)
{
#if 0
    int i;
    uint32_t scan_clock_mhz;

    scan_clock_mhz = hw_get_scan_clk_speed();
    dbg1( "%s scan_clock_mhz=%d\n", __FUNCTION__, scan_clock_mhz );

    for (i = 0; i < sizeof(regs)/sizeof(struct scan_register); i++) {
        dbg1( "%8s=0x%08x", regs[i].name, *regs[i].addr );
        if (i%4 == 3) {
            dbg1("\n");
        }
    }
    if (i%4 != 0) {
        dbg1("\n");
    }

    /* davep 16-Jul-2012 ; as of this writing, our current platforms have three
     * separate LDATA settings
     */
#ifdef SCAN_LDATA1_DS1_MASK
    dbg1( "%8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LDATA1", scif_regs->LDATA1,
            "LDATA2", scif_regs->LDATA2,
            "LDATA3", scif_regs->LDATA3,
            "LDATA4", scif_regs->LDATA4,
            "LDATA5", scif_regs->LDATA5,
            "LDATA6", scif_regs->LDATA6 );
#endif

#ifdef SCAN_LDATA10_DS1_MASK
    dbg1( "%8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LDATA10", scif_regs->LDATA10,
            "LDATA20", scif_regs->LDATA20,
            "LDATA30", scif_regs->LDATA30,
            "LDATA40", scif_regs->LDATA40,
            "LDATA50", scif_regs->LDATA50,
            "LDATA60", scif_regs->LDATA60 );
#endif

#ifdef SCAN_LDATA11_DS1_MASK
    dbg1( "%8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x %8s=0x%08x\n",
            "LDATA11", scif_regs->LDATA11,
            "LDATA21", scif_regs->LDATA21,
            "LDATA31", scif_regs->LDATA31,
            "LDATA41", scif_regs->LDATA41,
            "LDATA51", scif_regs->LDATA51,
            "LDATA61", scif_regs->LDATA61 );
#endif

    dbg1( "%8s=0x%08x %8s=0x%08x\n",
            "PSEQ1", scif_regs->PSEQ1,
            "PSEQ2", scif_regs->PSEQ2 );

    /* davep 16-Jul-2012 ; LEDs continue to diverge so pushed into separate
     * function
     */
    scif_led_dump();
#endif
    printk("STUBBED OUT: %s\n", __FUNCTION__);
}

void scif_reset(void)
{
    scanblk_soft_reset();
}

/**
 * \brief If scan block register exists in scands, use that value
 *
 * Overrides the named register's value. Allows us to change some of the scan
 * block registers while tuning clock edges.
 *
 * Originally part of merge_register_override() but moved to own function so
 * could add the LDATA registers
 *
 * \author David Poole
 * \date 24-Oct-2013
 */
#if 0
static void register_override( const char *name, volatile uint32_t *regaddr )
{
    scan_err_t scerr;
    uint32_t value;

    scerr = scands_get_integer_by_name( name, &value );
    if( scerr == SCANERR_NONE ) {
        dbg2( "%s using value=0x%08x for %s\n", __FUNCTION__, value, name );
        *regaddr = value;
    }
}
#endif
/**
 * \brief merge command line set register values
 *
 * Created this code to allow command line get/set of register values before we
 * start a scan.
 *
 * Feel free to shoot yourself in the foot.
 *
 * \author David Poole
 * \date 27-Sep-2010
 *
 */

static void merge_register_override( void )
{
    printk("STUBBED OUT: %s\n", __FUNCTION__);
#if 0
    int i;

    for( i=0 ; i < sizeof(regs)/sizeof(struct scan_register) ; i++) {
        register_override(regs[i].name, regs[i].addr );
    }

    /* davep 01-Nov-2013 ; adding LDATA override. LDATA registers not in the
     * scan_register list. LDATA registers are very different across ASICs.
     * Plus the new "scif_regs" style of code doesn't work in the compile-time
     * scan-register list.
     */
#ifdef SCAN_LDATA1_DS1_MASK
    register_override( "LDATA1", &scif_regs->LDATA1 );
    register_override( "LDATA2", &scif_regs->LDATA2 );
    register_override( "LDATA3", &scif_regs->LDATA3 );
    register_override( "LDATA4", &scif_regs->LDATA4 );
    register_override( "LDATA5", &scif_regs->LDATA5 );
    register_override( "LDATA6", &scif_regs->LDATA6 );
#endif

#ifdef SCAN_LDATA10_DS1_MASK
    register_override("LDATA10", &scif_regs->LDATA10 );
    register_override("LDATA20", &scif_regs->LDATA20 );
    register_override("LDATA30", &scif_regs->LDATA30 );
    register_override("LDATA40", &scif_regs->LDATA40 );
    register_override("LDATA50", &scif_regs->LDATA50 );
    register_override("LDATA60", &scif_regs->LDATA60 );
#endif

#ifdef SCAN_LDATA11_DS1_MASK
    register_override( "LDATA11", &scif_regs->LDATA11 );
    register_override( "LDATA21", &scif_regs->LDATA21 );
    register_override( "LDATA31", &scif_regs->LDATA31 );
    register_override( "LDATA41", &scif_regs->LDATA41 );
    register_override( "LDATA51", &scif_regs->LDATA51 );
    register_override( "LDATA61", &scif_regs->LDATA61 );
#endif
#endif
}

void scif_sensor_setup(SENSOR_CONF *sc)
{
    //*********************************************************************
    // NOTE: DO NOT change the LED gate times or PWM here. They've already been
    // set by calibration.
    //*********************************************************************
    struct scanblk_SCFG2_cfg   scfg2;
    struct scanblk_SCTRL_cfg   sctrl;
    struct scanblk_SCYCLE_cfg  scycle;
    struct scanblk_STCFG1_cfg  stcfg1;
    struct scanblk_STCFG2_cfg  stcfg2;
    struct scanblk_STCFG3_cfg  stcfg3;
    struct scanblk_STCFG4_cfg  stcfg4;
    struct scanblk_CCFG_cfg    ccfg;
    struct scanblk_SCLK1_cfg   sclk1;
    struct scanblk_SCLK2_cfg   sclk2;
    struct scanblk_SCLK3_cfg   sclk3;
    struct scanblk_SCLK4_cfg   sclk4;
    struct scanblk_SCLK5_cfg   sclk5;
    struct scanblk_SCLK6_cfg   sclk6;
    struct scanblk_ACLK1_cfg   aclk1;
    struct scanblk_ACLK2_cfg   aclk2;
    struct scanblk_ACLK3_cfg   aclk3;
    struct scanblk_ACLK4_cfg   aclk4;
    struct scanblk_ACLK5_cfg   aclk5;
    struct scanblk_ACLK6_cfg   aclk6;
    struct scanblk_LDATA10_cfg ldata10;
    struct scanblk_LDATA11_cfg ldata11;
    struct scanblk_LDATA20_cfg ldata20;
    struct scanblk_LDATA21_cfg ldata21;
    struct scanblk_LDATA30_cfg ldata30;
    struct scanblk_LDATA31_cfg ldata31;
    struct scanblk_LDATA40_cfg ldata40;
    struct scanblk_LDATA41_cfg ldata41;
    struct scanblk_LDATA50_cfg ldata50;
    struct scanblk_LDATA51_cfg ldata51;
    struct scanblk_LDATA60_cfg ldata60;
    struct scanblk_LDATA61_cfg ldata61;
    struct scanblk_PSEQ1_cfg   pseq1;
    struct scanblk_PSEQ2_cfg   pseq2;
    struct scanblk_SCFG1_cfg   scfg1;
    struct scanblk_SCFG3_cfg   scfg3;


    /* davep 22-Jul-2008 ; sanity checks on how we use the enum since we're
     * mapping it into the register setting
     */
    XASSERT( SCAN_CMODE_MONO==1, SCAN_CMODE_MONO );

    /* davep 17-Feb-2008 ; set SCIF_SCIFG2_CMODE to mono for now; must be
     * changed later as we're setting up the per-scan
     *
     * Set SSEL to zero since it depends on our DPI.  (Will set later.)
     *
     */
    memset(&scfg2, 0x0, sizeof(scfg2));
    scfg2.cmode          = SCAN_CMODE_MONO;
    scfg2.stype          = sc->STYPE;
    scfg2.ssel           = 0;
    scfg2.afewidth       = sc->AFEWIDTH;
    scfg2.pixper         = sc->PIXPER-1;
    scfg2.cmode_valid    = true;
    scfg2.stype_valid    = true;
    scfg2.ssel_valid     = true;
    scfg2.afewidth_valid = true;
    scfg2.pixper_valid   = true;
    scanblk_set_SCFG2(&scfg2);

    /* Note I'm enabling the scan command queue but leaving the other fields
     * disabled.
     *
     * See scif_control() to enable/disable SCTRL.RUN and SCTRL.SCMDIE fields.
     * See scif_run() to enable/disable SCTRL.SCANEN field.
     */
    memset(&sctrl, 0x0, sizeof(sctrl));
    sctrl.cmdqen       = 1;
    sctrl.cmdqen_valid = true;
    scanblk_set_SCTRL(&sctrl);

    /*
     * SCANX is set up later since it depends on our DPI
     */

    /* davep 26-Mar-2010 ; asic team replaced the MSTEPP field with the EXPPER
     * (Exposure Field Data)
     */
    memset(&scycle, 0x0, sizeof(scycle));
    scycle.scycle       = SCAN_SCYCLE_DEFAULT-1;
    scycle.scycle_valid = true;
    scanblk_set_SCYCLE(&scycle);

    /* ballen TODO -- where did expper / mstepp go? */
#if 0
#ifdef SCIF_SCYCLE_EXPPER
    *SCIF_SCYCLE =
        SCIF_SCYCLE_EXPPER(SCAN_SCYCLE_DEFAULT-1) |
        SCIF_SCYCLE_SCYCLE(SCAN_SCYCLE_DEFAULT-1);
#else
    *SCIF_SCYCLE =
        SCIF_SCYCLE_MSTEPP(SCAN_SCYCLE_DEFAULT-1) |
        SCIF_SCYCLE_SCYCLE(SCAN_SCYCLE_DEFAULT-1);
#endif
#endif

    /* STCFG1 */
    memset(&stcfg1, 0x0, sizeof(stcfg1));
    stcfg1.spgen       = sc->SPGEN == SPGEN_SYSTEM_CLOCK;
    stcfg1.ssme1       = sc->sensor_ssm_edges[0];
    stcfg1.ssme2       = sc->sensor_ssm_edges[1];
    stcfg1.spgen_valid = true;
    stcfg1.ssme1_valid = true;
    stcfg1.ssme2_valid = true;
    scanblk_set_STCFG1(&stcfg1);

    /* STCFG2 */
    memset(&stcfg2, 0x0, sizeof(stcfg2));
    stcfg2.spe1       = sc->sensor_sp_edges[0];
    stcfg2.spe2       = sc->sensor_sp_edges[1];
    stcfg2.spe1_valid = true;
    stcfg2.spe2_valid = true;
    /* ballen TODO: figure out spe1_1 and spe2_2 */
    scanblk_set_STCFG2(&stcfg2);

    /* STCFG3 */
    memset(&stcfg3, 0x0, sizeof(stcfg3));
    stcfg3.aph1e1       = sc->sensor_aph1clk_edges[0];
    stcfg3.aph1e2       = sc->sensor_aph1clk_edges[1];
    stcfg3.aph1e3       = sc->sensor_aph1clk_edges[2];
    stcfg3.aph1e4       = sc->sensor_aph1clk_edges[3];
    stcfg3.aph1e1_valid = true;
    stcfg3.aph1e2_valid = true;
    stcfg3.aph1e3_valid = true;
    stcfg3.aph1e4_valid = true;
    scanblk_set_STCFG3(&stcfg3);

    /* STCFG4 */
    memset(&stcfg4, 0x0, sizeof(stcfg4));
    stcfg4.aph1e5       = sc->sensor_aph1clk_edges[4];
    stcfg4.aph1e6       = sc->sensor_aph1clk_edges[5];
    stcfg4.aph1e7       = sc->sensor_aph1clk_edges[6];
    stcfg4.aph1e8       = sc->sensor_aph1clk_edges[7];
    stcfg4.aph1e5_valid = true;
    stcfg4.aph1e6_valid = true;
    stcfg4.aph1e7_valid = true;
    stcfg4.aph1e8_valid = true;
    scanblk_set_STCFG4(&stcfg4);

    /* CCFG */
    memset(&ccfg, 0x0, sizeof(ccfg));
    ccfg.cpmode       = sc->ccd_clamp_mode;
    ccfg.clmode       = sc->afe_clamp_mode;
    /* ballen TODO: clsel is new -- what to do, what to do */
    ccfg.cle1         = sc->sensor_clamp_edges[0];
    ccfg.ccle2        = sc->sensor_clamp_edges[1];
    ccfg.acle2        = sc->sensor_clamp_edges[2];
    ccfg.cpmode_valid = true;
    ccfg.clmode_valid = true;
    ccfg.cle1_valid   = true;
    ccfg.ccle2_valid  = true;
    ccfg.acle2_valid  = true;
    scanblk_set_CCFG(&ccfg);

    /* SCLK1 */
    memset(&sclk1, 0x0, sizeof(sclk1));
    sclk1.p1e1       = sc->sensor_p1clk_edges[0];
    sclk1.p1e2       = sc->sensor_p1clk_edges[1];
    sclk1.p1e3       = sc->sensor_p1clk_edges[2];
    sclk1.p1e4       = sc->sensor_p1clk_edges[3];
    sclk1.p1e1_valid = true;
    sclk1.p1e2_valid = true;
    sclk1.p1e3_valid = true;
    sclk1.p1e4_valid = true;
    scanblk_set_SCLK1(&sclk1);

    /* SCLK2 */
    memset(&sclk2, 0x0, sizeof(sclk2));
    sclk2.p1e5       = sc->sensor_p1clk_edges[4];
    sclk2.p1e6       = sc->sensor_p1clk_edges[5];
    sclk2.p1e7       = sc->sensor_p1clk_edges[6];
    sclk2.p1e8       = sc->sensor_p1clk_edges[7];
    sclk2.p1e5_valid = true;
    sclk2.p1e6_valid = true;
    sclk2.p1e7_valid = true;
    sclk2.p1e8_valid = true;
    scanblk_set_SCLK2(&sclk2);

    /* SCLK3 */
    memset(&sclk3, 0x0, sizeof(sclk3));
    sclk3.p2e1       = sc->sensor_p2clk_edges[0];
    sclk3.p2e2       = sc->sensor_p2clk_edges[1];
    sclk3.p2e3       = sc->sensor_p2clk_edges[2];
    sclk3.p2e4       = sc->sensor_p2clk_edges[3];
    sclk3.p2e1_valid = true;
    sclk3.p2e2_valid = true;
    sclk3.p2e3_valid = true;
    sclk3.p2e4_valid = true;
    scanblk_set_SCLK3(&sclk3);

    /* SCLK4 */
    memset(&sclk4, 0x0, sizeof(sclk4));
    sclk4.p2e5       = sc->sensor_p2clk_edges[4];
    sclk4.p2e6       = sc->sensor_p2clk_edges[5];
    sclk4.p2e7       = sc->sensor_p2clk_edges[6];
    sclk4.p2e8       = sc->sensor_p2clk_edges[7];
    sclk4.p2e5_valid = true;
    sclk4.p2e6_valid = true;
    sclk4.p2e7_valid = true;
    sclk4.p2e8_valid = true;
    scanblk_set_SCLK4(&sclk4);

    /* SCLK5 */
    memset(&sclk5, 0x0, sizeof(sclk5));
    sclk5.rse1       = sc->sensor_rsclk_edges[0];
    sclk5.rse2       = sc->sensor_rsclk_edges[1];
    sclk5.rse3       = sc->sensor_rsclk_edges[2];
    sclk5.rse4       = sc->sensor_rsclk_edges[3];
    sclk5.rse1_valid = true;
    sclk5.rse2_valid = true;
    sclk5.rse3_valid = true;
    sclk5.rse4_valid = true;
    scanblk_set_SCLK5(&sclk5);

    /* SCLK6 */
    memset(&sclk6, 0x0, sizeof(sclk6));
    sclk6.cpe1       = sc->sensor_cpclk_edges[0];
    sclk6.cpe2       = sc->sensor_cpclk_edges[1];
    sclk6.cpe3       = sc->sensor_cpclk_edges[2];
    sclk6.cpe4       = sc->sensor_cpclk_edges[3];
    sclk6.cpe1_valid = true;
    sclk6.cpe2_valid = true;
    sclk6.cpe3_valid = true;
    sclk6.cpe4_valid = true;
    scanblk_set_SCLK6(&sclk6);

    /* ACLK1 */
    memset(&aclk1, 0x0, sizeof(aclk1));
    aclk1.mce1       = sc->afe_mclk_edges[0];
    aclk1.mce2       = sc->afe_mclk_edges[1];
    aclk1.mce3       = sc->afe_mclk_edges[2];
    aclk1.mce4       = sc->afe_mclk_edges[3];
    aclk1.mce1_valid = true;
    aclk1.mce2_valid = true;
    aclk1.mce3_valid = true;
    aclk1.mce4_valid = true;
    scanblk_set_ACLK1(&aclk1);

    /* ACLK2 */
    memset(&aclk2, 0x0, sizeof(aclk2));
    aclk2.mce5       = sc->afe_mclk_edges[4];
    aclk2.mce6       = sc->afe_mclk_edges[5];
    aclk2.mce7       = sc->afe_mclk_edges[6];
    aclk2.mce8       = sc->afe_mclk_edges[7];
    aclk2.mce5_valid = true;
    aclk2.mce6_valid = true;
    aclk2.mce7_valid = true;
    aclk2.mce8_valid = true;
    scanblk_set_ACLK2(&aclk2);

    /* ACLK3 */
    memset(&aclk3, 0x0, sizeof(aclk3));
    aclk3.mce9        = sc->afe_mclk_edges[8];
    aclk3.mce10       = sc->afe_mclk_edges[9];
    aclk3.mce11       = sc->afe_mclk_edges[10];
    aclk3.mce12       = sc->afe_mclk_edges[11];
    aclk3.mce9_valid  = true;
    aclk3.mce10_valid = true;
    aclk3.mce11_valid = true;
    aclk3.mce12_valid = true;
    scanblk_set_ACLK3(&aclk3);

    /* Out of order because it uses afe_mclk_edges[12-15] (next in sequence) */
    /* ACLK6 */
    memset(&aclk6, 0x0, sizeof(aclk6));
    aclk6.mce13       = sc->afe_mclk_edges[12];
    aclk6.mce14       = sc->afe_mclk_edges[13];
    aclk6.mce15       = sc->afe_mclk_edges[14];
    aclk6.mce16       = sc->afe_mclk_edges[15];
    aclk6.mce13_valid = true;
    aclk6.mce14_valid = true;
    aclk6.mce15_valid = true;
    aclk6.mce16_valid = true;
    scanblk_set_ACLK6(&aclk6);

    /* ACLK4 */
    memset(&aclk4, 0x0, sizeof(aclk4));
    aclk4.vse1       = sc->afe_vsclk_edges[0];
    aclk4.vse2       = sc->afe_vsclk_edges[1];
    aclk4.vse3       = sc->afe_vsclk_edges[2];
    aclk4.vse4       = sc->afe_vsclk_edges[3];
    aclk4.vse1_valid = true;
    aclk4.vse2_valid = true;
    aclk4.vse3_valid = true;
    aclk4.vse4_valid = true;
    scanblk_set_ACLK4(&aclk4);

    /* ACLK5 */
    memset(&aclk5, 0x0, sizeof(aclk5));
    aclk5.cd1e1       = sc->afe_cd1clk_edges[0];
    aclk5.cd1e2       = sc->afe_cd1clk_edges[1];
    aclk5.cd1e3       = sc->afe_cd1clk_edges[2];
    aclk5.cd1e4       = sc->afe_cd1clk_edges[3];
    aclk5.cd1e1_valid = true;
    aclk5.cd1e2_valid = true;
    aclk5.cd1e3_valid = true;
    aclk5.cd1e4_valid = true;
    scanblk_set_ACLK5(&aclk5);

    /***************************/
    /***** LDATA registers *****/
    /***************************/

    /* LDATA10 */
    memset(&ldata10, 0x0, sizeof(ldata10));
    ldata10.ds1       = sc->afe_ldata[0].clock;
    ldata10.dc1       = sc->afe_ldata[0].code;
    ldata10.ds2       = sc->afe_ldata[1].clock;
    ldata10.dc2       = sc->afe_ldata[1].code;
    ldata10.ds1_valid = true;
    ldata10.dc1_valid = true;
    ldata10.ds2_valid = true;
    ldata10.dc2_valid = true;
    scanblk_set_LDATA10(&ldata10);

    /* LDATA11 */
    memset(&ldata11, 0x0, sizeof(ldata11));
    ldata11.ds1       = sc->afe_ldata_a2[0].clock;
    ldata11.dc1       = sc->afe_ldata_a2[0].code;
    ldata11.ds2       = sc->afe_ldata_a2[1].clock;
    ldata11.dc2       = sc->afe_ldata_a2[1].code;
    ldata11.ds1_valid = true;
    ldata11.dc1_valid = true;
    ldata11.ds2_valid = true;
    ldata11.dc2_valid = true;
    scanblk_set_LDATA11(&ldata11);

    /* LDATA20 */
    memset(&ldata20, 0x0, sizeof(ldata20));
    ldata20.ds3       = sc->afe_ldata[2].clock;
    ldata20.dc3       = sc->afe_ldata[2].code;
    ldata20.ds4       = sc->afe_ldata[3].clock;
    ldata20.dc4       = sc->afe_ldata[3].code;
    ldata20.ds3_valid = true;
    ldata20.dc3_valid = true;
    ldata20.ds4_valid = true;
    ldata20.dc4_valid = true;
    scanblk_set_LDATA20(&ldata20);

    /* LDATA21 */
    memset(&ldata21, 0x0, sizeof(ldata21));
    ldata21.ds3       = sc->afe_ldata_a2[2].clock;
    ldata21.dc3       = sc->afe_ldata_a2[2].code;
    ldata21.ds4       = sc->afe_ldata_a2[3].clock;
    ldata21.dc4       = sc->afe_ldata_a2[3].code;
    ldata21.ds3_valid = true;
    ldata21.dc3_valid = true;
    ldata21.ds4_valid = true;
    ldata21.dc4_valid = true;
    scanblk_set_LDATA21(&ldata21);

    /* LDATA30 */
    memset(&ldata30, 0x0, sizeof(ldata30));
    ldata30.ds5       = sc->afe_ldata[4].clock;
    ldata30.dc5       = sc->afe_ldata[4].code;
    ldata30.ds6       = sc->afe_ldata[5].clock;
    ldata30.dc6       = sc->afe_ldata[5].code;
    ldata30.ds5_valid = true;
    ldata30.dc5_valid = true;
    ldata30.ds6_valid = true;
    ldata30.dc6_valid = true;
    scanblk_set_LDATA30(&ldata30);

    /* LDATA31 */
    memset(&ldata31, 0x0, sizeof(ldata31));
    ldata31.ds5       = sc->afe_ldata_a2[4].clock;
    ldata31.dc5       = sc->afe_ldata_a2[4].code;
    ldata31.ds6       = sc->afe_ldata_a2[5].clock;
    ldata31.dc6       = sc->afe_ldata_a2[5].code;
    ldata31.ds5_valid = true;
    ldata31.dc5_valid = true;
    ldata31.ds6_valid = true;
    ldata31.dc6_valid = true;
    scanblk_set_LDATA31(&ldata31);

    /* LDATA40 */
    memset(&ldata40, 0x0, sizeof(ldata40));
    ldata40.ds7       = sc->afe_ldata[6].clock;
    ldata40.dc7       = sc->afe_ldata[6].code;
    ldata40.ds8       = sc->afe_ldata[7].clock;
    ldata40.dc8       = sc->afe_ldata[7].code;
    ldata40.ds7_valid = true;
    ldata40.dc7_valid = true;
    ldata40.ds8_valid = true;
    ldata40.dc8_valid = true;
    scanblk_set_LDATA40(&ldata40);

    /* LDATA41 */
    memset(&ldata41, 0x0, sizeof(ldata41));
    ldata41.ds7       = sc->afe_ldata_a2[6].clock;
    ldata41.dc7       = sc->afe_ldata_a2[6].code;
    ldata41.ds8       = sc->afe_ldata_a2[7].clock;
    ldata41.dc8       = sc->afe_ldata_a2[7].code;
    ldata41.ds7_valid = true;
    ldata41.dc7_valid = true;
    ldata41.ds8_valid = true;
    ldata41.dc8_valid = true;
    scanblk_set_LDATA41(&ldata41);

    /* LDATA50 */
    memset(&ldata50, 0x0, sizeof(ldata50));
    ldata50.ds9        = sc->afe_ldata[8].clock;
    ldata50.dc9        = sc->afe_ldata[8].code;
    ldata50.ds10       = sc->afe_ldata[9].clock;
    ldata50.dc10       = sc->afe_ldata[9].code;
    ldata50.ds9_valid  = true;
    ldata50.dc9_valid  = true;
    ldata50.ds10_valid = true;
    ldata50.dc10_valid = true;
    scanblk_set_LDATA50(&ldata50);

    /* LDATA51 */
    memset(&ldata51, 0x0, sizeof(ldata51));
    ldata51.ds9        = sc->afe_ldata_a2[8].clock;
    ldata51.dc9        = sc->afe_ldata_a2[8].code;
    ldata51.ds10       = sc->afe_ldata_a2[9].clock;
    ldata51.dc10       = sc->afe_ldata_a2[9].code;
    ldata51.ds9_valid  = true;
    ldata51.dc9_valid  = true;
    ldata51.ds10_valid = true;
    ldata51.dc10_valid = true;
    scanblk_set_LDATA51(&ldata51);

    /* LDATA60 */
    memset(&ldata60, 0x0, sizeof(ldata60));
    ldata60.ds11       = sc->afe_ldata[10].clock;
    ldata60.dc11       = sc->afe_ldata[10].code;
    ldata60.ds12       = sc->afe_ldata[11].clock;
    ldata60.dc12       = sc->afe_ldata[11].code;
    ldata60.ds11_valid = true;
    ldata60.dc11_valid = true;
    ldata60.ds12_valid = true;
    ldata60.dc12_valid = true;
    scanblk_set_LDATA60(&ldata60);

    /* LDATA61 */
    memset(&ldata61, 0x0, sizeof(ldata61));
    ldata61.ds11       = sc->afe_ldata_a2[10].clock;
    ldata61.dc11       = sc->afe_ldata_a2[10].code;
    ldata61.ds12       = sc->afe_ldata_a2[11].clock;
    ldata61.dc12       = sc->afe_ldata_a2[11].code;
    ldata61.ds11_valid = true;
    ldata61.dc11_valid = true;
    ldata61.ds12_valid = true;
    ldata61.dc12_valid = true;
    scanblk_set_LDATA61(&ldata61);

    /* PSEQ1 */
    memset(&pseq1, 0x0, sizeof(pseq1));
    pseq1.pseq1       = sc->afe_pixel_seq[0];
    pseq1.pseq2       = sc->afe_pixel_seq[1];
    pseq1.pseq3       = sc->afe_pixel_seq[2];
    pseq1.pseq4       = sc->afe_pixel_seq[3];
    pseq1.pseq1_valid = true;
    pseq1.pseq2_valid = true;
    pseq1.pseq3_valid = true;
    pseq1.pseq4_valid = true;
    scanblk_set_PSEQ1(&pseq1);

    /* PSEQ2 */
    memset(&pseq2, 0x0, sizeof(pseq2));
    pseq2.pseq5       = sc->afe_pixel_seq[4];
    pseq2.pseq6       = sc->afe_pixel_seq[5];
    pseq2.pseq5_valid = true;
    pseq2.pseq6_valid = true;
    scanblk_set_PSEQ2(&pseq2);

    //*********************************************************************
    // NOTE: DO NOT change the LED gate times or PWM here. They've already been
    // set by calibration.
    //*********************************************************************

    /* SCFG1 */
    /* ballen TODO : sptog is new. Note that we don't set bulben (following davep, who
     * seeminly intentionally left it */
    memset(&scfg1, 0x0, sizeof(scfg1));
    scfg1.p1clken       = sc->SCFG1.p1clken;
    scfg1.p2clken       = sc->SCFG1.p2clken;
    scfg1.rsclken       = sc->SCFG1.rsclken;
    scfg1.cpclken       = sc->SCFG1.cpclken;
    scfg1.mclken        = sc->SCFG1.mclken;
    scfg1.vsclken       = sc->SCFG1.vsclken;
    scfg1.clclken       = sc->SCFG1.clclken;
    scfg1.lclmpen       = sc->SCFG1.lclpen;
    scfg1.ssmen         = sc->SCFG1.ssmen;
    scfg1.smpol         = sc->SCFG1.smpol;
    scfg1.p1apol        = sc->SCFG1.p1apol;
    scfg1.sppol         = sc->SCFG1.sppol;
    scfg1.scpol         = sc->SCFG1.scpol;
    scfg1.acpol         = sc->SCFG1.acpol;
    scfg1.p1clken_valid = true;
    scfg1.p2clken_valid = true;
    scfg1.rsclken_valid = true;
    scfg1.cpclken_valid = true;
    scfg1.mclken_valid  = true;
    scfg1.vsclken_valid = true;
    scfg1.clclken_valid = true;
    scfg1.lclmpen_valid = true;
    scfg1.ssmen_valid   = true;
    scfg1.smpol_valid   = true;
    scfg1.p1apol_valid  = true;
    scfg1.sppol_valid   = true;
    scfg1.scpol_valid   = true;
    scfg1.acpol_valid   = true;
    scanblk_set_SCFG1(&scfg1);

    memset(&scfg3, 0x0, sizeof(scfg3));
    scfg3.p1rise        = sc->SCFG3.p1rise;
    scfg3.p1fall        = sc->SCFG3.p1fall;
    scfg3.p2rise        = sc->SCFG3.p2rise;
    scfg3.p2fall        = sc->SCFG3.p2fall;
    scfg3.rsrise        = sc->SCFG3.rsrise;
    scfg3.rsfall        = sc->SCFG3.rsfall;
    scfg3.cprise        = sc->SCFG3.cprise;
    scfg3.cpfall        = sc->SCFG3.cpfall;
    scfg3.mcrise        = sc->SCFG3.mcrise;
    scfg3.mcfall        = sc->SCFG3.mcfall;
    scfg3.vsrise        = sc->SCFG3.vsrise;
    scfg3.vsfall        = sc->SCFG3.vsfall;
    scfg3.cd1rise       = sc->SCFG3.cd1rise;
    scfg3.cd1fall       = sc->SCFG3.cd1fall;
    scfg3.sprise        = sc->SCFG3.sprise;
    scfg3.spfall        = sc->SCFG3.spfall;
    scfg3.p1rise_valid  = true;
    scfg3.p1fall_valid  = true;
    scfg3.p2rise_valid  = true;
    scfg3.p2fall_valid  = true;
    scfg3.rsrise_valid  = true;
    scfg3.rsfall_valid  = true;
    scfg3.cprise_valid  = true;
    scfg3.cpfall_valid  = true;
    scfg3.mcrise_valid  = true;
    scfg3.mcfall_valid  = true;
    scfg3.vsrise_valid  = true;
    scfg3.vsfall_valid  = true;
    scfg3.cd1rise_valid = true;
    scfg3.cd1fall_valid = true;
    scfg3.sprise_valid  = true;
    scfg3.spfall_valid  = true;
    scanblk_set_SCFG3(&scfg3);

    /* davep 27-Jun-2011 ; poke the new ASIC CFGARB register */
    scif_cbi_arbitration_config( sc->cbi_enable_external, sc->cbi_enable_parallel );


    /* davep 27-Sep-2010 ; adding command line feature to set override values
     * for registers
     */
    merge_register_override();
}

void
scif_set_scanx( uint32_t scanx )
{
    struct scanblk_SCANX_cfg scanx_reg;
    memset(&scanx_reg, 0x0, sizeof(scanx_reg));
    scanx_reg.scanx = scanx-1;         /* -1 because the SCANX register counts from zero */
    scanx_reg.scanx_valid = true;
    scanblk_set_SCANX(&scanx_reg);
}

void
scif_set_pixel_period( uint32_t pixel_period )
{
//    dbg2( "%s pixel_period=%d\n", __FUNCTION__, pixel_period );

    struct scanblk_SCFG2_cfg scfg2;
    memset(&scfg2, 0x0, sizeof(scfg2));
    scfg2.pixper       = pixel_period-1;
    scfg2.pixper_valid = true;
    scanblk_set_SCFG2(&scfg2);
}

uint32_t scif_calc_pixel_period( uint32_t sensor_freq_khz )
{
#if 0
    float sensor_mhz;

    sensor_mhz = sensor_freq_khz / 1000.0;

    /* check for initialized variable (have some new code I want to make sure
     * doesn't cause a divide-by-zero)
     */
    ASSERT( sensor_mhz > 0 );

    /* adding 0.5 to round up */
    return (uint32_t)(((float)hw_get_scan_clk_speed() / sensor_mhz) + 0.5 );
#else
    uint32_t scan_clk_khz;

    /* get scan block clock in Mhz, convert to khz */
    scan_clk_khz = hw_get_scan_clk_speed()*1000;

//    dbg2( "%s scan_clk_khz=%d sensor_freq_khz=%d pixper=%d\n", __FUNCTION__, scan_clk_khz, sensor_freq_khz,
//        safeint_divide_with_round( scan_clk_khz, sensor_freq_khz ));

    return safeint_divide_with_round( scan_clk_khz, sensor_freq_khz );
#endif
}

void
scif_set_ssel( uint8_t ssel )
{
    struct scanblk_SCFG2_cfg scfg2;
    memset(&scfg2, 0x0, sizeof(scfg2));
    scfg2.ssel       = ssel;
    scfg2.ssel_valid = true;
    scanblk_set_SCFG2(&scfg2);
}

bool scif_sstat_qempty(void)
{
    struct scanblk_SSTAT_cfg sstat;
    scanblk_get_SSTAT(&sstat);
    return sstat.qempty && true;
}

static bool scif_sstat_qfull(void)
{
    struct scanblk_SSTAT_cfg sstat;
    scanblk_get_SSTAT(&sstat);
    return sstat.qfull && true;
}

/*
 * ballen TODO - fix this
 * 06-2014
 *
 * I haven't yet un-entangled uint32_t commands from scancmdq.c, and they need a way to write
 * using the new driver. This is a quick compatability function to bridge the gap.
 *
 * Note that there are significant changes in the SCMD register
 */
void scif_scmd_write(uint32_t cmd, bool in_irq_context)
{
    // FIXME - we really need to be queueing the scanblk_SCMD_cfg structures, and not the uint32_t
    // reference scif_command() and all of the callers of it, and the cmdq that stores the uint32_t

    struct scanblk_SCMD_cfg scmd;

    memset(&scmd, 0x0, sizeof(scmd));
    scmd.eoi           = SCAN_SCMD_EOI_MASK_SHIFT(cmd);
    scmd.sdata         = SCAN_SCMD_SDATA_MASK_SHIFT(cmd);
    scmd.sptogval      = SCAN_SCMD_SPTOGVAL_MASK_SHIFT(cmd);
    scmd.sclkper       = SCAN_SCMD_SCLKPER_MASK_SHIFT(cmd);
    scmd.scany         = SCAN_SCMD_SCANY_MASK_SHIFT(cmd);
    
    scmd.eoi_valid   = true;
    scmd.sdata_valid   = true;
    scmd.sptogval_valid   = true;        
    scmd.sclkper_valid = true;
    scmd.scany_valid   = true;

    // We need to know if we are in interrupt context here: if the low level driver
    // uses the wrong protection method (spinlock) ... very bad things will happen. 
    if (in_irq_context) {
        scanblk_set_SCMD_isr(&scmd);
    }
    else {
        scanblk_set_SCMD(&scmd);
    }
}

void
scif_send_simple_scmd( bool sdata, uint32_t sclkper, uint16_t num_lines )
{
    /* make sure the scan block can accept another command (otherwise we'll get
     * a command queue overflow interrupt)
     */
    struct scanblk_SCMD_cfg scmd;
    bool qempty;
    bool qfull = scif_sstat_qfull();

    XASSERT( !qfull, qfull );

    memset(&scmd, 0x0, sizeof(scmd));
    scmd.sdata         = sdata;
    scmd.sclkper       = sclkper-1;
    scmd.scany         = num_lines-1;
    scmd.sdata_valid   = true;
    scmd.sclkper_valid = true;
    scmd.scany_valid   = true;
    scanblk_set_SCMD(&scmd);

    qempty = scif_sstat_qempty();
    XASSERT( !qempty, qempty );
}

// Used by scantest - not for general use by normal scanning
void scif_set_internal_scanline_mode( scan_cmode_t cmode, uint32_t dpi, uint32_t scanx, uint32_t *sclkper )
{
    scif_motor_setup(MOTOR1, SYNC_DISABLE, MOT_INTERNAL_SYNC, MOT_INTERNAL_SYNC);

    /* we need to send sclkper back for SCMD - fake out source as flatbed, since this is just for test */
    *sclkper = scanplat_calc_sclkper(cmode, dpi, scanx, SCAN_DOCUMENT_SOURCE_FLATBED);

//    dbg2( "%s cmode=%d dpi=%d scanx=%d sclkper=%d\n", __FUNCTION__,
//            cmode, dpi, scanx, *sclkper );
}

/**
 * \brief CBI Arbitration Configuration
 *
 * enable_external : "Enables the CBI arbitration block to accept data from the
 * respective external interface and multiplex it into the output datastream. "
 *
 * enable_parallel : "Enables the CBI arbitration block to accept data from the
 * respective parallel (sdata) interface and multiplex it into the output
 * datastream."
 *
 * \author David Poole
 * \date 29-Jun-2011
 *
 */

void scif_cbi_arbitration_config( bool enable_external, bool enable_parallel  )
{
    struct scanblk_CFGARB_cfg cfgarb;

    /* One or the other must be enabled. Hopefully trap old code which needs to
     * update scansen_get_sensor_conf()
     */
    XASSERT( enable_external || enable_parallel, enable_external );

    memset(&cfgarb, 0x0, sizeof(cfgarb));
    cfgarb.extenable       = enable_external ? 1 : 0;
    cfgarb.parenable       = enable_parallel ? 1 : 0;
    cfgarb.extenable_valid = true;
    cfgarb.parenable_valid = true;
    scanblk_set_CFGARB(&cfgarb);
}

/**
 * \brief  Tentative function to enable/disable dual scan
 *
 *  Not final code! See note below.
 *
 * \author David Poole
 * \date 23-Jan-2013
 */

void scif_cbi_enable_dual_channel( bool enable )
{
    /* davep 21-Jan-2013 ; XXX this function is just a temporary placehold
     * while turning on dual scan. Need something that will allow us to change
     * the parenable flags to 1, 2, or 3.
     */
    struct scanblk_CFGARB_cfg cfgarb;
    memset(&cfgarb, 0x0, sizeof(cfgarb));
    cfgarb.parenable       = enable ? 3 : 1;
    cfgarb.parenable_valid = true;
    cfgarb.channel       = 0x02;   // source 0 to channel 0, source 1 to channel 1
    cfgarb.channel_valid = true;
    scanblk_set_CFGARB(&cfgarb);
}

/**
 * \brief  Get Scan Block Dual Channel enable/disable state
 *
 * Created for sanity checking. All of Scan block, CISX, and PIC must be in
 * agreement for dual channel mode.
 *
 * \author David Poole
 * \date 03-Apr-2013
 */

bool scif_get_dual_channel_enabled( void )
{
    struct scanblk_CFGARB_cfg cfgarb;
    scanblk_get_CFGARB(&cfgarb);
    /* Note! I'm only checking the parallel sources. */
    return cfgarb.parenable == 3;
}

/**
 * \brief Build a PSEQ pixel register field value.
 *
 * \author David Poole
 * \date 15-Mar-2013
 *
 * From the programmer's guide:
 *
 * S = Sequence Complete
 * P = DataType Complete; also decrements the SCANX data count when P = 1.
 * V = Output Valid
 *
 * Color[2:0] = Specified by the PIC Common Bus Interface:
 * 000 = Color 0 non-staggered
 * 001 = Color 1 non-staggered
 * 010 = Color 2 non-staggered
 * 011 = Monochrome non-staggered 100 = Color 0 staggered
 * 101 = Color 1 staggered
 * 110 = Color 2 staggered
 * 111 = Monochrome staggered
 *
 */

#define SCIF_PSEQ_S(x) (((x)&1)<<5)
#define SCIF_PSEQ_P(x) (((x)&1)<<4)
#define SCIF_PSEQ_V(x) (((x)&1)<<3)
#define SCIF_PSEQ_COLOR(x) ((x)&0x07)  /* three bits */

uint32_t scif_pseq( uint32_t S, uint32_t P, uint32_t V, uint32_t color )
{
    return SCIF_PSEQ_S(S)
            | SCIF_PSEQ_P(P)
            | SCIF_PSEQ_V(V)
            | SCIF_PSEQ_COLOR(color);
}

