/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2011-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/** 
 * \file piedma_descrip.c
 *
 * \brief PIE descriptor based DMA functions (using PIE driver to access hardware)
 */

#include <stdint.h>

#include "scos.h"

#include "lassert.h"
#include "list.h"
#include "memAPI.h"
#include "cpu_api.h"

#include "scancore.h"
#include "scantypes.h"
#include "scandbg.h"
#include "safetylock.h"
#include "icedma.h"

#include "pie_handle_if.h"
#include "scansen.h"
#include "scanvars.h"
#include "pie.h"
#include "pie_if.h"
#include "pie_convenience_if.h"

#include "piedma_descrip.h"
#include "scanalyzer.h"

#define PIE_DESCRIPTOR_OWNED_BY_BLOCK(desc) ((desc)->config_flags & PIE_DMA_DESCRIPTOR_CONFIG_OWNER_BLOCK)

#define MAX_PIE_DMA_DESCRIPTORS 16
//#define MAX_PIE_DMA_DESCRIPTORS 8

#define PIE_DMA_NAME_MAX 12 

struct pie_ddma_channel {
    char name[PIE_DMA_NAME_MAX+1]; /* useful for debugging (+1 for NULL) */

    uint8_t channel;

    bool is_open;

    int num_descriptors;
    struct pie_dma_descriptor *desc_list_virt;   // Virtual pointer to descriptor list
    struct pie_dma_descriptor *desc_list_dma;    // Physical pointer to descriptor list

    int num_running;
    struct pie_dma_descriptor *head_desc_virt;   // Virtual head pointer
    struct pie_dma_descriptor *tail_desc_virt;   // Virtual tail pointer
};

static struct pie_ddma_channel pie_ddma_read_channel[PIE_RDMA_NUM_CHANNELS];
static struct pie_ddma_channel pie_ddma_write_channel[PIE_WDMA_NUM_CHANNELS];

static struct ice_dma_driver pie_rdma_desc_driver = { 
    .msg_data =   SMSG_PIE_RDMA_DATA,
    .max_queue_depth = 1,

    .reset =      pie_rdma_channel_reset,
    .enable =     pie_rdma_channel_enable,
    .disable =    pie_rdma_channel_disable,
    .load =       pie_rdma_channel_load, 
    .start =      pie_rdma_channel_start,
    .is_enabled = pie_rdma_channel_is_enabled,
    .icebuf_isr = NULL 
};

static struct ice_dma_driver pie_wdma_desc_driver = { 
    .msg_data =   SMSG_PIE_WDMA_DATA,
    .max_queue_depth = 1,

    .reset =      pie_wdma_channel_reset,
    .enable =     pie_wdma_channel_enable,
    .disable =    pie_wdma_channel_disable,
    .load =       pie_wdma_channel_load, 
    .start =      pie_wdma_channel_start,
    .is_enabled = pie_wdma_channel_is_enabled,
    .icebuf_isr = NULL 
};


#define MAX_DMA_DESCRIPTORS 64

/* davep 18-Feb-2011 ; PIE descriptor DMA with new! shiny! horror! show! only
 * has three read channels and one write channel; 
 *
 * TODO move this into piehw.h  (somehow?)
 */
#define PIE_RDMA_VALID_CHANNEL_MASK 0x07 /* b'00111 */
#define PIE_WDMA_VALID_CHANNEL_MASK (1<<4) /* b10000 */


struct {
    uint32_t total_count;

    uint32_t read_clear[PIE_RDMA_NUM_CHANNELS];
    uint32_t read_desc[PIE_RDMA_NUM_CHANNELS];

    uint32_t write_clear[PIE_RDMA_NUM_CHANNELS];
    uint32_t write_desc[PIE_RDMA_NUM_CHANNELS];
} interrupt_counters;


static void dump_pie_regs(int num_idma)
{
    struct pie_handle_t *pie_handle;
    
    pie_handle = pie_create_new_default_handle();
    pie_do_get_current(pie_handle);
    pie_dump_handle_regs(pie_handle, num_idma);
    pie_do_free_handle(pie_handle);
}

void pie_dma_dump_counters( void )
{
    dbg1( "%s %d  w=%d %d  r=%d %d\n", __FUNCTION__, 
                interrupt_counters.total_count,
                interrupt_counters.write_desc[0],
                interrupt_counters.write_clear[0],

                interrupt_counters.read_desc[0],
                interrupt_counters.read_clear[0]
         );
}

void pie_dma_dump_descriptor( struct pie_dma_descriptor *desc )
{
    dbg2( "desc=%p flags=%#x dsrc=%p len=%d dnext=%p dar1=%p:%p dar2=%p:%p vself=%p vnext=%p dself=%p\n", 
                desc,
                desc->config_flags,
                (void *)desc->dma_src_addr,
                desc->transfer_len_bytes,
                (void *)desc->dma_ptr_next_desc,
                (void *)desc->dar1_start, (void *)desc->dar1_end,
                (void *)desc->dar2_start, (void *)desc->dar2_end,
                (void *)desc->virt_ptr_self,
                (void *)desc->virt_ptr_next_desc,
                (void *)desc->dma_ptr_self);
}

static void desc_flush( struct pie_dma_descriptor *desc )
{
    /* flush the descriptor from the cache to main mem where dma can find it */
    cpu_dcache_writeback_region( desc, sizeof(struct pie_dma_descriptor) );
    cpu_dcache_invalidate_region( desc, sizeof(struct pie_dma_descriptor) );
}
        
void pie_dma_flush_descriptor( struct pie_dma_descriptor *desc )
{
    desc_flush( desc );
}

void pie_dma_desc_chain_dump( struct pie_ddma_channel *pch )
{
    int sanity_count;
    struct pie_dma_descriptor *head;
    struct pie_dma_descriptor *curr;

    dbg2( "%s %s-%d\n", __FUNCTION__, pch->name, pch->channel );
    dbg2( "%s head=%p tail=%p\n", __FUNCTION__, pch->head_desc_virt, pch->tail_desc_virt );

    head = pch->head_desc_virt;

    ASSERT( head );
    ASSERT( pch->desc_list_virt );

    sanity_count = 0;
    curr = head;
    do {
        cpu_dcache_invalidate_region( curr, sizeof(struct pie_dma_descriptor) );

        pie_dma_dump_descriptor( curr );

        /* make sure our list is still properly formed */
        XASSERT( curr->virt_ptr_next_desc!=0, (uint32_t)curr );

        curr = (struct pie_dma_descriptor *)curr->virt_ptr_next_desc;

        /* avoid infinite loops on corrupted lists */
        sanity_count += 1;
        XASSERT( sanity_count<=pch->num_descriptors, sanity_count );

    } while( curr != head );

}

static void pie_ddma_channel_isr( struct ice_dma_driver *drvr, struct pie_ddma_channel *pch )
{
    struct pie_dma_descriptor *curr;
    uint32_t desc_count;

    /* 
     * BIG FAT NOTE!
     *
     * This is an interrupt handler! 
     *
     */

    /* davep 06-May-2011 ; TODO move to ddma.c (somehow...) */

//    dbg2( "%s %s-%d\n", __FUNCTION__, pch->name, pch->channel );
//    pie_dma_desc_chain_dump( pch );

    /* use a convenience pointer; remember we still have to modify the list
     * itself
     */
    curr = pch->head_desc_virt;
    cpu_dcache_invalidate_region( curr, sizeof(struct pie_dma_descriptor) );

    /* davep 11-Oct-2010 ; add safety check to make sure the hardware is far
     * enough behind us we won't collide ("listening for the bulldozers")
     */
//    XASSERT( DESCRIPTOR_OWNED_BY_BLOCK((struct pie_dma_descriptor *)curr->fw_prev_descriptor_addr), 
//            curr->fw_prev_descriptor_addr );

    desc_count = 0;

    /* we could have more than one completed descriptor so we'll walk the list
     * until we find a descriptor with own bit set
     */
    while( !PIE_DESCRIPTOR_OWNED_BY_BLOCK( curr ) )
    {
        SCANALYZER_LOG( LOG_PIE_DDMA_CHANNEL_ISR, (uint32_t)curr );

//        dbg2( "%s %s-%d curr=%p\n", __FUNCTION__, pch->name, pch->channel, curr );

        desc_count += 1;

        XASSERT( (curr->config_flags & PIE_DMA_DESCRIPTOR_CONFIG_OWNER_BLOCK)==0, curr->config_flags );

        XASSERT( curr->config_flags & PIE_DMA_DESCRIPTOR_CONFIG_INT, curr->config_flags ); 

        /* mark the descriptor empty */
        curr->dma_src_addr = 0;
        curr->transfer_len_bytes = 0;

        pch->num_running -= 1;
        XASSERT( pch->num_running >=0, pch->channel );

        /* Set the "own" bit back on the descriptor so I know I've already
         * handled this buffer. We're in a circular list so need to know when
         * to stop.
         */
        curr->config_flags |= PIE_DMA_DESCRIPTOR_CONFIG_OWNER_BLOCK;
        desc_flush( curr );

        /* move head of list to next element */
        pch->head_desc_virt = (struct pie_dma_descriptor *)curr->virt_ptr_next_desc;

        /* move to next link in our chain so we can loop again if necessary */
        curr = pch->head_desc_virt;
        cpu_dcache_invalidate_region( curr, sizeof(struct pie_dma_descriptor) );

//        ddma_channel_sanity( pch );

        /* call the default method to do the icebuf buffer dance */
        if( drvr->icebuf_isr) {
            drvr->icebuf_isr( pch->channel );
        }
        else {
            dbg2( "%s WARNING no drvr->icebuf_isr\n", __FUNCTION__ );
        }
    }

//    dbg2( "%s %s-%d desc_count=%d\n", __FUNCTION__, pch->name, pch->channel, desc_count );
//    if (desc_count == 0) {
//        pie_dma_desc_chain_dump( pch );
//    }

    SCANALYZER_LOG( LOG_PIE_DDMA_CHANNEL_ISR, desc_count );
}

static void pie_ddma_channel_close( struct pie_ddma_channel *pch )
{
    int num_bytes;

    if( !pch->is_open ) {
        return;
    }

    /* davep 07-Jun-2011 ; num_running>=0 if we cancelled */
//    XASSERT( pch->num_running==0, pch->num_running );
    XASSERT( pch->num_descriptors==MAX_PIE_DMA_DESCRIPTORS, pch->num_descriptors );

    num_bytes = sizeof(struct pie_dma_descriptor) * MAX_PIE_DMA_DESCRIPTORS;

    /* poison the memory to catch anyone still using it */
    memset( pch->desc_list_virt, 0xaa, num_bytes );

#ifdef __KERNEL__
    dma_free_coherent( NULL, num_bytes, pch->desc_list_virt, (dma_addr_t)pch->desc_list_dma );
    pch->desc_list_virt = NULL;
    pch->desc_list_dma  = NULL;
#else
    PTR_FREE( pch->desc_list_virt );
    pch->desc_list_dma  = NULL;
#endif

    pch->is_open = false;
}

static scan_err_t pie_ddma_channel_open( uint8_t channel, struct pie_ddma_channel *pch, const char *name )
{
    struct pie_dma_descriptor *desc_virt;
    int i;
    int num_bytes;

    //dbg2("%s channel=%d pch=%p name=%s\n", __FUNCTION__, channel, pch, name);

    /* be paranoid about memory leaks */
    XASSERT( pch->desc_list_virt==NULL, (uint32_t)pch->desc_list_virt );

    memset( pch, 0, sizeof(struct pie_ddma_channel) );

    pch->channel = channel;
    strncpy( pch->name, name, PIE_DMA_NAME_MAX );

    num_bytes = sizeof(struct pie_dma_descriptor) * MAX_PIE_DMA_DESCRIPTORS;

#ifdef __KERNEL__
    pch->desc_list_virt = (struct pie_dma_descriptor *)dma_alloc_coherent( 
                                      NULL,
                                      num_bytes, 
                                      (dma_addr_t *)&pch->desc_list_dma, 
                                      GFP_DMA );
#else
    pch->desc_list_virt = MEM_MALLOC_ALIGN( num_bytes, e_32_byte );
    pch->desc_list_dma  = pch->desc_list_virt;
#endif

    if( pch->desc_list_virt==NULL ) {
        dbg2( "%s failed to alloc descriptor list\n", __FUNCTION__ );
        return SCANERR_OUT_OF_MEMORY;
    }

    memset( pch->desc_list_virt, 0, num_bytes );
    pch->num_descriptors = MAX_PIE_DMA_DESCRIPTORS;

    /* Link them all together. 
     *
     * I set the transfer int bit on each descriptor so I get an interrupt on
     * each completed transfer. 
     */
    for( i=0 ; i < pch->num_descriptors-1 ; i++ ) {
        desc_virt = &pch->desc_list_virt[i];

        desc_virt->config_flags = PIE_DMA_DESCRIPTOR_CONFIG_INT
                           | PIE_DMA_DESCRIPTOR_CONFIG_STOP
                           | PIE_DMA_DESCRIPTOR_CONFIG_OWNER_BLOCK;

        desc_virt->virt_ptr_self      = desc_virt;
        desc_virt->dma_ptr_self       = &pch->desc_list_dma[i];        
        desc_virt->virt_ptr_next_desc = &pch->desc_list_virt[i+1];
        desc_virt->dma_ptr_next_desc  = (uint32_t)&pch->desc_list_dma[i+1];
        desc_flush( desc_virt );
    }

    desc_virt = &pch->desc_list_virt[i];
    desc_virt->config_flags = PIE_DMA_DESCRIPTOR_CONFIG_INT
                       | PIE_DMA_DESCRIPTOR_CONFIG_STOP
                       | PIE_DMA_DESCRIPTOR_CONFIG_OWNER_BLOCK;
    desc_virt->virt_ptr_self      = desc_virt;
    desc_virt->dma_ptr_self       = &pch->desc_list_dma[i];        
    desc_virt->virt_ptr_next_desc = &pch->desc_list_virt[0];
    desc_virt->dma_ptr_next_desc  = (uint32_t)&pch->desc_list_dma[0];
    desc_flush( desc_virt );

    pch->head_desc_virt = &pch->desc_list_virt[0];
    pch->tail_desc_virt = &pch->desc_list_virt[0];

    /* Last link in the chain. We'll move this flag along to indicate the last
     * valid buffer in the chain. The DMA HW will stop after finishing this
     * descriptor. We don't want that. We want to keep feeding DMA, keep it
     * running as long as possible.
     */
    pch->tail_desc_virt->config_flags |= PIE_DMA_DESCRIPTOR_CONFIG_STOP; 
    desc_flush( pch->tail_desc_virt );

    // DEBUG ONLY
    //pie_dma_desc_chain_dump( pch );

    /* open for business! */
    pch->is_open = true;

    return SCANERR_NONE;
}

static void pie_ddma_channel_load2( struct pie_ddma_channel *pch, 
                                    uint32_t dar1, uint32_t drcr1, 
                                    uint32_t dar2, uint32_t drcr2, 
                                    uint32_t bytes_per_row )
{
    struct pie_dma_descriptor *next_virt;

    XASSERT( pch->is_open, pch->channel );

    XASSERT( (dar1 & ~ICE_DMA_BYTE_ALIGN_MASK) == 0, dar1 );
    XASSERT( (dar2 & ~ICE_DMA_BYTE_ALIGN_MASK) == 0, dar2 );
    XASSERT( (bytes_per_row & ~ICE_DMA_BYTE_ALIGN_MASK)==0, bytes_per_row );

    // FIXME:
    // We need to turn off IRQ's here to prevent corruption of our data structures.
    // Problem is that we may be loading channels before launch, in which case
    // IRQ's will be off.  The new PIE driver has no way to tell us if IRQ's are 
    // enabled like the old code (which could case them to be accidentally enabled
    // as we exit this function):
    //    was_enabled = pie_interrupt_disable();
    pie_disable_pie_common_irq(NULL);

    /* davep 06-May-2011 ; XXX temp debug ; dump bunch of debug on this
     * condition 
     */
//    if( !PIE_DESCRIPTOR_OWNED_BY_BLOCK(pch->tail_desc_virt) ) {
//        pie_dma_desc_chain_dump( pch );
//        pie_rdma_dump();
//        pie_wdma_dump();
//        ASSERT(0);
//    }

//    /* make sure the block hasn't been here yet */
//    XASSERT( PIE_DESCRIPTOR_OWNED_BY_BLOCK(pch->tail_desc_virt), pch->tail_desc_virt->config_flags );

    /* can hit this assert if we're trying to dma without allocating the
     * descriptors via pie_wdma_open() 
     */
    XASSERT( pch->num_descriptors>0, pch->channel );

    XASSERT( pch->num_running < pch->num_descriptors, pch->num_running );

//    /* make sure the block hasn't been here yet */
//    XASSERT( PIE_DESCRIPTOR_OWNED_BY_BLOCK(pch->tail_desc_virt), pch->tail_desc_virt->config_flags );

    /* set up tail->next transaction */
    if( pch->tail_desc_virt->dma_src_addr==0 ) {
        /* Empty list. Note head==tail in a list with one element AND an empty
         * list thus the check on dma_src_addr 
         */
        next_virt = pch->head_desc_virt;
    }
    else {
        next_virt = (struct pie_dma_descriptor *)(pch->tail_desc_virt->virt_ptr_next_desc);
    }

    /* beware overrunning existing members of the list (dma_src_addr set to zero in
     * ISR when the descriptor completes)
     */
    if (next_virt->dma_src_addr != 0 )
    {
        pie_dma_desc_chain_dump( pch );
        XASSERT( next_virt->dma_src_addr==0, (uint32_t)next_virt );
    }

    next_virt->dma_src_addr = dar1;
    next_virt->transfer_len_bytes = (drcr1 + drcr2) * bytes_per_row;
    next_virt->dar1_start = dar1;
    next_virt->dar1_end = dar1 + drcr1*bytes_per_row -1;
    next_virt->dar2_start = dar2;
    if( dar2 ) {
        next_virt->dar2_end = dar2 + drcr2*bytes_per_row -1;
    }
    else {
        next_virt->dar2_end = 0;
    }

    next_virt->config_flags |= PIE_DMA_DESCRIPTOR_CONFIG_STOP|PIE_DMA_DESCRIPTOR_CONFIG_OWNER_BLOCK;

    /* flush the descriptor from the cache to main mem where dma can find it */
    desc_flush( next_virt );

//    pch->tail_desc_virt->config_flags &= ~PIE_DMA_DESCRIPTOR_CONFIG_STOP;

    /* flush the descriptor from the cache to main mem where dma can find it */
    desc_flush( pch->tail_desc_virt );

    /* move to next */
    pch->tail_desc_virt = next_virt;

    pch->tail_desc_virt->config_flags |= PIE_DMA_DESCRIPTOR_CONFIG_STOP;
    desc_flush( pch->tail_desc_virt );
    
    pch->num_running += 1;

//    pie_dma_channel_sanity( pch );
//    pie_dma_desc_chain_dump( pch );

    // FIXME
    // How do I know if IRQs should be enabled here? Old code looked like:
    //    if( was_enabled ) {
    //        pie_interrupt_enable();
    //    }
    pie_enable_pie_common_irq(NULL);
}

/* ------------------------------------------
 *
 *  Read DMA Functions
 *
 * ------------------------------------------
 */

bool pie_rdma_channel_is_valid( uint8_t channel ) 
{
    /* Simple channel number range check */
    if (channel >= PIE_RDMA_NUM_CHANNELS) {
        return false;
    }

    /* Check if this channel number is valid on this hardware. Valid channels
     * have a bit set in the conditionally compiled mask.
     */
    return ( ((1<<channel) & (~PIE_RDMA_VALID_CHANNEL_MASK))==0 );
}

static int pie_rdma_desc_isr( void *irq_stuff )
{
    struct idma_interrupt_info *idma_data;
    uint32_t channel;

    /* 
     * BIG FAT NOTE!
     *
     * This is an interrupt handler! 
     */

    idma_data = irq_stuff;

    if (idma_data != NULL)
    {
        interrupt_counters.total_count++;
        channel = idma_data->instance;

        // IRQ Debug!
        //dbg1("%s: instance=%d, EOS=%d, DESC=%d, CC=%d, Own=%d, LE=%d, OORE=%d RRERR=%d BRERR=%d\n",
        //        __FUNCTION__, idma_data->instance, idma_data->EndOfStrip, idma_data->Desc,
        //        idma_data->ClearComplete, idma_data->Own, idma_data->LengthErr, idma_data->OutOfRangeErr,
        //        idma_data->RRespErr, idma_data->BRespErr);

        /* brutally fail if someone calls with an invalid channel (should not be
         * called unless setup is wrong)
         */
        XASSERT( pie_rdma_channel_is_valid(channel), channel );

        /* fail on anything unexpected */
        if (idma_data->RRespErr || idma_data->BRespErr || idma_data->OutOfRangeErr || idma_data->LengthErr || idma_data->Own)
        {
            dbg1("%s: unexpected irq\n", __FUNCTION__);
            dbg1("%s: instance=%d, EOS=%d, DESC=%d, CC=%d, Own=%d, LE=%d, OORE=%d RRERR=%d BRERR=%d\n",
                 __FUNCTION__, idma_data->instance, idma_data->EndOfStrip, idma_data->Desc,
                 idma_data->ClearComplete, idma_data->Own, idma_data->LengthErr, idma_data->OutOfRangeErr,
                 idma_data->RRespErr, idma_data->BRespErr);

            pie_dma_dump_counters();
            pie_rdma_dump();
            ASSERT( 0 );
        }

        /* Clear complete */
        if( idma_data->ClearComplete ) {
            interrupt_counters.read_clear[channel] += 1;
        }

        /* Descriptor complete! */
        if( idma_data->Desc ) {
            pie_ddma_channel_isr( &pie_rdma_desc_driver, &pie_ddma_read_channel[channel] );
            interrupt_counters.read_desc[channel] += 1;
        }
    }
    else
    {
        dbg1("%s: WARNING irq callback with no stuff!\n", __FUNCTION__);
    }

    return 0;
}

void pie_rdma_setup_config( struct pie_handle_t *pie_handle, 
                            uint8_t      num_idma_channels,
                            uint8_t      pie_bpp, 
                            scan_cmode_t cmode, 
                            pie_pixel_t  piein, 
                            uint32_t     strip_numrows,
                            uint32_t     bytes_per_row )
{
    uint32_t cfg_bpp = POGO_8BPP;
    uint32_t cfg_fmt = POGO_FMT_MONO;
    uint8_t handshake, ownership, own_polarity, enable;
    int     i;

    switch( pie_bpp ) {
        case 8 :
            cfg_bpp = POGO_8BPP;
            break;
        case 16 :
            cfg_bpp = POGO_16BPP;
            break;
        default:
            /* only 8 and 16-bpp pixels are supported (value ignored for
             * rgb,rgbx,xrgb)
             */
            XASSERT( 0, pie_bpp );
            break;
    }

    switch( piein ) {
        case PIE_PIXEL_XRGB :
            cfg_fmt = POGO_FMT_XRGB;
            break;
        case PIE_PIXEL_RGBX :
            cfg_fmt = POGO_FMT_RGBX;
            break;
        case PIE_PIXEL_RGB :
            cfg_fmt = POGO_FMT_RGB;
            break;
        case PIE_PIXEL_MONO :
            cfg_fmt = POGO_FMT_MONO;
            break;
        case PIE_PIXEL_3_PLANES :
            cfg_fmt = POGO_FMT_PLANAR;
            break;
        default:
            /* no other formats supported */
            XASSERT( 0, piein );
            break;
    }

    pie_pogoizer_set_config(pie_handle, cfg_fmt, cfg_bpp, POGO_NOCOLORSWAP,
                            PIE_NO_POGO_XSCALE, PIE_NO_POGO_YSCALE);

    // REVISIT: are these settings appropriate?
    handshake    = 0;
    ownership    = 0;
    own_polarity = 0;
    enable       = DMA_ENABLE;

    for (i=0; i<num_idma_channels; i++)
    {
        // setup the input dma for each input channel
        pie_input_dma_set_parms(pie_handle, handshake, ownership, own_polarity, enable,
                                bytes_per_row, strip_numrows, i);
        pie_enable_pogo_idma_irqs(pie_handle, NULL, i, true); // enable all idma interrupts
    }

    pie_register_idma_callback(pie_handle, pie_rdma_desc_isr, NULL);
}

/**
 * \brief  Return true if all of PIE WDMA is idle
 *
 * Created to assist multiplex PIE DMAs
 *
 * \author David Poole
 * \date 28-May-2013
 */
bool pie_rdma_is_idle( void )
{
    int channel;
    bool is_idle = true;

    for ( channel=0 ; channel<PIE_RDMA_NUM_CHANNELS ; channel++ ) {
        if( !pie_rdma_channel_is_enabled( channel ) ) {
            /* skip currently unused hardware channels */
            continue;
        }

        is_idle = pie_idma_is_idle(channel);
        if (is_idle == false) {
            /* found an active channel, bail out */
            break;
        }
    }

    return is_idle;
}

/**
 * \brief RDMA channel reset
 *
 * RDMA driver function to reset an rdma channel. Typically called during a cancel
 * operation.
 *
 * NOTE: users of this driver function (icedma) don't have pie handles, so we
 *       can't interact with hardware unless the pie driver provides the needed
 *       direct access functions (which can be done if needed, dma is 'special').
 */
void pie_rdma_channel_reset( uint8_t channel )
{
    if( !pie_rdma_channel_is_valid(channel) ) {
        /* gently ignore because called from icedma */
        return;
    }

    // FIXME
    // Disabling IRQs for the specific channel would be nice. Because we know this
    // comes from a cancel, just club the entire PIE (once for each channel, doh).
    pie_do_reset();

#if 0
    volatile UDMA_REGS_t *udma;
    volatile POGO_IDMA_CORE_REGS_t *idma;

    get_iudma( channel, &udma );
    get_idma( channel, &idma );

//    dbg2( "%s %p %p\n", __FUNCTION__, udma, idma );

    /* disable all interrupts */
    udma->UIER = 0;
    idma->IIER = 0;

    /* ack all pending interrupts */
    udma->UICR = ~0;
    idma->IICR = ~0;
    
    /* restore registers to power-on defaults */
    udma->UCR = 0;
    idma->ICR = POGO_IDMA_CORE_ICR_BURSTSIZE_REPLACE_VAL(idma->ICR,0);
    idma->ILWR = 0; /* Line Width Register */
    idma->IRHR = 0; /* Line Height Register */

    /* sanity check on register field */
    XASSERT( udma->UIER==0, udma->UIER );
    XASSERT( idma->IIER==0, idma->IIER );

//    pie_rdma_channel_enable(channel);
//    uint32_t count;
//    count = 0;
//    while( !(udma->UIPR & UDMA_UIPR_CLEARCOMPLETE_MASK) ) {
//        dbg2( "%s waiting count=%d\n", __FUNCTION__, count );
//        XASSERT( count++ < 100000, count );
//        posix_sleep_seconds(1);
////        cpu_spin_delay(1);
//    }
//
//    XASSERT( udma->UIPR==0, udma->UIPR );
//    pie_rdma_channel_disable(channel);
#endif
}

/**
 * \brief RDMA channel start
 *
 * RDMA driver function to start an rdma channel
 *
 * NOTE: users of this driver function (icedma) don't have pie handles, so we
 *       can't interact with hardware unless the pie driver provides the needed
 *       direct access functions (which can be done if needed, dma is 'special').
 */
void pie_rdma_channel_start( uint8_t channel )
{
    struct pie_ddma_channel *pch;

    XASSERT( pie_rdma_channel_is_valid(channel), channel );
    pch = &pie_ddma_read_channel[channel];

    XASSERT( pch->channel==channel, channel );
    XASSERT( pch->is_open, pch->channel );

    XASSERT( pch->num_running > 0, pch->num_running );
    XASSERT( pch->head_desc_virt != NULL, pch->channel );

    XASSERT( PIE_DESCRIPTOR_OWNED_BY_BLOCK(pch->head_desc_virt), pch->head_desc_virt->config_flags );

    // FIXME:
    // Do we need to check the USR busy bit before starting a new operation?
    // The old code used to check and ASSERT if already busy as a sanity check.

    //dbg2("%s RDMA Fire in the hole - channel=%d\n", __FUNCTION__, channel);

    pie_start_pogo_input_dma((uint32_t)pch->head_desc_virt->dma_ptr_self, channel);
}

/**
 * \brief RDMA channel enable
 *
 * RDMA driver function to enable an rdma channel.
 *
 * NOTE: users of this driver function (icedma) don't have pie handles, so we
 *       can't interact with hardware unless the pie driver provides the needed
 *       direct access functions (which can be done if needed, dma is 'special').
 */
void pie_rdma_channel_enable( uint8_t channel )
{
    // FIXME
    // No pie handle here ... so can't enable the channel. Turns out that
    // this function isn't used, as channels are enabled as part of the initial
    // setup process for the scan.
    XASSERT( pie_rdma_channel_is_valid(channel), channel );

#if 0
    volatile UDMA_REGS_t *udma;
    volatile POGO_IDMA_CORE_REGS_t *idma;

    get_iudma( channel, &udma );
    get_idma( channel, &idma );

    udma->UCR = UDMA_UCR_ENABLE_REPLACE_VAL( udma->UCR, 1 );

    /* enable (almost) all interrupts. EOS (End of Strip) apparently exactly
     * like UIER DESC (descriptor complete) so don't want the dup
     */
    udma->UIER = ~0;
    idma->IIER = ~(POGO_IDMA_CORE_IIER_EOS_MASK);
#endif
}

/**
 * \brief Is RDMA channel enabled?
 *
 * RDMA driver function to see if a channel is enabled. Typically used
 * for sanity checks when launching or closing channels.
 */
bool pie_rdma_channel_is_enabled( uint8_t channel )
{
    if( !pie_rdma_channel_is_valid(channel) ) {
        /* gently ignore because called from icedma */
        return false;
    }

    return pie_idma_is_enabled(channel);
}

/**
 * \brief RDMA channel disable
 *
 * RDMA driver function to disable an rdma channel. Typically called
 * during a cancel operation.
 *
 * NOTE: users of this driver function (icedma) don't have pie handles, so we
 *       can't interact with hardware unless the pie driver provides the needed
 *       direct access functions (which can be done if needed, dma is 'special').
 */
void pie_rdma_channel_disable( uint8_t channel )
{
    // FIXME
    // No pie handle here ... so can't disable the channel.  This is usually called
    // when icedma does a cancel (reset followed by disable). We clubbed pie in the
    // reset call, which should leave all channels disabled.

   if( !pie_rdma_channel_is_valid(channel) ) {
        /* gently ignore because called from icedma */
        return;
    }

#if 0
    volatile UDMA_REGS_t *udma;

    get_iudma( channel, &udma );
    udma->UCR = UDMA_UCR_ENABLE_REPLACE_VAL( udma->UCR, 0 );
#endif
}

/**
 * \brief RDMA channel add buffer
 *
 * RDMA driver function to add a buffer to an rdma channel.
 *
 * NOTE: users of this driver function (icedma) don't have pie handles, so we
 *       can't interact with hardware unless the pie driver provides the needed
 *       direct access functions (which can be done if needed, dma is 'special').
 */
void pie_rdma_channel_load( uint8_t channel, uint8_t *data_ptr, dma_addr_t dma_dest, 
                            uint32_t rows, uint32_t bytes_per_row )
{
    struct pie_ddma_channel *pch;

//    dbg2( "%s %d %#x %d %d\n", __FUNCTION__, channel, dma_dest, rows, bytes_per_row );

    XASSERT( pie_rdma_channel_is_valid(channel), channel );

    pch = &pie_ddma_read_channel[ channel ];
    XASSERT( pch->channel==channel, channel );

    pie_ddma_channel_load2( pch, (uint32_t)dma_dest, rows, 0, 0, bytes_per_row );

    XASSERT( PIE_DESCRIPTOR_OWNED_BY_BLOCK(pch->head_desc_virt), pch->head_desc_virt->config_flags );

    // FIXME/REVISIT
    // These values should NOT change during a scan, now set in pie_rdma_setup_config.
#if 0
    /* IDMA wants bytes_per_row so let's hope bytes_per_row stays constant
     * across the entire image 
     */
    get_idma( channel, &idma );
    idma->ILWR= bytes_per_row;
    idma->IRHR = rows;
#endif
}

void pie_rdma_channel_load2( uint8_t channel, uint32_t dar1, uint32_t drcr1, 
                             uint32_t dar2, uint32_t drcr2, uint32_t bytes_per_row )
{
    struct pie_ddma_channel *pch;

//    dbg2( "%s %d %p %p %p %p %d\n", __FUNCTION__, channel, dar1, drcr1, dar2, drcr2, bytes_per_row );

    XASSERT( pie_rdma_channel_is_valid(channel), channel );

    pch = &pie_ddma_read_channel[ channel ];
    XASSERT( pch->channel==channel, channel );

    pie_ddma_channel_load2( pch, dar1, drcr1, dar2, drcr2, bytes_per_row );

    XASSERT( PIE_DESCRIPTOR_OWNED_BY_BLOCK(pch->head_desc_virt), pch->head_desc_virt->config_flags );

    // FIXME
    // No pie handle here, can't touch hardware.  These values should NOT change
    // during a scan, so see if we can move to the initial pie setup.
#if 0
    /* IDMA wants bytes_per_row so let's hope bytes_per_row stays constant
     * across the entire image 
     */
    get_idma( channel, &idma );
    idma->ILWR= bytes_per_row;
    idma->IRHR = drcr1 + drcr2;
#endif
}

void pie_rdma_dump( void )
{
    int channel;

    // FIXME
    // May want to get some more focused dump functions from the pie driver, this dumps
    // it all. This method used to do core and rdma only.
    dump_pie_regs(PIE_RDMA_NUM_CHANNELS);

    /* davep 05-May-2011 ; XXX temp debug */
    for( channel=0 ; channel<PIE_RDMA_NUM_CHANNELS ; channel++ ) {
        if( pie_ddma_read_channel[channel].desc_list_virt ) {
            pie_dma_desc_chain_dump( &pie_ddma_read_channel[channel] );
        }
    }
}

const struct ice_dma_driver * pie_rdma_get_driver( void )
{
    return &pie_rdma_desc_driver;
}

void pie_rdma_set_icebuf_isr( iceisr_f icebuf_isr_f )
{
    pie_rdma_desc_driver.icebuf_isr = icebuf_isr_f;
}

scan_err_t pie_rdma_channel_open( uint8_t channel )
{
    XASSERT( pie_rdma_channel_is_valid(channel), channel );
    //dbg2("%s channel=%d\n", __FUNCTION__, channel);

    return pie_ddma_channel_open( channel, &pie_ddma_read_channel[channel], "pie_rdma" );
}

void pie_rdma_channel_close( uint8_t channel )
{
    XASSERT( channel < PIE_RDMA_NUM_CHANNELS, channel );
    //XASSERT( !pie_rdma_channel_is_enabled(channel), channel );

    pie_ddma_channel_close( &pie_ddma_read_channel[channel] );
}


/* ------------------------------------------
 *
 * Write DMA Functions
 *
 * ------------------------------------------
 */

static int pie_wdma_desc_isr( void *irq_stuff )
{
    struct odma_interrupt_info *odma_data;
    uint32_t channel = 0;

    /* 
     * BIG FAT NOTE!
     *
     * This is an interrupt handler! 
     */

    odma_data = irq_stuff;

    if (odma_data != NULL)
    {
        // IRQ Debug!
        //dbg2("%s: instance=0, EOS=%d, DESC=%d, CC=%d, Own=%d, LE=%d, OORE=%d RRERR=%d BRERR=%d\n",
        //       __FUNCTION__, odma_data->EndOfStrip, odma_data->Desc,
        //       odma_data->ClearComplete, odma_data->Own, odma_data->LengthErr, odma_data->OutOfRangeErr,
        //       odma_data->RRespErr, odma_data->BRespErr);

        XASSERT( channel<PIE_WDMA_NUM_CHANNELS, channel );
        interrupt_counters.total_count++;

        /* fail on anything unexpected */
        if (odma_data->RRespErr || odma_data->BRespErr || odma_data->OutOfRangeErr || odma_data->LengthErr || odma_data->Own)
        {
            dbg1("%s: unexpected irq\n", __FUNCTION__);
            dbg1("%s: instance=0, EOS=%d, DESC=%d, CC=%d, Own=%d, LE=%d, OORE=%d RRERR=%d BRERR=%d\n",
                 __FUNCTION__, odma_data->EndOfStrip, odma_data->Desc,
                 odma_data->ClearComplete, odma_data->Own, odma_data->LengthErr, odma_data->OutOfRangeErr,
                 odma_data->RRespErr, odma_data->BRespErr);

            pie_dma_dump_counters();
            pie_wdma_dump();
            ASSERT( 0 );
        }

        /* Clear complete */
        if( odma_data->ClearComplete ) {
            interrupt_counters.write_clear[channel] += 1;
        }

        /* Descriptor complete! */
        if( odma_data->Desc ) {
            pie_ddma_channel_isr( &pie_wdma_desc_driver, &pie_ddma_write_channel[channel] );
            interrupt_counters.write_desc[channel] += 1;
        }
    }
    else
    {
        dbg1("%s: WARNING irq callback with no stuff!\n", __FUNCTION__);
    }

    return 0;
}

void pie_wdma_setup_config( struct pie_handle_t *pie_handle,
                            scan_cmode_t cmode, 
                            pie_pixel_t  pieout, 
                            uint32_t     strip_numrows,
                            uint32_t     bytes_per_row )
{
    uint32_t depogo_cfg_fmt = POGO_FMT_MONO;
    uint8_t  color_swap     = POGO_NOCOLORSWAP;
    uint8_t handshake, ownership, own_polarity, enable;

    XASSERT( strip_numrows <= PIE_WDMA_MAX_ROWS, strip_numrows );

    switch( pieout ) 
    {
        case PIE_PIXEL_XRGB :
            depogo_cfg_fmt = POGO_FMT_XRGB;
            break;

        case PIE_PIXEL_RGBX :
            depogo_cfg_fmt = POGO_FMT_RGBX;
            break;

        case PIE_PIXEL_RGB :
            depogo_cfg_fmt = POGO_FMT_RGB;
            /* davep 13-Sep-2012 ; swap BGR output to RGB; DEPOGO colorswap is related to pixel endian */
            color_swap = POGO_COLORSWAP;
            break;

        case PIE_PIXEL_MONO :
            depogo_cfg_fmt = POGO_FMT_MONO;
            break;

        default :
            XASSERT( 0, pieout );
            break;
    } 

    // setup the configuration values for the depogoizer
    pie_depogoizer_set_config(pie_handle, depogo_cfg_fmt, POGO_8BPP,
                              color_swap, PIE_DEPOGO_MONOCHAN_DEF);
 
    // REVISIT: are these settings appropriate?
    handshake    = 0;
    ownership    = 0;
    own_polarity = 0;
    enable       = DMA_ENABLE;
    pie_output_dma_set_parms(pie_handle, handshake, ownership, own_polarity, enable, bytes_per_row);

    pie_register_odma_callback(pie_handle, pie_wdma_desc_isr, NULL);
    pie_enable_pogo_odma_irqs(pie_handle, NULL, true); // enable all odma interrupts
}

void pie_wdma_reset( void )
{
    int i;

    for( i=0 ; i<PIE_WDMA_NUM_CHANNELS ; i++ ) {
        pie_wdma_channel_reset(i);
    }
}

/**
 * \brief  Return true if all of PIE WDMA is idle
 *
 * Created to assist multiplex PIE DMAs
 *
 * \author David Poole
 * \date 28-May-2013
 */
bool pie_wdma_is_idle( void )
{
    return pie_odma_is_idle();
}

/**
 * \brief WDMA channel reset
 *
 * WDMA driver function to reset a wdma channel. Typically called during a cancel
 * operation.
 *
 * NOTE: users of this driver function (icedma) don't have pie handles, so we
 *       can't interact with hardware unless the pie driver provides the needed
 *       direct access functions (which can be done if needed, dma is 'special').
 */
void pie_wdma_channel_reset( uint8_t channel )
{
    // FIXME
    // Disabling IRQs for the specific channel would be nice. Because we know this
    // comes from a cancel, just club the entire PIE (once for each channel, doh).
    pie_do_reset();

#if 0
    volatile UDMA_REGS_t *udma;
    volatile POGO_ODMA_CORE_REGS_t *odma;

    XASSERT( channel<PIE_WDMA_NUM_CHANNELS, channel );

    /* davep 22-Feb-2011 ; shotgun debug for reset problems */
    pie_wdma_channel_enable(channel);

    get_oudma( channel, &udma );
    get_odma( channel, &odma );

    /* disable all interrupts */
    udma->UIER = 0;
    odma->OIER = 0;

    /* ack all stray interrupts */
    udma->UICR = ~0;
    odma->OICR = ~0;

    /* restore to power-on defaults */
    udma->UCR = 0;
    odma->OLWR = 0; 
    
//    /* davep 22-Feb-2011 ; shotgun debug for reset problems */
//    pie_wdma_channel_enable(channel);
    pie_wdma_channel_disable(channel);
#endif
}

/**
 * \brief WDMA channel start
 *
 * WDMA driver function to start a wdma channel
 *
 * NOTE: users of this driver function (icedma) don't have pie handles, so we
 *       can't interact with hardware unless the pie driver provides the needed
 *       direct access functions (which can be done if needed, dma is 'special').
 */
void pie_wdma_channel_start( uint8_t channel )
{
    struct pie_ddma_channel *pch;

    XASSERT( channel < PIE_WDMA_NUM_CHANNELS, channel );
    pch = &pie_ddma_write_channel[channel];

    XASSERT( pch->channel==channel, channel );
    XASSERT( pch->is_open, pch->channel );

    XASSERT( pch->num_running > 0, pch->num_running );
    XASSERT( pch->head_desc_virt != NULL, pch->channel );

    XASSERT( PIE_DESCRIPTOR_OWNED_BY_BLOCK(pch->head_desc_virt), pch->head_desc_virt->config_flags );

    // FIXME:
    // Do we need to check the USR busy bit before starting a new operation?
    // Old code used to check and ASSERT if already busy.

    //dbg2("%s WDMA Fire in the hole - channel=%d\n", __FUNCTION__, channel);

    pie_start_pogo_output_dma((uint32_t)pch->head_desc_virt->dma_ptr_self);
}

/**
 * \brief WDMA channel enable
 *
 * WDMA driver function to enable a wdma channel.
 *
 * NOTE: users of this driver function (icedma) don't have pie handles, so we
 *       can't interact with hardware unless the pie driver provides the needed
 *       direct access functions (which can be done if needed, dma is 'special').
 */
void pie_wdma_channel_enable( uint8_t channel )
{
    // FIXME
    // No pie handle here ... so can't enable the channel. Turns out that
    // this function isn't used, as channels are enabled as part of the initial
    // setup process for the scan.
#if 0
    volatile UDMA_REGS_t *udma;
    volatile POGO_ODMA_CORE_REGS_t *odma;

    XASSERT( channel<PIE_WDMA_NUM_CHANNELS, channel );

    get_oudma( channel, &udma );
    get_odma( channel, &odma );

    udma->UCR = UDMA_UCR_ENABLE_REPLACE_VAL( pie_odma_udma_registers->UCR, 1 );

    /* turn on (almost) all interrupts. EOS (End of Strip) apparently exactly
     * like UIER DESC (descriptor complete) so don't want the dup
     */
    udma->UIER = ~0;
    odma->OIER = ~(POGO_ODMA_CORE_OIER_EOS_MASK);
#endif
}

/**
 * \brief Is WDMA channel enabled?
 *
 * WDMA driver function to see if a channel is enabled. Typically used
 * for sanity checks when launching or closing channels.
 */
bool pie_wdma_channel_is_enabled( uint8_t channel )
{
    // FIXME: we only have one wdma channel ... what channel is passed in?
    return pie_odma_is_enabled();
}

/**
 * \brief WDMA channel disable
 *
 * WDMA driver function to disable a wdma channel. Typically called
 * during a cancel operation.
 *
 * NOTE: users of this driver function (icedma) don't have pie handles, so we
 *       can't interact with hardware unless the pie driver provides the needed
 *       direct access functions (which can be done if needed, dma is 'special').
 */
void pie_wdma_channel_disable( uint8_t channel )
{
    // FIXME
    // No pie handle here ... so can't disable the channel.  This is usually called
    // when icedma does a cancel (reset followed by disable). We clubbed pie in the
    // reset call, which should leave all channels disabled.
#if 0
    volatile UDMA_REGS_t *udma;

    XASSERT( channel<PIE_WDMA_NUM_CHANNELS, channel );

    get_oudma( channel, &udma );
    udma->UCR = UDMA_UCR_ENABLE_REPLACE_VAL( udma->UCR, 0 );
#endif
}

/**
 * \brief WDMA channel add buffer
 *
 * WDMA driver function to add a buffer to a wdma channel.
 *
 * NOTE: users of this driver function (icedma) don't have pie handles, so we
 *       can't interact with hardware unless the pie driver provides the needed
 *       direct access functions (which can be done if needed, dma is 'special').
 */
void pie_wdma_channel_load( uint8_t channel, uint8_t *data_ptr, dma_addr_t dma_dest, 
                            uint32_t rows, uint32_t bytes_per_row )
{
    struct pie_ddma_channel *pch;

//    dbg2( "%s %d %#x %d %d\n", __FUNCTION__, channel, dma_dest, rows, bytes_per_row );

    XASSERT( channel < PIE_WDMA_NUM_CHANNELS, channel );
    pch = &pie_ddma_write_channel[ channel ];
    XASSERT( pch->channel==channel, channel );

    pie_ddma_channel_load2( pch, (uint32_t)dma_dest, rows, 0, 0, bytes_per_row );

    XASSERT( PIE_DESCRIPTOR_OWNED_BY_BLOCK(pch->head_desc_virt), pch->head_desc_virt->config_flags );

    // FIXME/REVISIT
    // These values should NOT change during a scan, now set in pie_wdma_setup_config.
#if 0
    /* ODMA wants bytes_per_row so let's hope bytes_per_row stays constant
     * across the entire image 
     */
    get_odma( channel, &odma );
    odma->OLWR = bytes_per_row;
#endif
}

void pie_wdma_dump( void )
{
    int channel;

    // FIXME
    // May want to get some more focused dump functions from the pie driver, this dumps
    // it all. This method used to do core and wdma only.
    dump_pie_regs(PIE_RDMA_NUM_CHANNELS);

    /* davep 05-May-2011 ; XXX temp debug */
    for( channel=0 ; channel<PIE_WDMA_NUM_CHANNELS ; channel++ ) {
        if( pie_ddma_write_channel[channel].head_desc_virt ) {
            pie_dma_desc_chain_dump( &pie_ddma_write_channel[channel] );
        }
    }
}

const struct ice_dma_driver * pie_wdma_get_driver( void )
{
    return &pie_wdma_desc_driver;
}

void pie_wdma_set_icebuf_isr( iceisr_f icebuf_isr_f )
{
    pie_wdma_desc_driver.icebuf_isr = icebuf_isr_f;
}

scan_err_t pie_wdma_channel_open( uint8_t channel )
{
    XASSERT( channel < PIE_WDMA_NUM_CHANNELS, channel );

    return pie_ddma_channel_open( channel, &pie_ddma_write_channel[channel], "pie_wdma" );
}

void pie_wdma_channel_close( uint8_t channel )
{
    XASSERT( channel < PIE_WDMA_NUM_CHANNELS, channel );
    //XASSERT( !pie_wdma_channel_is_enabled(channel), channel );

    pie_ddma_channel_close( &pie_ddma_write_channel[channel] );
}

