/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2006-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * \file pie.h
 *
 * \brief Header file for PIE and HalfPack interface functions (DMA, etc).
 *
 */

#ifndef PIE_H
#define PIE_H

/* 6-Mar-05 ; these might more properly live in piehw.h but I need them here in
* pie.h for a lot of data structures.
*/
#define PIE_RDMA_NUM_CHANNELS  3
#define PIE_WDMA_NUM_CHANNELS  1

/* 13-Apr-05 ; PIE Mono HalfPack doesn't really have the concept of channels
 * so it only has one.
 */
#define PIEHP_RDMA_NUM_CHANNELS  1
#define PIEHP_WDMA_NUM_CHANNELS  1

typedef enum {
    HP_DMA_DIR_FORWARD,
    HP_DMA_DIR_REVERSE
} HP_DMA_DIR;

typedef enum {
    HP_BIAS_CONFIG_1,
    HP_BIAS_CONFIG_8
} HP_BIAS_CONFIG;

struct pie_interrupt_stats {
    uint32_t count;

    /* davep 18-Feb-2011 ; changed these arrays from using #def to a hard wired
     * big magic number. needed to move the PIE_[RW]_NUM_CHANNELS into piehw.h
     */
    /* PIE read/write DMA */
    uint32_t rdma[32]; /* +=1 for every interrupt on this channel */
    uint32_t wdma[32]; /* +=1 for every interrupt on this channel */

    uint32_t rdma_last_drr[32];
    uint32_t wdma_last_drr[32];

    /* Mono HalfPack read/write DMA */
    uint32_t hp_rdma;
    uint32_t hp_wdma;

    uint32_t hp_rdma_last_drr;
    uint32_t hp_wdma_last_drr;
};

/** Convenience structure holding row information. Used to balance the DMA
 * requirements of PIE and HalfPack.
 *
 *
 * See also pie_calc_row() and piehp_calc_row() 
 */

struct pie_dma_row {
    int pixels_in;
    int pixels_out;
    int bytes_in;
    int bytes_out;
};

/** convenience structure holding PIE XYScale scaling information. The scaler
 * values are passed as two fractions, an X scale and a Y scale. For example,
 * to scale down 50%, XYScale needs X=1/2 and Y=1/2.
 */
struct pie_scale {
    int x_numerator;
    int x_denominator;
    int y_numerator;
    int y_denominator;

    /* xyscaler hardware values */
    int x_scale;
    int y_scale;
};

/* davep 09-Jun-2011 ; centralizing a collection of hodge-podge strip size
 * calculations
 */
struct pie_strip {
    bool use_cs;
    bool use_mf;
    bool use_xy;

    int rdma_in; /* input dma strip size */

    int cs_in;   /* colorshift; eats two lines */
    int cs_out;

    int mf_in;   /* multifilter; eats two or four lines */
    int mf_out; 

    int xy_in;   /* very complicated */
    int xy_out;   

    int wdma_out; /* output dma strip size */

    int total_extra_rows; /* cs+mf */
};

/* function prototypes */

/* functions in pie.c */
void pie_reset(void);
scan_err_t pie_onetime_init(void);
scan_err_t pie_soft_setup(void);
int pie_get_bits_per_pixel(pie_pixel_t ppt);
int pie_get_bytes_per_pixel(pie_pixel_t ppt);

scan_err_t pie_calc_stripsize(struct pie_strip *strip, int fracY, int scaler_mode);
void pie_pixel_type_to_scan_data_type(pie_pixel_t ppt, scan_data_type *sdt);

void pie_dbg2_strip(struct pie_strip *strip);
void pie_dbg2_scale(struct pie_scale *scale);
void pie_dbg2_row(struct pie_dma_row *row);

/* Color Space Convert (CSC) */
void pie_csc_setup(struct pie_handle_t *pie_handle, uint32_t sensor_num);

/* MultiFilter (MF) */
void pie_filter_setup(struct pie_handle_t *pie_handle, const struct dsmf_settings *dsmf);

/* XYScale */
#define PIE_XY_SCALER_BYPASS 0
#define PIE_XY_SCALER_RAPR   1
#define PIE_XY_SCALER_LINEAR 2

/* smallest our current HW can support is 1/16. Largest is 8/1 */
#define PIE_XY_SMALLEST_SCALE  0.0625
#define PIE_XY_LARGEST_SCALE    8

void pie_xy_calc_row( int pixels_per_row, 
                      uint32_t pixel_size, 
                      pie_pixel_t pie_output_type,
                      struct pie_scale *scale,
                      int extra_pad,
                      int scaler_mode,
                      struct pie_dma_row *row );
void pie_xy_scale(struct pie_handle_t *pie_handle,
                  struct pie_scale *scale,
                  int pixels_in,
                  int strip_size_in,
                  int total_rows_in,
                  int extra_pad,
                  int scaler_mode,
                  int *pixels_out,
                  int strip_size_out,
                  int *total_rows_out);
void pie_xy_calc_expected_rows(int rows_in, int fracY, int scaler_mode, int *rows_out);
void pie_xy_calc_expected_pixels(int pixels_in, int fracY, int scaler_mode, int *pixels_out);
void pie_xy_calc_xyscale(struct pie_scale *scale, int scaler_mode);
scan_err_t pie_xy_calc_stripsize(int *read_stripsize, int fracY, 
                                 int scaler_mode, int *write_stripsize);
uint32_t pie_xy_calc_linear_mode_edge_pixels(uint32_t numerator, 
                                             uint32_t denominator);

uint32_t pie_interrupt_disable(void);
void pie_interrupt_enable(void);

/* 11-June-05 davep ; run-time swappable DMA driver */
#ifdef ICEDMA_H
/* include icedma.h before pie.h to get access to these functions 
 * XXX - this is a kludge; need to split up pie.h into pie.h and piedma.h
 */
const struct ice_dma_driver *pie_rdma_get_driver(void);
void pie_rdma_set_icebuf_isr(iceisr_f icebuf_isr_f);

const struct ice_dma_driver * pie_wdma_get_driver(void);
void pie_wdma_set_icebuf_isr(iceisr_f icebuf_isr_f);
#endif

/* read DMA - piedma.c */
void pie_rdma_setup_config(struct pie_handle_t *pie_handle, 
                           uint8_t      num_idma_channels,
                           uint8_t      pie_bpp, 
                           scan_cmode_t cmode, 
                           pie_pixel_t  piein, 
                           uint32_t     strip_numrows,
                           uint32_t     bytes_per_row);

bool pie_rdma_is_idle(void);
scan_err_t pie_rdma_channel_open(uint8_t channel);
void pie_rdma_channel_close(uint8_t channel);
void pie_rdma_channel_reset(uint8_t channel);
void pie_rdma_channel_start(uint8_t channel);
void pie_rdma_channel_enable(uint8_t channel);
bool pie_rdma_channel_is_enabled(uint8_t channel);
void pie_rdma_channel_disable(uint8_t channel);
void pie_rdma_channel_load(uint8_t channel, uint8_t *data_ptr, dma_addr_t dma_dest, 
                           uint32_t rows, uint32_t bytes_per_row);
void pie_rdma_channel_load2(uint8_t channel, uint32_t dar1, uint32_t drcr1, 
                            uint32_t dar2, uint32_t drcr2, uint32_t bytes_per_row);
void pie_rdma_dump(void);


////////// SANDRA FIXME, MOVE THESE TO pie_if.h????
/* Read DMA channel assignments */

/* From the PIE Read DMA MA:
 *
 * "In monochrome mode, only data in channel 0 is used in non-staggered mode
 * and the channel 0 and channel 1 in staggered mode. The other channels are
 * ignored and bogus data is handed to the pipe for the upper bits of the
 * output data."
 */
#define PIE_RDMA_MONO_EVEN 0
#define PIE_RDMA_MONO_ODD  1

/* "In staggered color mode, the data from Channels 0, 2 and 4 are interleaved
 * with the data on channels 1, 3 and 5 on a column basis. In monochrome mode,
 * the data for channels 0 and 1 are interleaved.  In non-staggered color mode,
 * the data is fetched from channels 0, 2 and 4. For non-staggered monochrome
 * mode, the data is taken from channel 0 only."
 */
#if defined HAVE_STAGGERED_SENSOR_SUPPORT
    #define PIE_RDMA_RED_EVEN   0
    #define PIE_RDMA_GREEN_EVEN 2
    #define PIE_RDMA_BLUE_EVEN  4
    #define PIE_RDMA_RED_ODD    1
    #define PIE_RDMA_GREEN_ODD  3 
    #define PIE_RDMA_BLUE_ODD   5
#else
    #if PIE_RDMA_NUM_CHANNELS==6
        /* davep 29-Mar-2010 ; as of this writing, PIE channels are still 0,2,4
         * in non-staggered ASICs. We'll need "holes" in our map.
         */
        #define PIE_RDMA_RED_EVEN   0
        #define PIE_RDMA_GREEN_EVEN 2
        #define PIE_RDMA_BLUE_EVEN  4
    #else 
        /* davep 22-Mar-2011 ; finally a PIE with channels 1,2,3 */
        #define PIE_RDMA_RED_EVEN   0
        #define PIE_RDMA_GREEN_EVEN 1
        #define PIE_RDMA_BLUE_EVEN  2
    #endif
#endif

/* write DMA - piedma.c */
void pie_wdma_setup_config(struct pie_handle_t *pie_handle,
                           scan_cmode_t cmode, 
                           pie_pixel_t  pieout, 
                           uint32_t     strip_numrows,
                           uint32_t     bytes_per_row);
bool pie_wdma_is_idle(void);
void pie_wdma_reset(void);
scan_err_t pie_wdma_channel_open(uint8_t channel);
void pie_wdma_channel_close(uint8_t channel);
void pie_wdma_channel_reset(uint8_t channel);
void pie_wdma_channel_start(uint8_t channel);
void pie_wdma_channel_enable(uint8_t channel);
bool pie_wdma_channel_is_enabled(uint8_t channel);
void pie_wdma_channel_disable(uint8_t channel);
void pie_wdma_channel_load(uint8_t channel, uint8_t *data_ptr, dma_addr_t dma_dest, 
                           uint32_t rows, uint32_t bytes_per_row);
void pie_wdma_dump(void);

void pie_wdma_debug_log(void);


////////// SANDRA FIXME, MOVE THESE TO pie_if.h????
/* Write DMA channel assignments */

/* xRGB output all comes out channel 0 */
#define PIE_WDMA_CHANNEL_xRGB  0

/* RGBx output also all comes out channel 0 */
#define PIE_WDMA_CHANNEL_RGBx  0

/* As you might have guessed, RGB output also all comes out channel 0 */
#define PIE_WDMA_CHANNEL_RGB   0

/* mono output comes out channel 0 */
#define PIE_WDMA_CHANNEL_MONO  0

/* red,green,blue come out separate channels */
#define PIE_WDMA_CHANNEL_RED   0
#define PIE_WDMA_CHANNEL_GREEN 1
#define PIE_WDMA_CHANNEL_BLUE  2

//////////////  WARNING - NEEDED HACKS from PIEHW.H just to compile - GET THESE REMOVED ASAP FIXME!!
//#define PIE_FILTER_LUT_SIZE 64
// FIXME - HACK - put PIE_FILTER_LUT_SIZE HERE AS A STOPGAP - all
// the filter code needs rewritten for the new asic
#define PIE_TCNS_LUT_SIZE  1024
#define PIE_TCNS_LUT_Y(x)         (((x) & 0xff) << 22)
#define PIE_TCNS_LUT_CR(x)        (((x) & 0xff) << 11)
#define PIE_TCNS_LUT_CB(x)        (((x) & 0xff) << 0)
#define PIE_TCNS_TUPLE(y,cb,cr)  (PIE_TCNS_LUT_Y(y) | PIE_TCNS_LUT_CB(cb) | PIE_TCNS_LUT_CR(cr))
#define PIE_TCNS_LUT_Y_SHIFT  22
#define PIE_TCNS_LUT_CR_SHIFT 11
#define PIE_TCNS_LUT_CB_SHIFT 0
#define PIE_RDMA_DSR_CMPL         (0x1 << 31)
#define PIE_RDMA_DSR_READY        (0x1 << 1)
#define PIE_RDMA_DSR_ACTIVE       (0x1 << 0)
//#define PIE_CSC_RGB_LUT_SIZE 257 /* number of 32-bit values (not bytes) */
                                 /* 257 is NOT a typo */
#define PIE_RDMA_CHANNEL_OFFSET 0x80
#define PIE_WDMA_DSR_CMPL         (0x1 << 31)
#define PIE_WDMA_DSR_READY        (0x1 << 1)
#define PIE_WDMA_DSR_ACTIVE       (0x1 << 0)

#define PIE_RDMA_DCR_CIE          (0x1 << 31)
#define PIE_RDMA_DCR_BURST_32     (0x1 << 4)
#define PIE_RDMA_DCR_RESET        (0x1 << 1)
#define PIE_RDMA_DCR_EN           (0x1 << 0)

// Mono HalfPack block - piehp.c 
void piehp_reset( void );
void piehp_dump( void );
// HalfPack long gone. Only survives as a firmware
// halftoner used with Fax. Time to remove.  

#define INT_DISABLE_HALFPACK  
#define INT_ENABLE_HALFPACK   
//#define INT_DISABLE_HALFPACK  intDisable(INTNUM_PIE)
//#define INT_ENABLE_HALFPACK   intEnable(INTNUM_PIE)

void piehp_rdma_reset( uint8_t channel );
void piehp_rdma_start( uint8_t channel );
void piehp_rdma_enable( uint8_t channel );
void piehp_rdma_disable( uint8_t channel );
void piehp_rdma_dump( void );
void piehp_rdma_load( uint8_t channnel, uint8_t *data_ptr, dma_addr_t dma_dest, 
                        uint32_t lines, uint32_t bytes_per_line );

void piehp_wdma_reset( uint8_t channel );
void piehp_wdma_start( uint8_t channel );
void piehp_wdma_enable( uint8_t channel );
void piehp_wdma_disable( uint8_t channel );
void piehp_wdma_dump( void );
void piehp_wdma_load( uint8_t channel, uint8_t *data_ptr, dma_addr_t dma_dest, 
                        uint32_t lines, uint32_t bytes_per_line );

void piehp_set_size_sram(uint32_t *size_values, int num_values );
void piehp_set_signed_bias_sram(uint32_t *signed_bias_values, int num_values );
void piehp_set_density_sram(uint32_t *density_values, int num_values);
void piehp_set_error_diffusion_sram(uint32_t *error_diff_values, int num_values );
void piehp_set_szdd_sram( uint32_t *size_values, uint32_t *error_diff_values, 
                          uint32_t *density_values, int num_values );
void piehp_set_error_diffusion_mode( uint32_t mode );
void piehp_set_wdma_direction(HP_DMA_DIR dir);
void piehp_set_rdma_direction(HP_DMA_DIR dir);
void piehp_set_start_of_image( bool enable );
void piehp_set_replicate_vert_data(bool replicate);
void piehp_set_replicate_horiz_data(bool replicate);
void piehp_set_bit_depth(int depth);
void piehp_set_bypass(bool bypass);
void piehp_set_invert_data(bool invert);
void piehp_set_include_accumerror_idx(bool include);
void piehp_set_include_signed_bias_idx(bool include);
void piehp_set_include_signed_bias_do_a_dot(bool include);
void piehp_set_do_a_dot_override(bool do_a_dot);
void piehp_set_shift(uint8_t shift);
void piehp_set_error_weight_1(uint8_t weight);
void piehp_set_error_weight_2(uint8_t weight);
void piehp_set_error_weight_3(uint8_t weight);
void piehp_set_bias_sram_config(HP_BIAS_CONFIG config);
void piehp_set_bias_reg0(uint8_t value);
void piehp_set_bias_reg1(uint8_t value);
void piehp_calc_row( int pixels_per_row_in,
                     int bpp, bool horiz_rep,
                     struct pie_dma_row *row );

/* 8-Apr-05 ; new ICE DMA driver */

/* forward declaration for some prototypes below */
struct ice_dma_buffer;

/* 
 * PIE read DMA 
 */
void pie_rdma_init(void);
int pie_rdma_open(uint8_t channels[], uint8_t num_channels,
                  int num_empties, int total_rows,
                  int bytes_per_row);
scan_err_t pie_rdma_add_buffer(uint8_t channel, uint8_t *data, uint32_t datalen,
                               uint32_t rows, uint32_t bytes_per_row);
void pie_rdma_add_ready(uint8_t channel, struct ice_dma_buffer **addme);
void pie_rdma_channel_launch(uint8_t channel);
void pie_rdma_sanity(void);
void pie_rdma_debug_log(void);
void pie_rdma_isr(uint8_t channel);
void pie_rdma_close(void);

/* 
 * PIE write DMA
 */
void pie_wdma_init(void);
int pie_wdma_open(uint8_t channels[], uint8_t num_channels,
                  int num_empties, int total_rows,
                  int bytes_per_row);
scan_err_t pie_wdma_add_buffer(uint8_t channel, uint8_t *data, uint32_t datalen,
                               uint32_t rows, uint32_t bytes_per_row);
void pie_wdma_add_ready(uint8_t channel, struct ice_dma_buffer **addme);
void pie_wdma_channel_launch(uint8_t channel);
void pie_wdma_sanity(void);
void pie_wdma_debug_log(void);
void pie_wdma_isr(uint8_t channel);
void pie_wdma_close(void);
void pie_wdma_cancel( void );

int pie_wdma_setup_buffers(uint8_t channel, int num_buffers, 
                           int rows_per_buffer, int bytes_per_row);


/* PIEDMA_FRIEND is for testing code which needs to be able to see the insides
 * of the structures; we don't have a ridiculous proliferation of small utility
 * functions
 */
#ifdef PIEDMA_FRIEND
struct ice_dma_mm *pie_rdma_get_mm( void );
struct ice_dma_mm *pie_wdma_get_mm( void );
#endif

/** PIE Mono HalfPack prototypes **/

/* 
 * PIE Mono HalfPack read DMA 
 */
void piehp_rdma_init( void );
int piehp_rdma_open( uint8_t channels[], uint8_t num_channels,
               int num_empties, int total_rows,
               int bytes_per_row );
scan_err_t piehp_rdma_add_buffer( uint8_t channel, uint8_t *data, uint32_t datalen,
                     uint32_t rows, uint32_t bytes_per_row);
void piehp_rdma_add_ready( uint8_t channel, struct ice_dma_buffer **addme);
void piehp_rdma_free_empty( uint8_t channel, struct ice_dma_buffer **freeme );
void piehp_rdma_channel_launch( uint8_t channel );
void piehp_rdma_sanity( void );
void piehp_rdma_debug_log( void );
void piehp_rdma_isr( uint8_t channel );
void piehp_rdma_close( void );
void piehp_rdma_cancel( void );
void piehp_rdma_run_forever( void );

/* 
 * PIE Mono HalfPack write DMA
 */
void piehp_wdma_init( void );
int piehp_wdma_open( uint8_t channels[], uint8_t num_channels,
               int num_empties, int total_rows,
               int bytes_per_row );
scan_err_t piehp_wdma_add_buffer( uint8_t channel, uint8_t *data, uint32_t datalen,
                     uint32_t rows, uint32_t bytes_per_row);
void piehp_wdma_add_ready( uint8_t channel, struct ice_dma_buffer **addme);
void piehp_wdma_free_empty( uint8_t channel, struct ice_dma_buffer **freeme );
void piehp_wdma_channel_launch( uint8_t channel );
void piehp_wdma_sanity( void );
void piehp_wdma_debug_log( void );
void piehp_wdma_isr( uint8_t channel );
void piehp_wdma_close( void );
void piehp_wdma_refill( int rows_per_buffer );
void piehp_wdma_cancel( void );
void piehp_wdma_run_forever( void );

int piehp_wdma_setup_buffers( uint8_t channel, int num_buffers, 
                               int rows_per_buffer, int bytes_per_row );

/* PIEDMA_FRIEND is for testing code which needs to be able to see the insides
 * of the structures; we don't have a ridiculous proliferation of small utility
 * functions
 */
#ifdef PIEDMA_FRIEND
struct ice_dma_mm *piehp_rdma_get_mm( void );
struct ice_dma_mm *piehp_wdma_get_mm( void );
#endif

/* davep 24-May-2013 ; adding capability to multiplex pie */
void piems_dma_set_bank(uint8_t banknum);
scan_err_t piems_dma_open_bank(uint8_t banknum);
void piems_dma_close_bank(uint8_t banknum);
void piems_dma_stall(void);
void piems_dma_reload(void);

/* 10-June-05 ; PIE MultiStrip Read DMA (when in doubt, use the single strip
 * version above)
 */
void piems_rdma_init(void);
int piems_rdma_open(uint8_t channels[], 
                    uint8_t num_channels,
                    int num_empties, 
                    int total_rows,
                    int bytes_per_row,
                    int strip_size);
scan_err_t piems_rdma_add_buffer(uint8_t channel, uint8_t *data, uint32_t datalen,
                                 uint32_t rows, uint32_t bytes_per_row);
void piems_rdma_add_ready(uint8_t channel, struct ice_dma_buffer **addme);
void piems_rdma_free_empty(uint8_t channel, struct ice_dma_buffer **freeme);
void piems_rdma_channel_launch(uint8_t channel);
void piems_rdma_launch(void);
void piems_rdma_sanity(void);
void piems_rdma_debug_log(void);
void piems_rdma_isr(uint8_t channel);
void piems_rdma_close(void);
void piems_rdma_cancel(void);
/* the following functions are unique to the PIE multistrip DMA driver */
void piems_rdma_init_scaler(int scaler_strip_size, 
                            int y_numerator, 
                            int y_denominator, 
                            int fracY);
void piems_rdma_intmsg(uint8_t channel);
void piems_rdma_set_image_offsets(int image_offsets[]);
void piems_rdma_init_mf(void);
void piems_rdma_init_colorshift(void);
void piems_rdma_run_forever(void);
bool piems_rdma_is_idle(void);
bool piems_rdma_is_empty(void);


/* 
 * PIE MultiStrip write DMA
 */
void piems_wdma_isr(uint8_t channel);
void piems_wdma_init(void);
int piems_wdma_open(uint8_t channels[], uint8_t num_channels,
                    int num_empties, 
                    int total_rows,
                    int bytes_per_row,
                    int strip_size);
scan_err_t piems_wdma_add_buffer(uint8_t channel, uint8_t *data, uint32_t datalen,
                                 uint32_t rows, uint32_t bytes_per_row);
void piems_wdma_add_ready(uint8_t channel, struct ice_dma_buffer **addme);
void piems_wdma_free_empty(uint8_t channel, struct ice_dma_buffer **freeme);
void piems_wdma_channel_launch(uint8_t channel);
void piems_wdma_launch(void);
void piems_wdma_sanity(void);
void piems_wdma_close(void);
void piems_wdma_cancel(void);
/* the following functions are unique to the PIE multistrip DMA driver */
void piems_wdma_init_scaler(int scaler_strip_size, 
                            int y_numerator, 
                            int y_denominator, 
                            int fracY);
void piems_wdma_intmsg(uint8_t channel);
void piems_wdma_refill(int rows_per_buffer);
void piems_wdma_init_colorshift(void);
void piems_wdma_run_forever(void);

int piems_wdma_setup_buffers(uint8_t channel, int num_buffers, 
                             int rows_per_buffer, int bytes_per_row);
bool piems_wdma_is_idle(void);
bool piems_wdma_is_empty(void);


/* 13-June-05 ; shotgun debug. Having a hard crash when leaving the channel
 * launch function. If I only load one DMA transaction into the system at a
 * time, it doesn't happen. I'm fiddling with the number of strips I'm leaving
 * queued thinking it has to do with how many transactions I have running at
 * a time and the fact that I'm queuing the hardware instead of my own internal
 * counter to determine that number.
 *
 * Do not set this less than three! Three shall the number you shall count and the
 * number you should count shall be three. Do not set to two. Zero is right
 * out.
 */
#define PIEMS_DMA_FLUSH_THRESHOLD  3


/* PIEDMA_FRIEND is for testing code which needs to be able to see the insides
 * of the structures; we don't have a ridiculous proliferation of small utility
 * functions
 */
#ifdef PIEDMA_FRIEND
struct ice_dma_mm *piems_rdma_get_mm( void );
struct ice_dma_mm *piems_wdma_get_mm( void );
#endif

#endif /* PIE_H */

