/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2010-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



/**
 * \file picpcdma_descrip.c
 *
 * \brief  Descriptor based PIC PC (Pixel Correction, aka PRNU/DSNU) LUT DMA
 *
 * Also known as "IDMA 2D DMA for PRNU/DSNU"
 *
 * davep 06-Aug-2010
 *
 **/

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "scos.h"

#include "hwconfig_api.h"
#include "cpu_api.h"
#include "lassert.h"
#include "regAddrs.h"
#include "debug.h"
#include "list.h"

#include "scantypes.h"
#include "scancore.h"
#include "scandbg.h"
#include "safetylock.h"
#include "icedma.h"
#include "ddma.h"

#include "pic_handle_if.h"
#include "pic_if.h"
#include "pic_convenience_if.h"

#include "pic.h"
#include "picdma_descrip.h"
#include "scansen.h"

/* PIC WDMA and PIC PC LUT DMA use the same descriptor format so borrow that
 * data struture. We only need one descriptor because the PC LUT DMA simply
 * repeats
 */
// created descriptor array for nsensor
static struct ddma_descriptor *idma2d_desc_array[SCANSEN_MAX_SENSORS];
//static struct ddma_descriptor *idma2d_desc = NULL; // cause error if block not properly set
//static int current_block_num=0; // icky, but we need to let callers restore the block after resets

void pic_prnudsnu_dma_onetime_init( uint8_t pic_instance )
{
    int scerr;

    if( idma2d_desc_array[pic_instance]==NULL ) {
        scerr = ddma_descriptor_alloc( &idma2d_desc_array[pic_instance] );
        if( scerr!=SCANERR_NONE ) {
            dbg2( "%s unable to alloc prnudsnu descriptor scerr=%d\n", 
                  __FUNCTION__, scerr );
            XASSERT(0,scerr);
            return;
        }
    }
}

void pic_prnudsnu_dma_dump( uint8_t pic_instance )
{
    pic_pd_idma_dump( pic_instance );
    
    ddma_desc_dump( idma2d_desc_array[ pic_instance ] );
}

void pic_prnudsnu_set_lut_addr( uint8_t pic_instance, dma_addr_t lut_addr )
{
    /* make sure pointer is correctly longword (32-bit) aligned */
    XASSERT( ((uint32_t)lut_addr & ~ICE_DMA_BYTE_ALIGN_MASK) == 0, (uint32_t)lut_addr );

    idma2d_desc_array[ pic_instance ]->src_addr = (uint32_t)lut_addr;
    idma2d_desc_array[ pic_instance ]->fw_next_descriptor_addr = idma2d_desc_array[ pic_instance ];
    idma2d_desc_array[ pic_instance ]->config_flags |= DDMA_DESCRIPTOR_CONFIG_OWNER_BLOCK;
}

void pic_prnudsnu_set_lut_xfer_count( uint8_t pic_instance, struct pic_handle_t *pic_handle, unsigned long byte_count )
{
    uint8_t repeat_forever, repeat_line_cnt;
    uint32_t line_width;

    idma2d_desc_array[ pic_instance ]->transfer_len_bytes = byte_count;

    /* set to repeat forever */
    pic_pd_idma2d_get_line_width(pic_handle, &repeat_forever, &repeat_line_cnt, &line_width);
    repeat_forever = 1;
    line_width     = byte_count;
    pic_pd_idma2d_set_line_width(pic_handle, repeat_forever, repeat_line_cnt, line_width);
}

void pic_prnudsnu_start_lut_dma(uint8_t pic_instance)
{
    uint32_t desc;

//    pic_prnudsnu_dump();
//    pic_prnudsnu_dma_dump();
//    pic_prnudsnu_desc_dump();

    /* davep 08-Jan-2013 ; turn on error interrupt(s) */
    // NOTE: turns out that the scanpipe code does not actually look at any
    //       interrupts coming from the IDMA_2D block. Guess the idea was to
    //       enable 'who' to detect errors.  No point in turning it on ...
    //memset(&irqs, 0, sizeof(struct pic_idma2d_interrupt_info));
    //irqs.who = true;
    //pic_pd_idma2d_enable_irqs(pic_instance, &irqs);

    ddma_desc_flush( idma2d_desc_array[ pic_instance ] );

    desc = idma2d_desc_array[ pic_instance ]->dma_ptr_self;
     
    /* fire in the hole! */    
    pic_start_idma2d_dma(pic_instance, desc);
    
//    pic_prnudsnu_dump();
//    pic_prnudsnu_dma_dump();
//    pic_prnudsnu_desc_dump();
}

unsigned long pic_prnudsnu_get_lut_xfer_count( uint8_t pic_instance )
{
    return idma2d_desc_array[ pic_instance ]->transfer_len_bytes;
}

