/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2011-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



/**
 * \file picmarg.c
 *
 * \brief PIC Margins for ASICs with margins outside the Bulb Montitor block.
 *
 **/

#include <stdint.h>
#include <stdbool.h>

#include "lassert.h"
#include "regAddrs.h"

#include "scancore.h"
#include "scantypes.h"
#include "scandbg.h"

#include "pic_handle_if.h"
#include "pic_if.h"
#include "pic_convenience_if.h"

#include "pic.h"
#include "picreset.h"


void pic_marg_reset(struct pic_handle_t *pic_handle)
{
    int i;

    // first set the top margin block, then the bottom margin block
    for (i=0; i<PIC_LRMARGIN_NUM_BLOCKS; i++)
    {
        // bypass lrmargin block
        pic_lrmargin_set_cfg(pic_handle, i, true);
        // set margins to 0
        pic_lrmargin_set_margin(pic_handle, i, 0, 0, 0);  // color 0
        pic_lrmargin_set_margin(pic_handle, i, 0, 0, 1);  // color 1
        pic_lrmargin_set_margin(pic_handle, i, 0, 0, 2);  // color 2
    }
}

void pic_marg_dump(uint8_t pic_instance, uint8_t lrmargin_block)
{
    pic_lrmargin_dump(pic_instance, lrmargin_block);
}

void pic_marg_set_bypass(struct pic_handle_t *pic_handle, bool bypass, uint8_t lrmargin_block)
{
    pic_lrmargin_set_cfg(pic_handle, lrmargin_block, bypass);
}

bool pic_marg_get_bypass( struct pic_handle_t *pic_handle, uint8_t lrmargin_block )
{
    bool bypass_val;

    pic_lrmargin_get_cfg(pic_handle, lrmargin_block, &bypass_val);
    return bypass_val;
}

uint32_t pic_marg_get_right( struct pic_handle_t *pic_handle, uint8_t lrmargin_block )
{
    uint16_t r0, l0, r1, l1, r2, l2;

    pic_lrmargin_get_margin(pic_handle, lrmargin_block, &r0, &l0, 0);  // color 0
    pic_lrmargin_get_margin(pic_handle, lrmargin_block, &r1, &l1, 1);  // color 1
    pic_lrmargin_get_margin(pic_handle, lrmargin_block, &r2, &l2, 2);  // color 2

    /* note: assuming all three margins set to the same values (see
     * pic_marg_set_xxx())
     */
    XASSERT(r0 == r1, r1);
    XASSERT(r1 == r2, r2);

    return r0;
}

uint32_t pic_marg_get_left( struct pic_handle_t *pic_handle, uint8_t lrmargin_block )
{
    uint16_t r0, l0, r1, l1, r2, l2;

    pic_lrmargin_get_margin(pic_handle, lrmargin_block, &r0, &l0, 0);  // color 0
    pic_lrmargin_get_margin(pic_handle, lrmargin_block, &r1, &l1, 1);  // color 1
    pic_lrmargin_get_margin(pic_handle, lrmargin_block, &r2, &l2, 2);  // color 2

    /* note: assuming all three margins set to the same values (see
     * pic_marg_set_xxx())
     */
    XASSERT(l0 == l1, l1);
    XASSERT(l1 == l2, l2);

    return l0;
}

/**
 * \brief  set all PIC margins' left position
 *
 * Sets all channels' margins to ****the same**** value.
 *
 * NOTE! PIC Margins work in PIXELS, not BYTES!  If you're using 16-bit pixels,
 * the margins will be 2x the size of the PIC WDMA settings. 
 *
 * Remember: Margins work on PIXELS.  PIC WDMA works on BYTES.
 *
 *
 * The Left margin is the first pixel lrmargin will pass, counting from zero.  So if
 * you want to trim the first 8 pixels, set the left margin to 8.  When the
 * pixel counter hits 8, the pixels will pass.
 *
 * The Right margin is the LAST pixel lrmargin will pass, counting from zero. 
 *
 * (I'm pretty sure this is how it works. 
 * It isn't entirely clear.)
 *
 * \author David Poole
 * \date 23-Jan-2008
 *
 *  davep 18-Jan-2011 ; adding support for new margin block
 *
 */

void pic_marg_set_left(struct pic_handle_t *pic_handle, uint32_t left, uint8_t lrmargin_block)
{
    uint16_t r0, r1, r2, lmargin;
    
    XASSERT(left<=65535, left);

    pic_lrmargin_get_margin(pic_handle, lrmargin_block, &r0, &lmargin, 0);
    pic_lrmargin_get_margin(pic_handle, lrmargin_block, &r1, &lmargin, 1);
    pic_lrmargin_get_margin(pic_handle, lrmargin_block, &r2, &lmargin, 2);

    pic_lrmargin_set_margin(pic_handle, lrmargin_block, r0, left, 0);  // color 0
    pic_lrmargin_set_margin(pic_handle, lrmargin_block, r1, left, 1);  // color 1
    pic_lrmargin_set_margin(pic_handle, lrmargin_block, r2, left, 2);  // color 2
}

/**
 * \brief  set all PIC margins' right position
 *
 * Sets all channels' margins to ****the same**** value.
 *
 * For in-depth docs, see pic_marg_set_left().
 *
 * \author David Poole
 * \date 23-Jan-2008
 *
 */

void pic_marg_set_right(struct pic_handle_t *pic_handle, uint32_t right, uint8_t lrmargin_block)
{
    uint16_t l0, l1, l2, rmargin;
    
    XASSERT(right >=4 && right <= 65535, right);

    pic_lrmargin_get_margin(pic_handle, lrmargin_block, &rmargin, &l0, 0);
    pic_lrmargin_get_margin(pic_handle, lrmargin_block, &rmargin, &l1, 1);
    pic_lrmargin_get_margin(pic_handle, lrmargin_block, &rmargin, &l2, 2);
    
    pic_lrmargin_set_margin(pic_handle, lrmargin_block, right, l0, 0);  // color 0
    pic_lrmargin_set_margin(pic_handle, lrmargin_block, right, l1, 1);  // color 1
    pic_lrmargin_set_margin(pic_handle, lrmargin_block, right, l2, 2);  // color 2
}

